from __future__ import print_function
from __future__ import absolute_import

import os
import sys
import copy
import signal
import logging
# FIXME: temporary turn off container delta limit [SANDBOX-5069]
# import platform
import subprocess as sp

from kernel.util import console

from sandbox.common import os as common_os
from sandbox.common import fs as common_fs
from sandbox.common import log as common_log
from sandbox.common import atfork as common_atfork
from sandbox.common import config as common_config
from sandbox.common import format as common_format
from sandbox.common import threading as common_threading
from sandbox.common import statistics as common_statistics

from sandbox.common.atfork import stdlib_fixer

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

import sandbox.client as sc
import sandbox.agentr.utils
import sandbox.agentr.client


logger = logging.getLogger("client")


def prepare_main_process():
    sc.system.setup_users()

    # fix possible deadlocks while using os.fork with threads
    common_atfork.monkeypatch_os_fork_functions()
    stdlib_fixer.fix_logging_module()

    console.setProcTitle(sc.system.PROCESS_TITLE)

    # FIXME: temporary turn off container delta limit [SANDBOX-5069]
    # sandbox.agentr.utils.ProjectQuota.check_support()
    #
    # if os.uname()[0] == "Linux" and "Microsoft" not in platform.platform():
    #     # drop CAP_SYS_RESOURCE to apply disk quotas for privileged user
    #     cap = common.os.Capabilities.get()
    #     cap.unset_effective(cap.Bits.CAP_SYS_RESOURCE)
    #     cap.unset_permitted(cap.Bits.CAP_SYS_RESOURCE)
    #     common.os.Capabilities.set(cap)
    #     common.os.Capabilities.drop_bound(cap.Bits.CAP_SYS_RESOURCE)


def check_client_stop():
    if os.path.exists(os.path.join(common_config.Registry().client.dirs.run, "client_check_stop")):
        print("Service was disabled", file=sys.stderr)
        logger.exception("Service was disabled")
        sys.exit(2)


def initialize():
    # setup logging
    log_path = os.path.join(common_config.Registry().client.log.root, common_config.Registry().client.log.name)
    with sc.system.UserPrivileges():
        if os.path.exists(log_path):
            os.chown(log_path, sc.system.SERVICE_USER.uid, -1)
    common_statistics.Signaler(common_statistics.ClientSignalHandler(), component=ctm.Component.CLIENT)
    common_log.setup_log(
        log_path, common_config.Registry().client.log.level,
        format="%(asctime)s %(levelname)-7s | %(module)20s:%(lineno)-4s | %(message)s"
    )
    logger.info(
        "Initializing %s root privileges. PID is %d", "WITH" if common_os.User.has_root else "without", os.getpid()
    )

    # ensure working dirs
    common_fs.make_folder(common_config.Registry().client.tasks.data_dir)
    common_fs.make_folder(common_config.Registry().client.dirs.run)
    common_fs.make_folder(common_config.Registry().client.log.root)
    common_fs.make_folder(common_config.Registry().client.executor.dirs.run)
    common_fs.make_folder(common_config.Registry().client.executor.log.root)

    # set signals
    signal.signal(signal.SIGINT, lambda *_: sc.pinger.PingSandboxServerThread().stop(True))
    signal.signal(signal.SIGTERM, lambda *_: sc.pinger.PingSandboxServerThread().stop(True))
    signal.signal(signal.SIGUSR2, lambda *_: common_threading.dump_threads(logger))

    if common_config.Registry().this.system.family == ctm.OSFamily.LINUX and common_os.User.has_root:
        sc.network.prepare_resolv_conf()

    # Set up CPU shares for MULTISLOT clients. Assuming client has N CPU cores,
    # it has N-1 job slots. Another core is used exclusively by service processes.
    # In order to give each slot equal CPU shares, the root /lxc group is set up to
    # have (N-1)*1024 shares while service processes get the default value of 1024 shares.
    if common_config.Registry().client.max_job_slots > 1 and ctc.Tag.LXC in common_config.Registry().client.tags:
        with sc.system.UserPrivileges():
            lxc_root = common_os.CGroup("/lxc")
            lxc_root.cpu["shares"] = 1024 * common_config.Registry().client.max_job_slots

    if sc.system.local_mode():
        # Point user base directory to the non-default path.
        local_path = os.environ["PYTHONUSERBASE"] = os.path.expanduser("~/.sandbox/local")
        sp_path = os.path.join(local_path, "lib", "python2.7", "site-packages")
        if not os.path.exists(sp_path):
            os.makedirs(sp_path, 0o755)
        os.environ["PATH"] = ":".join((os.path.join(local_path, "bin"), os.environ["PATH"]))
        sc.system.PROCMAN_TAG = "{}_{}".format(sc.system.PROCMAN_TAG, common_config.Registry().client.port)
        os.environ["EXECUTABLE"] = sys.executable
        os.environ[common_config.Registry.CONFIG_ENV_VAR] = str(common_config.Registry().custom)
        os.environ["SANDBOX_DIR"] = os.path.realpath(
            os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))
        )

    # Runtime-specific initialization (network, environment, etc)
    if common_config.Registry().client.porto.enabled:
        sc.platforms.porto.PortoNetwork.setup_porto_network()
        sc.platforms.porto.PortoContainerRegistry().initialize()

    if (
        common_config.Registry().client.lxc.enabled and
        common_config.Registry().client.lxc.network.type == ctm.Network.Type.NAT
    ):
        sc.platforms.lxc.LXCNetwork.setup_nat()


def log_system_info(start_type="binary"):
    logger.info("Start Sandbox client %s", start_type)
    logger.info("Current system limits:\n%s", sp.check_output("ulimit -a", shell=True))
    logger.info("\n".join(["Current environment:"] + ["{}: {}".format(k, os.environ[k]) for k in sorted(os.environ)]))


def get_logged_jobs(jobs):
    result = []
    for id, data in jobs.iteritems():
        logged_id = common_format.obfuscate_token(id)
        logged_data = copy.deepcopy(data)
        try:
            logged_data["__state"]["token"] = common_format.obfuscate_token(logged_data["__state"]["token"])
        except Exception:
            pass
        result.append({logged_id: logged_data})
    return result


def start(workers=None):
    pt = sc.pinger.PingSandboxServerThread()
    service = sandbox.agentr.client.Service(logger)
    with pt.lock:
        jobs = service.jobs()
        logger.debug("Got jobs from AgentR: %s", get_logged_jobs(jobs))

        stopped = service.stopped
        if stopped:
            service.stopped = False

        # getajob token might fail to reset after its job has started
        getajob_token = service.getajob_token
        if getajob_token in jobs:
            service.getajob_token = None

        pt.load_state(jobs)

        logger.info(
            "Client state restored from AgentR. Jobs restored: %r, available slots: %r",
            len(sc.commands.Command.registry), pt._free_slots,
        )
        pt.start().run()
        if workers:
            workers(logger).stop()
