import os
import stat
import time
import logging

import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
import sandbox.agentr.errors as ar_errors

from sandbox.common import fs as common_fs
from sandbox.common import os as common_os
from sandbox.common import rest as common_rest
from sandbox.common import config as common_config
from sandbox.common import patterns as common_patterns
from sandbox.common import itertools as common_itertools

from sandbox.client import platforms, system

from . import base


logger = logging.getLogger(__name__)


class TaskCommand(base.Command):
    """ Task processing command """

    def __init__(self, *_):
        super(TaskCommand, self).__init__()
        self.task_id = self.args.get("task_id")
        self.iteration = self.args.get("iteration", 0)
        self.tasks_rid = self.args.get("tasks_rid")
        self.platform = platforms.Platform(self)
        self.task_id = self.args["task_id"]
        self.args["iteration"] = self.iteration

    @common_patterns.singleton_classproperty
    def ssh_private_key(cls):
        return common_fs.read_settings_value_from_file(common_config.Registry().client.sdk.svn.arcadia.key, True)

    @common_patterns.singleton_classproperty
    def common_arc_token(self):
        return common_fs.read_settings_value_from_file(common_config.Registry().client.sdk.arc.token, True)

    @common_patterns.singleton_classproperty
    def docker_registry_token(self):
        return common_fs.read_settings_value_from_file(
            common_config.Registry().client.auth.get("docker_registry_token"),
            True
        )

    def __repr__(self):
        return "<{} of #{} :: {}>".format(type(self).__name__, self.task_id, self.platform and self.platform.name)

    def get_status(self):
        result = super(TaskCommand, self).get_status()
        result["task_id"] = self.task_id
        return result

    @property
    def exec_type(self):
        return self.args.get("exec_type", ctt.ImageType.REGULAR_ARCHIVE)

    def spawn(self):
        self.platform.prepare(None if system.local_mode() else self.tasks_rid)
        super(TaskCommand, self).spawn()

    def update(self, options):
        client = common_rest.Client(auth=self.token, component=ctm.Component.CLIENT)
        try:
            client.task.current.commit()
        except client.errors.SessionExpired:
            pass


class TerminateTaskCommand(TaskCommand):
    """ Terminate command """
    command = ctc.Command.TERMINATE

    def __init__(self, *_):
        self.args.pop("command_id", None)
        timestamp_start = self.args.pop("timestamp_start", time.time())
        super(TerminateTaskCommand, self).__init__()
        if self.killed_by_timeout:
            self.kill_timeout = common_config.Registry().common.task.execution.terminate_timeout
            self.killed_by_timeout = False
        else:
            self.timestamp_start = timestamp_start

    def spawn(self):
        # Skip platform prepare and packages actualization.
        super(TaskCommand, self).spawn()

    def mark_as_ready(self):
        if self.status in ctt.Status.Group.SUCCEED:
            return True
        elif self.status in common_itertools.chain(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH):
            return False
        elif self.status in ctt.Status.Group.WAIT:
            return None
        return True

    def on_terminate(self):
        try:
            self.agentr.finished(drop_session=False, mark_as_ready=self.mark_as_ready())
        except ar_errors.ARException:
            self.logger.exception("Error on communicating with AgentR.")
        if self.killed_by_timeout:
            # Use admin API client instance because session token can be expired already
            self.audit(ctt.Status.TIMEOUT, "Timed out on postprocessing")


class ExecuteTaskCommand(TaskCommand):
    """ Execute task command """
    command = ctc.Command.EXECUTE

    def spawn(self):
        # supply args for terminate stage only for regular EXECUTE commands
        if self.command == ExecuteTaskCommand.command:
            terminator = self.terminator
            self.args["subsequent"] = {
                "command": terminator.command,
                "args": terminator.args
            }

        super(ExecuteTaskCommand, self).spawn()
        self.agentr.monitoring_setup()

        with system.UserPrivileges():
            # attempt to make system log readable by all
            syslog_path = common_os.system_log_path()
            if syslog_path:
                try:
                    os.chmod(syslog_path, os.stat(syslog_path).st_mode | stat.S_IROTH)
                except OSError as ex:
                    self.logger.warning("Failed to give read permissions to %s: %s", syslog_path, ex)

    @property
    def terminator(self):
        args = {
            "id": self.token,
            "task_id": self.task_id,
            "tasks_rid": self.tasks_rid,
            "status": self.status,
            "exec_type": self.exec_type,
            "ctx": self.executor_ctx,
            "iteration": self.iteration,
            "timestamp_start": self.timestamp_start,
            "kill_timeout": self.kill_timeout,
            "killed_by_timeout": self.killed_by_timeout,
            "arch": self.arch,
            "command_id": self.id,
            "ramdrive": self.args.get("ramdrive"),
            "vault_key": self.args.get("vault_key"),
            "dns": self.args.get("dns"),
        }
        if self.status == ctt.Status.STOPPED:
            # Canceled job will be terminated also here (on the same Client).
            return base.Command(StopTaskCommand.command, args)

        return base.Command(TerminateTaskCommand.command, args)

    def on_terminate(self):
        term_cmd = self.terminator
        term_cmd.agentr = self.agentr

        if self.executor_ctx.get("ran_subsequent") and not self.killed_by_timeout:
            term_cmd.status = self.status
            return term_cmd.on_terminate()

        term_cmd.platform = self.platform
        term_cmd.platform._cmd = term_cmd
        return term_cmd

    def update(self, options):
        new_kill_timeout = options.get("kill_timeout") if options else None
        if new_kill_timeout and not self.timeout_extension and self.kill_timeout < new_kill_timeout:
            self.logger.info("Updating kill timeout from %s to %s", self.kill_timeout, new_kill_timeout)
            self.kill_timeout = new_kill_timeout
            self.args["expiration_time"] = (self.timestamp_start or time.time()) + self.kill_timeout
            self.save_state()

        super(ExecuteTaskCommand, self).update(options)


class ExecutePrivilegedTaskCommand(ExecuteTaskCommand):
    """ Execute task command with privileged LXC """
    command = ctc.Command.EXECUTE_PRIVILEGED

    def on_terminate(self):
        self.platform.terminate()
        term_cmd = self.terminator
        term_cmd.agentr = self.agentr
        return term_cmd


class ReleaseTaskCommand(TaskCommand):
    """ Release task """
    command = ctc.Command.RELEASE

    def cancel(self, status=None, reason=None):
        if reason is None:
            if self.killed_by_timeout:
                reason = "Timed out on releasing"
            else:
                reason = "Release cancelled"

        # Use session token because this should only be done if session is still active.
        # If session is already expired, status change will be done by server.
        try:
            self.audit(ctt.Status.NOT_RELEASED, reason, force=False, expected_status=ctt.Status.RELEASING)
        except common_rest.Client.HTTPError:
            logger.exception("Error on setting NOT_RELEASED status")

        super(ReleaseTaskCommand, self).cancel(status=status, reason=reason)


class StopTaskCommand(TaskCommand):
    """ Stop task """
    command = ctc.Command.STOP
