import os
import shutil
import logging

import subprocess as sp

from . import system


logger = logging.getLogger(__name__)


def ensure_ip6tables_rule(rule):
    with open(os.devnull, "w") as devnull:
        no_rule = sp.call(["ip6tables", "-t", "nat", "-C"] + rule, stderr=devnull)

    if no_rule:
        logger.info("Default NAT ip6tables rule not found, add one: '%s'", " ".join(rule))
        sp.check_call(["ip6tables", "-t", "nat", "-A"] + rule)


def ensure_sysctl_forwarding():
    sysctl_vars = [
        ("net.ipv6.conf.all.forwarding", 1),
        ("net.ipv6.conf.all.proxy_ndp", 1),
        ("net.ipv6.conf.all.accept_ra", 2),
    ]
    for k, v in sysctl_vars:
        if sp.check_output(["sysctl", "-nb", k]) == str(v):
            continue
        logger.info("Set sysctl %s = %s", k, v)
        sp.check_call(["sysctl", "-q", "{}={}".format(k, v)])


def switch_off_resolv_conf_updates():
    rc_conf_path = "/etc/rc.conf.local"
    mandatory_lines = ['fix_resolvconf="no"\n', 'fix_hosts="no"\n']
    try:
        if os.path.isfile(rc_conf_path):
            with open(rc_conf_path, "r") as rc_conf_file:
                rc_conf_lines = rc_conf_file.readlines()
            add_lines = [line for line in mandatory_lines if line not in rc_conf_lines]
            if add_lines:
                rc_conf_lines.extend(add_lines)
            else:
                rc_conf_lines = None
        else:
            rc_conf_lines = mandatory_lines
        if rc_conf_lines:
            with open(rc_conf_path, "w") as rc_conf_file:
                rc_conf_file.writelines(rc_conf_lines)
    except Exception as ex:
        logger.exception("Failed to update %r: %r", rc_conf_path, ex)


def prepare_resolv_conf():
    from sandbox.client.platforms.base import ResolvConfMixin
    resolv_conf_path = ResolvConfMixin.resolv_conf_path
    if not os.path.exists(resolv_conf_path):
        return
    with system.UserPrivileges():
        switch_off_resolv_conf_updates()

        try:
            with open(os.devnull, "w") as devnull:
                sp.check_call(["chattr", "-i", resolv_conf_path], stderr=devnull)
        except sp.CalledProcessError:
            pass

        try:
            if not os.path.exists(resolv_conf_path + ".orig"):
                shutil.copy(resolv_conf_path, resolv_conf_path + ".orig")
        except Exception as er:
            logger.exception("Fail to save original file %r: %r", resolv_conf_path, er)
