import os
import re
import random
import shutil
import socket
import struct
import logging
import tarfile
import textwrap
import itertools as it
import threading
import contextlib
import collections

import yaml
import netaddr
import subprocess32 as sp

from six.moves import queue

import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt

from sandbox.common import os as common_os
from sandbox.common import fs as common_fs
from sandbox.common import hash as common_hash
from sandbox.common import config as common_config
from sandbox.common import format as common_format
from sandbox.common import context as common_context
from sandbox.common import patterns as common_patterns
from sandbox.common import platform as common_platform
from sandbox.common import itertools as common_itertools

from sandbox.client import base, errors, network, system

from sandbox.agentr import utils as agentr_utils

from . import linux


logger = logging.getLogger(__name__)


class Container(common_patterns.Abstract, base.Serializable):
    """ Container assigned information - template ID, instance ID, name. """
    __slots__ = ("template", "instance", "no", "ip", "hostname", "resource_id")
    __defs__ = [None] * 6

    SUFFIX = collections.namedtuple("Suffix", ("workdir", "delta"))("workdir", "delta")
    rc_local_flag = "/etc/rc_local_flag"

    def __init__(self, template=None, instance=None, no=None, ip=None, hostname=None, resource_id=None, name=None):
        """
        Container can be initialized by its name, template or resource_id.

        If name argument is supplied, container's name is parsed into components.
        ValueError is raised if the name is invalid.

        If template is supplied, resource_id is deduced from the template name, unless for privileged containers.

        If resource_id is supplied, template name is constructed as <client_port>.<resource_id> for local sandbox
        and simply <resource_id> for production sandbox.
        """
        if name is not None:
            port, template, instance = self.parse_name(name)
            if system.local_mode() and port is not None and port != common_config.Registry().client.port:
                raise ValueError(
                    "Container has port {}, expected {}".format(port, common_config.Registry().client.port)
                )

        if resource_id is not None and template is None:
            template = (
                "{}.{}".format(common_config.Registry().client.port, resource_id)
                if system.SERVICE_USER == system.UNPRIVILEGED_USER
                else str(resource_id)
            )

        if resource_id is None and template and "privileged" not in template:
            resource_id = int(template.split(".")[-1])

        super(Container, self).__init__(template, instance, no, ip, hostname, resource_id)

    def __setstate__(self, state):
        self.__init__(*state[:len(self.__slots__)])

    def __getstate__(self):
        return list(self.itervalues())

    def __repr__(self):
        return self.name

    @common_patterns.classproperty
    def settings(self):
        return common_config.Registry().client.lxc

    @property
    def fstab(self):
        return self.settings.fstab_template.format(self.name)

    @property
    def config(self):
        return self.settings.config_template.format(self.name)

    @property
    def name(self):
        if self.instance is None:
            return self.template
        else:
            return ".".join((self.template, str(self.instance)))

    @staticmethod
    def parse_name(name):
        """container name => (port, template, instance)"""
        split = name.split(".")
        if split[-1] == "privileged":
            split += ["0"]
        if len(split) == 3:
            return (int(split[0]), "{}.{}".format(split[0], split[1]), split[2])
        elif len(split) == 2:
            return (None, split[0], split[1])
        else:
            raise ValueError("Container name must consist of 2 or 3 components)")

    @property
    def basedir(self):
        return os.path.join(self.settings.rootfs.basedir, self.name)

    def path(self, *args):
        return os.path.join(self.basedir, self.settings.rootfs.base, *(_.lstrip("/") for _ in args))

    @property
    def root(self):
        return self.path()

    @property
    def template_root(self):
        return os.path.join(self.settings.rootfs.template, self.template, self.settings.rootfs.base)

    @property
    def workdir(self):
        return os.path.join(self.basedir, self.SUFFIX.workdir)

    @property
    def deltadir(self):
        return os.path.join(self.basedir, self.SUFFIX.delta)

    @property
    def delta_usage(self):
        return agentr_utils.ProjectQuota(self.deltadir, create=False).usage

    @property
    def delta_limit(self):
        return agentr_utils.ProjectQuota(self.deltadir, create=False).limit

    @delta_limit.setter
    def delta_limit(self, limit):
        logger.debug(
            "Set limit on delta directory %s to %s",
            self.deltadir, common_format.size2str(limit)
        )
        quota = agentr_utils.ProjectQuota(self.deltadir)
        quota.limit = limit
        agentr_utils.ProjectQuota(os.path.join(self.workdir, "work"), create=False).project = quota.project

    @delta_limit.deleter
    def delta_limit(self):
        logger.debug("Remove limit from delta directory %s", self.deltadir)
        agentr_utils.ProjectQuota(os.path.join(self.workdir, "work"), create=False).project = 0
        agentr_utils.ProjectQuota(self.deltadir, create=False).destroy(ignore_errors=True)


class PrivilegedContainer(Container):
    """ Privileged container metainformation. """
    __slots__ = Container.__slots__ + ("task_id",)
    __defs__ = Container.__defs__ + [None]

    def __init__(self, base=None, task_id=None):
        super(Container, self).__init__(**dict(base or ()))
        self.task_id = task_id
        self.no = 99

    def __setstate__(self, state):
        base_slots = len(self.__slots__) - 1
        super(Container, self).__init__(*state[:base_slots])
        self.task_id = state[base_slots]

    @common_patterns.classproperty
    def name(cls):
        if system.local_mode():
            return ".".join([str(common_config.Registry().client.port), "privileged"])
        return "privileged"

    @property
    def _deltas_dir(self):
        return os.path.join(self.settings.rootfs.basedir_privileged, str(self.task_id))

    @property
    def workdir(self):
        return os.path.join(self._deltas_dir, self.SUFFIX.workdir)

    @property
    def deltadir(self):
        return os.path.join(self._deltas_dir, self.SUFFIX.delta)


class LXCNetwork(object):
    """
    This class incapsulates (almost) all network-related operations for LXC, such as network setup,
    network interface common.config.Registry() and routing table generation and so on.
    """

    #: Default NAT rule to proxy all the traffic which comes out of containers and replace their IP address
    #: with that of a host machine
    IP6TABLES_RULE = "POSTROUTING -s fc00::/64 ! -d fc00::/64 -j MASQUERADE".split()

    #: Command to allocate a network for containers' needs
    IFACE_ADDRESS_CMD = "/sbin/ip -6 addr add fc00::1/64 dev lxcbr0".split()

    #: Regexp for filtering and removing down interfaces
    DOWN_INTERFACE_RE = re.compile(r"^\d+:\s+(veth\w+):\s+.*state DOWN")

    # `sysctl.conf` IPv6 configuration for the container
    LXC_SYSCTL = textwrap.dedent("""
        # Sandbox-generated options:
        # This host's address statically configured
        net.ipv6.conf.{dev}.accept_ra = 0
        net.ipv6.conf.default.accept_ra = 0
        net.ipv6.conf.all.accept_ra = 0
        # This is host and not router
        net.ipv6.conf.{dev}.router_solicitations = 0
        net.ipv6.conf.default.router_solicitations = 0
        # Accept Router Preference in RA?
        net.ipv6.conf.{dev}.accept_ra_rtr_pref = 0
        net.ipv6.conf.default.accept_ra_rtr_pref = 0
        # Learn Prefix Information in Router Advertisement
        net.ipv6.conf.{dev}.accept_ra_pinfo = 0
        net.ipv6.conf.default.accept_ra_pinfo = 0
        # Setting controls whether the system will accept Hop Limit config.Registry() from a router advertisement
        net.ipv6.conf.{dev}.accept_ra_defrtr = 0
        net.ipv6.conf.default.accept_ra_defrtr = 0
        # Router advertisements can cause the system to assign a global unicast address to an interface
        net.ipv6.conf.{dev}.autoconf = 0
        net.ipv6.conf.default.autoconf = 0
        net.ipv6.conf.all.autoconf = 0
        # how many neighbor solicitations to send out per address?
        net.ipv6.conf.{dev}.dad_transmits = 0
        net.ipv6.conf.default.dad_transmits = 0
    """)

    # Container's `/etc/rc.local` devices setup file. Ensures special devices are created.
    LXC_RC_LOCAL_DEVICES = textwrap.dedent("""
        #!/bin/bash -e

        for i in {{0..511}}; do
            [ ! -b /dev/loop$i ] && mknod -m660 /dev/loop$i b 7 $i || true
        done
        [ ! -b /dev/loop-control ] && mknod -m660 /dev/loop-control c 10 237 || true

        [ ! -e /dev/kvm ] && mknod /dev/kvm c 10 232 || true
        [ -e /dev/kvm ] && chown root:kvm /dev/kvm || true
        [ -e /dev/kvm ] && chmod ug+rw /dev/kvm

        [ -e /dev/fuse ] && chown root:fuse /dev/fuse || true
        [ -e /dev/fuse ] && chmod a+rw /dev/fuse

        {network}

        touch {rc_local_flag}
        exit 0
    """).lstrip()

    # Container's `/etc/rc.local` network setup file. Required for MACVLAN schema
    LXC_RC_LOCAL_NETWORK_MACVLAN = textwrap.dedent("""
        /sbin/ip link set {dev} down || true
        /sbin/sysctl -p || true
        /sbin/ip link set {dev} up || true

        /sbin/ip addr add {ip}/{netmask} dev {dev}
        {routes}
    """).lstrip()

    # Container's `/etc/rc.local` network setup file. Required for NAT schema
    LXC_RC_LOCAL_NETWORK_NAT = textwrap.dedent("""
        /sbin/ip -6 ro add default via fc00::1 dev eth0 metric 1024
    """).lstrip()

    # noinspection PyPep8Naming
    @common_patterns.classproperty
    def LXC_CONFIG_TEMPLATE(cls):
        net_templates = {
            common_platform.UbuntuRelease.XENIAL: {
                ctm.Network.Type.NAT: textwrap.dedent("""
                    lxc.network.type = veth
                    lxc.network.flags = up
                    lxc.network.link = lxcbr0
                    lxc.network.name = {iface_name}
                    lxc.network.mtu = {lxc_mtu}
                    lxc.network.ipv6 = fc00::{no}
                    lxc.network.hwaddr = {lxc_mac}
                """),
                ctm.Network.Type.MACVLAN: textwrap.dedent("""
                    lxc.network.type = macvlan
                    lxc.network.macvlan.mode = bridge
                    lxc.network.flags = up
                    lxc.network.link = {iface_name}
                    lxc.network.name = {iface_name}
                    lxc.network.mtu = {lxc_mtu}
                    lxc.network.hwaddr = {lxc_mac}
                    # Internal network
                    lxc.network.type = veth
                    lxc.network.flags = up
                    lxc.network.link = lxcbr0
                    lxc.network.name = lo1
                    lxc.network.mtu = 8950
                    lxc.network.ipv6 = fc00::{no}
                    lxc.network.hwaddr = e2:e4:e7:{o4:02x}:{o5:02x}:{o6:02x}
                """)
            },
            common_platform.UbuntuRelease.FOCAL: {
                ctm.Network.Type.NAT: textwrap.dedent("""
                    lxc.net.0.type = veth
                    lxc.net.0.flags = up
                    lxc.net.0.link = lxcbr0
                    lxc.net.0.name = {iface_name}
                    lxc.net.0.mtu = {lxc_mtu}
                    lxc.net.0.ipv6.address = fc00::{no}
                    lxc.net.0.hwaddr = {lxc_mac}
                """),
                ctm.Network.Type.MACVLAN: textwrap.dedent("""
                    lxc.net.0.type = macvlan
                    lxc.net.0.macvlan.mode = bridge
                    lxc.net.0.flags = up
                    lxc.net.0.link = {iface_name}
                    lxc.net.0.name = {iface_name}
                    lxc.net.0.mtu = {lxc_mtu}
                    lxc.net.0.hwaddr = {lxc_mac}
                    # Internal network
                    lxc.net.1.type = veth
                    lxc.net.1.flags = up
                    lxc.net.1.link = lxcbr0
                    lxc.net.1.name = lo1
                    lxc.net.1.mtu = 8950
                    lxc.net.1.ipv6.address = fc00::{no}
                    lxc.net.1.hwaddr = e2:e4:e7:{o4:02x}:{o5:02x}:{o6:02x}
                """),
            }
        }
        dist = common_platform.dist_version()
        template = net_templates.get(dist, {}).get(Container.settings.network.type)
        if not template:
            raise errors.CriticalInfraError(
                "No network config for {} (net type: {})".format(dist, Container.settings.network.type)
            )
        return template

    # noinspection PyPep8Naming
    @common_patterns.classproperty
    def CONTAINER_NO_REGEXP(cls):
        expressions = {
            common_platform.UbuntuRelease.XENIAL: {
                ctm.Network.Type.NAT: re.compile(r"lxc\.network\.ipv6 = fc00::(\d+)"),
                ctm.Network.Type.MACVLAN: re.compile(
                    r"(lxc\.network\.ipv4 = 10\.0\.3\.(\d+)|lxc\.network\.ipv6 = fc00::(\d+))"
                )},
            common_platform.UbuntuRelease.FOCAL: {
                ctm.Network.Type.NAT: re.compile(r"lxc\.net\.0\.ipv6\.address = fc00::(\d+)"),
                ctm.Network.Type.MACVLAN: re.compile(
                    r"(lxc\.network\.ipv4 = 10\.0\.3\.(\d+)|lxc\.net\.1\.ipv6\.address = fc00::(\d+))"
                )},
        }

        dist = common_platform.dist_version()
        regex = expressions.get(dist, {}).get(Container.settings.network.type)
        if not regex:
            raise errors.CriticalInfraError(
                "No container_num regex for {} (net type {}).".format(dist, Container.settings.network.type)
            )
        return regex

    @classmethod
    def setup_nat(cls):
        with system.PrivilegedSubprocess("initial network configuration for NAT"):
            network.ensure_ip6tables_rule(cls.IP6TABLES_RULE)
            network.ensure_sysctl_forwarding()
            try:
                output = sp.check_output(cls.IFACE_ADDRESS_CMD, stderr=sp.STDOUT)
                if output:
                    logger.info("Update command output:\n%s", output)
            except sp.CalledProcessError as exc:
                logger.warning("ip addr add failed: %s", exc.output.strip())

    @classmethod
    def setup_devices(cls, container, macvlan=True):
        network_part = ""
        # Resolve main host network device
        if macvlan:
            dev, _, ip = cls.lxc_iface(container.no)
            logger.info("Found ip {} for dev {} in container {}.{}".format(
                ip,
                dev,
                container.resource_id,
                container.no
            ))
            # Disable network autoconfiguration
            with open(container.path("etc", "sysctl.conf"), "a+") as sysctl:
                sysctl.write(cls.LXC_SYSCTL.format(dev=dev))

            # Setup network manually after container is initialized
            routes = "\n".join("/sbin/ip route add {} || true".format(r) for r in cls.lxc_route_table(dev))
            network_part = cls.LXC_RC_LOCAL_NETWORK_MACVLAN.format(dev=dev, ip=ip, netmask=64, routes=routes)
        else:
            network_part = cls.LXC_RC_LOCAL_NETWORK_NAT.format()

        rc_local_path = container.path("etc", "rc.local")
        with open(rc_local_path, "w") as rc_local:
            rc_local.write(
                cls.LXC_RC_LOCAL_DEVICES.format(
                    network=network_part,
                    rc_local_flag=container.rc_local_flag
                )
            )
        os.chmod(rc_local_path, 0o755)

    @classmethod
    def setup_hostname_and_hosts(cls, container):
        with open(container.path("etc", "hostname"), "w") as outfh:
            outfh.write(container.hostname + "\n")

        with open("/etc/hosts", "r") as infh, open(container.path("etc", "hosts"), "a") as outfh:
            outfh.write("# Generated by Sandbox client\n")
            outfh.write("# The following block copied from the host\n")
            for line in infh:
                parts = line.split()
                if not parts or "." not in parts[0] or "localhost" in line:
                    outfh.write(line)
            outfh.write("# The following block is individual for the container\n")
            outfh.write("{lxc_ip}\t{hostname}\n".format(lxc_ip=container.ip, hostname=container.hostname))
            if Container.settings.network.type == ctm.Network.Type.MACVLAN:
                outfh.write("fc00::1\t{hostname}\n".format(hostname=common_config.Registry().this.fqdn))

    @classmethod
    def lxc_iface(cls, container_no):
        ipaddrs = sp.check_output("ip -o -6 addr sh scope global".split()).splitlines()
        for iface in ipaddrs:
            iface = iface.split()
            if "lxc" not in iface[1] and "vlan" not in iface[1] and not iface[3].startswith("f"):
                dev = iface[1]
                addr = netaddr.IPNetwork(iface[3])
                logger.debug("Detected backbone IP address '%s' on dev '%s'", addr, dev)

                # MAC generation:
                # * convert last 4 bytes to 2 bytes via crc16
                # * use container number as last byte
                data = common_hash.crc16(bytearray(struct.pack("!HH", *addr.ip.words[-2:])))
                hi = data >> 8
                lo = data & 0x00FF
                mac_as_bytes = [0x02, 0xDD, 0xDD, hi, lo, container_no]
                mac = ":".join("{:02x}".format(byte) for byte in mac_as_bytes)

                # IP generation:
                # * convert last 4 bytes to 3 bytes via crc24
                # * use container number as last byte
                data = common_hash.crc24(bytearray(struct.pack("!HH", *addr.ip.words[-2:])))
                hi = data >> 8
                lo = ((data & 0x0000FF) << 8) + container_no
                ip_as_number = netaddr.strategy.ipv6.words_to_int(addr.ip.words[:-2] + (hi, lo))
                ip = str(netaddr.IPAddress(ip_as_number))

                return dev, mac, ip

        logger.error("Cannot determine backbone interface for new LXC container.")
        raise errors.ExecutorFailed

    @classmethod
    def lxc_route_table(cls, dev, global_only=True):
        """
        Return string with dev interface dst routes sutable for "ip route add command"
        Works only in binary mode due to arcadia-dependency
        """
        import pyroute2
        with pyroute2.IPRoute() as ipr:
            dev_num = ipr.link_lookup(ifname=dev)[-1]
            routes = ipr.get_routes(family=socket.AF_INET6, oif=dev_num)
        for route in routes:
            gw = route.get_attr("RTA_GATEWAY")
            dst = route.get_attr("RTA_DST")
            dst_net = route.get("dst_len")
            metric = route.get_attr("RTA_PRIORITY")
            mtu, admss = None, None
            if route.get_attr("RTA_METRICS"):
                mtu = route.get_attr("RTA_METRICS").get_attr("RTAX_MTU")
                admss = route.get_attr("RTA_METRICS").get_attr("RTAX_ADVMSS")

            # Drop all routes w/o dst and dst/128 routes w/o gw (link-only route), also multicast routes (ff00)
            if global_only and (
                not dst or not dst_net or (dst_net == 128 and gw is None) or dst.startswith("ff00")
            ):
                continue

            route = [
                "{}/{}".format(dst, dst_net),
                "via {}".format(gw) if gw else "",
                "metric {}".format(metric) if metric else "",
                "mtu {}".format(mtu) if mtu else "",
                "advmss {}".format(admss) if admss else "",
                "dev {}".format(dev)
            ]
            yield " ".join(" ".join(route).split())
        yield "default via fe80::1 metric 1024 dev {}".format(dev)

    @common_patterns.classproperty
    def lxc_mtu(cls):
        s = socket.socket(socket.AF_INET6, socket.SOCK_DGRAM)
        s.connect(("sandbox.yandex-team.ru", 80))
        s.setsockopt(socket.IPPROTO_IP, 10, 2)
        mtu = 65536  # https://st.yandex-team.ru/SANDBOX-3119#1463166902000
        try:
            s.send("#" * 512000)
        except socket.error:
            option = 14
            mtu = s.getsockopt(socket.IPPROTO_IP, option)
        return mtu

    @classmethod
    def clear_down_interfaces(cls):
        logger.debug("Checking for down virtual LXC bridge interfaces")
        with system.UserPrivileges():
            ifaces = set()
            for l in sp.check_output(["/sbin/ip", "li"]).splitlines():
                m = cls.DOWN_INTERFACE_RE.match(l)
                if m:
                    ifaces.add(m.group(1))
            with open(os.devnull, "wb") as _devnull:
                for iface in ifaces:
                    logger.info("Dropping virtual LXC bridge interface %r", iface)
                    sp.call(["/sbin/ip", "li", "del", iface], stdout=_devnull, stderr=_devnull)


class ContainerCorrupted(errors.ExecutorFailed):
    """ Signals that a container is corrupted and must be destroyed """


class LXCPlatform(linux.LinuxPlatform):
    SERIALIZABLE_ATTRS = linux.LinuxPlatform.SERIALIZABLE_ATTRS + ("_container", "cgroup_prefix", "_venv_rid")
    # Container's status - name and status string ('running', 'stopped', etc)
    ContainerStatus = collections.namedtuple("ContainerStatus", ("name", "status"))

    # Invalid cgroup name regexp
    INVALID_CGROUP_NAME = re.compile(r"\A(/lxc/)(([\d]+).([\d]+)|privileged)-(\d)\Z")

    # Maximum amount of host's containers
    MAX_CONTAINERS = 5

    # how long to wait until container is validated (seconds)
    CONTAINER_VALIDATE_TIMEOUT = 5

    # limit for container's delta directory size in bytes
    # FIXME: temporary increase limit to 30GiB due to problems with task BUILD_DISK_DOCKER_IMAGE
    CONTAINER_DELTA_HARD_LIMIT = 30 << 30  # 30 GiB
    # max delta size before destroy container
    CONTAINER_DELTA_SOFT_LIMIT = 5 << 30  # 5 GiB

    # mount arc with sandbox code to lxc
    LXC_ARC_MOUNT = "lxc.mount.entry = {arc_repo} {rootfs}{arc_repo} none defaults,ro,bind 0 0"

    # Additional options to LXC container's configuration related to Linux capabilities
    LXC_CONFIG_CAPABILITIES = textwrap.dedent("""
        # Drop CAP_SYS_RESOURCE to allow disk quotas for root
        lxc.cap.drop = sys_resource
    """)

    # LXC container's "fstab" file template. Placeholders:
    #   "rootfs"
    #   "datadir", "datadir_mount_mode", "logdir", "rundir", "skydir"
    LXC_FSTAB_BASE_TEMPLATE = textwrap.dedent("""
        proc                        {rootfs}/proc       proc                nodev,noexec,nosuid 0 0
        sysfs                       {rootfs}/sys        sysfs               defaults  0 0
        /lib/modules                {rootfs}/lib/modules none               defaults,ro,bind 0 0
        {datadir}                   {rootfs}{datadir}   none                defaults,{datadir_mount_mode},bind 0 0
        {base_cache}/arc_vcs        {rootfs}{base_cache}/arc_vcs {arc_vcs_fs} {arc_vcs_opts} 0 0
        {logdir}                    {rootfs}{logdir}    none                defaults,rw,bind 0 0
        {rundir}                    {rootfs}{rundir}    none                defaults,rw,bind 0 0
        {skydir}                    {rootfs}{skydir}    none                defaults,ro,bind 0 0

    """)
    # Additional section for "fstab" file of regular container. Placeholders: "rootfs"
    LXC_FSTAB_REGULAR_TEMPLATE = textwrap.dedent("""
        /place/vartmp               {rootfs}/place/vartmp none              defaults,rw,bind 0 0
        /place/coredumps            {rootfs}/place/coredumps none           defaults,rw,bind 0 0
    """)
    # Additional section for "fstab" file for service user's home directory. Placeholders: "rootfs", "home"
    LXC_FSTAB_SU_HOME = textwrap.dedent("""
        {home}                      {rootfs}{home}      none                defaults,rw,bind 0 0
    """)
    # Additional section for "fstab" file for host tmp directory. Placeholders: "rootfs", "tmp"
    LXC_FSTAB_TMP_DIR = textwrap.dedent("""
        {tmp}                       {rootfs}{tmp}       none                defaults,create=dir,rw,bind 0 0
    """)
    # Additional section for "fstab" file of privileged container. Placeholders: "rootfs", "workdir"
    LXC_FSTAB_PRIVILEGED_TEMPLATE = textwrap.dedent("""
        {workdir}                   {rootfs}{workdir}   none                defaults,rw,bind 1 0
    """)

    # System files to copy from host to container
    SYSTEM_FILES = ["/etc/security/limits.d/sandbox.conf"]

    # System systemd services to disable
    UNWANTED_SYSTEMD_SERVICES = ("udevd", "udev-trigger")

    # List of running container names as keys and platform as values.
    running = {}
    # List of per-template instances with busy information per instance, i.e., the structure is the following:
    # `{'<TEMPLATE_ID>': [<TASK_ID> or 0, ...], ...}`
    instances = {}
    _container = None
    cgroup_prefix = None
    # Container operation process-level lock
    _lock = threading.RLock()
    # lock for individual container templates
    _template_locks = collections.defaultdict(threading.RLock)
    # queue for lxc containers to be destroyed
    _destroy_queue = queue.Queue()

    def __init__(self, cmd):
        super(LXCPlatform, self).__init__(cmd)
        container = self._cmd.args.pop("container")
        self._container = Container(resource_id=container["id"])
        self._venv_rid = container.get("venv_rid")
        self._cmd.arch = container["alias"]

    @common_patterns.classproperty
    def LXC_CONFIG_TEMPLATE(cls):
        # LXC container's configuration file template.
        # Placeholders: "hostname", "no", "rootfs", "fstab_path"
        templates = {
            common_platform.UbuntuRelease.XENIAL: textwrap.dedent("""
                lxc.utsname = {hostname}
                lxc.tty = 4
                lxc.pts = 1024
                lxc.rootfs = {rootfs}
                lxc.mount  = {fstab_path}
                {arc_mount}
                lxc.cgroup.devices.deny = a
                lxc.cgroup.devices.allow = a *:* m
                # /dev/null and zero
                lxc.cgroup.devices.allow = c 1:3 rwm
                lxc.cgroup.devices.allow = c 1:5 rwm
                # consoles
                lxc.cgroup.devices.allow = c 5:1 rwm
                lxc.cgroup.devices.allow = c 5:0 rwm
                lxc.cgroup.devices.allow = c 4:0 rwm
                lxc.cgroup.devices.allow = c 4:1 rwm
                # /dev/{{,u}}random
                lxc.cgroup.devices.allow = c 1:9 rwm
                lxc.cgroup.devices.allow = c 1:8 rwm
                lxc.cgroup.devices.allow = c 136:* rwm
                lxc.cgroup.devices.allow = c 5:2 rwm
                # rtc
                lxc.cgroup.devices.allow = c 254:0 rwm
                # loop devices
                lxc.cgroup.devices.allow = b 7:* rwm
                lxc.cgroup.devices.allow = b 8:* rwm
                lxc.cgroup.devices.allow = c 10:237 rwm
                # tun, kvm, fuse
                lxc.cgroup.devices.allow = c 10:200 rwm
                lxc.cgroup.devices.allow = c 10:232 rwm
                lxc.cgroup.devices.allow = c 10:229 rwm
                # nbd
                lxc.cgroup.devices.allow = b 43:* rw
                # Set low blkio weight for all user processess
                lxc.cgroup.blkio.weight = 100
            """),
            common_platform.UbuntuRelease.FOCAL: textwrap.dedent("""
                lxc.uts.name = {hostname}
                lxc.tty.max = 4
                lxc.pty.max = 1024
                lxc.rootfs.path = {rootfs}
                lxc.mount.fstab  = {fstab_path}
                {arc_mount}

                # Set low blkio weight for all container processes
                limits.disk.priority = 1
                # Limit nofiles in container
                limits.kernel.nofile = 1048576
                lxc.prlimit.nofile = 1048576
                lxc.sysctl.fs.file-max = 1048576

                lxc.cgroup.devices.deny = a
                lxc.cgroup.devices.allow = a *:* m
                # /dev/null and zero
                lxc.cgroup.devices.allow = c 1:3 rwm
                lxc.cgroup.devices.allow = c 1:5 rwm
                # consoles
                lxc.cgroup.devices.allow = c 5:1 rwm
                lxc.cgroup.devices.allow = c 5:0 rwm
                lxc.cgroup.devices.allow = c 4:0 rwm
                lxc.cgroup.devices.allow = c 4:1 rwm
                # /dev/{{,u}}random
                lxc.cgroup.devices.allow = c 1:9 rwm
                lxc.cgroup.devices.allow = c 1:8 rwm
                lxc.cgroup.devices.allow = c 136:* rwm
                lxc.cgroup.devices.allow = c 5:2 rwm
                # rtc
                lxc.cgroup.devices.allow = c 254:0 rwm
                # loop devices
                lxc.cgroup.devices.allow = b 7:* rwm
                lxc.cgroup.devices.allow = b 8:* rwm
                lxc.cgroup.devices.allow = c 10:237 rwm
                # tun, kvm, fuse
                lxc.cgroup.devices.allow = c 10:200 rwm
                lxc.cgroup.devices.allow = c 10:232 rwm
                lxc.cgroup.devices.allow = c 10:229 rwm
                # nbd
                lxc.cgroup.devices.allow = b 43:* rw
            """)
        }
        template = templates.get(common_platform.dist_version())
        if not template:
            raise errors.CriticalInfraError("No lxc config for ubuntu {} found".format(common_platform.dist_version()))
        return template

    @property
    @contextlib.contextmanager
    def lock(self):
        with common_context.Timer() as t:
            self.logger.info("Acquiring container operation lock.")
            with t["acquire"]:
                self._lock.acquire()
            with t["hold"]:
                try:
                    yield self
                finally:
                    self._lock.release()
                    self.logger.info("Releasing container operation lock. Timings: %s", t)

    @property
    def name(self):
        return " : ".join((super(LXCPlatform, self).name, self._alias))

    @classmethod
    def _logged_command_check_critical_errors(cls, errs):
        # TODO: remove after SANDBOX-5901
        lxc_monitor_open_error_msg = "lxc_monitor_open: 237 Failed to connect to monitor socket"
        if lxc_monitor_open_error_msg in errs:
            raise errors.CriticalInfraError("lxc_monitor_open error")
        unknown_squashfs_msg = "unknown filesystem type 'squashfs'"
        if unknown_squashfs_msg in errs:
            raise errors.CriticalInfraError(unknown_squashfs_msg)

    @classmethod
    def _lxc_execute(cls, name, cmd, **popen_args):
        with system.UserPrivileges():
            return sp.Popen(
                [
                    "lxc-attach", "-n", name,
                    "--"
                ] + cmd,
                **popen_args
            )

    @classmethod
    def _list_containers(cls, filter_status=None):
        """ Generates a list of tuples(name, status) of available containers. """
        try:
            with system.UserPrivileges():
                with open(os.devnull, "w") as devnull:
                    lxc_ls = sp.check_output(["/usr/bin/lxc-ls", "--fancy"], stderr=devnull)
                    lxc_ls = [
                        map(str.lower, line.split(None, 2)[:2])
                        for line in lxc_ls.split("\n")[1:]
                        if line and not line.startswith("---")
                    ]
        except (sp.CalledProcessError, OSError):
            lxc_ls = []

        for name, status in lxc_ls:
            if (
                name and ".yandex." not in name and
                status in ("running", "stopped") and
                not (filter_status or status == filter_status)
            ):
                yield cls.ContainerStatus(name, status)

    def cleanup_home(self):
        if system.local_mode():
            return
        home_dir = self._container.path(self._container.settings.dirs.root.lstrip(os.path.sep))
        with system.PrivilegedSubprocess(("Cleaning home for container", self._container.name)):
            self._remove_path(home_dir)
            self._restore_home(home_dir, job_logger=self.logger)

    def cleanup(self):
        super(LXCPlatform, self).cleanup()

        # FIXME: temporary turn off container delta limit [SANDBOX-5069]
        # if not system.local_mode():
        #     with common_os.Capabilities(common_os.Capabilities.Cap.Bits.CAP_SYS_ADMIN):
        #         delta_usage = self._container.delta_usage
        #     if delta_usage > self.CONTAINER_DELTA_SOFT_LIMIT:
        #         self.logger.warning(
        #             "Delta directory %s has exceeded limit %s / %s, container %s will be destroyed",
        #             self._container.deltadir, delta_usage, self.CONTAINER_DELTA_SOFT_LIMIT, self._container
        #         )
        #         self._destroy_container(self._container, check=False, stop=True)
        #         return

        self.restore_system_files()
        self.cleanup_home()
        with system.PrivilegedSubprocess(("Freezing container ", self._container.name)):
            self._freeze_container(self._container.name)
        if self._container.instance is not None:
            self.instances[self._container.template][self._container.instance] = 0

    @classmethod
    def _copy_capabilities(cls, container, binary):
        dst_binary = container.path(*filter(None, binary.split(os.sep)))
        if not all(it.imap(os.path.exists, (binary, dst_binary, "/sbin/getcap"))):
            return
        try:
            cap = sp.check_output(["/sbin/getcap", binary]).partition("=")[2].strip()
        except sp.CalledProcessError as ex:
            logger.error("Error while getting capability for %r: %s", binary, ex)
            return
        if not cap:
            return
        try:
            sp.check_call(["/sbin/setcap", cap, dst_binary])
        except sp.CalledProcessError as ex:
            logger.error("Error while setting capability %r for %r: %s", cap, dst_binary, ex)

    @classmethod
    def _check_cgroups(cls, container):
        try:
            out = sp.check_output([
                "lxc-attach", "-n", container.name, "--",
                "ls", os.path.join(common_os.CGroup.ROOT, "freezer/tasks")
            ])
        except sp.CalledProcessError:
            return None
        return bool(out)

    @classmethod
    def _restore_cgroups(cls, container):
        # attempt to configure minimal cgroups for precise containers
        # see SANDBOX-4878 for more details
        cgconf_path = os.path.join(container.path(), "etc/cgconfig.conf")
        if os.path.exists(cgconf_path):
            logger.debug("Try to configure cgroups for %r", container)
            with open(cgconf_path, "w") as f:
                f.writelines([
                    "mount {",
                    "\tmemory = /sys/fs/cgroup/memory;"
                    "\tfreezer = /sys/fs/cgroup/freezer;",
                    "}"
                ])
            try:
                out = sp.check_output(
                    ["lxc-attach", "-n", container.name, "--", "service", "cgconfig", "restart"],
                    stderr=sp.STDOUT
                )
            except sp.CalledProcessError as exc:
                logger.warning("Error when running cgconfig restart, output follows:\n%s", exc.output)
                return None
            logger.debug("cgconfig restart succeeded with the following output:\n%s", out)
            result = cls._check_cgroups(container)
            if result:
                logger.debug("Successfully configured cgroups in %r", container)
            else:
                logger.warning("Failed to configure cgroups in %r", container)
            return result
        logger.warning("Unable to configure cgroups in container: cgconfig.conf not found")
        return False

    @classmethod
    def _get_mount(cls, container):
        try:
            out = sp.check_output([
                "lxc-attach", "-n", container.name, "--",
                "/bin/mount"
            ], stderr=sp.STDOUT, timeout=300)
        except (sp.CalledProcessError, sp.TimeoutExpired) as exc:
            return {"valid": False, "output": exc.output}
        return {"valid": len(filter(None, out.split("\n"))) > 2, "output": out}

    @staticmethod
    def _check_mtab(container):  # type: (Container) -> bool
        mtab = os.path.join(container.root, "etc", "mtab")
        return (
            os.path.islink(mtab) and os.readlink(mtab) == "../proc/self/mounts" or
            os.path.exists(mtab) and os.path.getsize(mtab)
        )

    @classmethod
    def _set_freezer_state(cls, freezer, state, recursive=False):
        logger.info("Set freezer state %s to %s", freezer, state)
        freezer["state"] = state
        if not recursive:
            return
        for subgroup in freezer:
            cls._set_freezer_state(subgroup, state, recursive=True)

    @classmethod
    def _freeze_container(cls, container_name):
        cg = common_os.CGroup("/lxc/{}/sysdefault".format(container_name))
        if cg is None:
            return
        freezer = cg.freezer
        if not freezer.exists:
            logger.warning("%s does not exist", freezer)
            return
        cls._set_freezer_state(freezer, common_os.FreezerState.FROZEN)

    @classmethod
    def _unfreeze_container(cls, container_name):
        cg = common_os.CGroup("/lxc/{}".format(container_name))
        if cg is None:
            return
        freezer = cg.freezer
        if not freezer.exists:
            logger.warning("%s does not exist", freezer)
            return
        sysdefault = freezer >> "sysdefault"
        frozen = sysdefault.exists and sysdefault["state"] == common_os.FreezerState.FROZEN
        for subgroup in freezer:
            cls._set_freezer_state(subgroup, common_os.FreezerState.THAWED, recursive=True)
        return frozen

    def _validate_container(self, container, create=False):  # type: (Container, bool) -> bool
        """
        Validates that container is in normal state.

        Called when container is created, and every time a task is assigned to container.
        When container is created we wait until mtab and mount are present, and then attempt to configure cgroups
        if they are not available, once. If that fails, we ensure that validation procedure is run again
        by returning False. File `.cgroup_check_failed` ensures that we do not attempt to configure cgroups again.

        When a task is assigned to container, we check that mtab and mount are present, and if cgroups were present
        previously, they still exist. This is checked by presence of `.cgroup_check` file. Otherwise a
        ContainerCorrupted error is raised which causes destruction of the container.

        """
        self.logger.debug("Validating container %r", container)
        root = container.root
        cgroup_check = os.path.join(root, "opt", ".cgroup_check")
        cgroup_check_failed = os.path.join(root, "opt", ".cgroup_check_failed")

        check_results = {}
        for check_name, validate in (
            ("cgroups", self._check_cgroups),
            ("mtab", self._check_mtab),
            ("mount", lambda _: self._get_mount(_)["valid"]),
        ):
            self.logger.info("Run '%s' check on container %s", check_name, container.name)
            check_results[check_name] = validate(container)
            if not check_results[check_name]:
                self.logger.warning("[-] Container %s has no %s", container.name, check_name)

        self.logger.info("Container %s validation result: %s", container.name, check_results)
        ready = check_results["mtab"] and check_results["mount"]
        if ready:
            restored = False
            if not check_results["cgroups"]:
                if create and not os.path.exists(cgroup_check_failed):
                    restored = self._restore_cgroups(container)
                    if restored is None:
                        with open(cgroup_check_failed, "a"):
                            pass
                        return False
                elif os.path.exists(cgroup_check):
                    raise ContainerCorrupted("Container {} had cgroups, but now doesn't".format(container))

            if check_results["cgroups"] or restored:
                with open(cgroup_check, "a"):
                    pass
        elif not create:
            raise ContainerCorrupted("Container {} failed validation".format(container))

        cgroup_prefix = self.get_cgroup_prefix(container.name)
        if cgroup_prefix and self.INVALID_CGROUP_NAME.match(cgroup_prefix):
            raise errors.InvalidCGroups("Cgroup has invalid prefix: {}.".format(cgroup_prefix))

        self.logger.info("Returning %s for container %s", ready, container.name)

        return ready

    def _start_container_inner(self, container):
        # this scripts cleans tmp and remounts it as tmpfs
        init_mounted_tmp = container.path("etc", "init", "mounted-tmp.conf")
        if Container.settings.mount_tmp_dir and os.path.exists(init_mounted_tmp):
            self.logger.debug("Mounting host tmp directory is enabled, remove 'etc/init/mounted-tmp.conf'")
            os.unlink(init_mounted_tmp)

        # this script sets cpu scaling_governor to ondemand, but we want PERFORMANCE
        init_d_ondemand = container.path("etc", "init.d", "ondemand")
        if os.path.exists(init_d_ondemand):
            self.logger.debug("Found 'etc/init.d/ondemand' inside container, remove it")
            os.unlink(init_d_ondemand)

        if os.path.exists("/usr/bin/perf") or common_config.Registry().common.installation != ctm.Installation.LOCAL:
            shutil.copy("/usr/bin/perf", container.path("usr", "bin", "perf"))

        LXCNetwork.setup_devices(container, Container.settings.network.type == ctm.Network.Type.MACVLAN)

        # Set core_pattern to host value
        sysctld = container.path("etc", "sysctl.d")
        if os.path.exists(sysctld):
            with open("/proc/sys/kernel/core_pattern") as _:
                host_core_pattern = _.read()
            with open(os.path.join(sysctld, "99-kernel.conf"), "w") as sysctld_kernel:
                sysctld_kernel.write("kernel.core_pattern={}".format(host_core_pattern))

        self._copy_capabilities(container, "/usr/sbin/tcpdump")

        cgroup_prefix = self._cgroup_prefix(container.name)
        self.logger.debug("Dropping control subgroups %s", cgroup_prefix)
        common_os.CGroup(cgroup_prefix, iterate_all_subsystems=True).delete()

        for service_name in self.UNWANTED_SYSTEMD_SERVICES:
            path = container.path("lib/systemd/system/sysinit.target.wants/systemd-{}.service".format(service_name))
            if os.path.exists(path):
                os.unlink(path)
        etc_sysctl_d = container.path("etc/sysctl.d")
        for name in os.listdir(etc_sysctl_d):
            if name.endswith("-fs.conf"):
                os.unlink(os.path.join(etc_sysctl_d, name))

        for attempt in range(-2, 1):
            self.logger.debug("Trying to start container '%s'", container)
            logfile = "/var/log/lxc/" + ".".join((container.name, "log"))
            system.TaskProc.create([
                "/usr/bin/lxc-start", "-d", "-n", container.name, "-o", logfile, "-l", "DEBUG",
                "--", "/sbin/init", "--verbose"
            ], user="root", cgroup="")
            cmd = ["/usr/bin/lxc-wait", "-n", container.name, "-s", "RUNNING"]
            try:
                # Wait for container goes to RUNNING state
                self.logged_command("Waiting for container '{}' start".format(container), cmd + ["-t", "15"])
                # Wait for rc.local finished successfully

                ret, slept = common_itertools.progressive_waiter(
                    0.1, 1, 60,
                    lambda: os.path.exists(
                        container.path(
                            container.rc_local_flag
                        )
                    )
                )
                if ret:
                    break
                self.logger.debug(
                    "Container %s failed to start because rc.local exec failed (no %s file). "
                    "More info in %s",
                    container.name,
                    container.path(container.rc_local_flag),
                    logfile
                )
                sp.call(["/usr/bin/lxc-stop", "-n", container.name, "-k"])
                raise errors.RcLocalError
            except (errors.ExecutorFailed, errors.RcLocalError):
                if not attempt:
                    self.logger.exception("Failed to start container")
                    raise

        LXCNetwork.setup_hostname_and_hosts(container)
        self.logger.debug("Creating control subgroups")
        (common_os.CGroup(
            self._cgroup_name(container.name), owner=system.UNPRIVILEGED_USER.login
        ) << 1).create()

    def _start_container(self, container):
        try:
            for attempt in range(1, 4):
                self._start_container_inner(container)

                ret = common_itertools.progressive_waiter(
                    0.1, 1, self.CONTAINER_VALIDATE_TIMEOUT,
                    lambda: self._validate_container(container, create=True)
                )[0]
                if ret:
                    self.logger.debug("Container %s mount table:\n%s", container, self._get_mount(container)["output"])
                    # TODO: remove after https://st.yandex-team.ru/SANDBOX-9502
                    self.logged_command("Reset sysctl settings", ["sysctl", "--system"], privileged=True)
                    # FIXME: temporary turn off container delta limit [SANDBOX-5069]
                    # if not system.local_mode():
                    #     container.delta_limit = self.CONTAINER_DELTA_HARD_LIMIT
                    return

                self.logger.warning("Container %s failed validation (attempt %s)", container, attempt)
                self._stop_container(container.name)
        except errors.ContainerTemporaryError:
            self.logger.exception("Fail to start container %s", container.name)
            self._stop_container(container.name)
            self._destroy_container(container)
            raise errors.InfraError
        except Exception:
            self.logger.exception("Exception when starting container %s", container)
            self._stop_container(container.name)

        self._destroy_container(container)
        raise errors.ExecutorFailed

    @classmethod
    def _stop_container(cls, name, graceful=False):
        cls._unfreeze_container(name)
        cmd = ["/usr/bin/lxc-stop", "-n", str(name)]
        if not graceful:
            cmd.append("-k")
        try:
            cls.logged_command("Trying to stop container '{}'".format(name), cmd)
        except errors.ExecutorFailed:
            pass
        except errors.InfraError as ex:
            # do not fail if container is not running (man lxc-stop)
            if ex.returncode != 2:
                raise
        logger.debug("Dropping control subgroups")
        (common_os.CGroup(cls._cgroup_name(name), iterate_all_subsystems=True) << 1).delete()

    @classmethod
    def _remove_template(cls, container, template_lock):
        template = container.template
        logger.debug("Acquiring template lock for %s to destroy container %s", template, container)
        with template_lock or threading.RLock():
            instances = sum(1 for cn in cls.running.keys() if Container(name=cn).template == template)
            logger.debug("There are %d instances left for template %r", instances, template)
            if instances <= 1:
                # check that there are no active jobs for this template
                jobs = cls.instances.get(template, [])
                if all(x <= 0 for x in jobs):
                    with system.PrivilegedSubprocess(("removing template", repr(template))):
                        sp.call(["/bin/umount", container.template_root])
                        try:
                            cls.logged_command("Removing container's template", ["rm", "-rf", template])
                        except errors.ExecutorFailed:
                            pass
                else:
                    logger.debug("There are active jobs for template %s: %s", template, jobs)

    @classmethod
    def _destroy_container(
        cls, container, stop=False, check=False, graceful=False,
        keep_template=False, template_lock=None
    ):
        if check:
            cnt = next((_ for _ in cls._list_containers() if _.name == container.name), None)
            if not cnt:
                return
            stop = cnt.status == "running"
        logger.debug("Destroying container %r", container)
        with system.PrivilegedSubprocess(("Destroy container", container.name), watchdog=300):
            # FIXME: temporary turn off container delta limit [SANDBOX-5069]
            # if not system.local_mode():
            #     del container.delta_limit
            if stop:
                cls._stop_container(container.name, graceful=graceful)
            else:
                logger.debug("Dropping control subgroups")
                cgroup = common_os.CGroup(cls._cgroup_name(container.name), iterate_all_subsystems=True)
                if cgroup:
                    (cgroup << 1).delete()
            try:
                cls.logged_command("Unmounting overlayed filesystem", ["/bin/umount", container.root])
            except errors.InfraError as err:
                if not err.stderr or not ("not mounted" in err.stderr or "mountpoint not found" in err.stderr):
                    raise
            except errors.ExecutorFailed:
                pass

            shutil.rmtree(os.path.dirname(container.config))
            try:
                cls.logged_command("Removing container's root filesystem", ["rm", "-rf", container.basedir])
            except errors.ExecutorFailed:
                pass

        if not keep_template and container.template:
            cls._remove_template(container, template_lock)
        cls.running.pop(container.name, None)
        logger.debug("Container %r destroyed", container)

    @staticmethod
    def _patch_passwd(fname, user, newline):
        marker = user + ":"
        with open(fname, "r") as fh:
            for l in fh:
                if l.startswith(marker):
                    return
        with open(fname, "a") as fh:
            fh.write(newline + "\n")

    @staticmethod
    def _add_user_to_group(fname, user, group):
        with open(fname, "r") as f:
            lines = map(lambda _: _.strip().split(":"), f.readlines())
        for line in lines:
            if line[0] != group:
                continue
            users = filter(None, line[-1].split(","))
            if user in users:
                continue
            users.append(user)
            line[-1] = ",".join(users)
            with open(fname, "w") as f:
                f.write("\n".join(common_itertools.chain((":".join(line) for line in lines), "")))
            break

    @common_patterns.classproperty
    def _skynet_root(self):
        with open("/skynet/.info", "rb") as fh:
            root = yaml.safe_load(fh)["paths"]["prefix"]
        while os.path.islink(root):
            root = os.readlink(root)
        return root

    @common_patterns.classproperty
    def _skynet_link(self):
        with open("/skynet/.info", "rb") as fh:
            return yaml.safe_load(fh)["links"]["/skynet"]

    @classmethod
    def _unpack_template(cls, container, src):
        rootfs = container.template_root
        tmpldir = os.path.dirname(rootfs)
        with system.PrivilegedSubprocess(("container template", repr(container), "unpacker")):
            # `/var/lib/lxc` is readable on prod clients, but not necessarily on local installations.
            # But it should be with root privileges.
            if os.path.isdir(rootfs):
                logger.debug("Container %r template already exists.", container)
                return rootfs

            tmpdir = cls.empty_dir(tmpldir + ".tmp")
            cls.logged_command("Unpacking template for '{}'".format(container), ["tar", "-C", tmpdir, "-zxf", src])
            os.rename(tmpdir, tmpldir)
        return rootfs

    @classmethod
    def _mount_template(cls, container, src):
        rootfs = container.template_root
        tmpldir = os.path.dirname(rootfs)
        if os.path.ismount(tmpldir):
            logger.debug("{} already mounted".format(tmpldir))
            return rootfs
        with system.UserPrivileges():
            cls.empty_dir(tmpldir)
        cls.logged_command("Mounting container template", ["/bin/mount", src, tmpldir], privileged=True)
        return rootfs

    @classmethod
    def _mount_container(cls, container):
        template = container.template_root
        basedir = container.basedir
        delta = container.deltadir
        work = container.workdir
        root = container.root
        with system.UserPrivileges():
            map(cls.empty_dir, (basedir, delta, work, root))
        cls.logged_command("Mounting overlayed filesystem", [
            "/bin/mount", "-t", "overlay", "overlay",
            "-olowerdir={0},upperdir={1},workdir={2}".format(template, delta, work),
            root
        ], privileged=True)

    @classmethod
    def _patch_container(cls, root):
        su = (system.SERVICE_USER.uid, system.SERVICE_USER.gid, 0o755)
        uu = (system.UNPRIVILEGED_USER.uid, system.UNPRIVILEGED_USER.gid, 0o755)
        skydir = cls._skynet_root
        for path, (uid, gid, mode) in (
            (os.path.join(root, system.SERVICE_USER.home.lstrip(os.path.sep)), su),
            (os.path.join(root, system.SERVICE_USER.home.lstrip(os.path.sep), "tasks"), su),
            (os.path.join(root, common_config.Registry().client.dirs.run.lstrip(os.path.sep)), uu),
            (os.path.join(root, common_config.Registry().client.log.root.lstrip(os.path.sep)), uu),
            (os.path.join(root, common_config.Registry().client.dirs.data.lstrip(os.path.sep)), uu),
            (os.path.join(root, skydir.lstrip(os.path.sep)), (0, 0, 0o755)),
            (os.path.join(root, "lib", "modules"), (0, 0, 0o755)),
            (os.path.join(root, "place", "vartmp"), (0, 0, 0o1755)),
            (os.path.join(root, "place", "coredumps"), (0, 0, 0o1755)),
        ):
            logger.info("Creating %r", path)
            if os.path.islink(path):
                os.unlink(path)
            sp.check_call(["install", "-d", "-o", str(uid), "-g", str(gid), "-m", oct(mode), path])

        cls._patch_passwd(
            os.path.join(root, "etc", "passwd"),
            system.SERVICE_USER.login, "{}:x:{}:{}:{}:{}:/bin/bash".format(
                system.SERVICE_USER.login, system.SERVICE_USER.uid,
                system.SERVICE_USER.gid, system.SERVICE_USER.group, system.SERVICE_USER.home
            )
        )
        cls._patch_passwd(
            os.path.join(root, "etc", "group"),
            system.SERVICE_USER.group, "{}:x:{}:".format(system.SERVICE_USER.group, system.SERVICE_USER.gid)
        )
        cls._add_user_to_group(os.path.join(root, "etc", "group"), system.UNPRIVILEGED_USER.login, "fuse")

        origin = cls._skynet_link
        cnt_origin = os.path.join(root, "skynet")
        dst = os.path.join(root, origin.lstrip(os.path.sep))
        logger.debug("Mirroring skynet link pointing to '%s' at '%s'", dst, cnt_origin)
        if os.path.lexists(cnt_origin):
            os.unlink(cnt_origin)
        os.symlink(origin, cnt_origin)

        logger.debug("Disabling cron at '%s'", root)
        for f in ("etc/init.d/cron", "etc/init/cron.conf", "lib/systemd/system/cron.service"):
            fname = os.path.join(root, f)
            if os.path.exists(fname):
                os.unlink(fname)

    @classmethod
    def _create_configs(cls, name, hostname, rootfs):
        containers = []
        configs_dir = os.path.dirname(os.path.dirname(Container.settings.config_template))

        reserved_no = set()
        for dirname in os.listdir(configs_dir):
            cfg = Container.settings.config_template.format(dirname)
            if os.path.exists(cfg):
                try:
                    with open(cfg, "r") as fh:
                        for l in fh:
                            m = LXCNetwork.CONTAINER_NO_REGEXP.search(l)
                            if m:
                                no = int(m.group(2) or m.group(3))
                                reserved_no.add(no)
                                try:
                                    container = Container(name=dirname, no=no)
                                except ValueError as e:
                                    logger.debug("Somebody else's container %s: %s", dirname, str(e))
                                else:
                                    if dirname in cls.running:
                                        containers.append(container)
                                    else:
                                        logger.debug("Container has config but not running: %s", dirname)
                                break
                except (OSError, IOError):
                    logger.debug("Container's config could not be read: %s", dirname)
        prev = 1
        logger.debug("Detected containers: %r, reserved no: %r", containers, reserved_no)
        for no in sorted(reserved_no):
            if no - prev > 1:
                break
            prev = no
        no = prev + 1
        logger.debug("Container %r number assigned %r, hostname: %r.", name, no, hostname)
        lxc_if, lxc_mac, lxc_ip = LXCNetwork.lxc_iface(no)
        cnt = Container(name, no=no, ip=lxc_ip)

        cls.empty_dir(os.path.dirname(cnt.config))

        with open(cnt.config, "w") as fh:
            config_contents = "\n".join((
                cls.LXC_CONFIG_TEMPLATE,
                # FIXME: temporary turn off container delta limit [SANDBOX-5069]
                # cls.LXC_CONFIG_CAPABILITIES,
                LXCNetwork.LXC_CONFIG_TEMPLATE,
                "lxc.console.logfile = /var/log/lxc/{container_name}.console.log",
            )).format(
                hostname=hostname,
                no=no,
                rootfs=rootfs,
                fstab_path=cnt.fstab,
                iface_name=lxc_if,
                lxc_mtu=LXCNetwork.lxc_mtu,
                lxc_mac=lxc_mac,
                container_name=name,
                o4=random.randint(0, 255),
                o5=random.randint(0, 255),
                o6=random.randint(0, 255),
                arc_mount=cls.LXC_ARC_MOUNT.format(
                    arc_repo=common_config.Registry().devbox.arc_repo,
                    rootfs=rootfs
                ) if system.local_mode() and common_config.Registry().devbox.arc_repo else ""
            )
            fh.write(config_contents)

            # In MULTISLOT clients CPU core #0 is not used by LXC containers to free it up
            # for service processes (Client, AgentR etc.)
            if common_config.Registry().client.max_job_slots > 1 and common_config.Registry().this.cpu.cores > 1:
                fh.write("\nlxc.cgroup.cpuset.cpus = 1-{}".format(common_config.Registry().this.cpu.cores - 1))

        with open(cnt.fstab, "w") as fh:
            if system.SERVICE_USER == system.UNPRIVILEGED_USER:
                fh.write(cls.LXC_FSTAB_SU_HOME.format(rootfs=rootfs, home=system.SERVICE_USER.home))
            if cnt.settings.mount_tmp_dir:
                fh.write(cls.LXC_FSTAB_TMP_DIR.format(rootfs=rootfs, tmp=cnt.settings.dirs.tmp))
            fh.write(cls.LXC_FSTAB_BASE_TEMPLATE.format(
                rootfs=rootfs,
                datadir_mount_mode="rw",
                rundir=common_config.Registry().client.dirs.run,
                logdir=common_config.Registry().client.log.root,
                base_cache=common_config.Registry().client.vcs.dirs.base_cache,
                datadir=common_config.Registry().client.dirs.data,
                skydir=cls._skynet_root,
                arc_vcs_fs="none",
                arc_vcs_opts="defaults,rw,bind",
            ))
            fh.write(cls.LXC_FSTAB_REGULAR_TEMPLATE.format(rootfs=rootfs))

        def key(c):
            try:
                return os.stat(c.config).st_mtime
            except OSError:
                return None
        containers = sorted(containers, key=key)
        containers.append(cnt)
        return containers

    def _ensure_container(self):
        with self.lock:
            busy, empty_instance, task_instance = self.instances.setdefault(self._container.template, []), None, None

            for instance, task in enumerate(busy):
                if task == getattr(self._cmd, "task_id", 0):
                    task_instance = instance
                elif not task:
                    empty_instance = instance

            if task_instance is not None:
                instance = task_instance
            elif empty_instance is not None:
                instance = empty_instance
            else:
                instance = len(busy)
                busy.append(0)
            self.logger.debug(
                "Selected copy %r of container %r. Copies state: %r", instance, self._container.template, busy
            )
            self._container.instance = instance
            busy[instance] = getattr(self._cmd, "task_id", 0)
            template_lock = self._template_locks[self._container.template]

        from sandbox.client.pinger import PingSandboxServerThread
        pt = PingSandboxServerThread()
        pt._kamikadze_thread.ttl = common_config.Registry().client.idle_time * 100

        if self._container.name in self.running:
            self.logger.debug("Container %r (%r) exists and running.", self._container, self._cmd.arch)
            try:
                with system.PrivilegedSubprocess(("Container", repr(self._container), "check"), watchdog=300):
                    os.utime(self._container.config, None)
                    self._unfreeze_container(self._container.name)
                    self._validate_container(self._container)
                # TODO: SANDBOX-3403: Here we should fill container number
                # TODO: SANDBOX-3403: otherwise it will not be available for the executing task's object
                return
            except ContainerCorrupted as exc:
                self.logger.warning("Container %s is corrupted: %s", self._container, str(exc))
            except errors.InvalidCGroups:
                self.logger.warning("Container %s has invalid cgroups", self._container)

            self._destroy_container(self._container, check=False, stop=True)

        self.logger.debug("Creating container %r (%r).", self._container, self._cmd.arch)
        path = self.agentr.resource_sync(self._container.resource_id, fastbone=False)
        container_meta = self.agentr.resource_meta(self._container.resource_id)

        self.logger.debug("Acquiring template lock for %s", self._container.template)
        with template_lock:
            if container_meta["attributes"].get("type") != ctt.ImageType.Group.IMAGE:
                self._unpack_template(self._container, path)
            else:
                self._mount_template(self._container, path)
            self._mount_container(self._container)
            with system.UserPrivileges():
                self._patch_container(self._container.root)
        with self.lock:
            self._container.hostname = self._alias.replace("_", "-")
            with system.UserPrivileges():
                containers = self._create_configs(
                    self._container.name, self._container.hostname,
                    self._container.path()
                )
                self._container.ip, self._container.no = containers[-1].ip, containers[-1].no
            drop = len(containers) - self.MAX_CONTAINERS - common_config.Registry().client.max_job_slots
            self.logger.debug(
                "Current containers amount: %d, max containers: %d, max slots: %d, drop: %d",
                len(containers), self.MAX_CONTAINERS, common_config.Registry().client.max_job_slots, drop - 1
            )
            to_destroy = []
            for i in range(drop):
                container = containers[i]
                if self.instances.get(container.template)[int(container.instance)]:
                    self.logger.warning("Skip container %r destroying - it has a running job!", container)
                    continue
                self.logger.info("Destroying extra running container %r", container)
                to_destroy.append((container, self._template_locks[container.template]))
                # lock container so that nothing is assigned to it
                self.instances.get(container.template)[int(container.instance)] = -1

        for container, template_lock in to_destroy:
            self._destroy_queue.put((
                container,
                dict(
                    check=False, stop=True,
                    keep_template=(self._container.template == container.template),
                    template_lock=template_lock
                )
            ))

        with system.PrivilegedSubprocess(("Starting container", self._container.name), watchdog=300):
            self._start_container(self._container)

        with self.lock:
            self.running[self._container.name] = ""

        self.restore_system_files()
        with system.PrivilegedSubprocess(("Cleaning home for container", self._container.name)):
            self.cleanup_home()
        self.logger.info("Created container #%r (%r).", self._container, self._cmd.arch)

    def restore_system_files(self):
        if system.local_mode():
            return
        with system.PrivilegedSubprocess("restoring system files"):
            for path in self.SYSTEM_FILES:
                if not os.path.exists(path):
                    self.logger.warning("Path %s not found", path)
                    continue
                inner_path = os.path.join(self._container.root, path.lstrip(os.sep))
                inner_dir = os.path.dirname(inner_path)
                if not os.path.exists(inner_dir):
                    os.makedirs(inner_dir, mode=0o755)
                shutil.copy(path, inner_path)

    @property
    def executable(self):
        if system.local_mode():
            return os.path.join(system.SERVICE_USER.home, "venv." + str(self._venv_rid), "bin", "python")
        return os.path.join(system.SERVICE_USER.home, "venv", "bin", "python")

    def read_resolv_conf(self, names=None):
        if not names:
            names = ("lxc", "orig", "")
        return super(LXCPlatform, self).read_resolv_conf(names)

    def prepare(self, tasks_rid):
        res_id = self._container.resource_id
        if not self._cmd.arch:
            raise errors.InvalidJob("Container resource #{} has no 'platform' attribute".format(res_id))
        elif not self.venv:
            raise errors.InvalidJob("No virtual environment resource for container resource #{}".format(res_id))
        try:
            self._ensure_container()
        except (OSError, IOError, socket.gaierror, common_os.SubprocessAborted) as ex:
            self.logger.error("Error while ensuring container for task #%s: %s", self._cmd.task_id, ex, exc_info=ex)
            raise errors.InfraError(str(ex))
        super(LXCPlatform, self).prepare(tasks_rid)

    @classmethod
    def get_cgroup_prefix(cls, container_name):
        try:
            sp.check_output([
                "lxc-attach", "-n", container_name, "--",
                "ls", os.path.join(common_os.CGroup.ROOT, "freezer/freezer.state")
            ])
        except sp.CalledProcessError:
            return None
        pid = sp.check_output(["lxc-info", "-n", container_name, "-pH"]).strip()
        cgroup_prefix = common_os.CGroup(pid=pid).freezer.path
        if not cgroup_prefix:
            return None
        match = re.search(r"(/lxc/|lxc\.payload\.)([\d\.]+[^/])", cgroup_prefix)
        if match:
            return cls._cgroup_prefix(container_name)
        else:
            return None

    def set_cgroup_prefix(self):
        """
        Here we try to determine whether the container can see the root cgroup of the host.
        Starting from linux kernel version 4.9.51-13 it cannot.
        First we check the presence of /sys/fs/cgroup/freezer/freezer.state which should not be in the root cgroup.
        See: https://www.kernel.org/doc/Documentation/cgroup-v1/freezer-subsystem.txt
        Then we determine the container's cgroup and set it as cgroup prefix for this platform.

        """
        self.cgroup_prefix = self.get_cgroup_prefix(self._container.name)
        if self.cgroup_prefix:
            self.logger.debug("Using cgroup prefix: '%s'", self.cgroup_prefix)

    def set_container_limits(self):
        ram = self._cmd.args.get("ram")
        ram = ram << 20 if ram else 0
        cpu_shares = (self._cmd.args.get("cores") or 1) << 10
        for subsys, name, value in (
            ("memory", "low_limit_in_bytes", ram),
            ("cpu", "shares", cpu_shares),
        ):
            if not os.path.exists("/sys/fs/cgroup/{subsys}/{subsys}.{name}".format(subsys=subsys, name=name)):
                self.logger.warning(
                    "Unable to set up LXC container {subsys} guarantee: cgroup variable {subsys}.{name} "
                    "is not supported on this system.".format(subsys=subsys, name=name)
                )
                continue
            try:
                sp.check_output(
                    [
                        "lxc-cgroup", "-n", self._container.name,
                        "{}.{}".format(subsys, name), str(value)
                    ],
                    stderr=sp.STDOUT
                )
            except sp.CalledProcessError as exc:
                self.logger.warning("Failed to set up LXC container {} guarantee: %s".format(subsys), exc.output)
                raise errors.ExecutorFailed

    def setup_agentr(self, agentr):
        # Provide container name for AgentR
        agentr.lxc = self._container.name
        lxc_system_log = common_os.system_log_path(self._container.root)
        if lxc_system_log:
            agentr.monitoring_add_log(lxc_system_log, "system_{}.log".format(self._container.name))
        # Lock container resource for the task session
        agentr.resource_sync(self._container.resource_id)
        return super(LXCPlatform, self).setup_agentr(agentr)

    def _container_info(self):
        # SANDBOX-5358: We have to keep dictionary here because of backward compatibility issue with binary tasks
        return {
            "rootfs": self._container.root,
            "platform": self._cmd.arch,
            "name": self._container.name,
            "executable": self.executable,
            "template": self._container.resource_id,
            "instance": self._container.instance,
            "no": self._container.no,
            "ip": self._container.ip,
        }

    def on_system_error(self):
        self.logger.info("Destroy container %s on system error", self._container.name)
        self._destroy_container(self._container, check=False, stop=True)

    def spawn(self, executor_args):
        self.set_resolv_conf_arg(executor_args)
        with system.UserPrivileges():
            self.set_cgroup_prefix()
            executor_args["cgroup"] = self.cgroup and self._deprefix_cgroup(self.cgroup.name)
            self.set_container_limits()

        executor_args["container"] = self._container_info()
        self._cmd.executor_args = executor_args
        self._cmd.save_state()

        local_path = None
        if system.local_mode():
            local_path = os.path.join(os.path.expanduser("~/.sandbox/lxc"), self._container.name)
            sp_path = os.path.join(local_path, "lib", "python2.7", "site-packages")
            with system.UserPrivileges.lock:
                if not os.path.exists(sp_path):
                    os.makedirs(sp_path, 0o755)

        env = os.environ.copy()
        env["HOME"] = system.UNPRIVILEGED_USER.home
        env["USER"] = env["LOGNAME"] = system.UNPRIVILEGED_USER.login
        env["LANG"] = "en_US.UTF8"
        if local_path:
            env["PYTHONUSERBASE"] = local_path
        env.pop("PYTHONPATH", None)
        env[common_config.Registry.CONFIG_ENV_VAR] = self.config_path
        return system.TaskLiner(
            common_format.obfuscate_token(self._cmd.token),
            self.logger,
            [
                "/usr/bin/lxc-attach", "-n", self._container.name, "--",
                self.preexecutor_path, self._cmd.executor_args
            ],
            env,
            "root",
            self.cgroup,
        ), executor_args

    @classmethod
    def maintain(cls):
        """ Fills running containers collection and also tries to start all stopped containers. """
        if not cls.available:
            logger.error("No LXC containers available.")
            return

        LXCNetwork.clear_down_interfaces()

        logger.debug("Checking available containers.")
        cls.running.clear()

        destroy = set()
        for retry in range(2):
            for name, status in cls._list_containers():
                stop = False
                if system.local_mode():
                    try:
                        container = Container(name=name)
                    except ValueError as e:
                        logger.debug("Skip %s container %r check: %s", status, name, str(e))
                        continue

                    if "." not in container.template and name not in cls.running:
                        logger.debug("Destroying outdated %s container %r", status, name)
                        destroy.add((name, status == "running"))
                        continue

                logger.debug("Checking %s container %r.", status, name)
                if status == "running" and name not in cls.running:
                    stop = True
                    with system.UserPrivileges():
                        frozen = cls._unfreeze_container(name)
                        pl = cls._lxc_execute(
                            name,
                            [
                                "/usr/bin/sudo", "-nHu", system.UNPRIVILEGED_USER.login,
                                "/skynet/python/bin/python",
                                "-c", "import platform; print platform.platform()"
                            ],
                            stdout=sp.PIPE,
                            close_fds=True
                        ).communicate()[0].strip()
                        if frozen:
                            cls._freeze_container(name)
                    logger.debug("Container %r platform: %r", name, pl)
                    if pl:
                        cls.running[name] = pl

                if name not in cls.running:
                    destroy.add((name, stop))

        instances = {}
        for name in cls.running.keys():
            container = Container(name=name)
            template, instance = container.template, container.instance
            instances[template] = max(instances.get(template, 0), int(instance))
        for template, instance in instances.iteritems():
            if template not in cls.instances:
                cls.instances[template] = [0] * (instance + 1)
            elif len(cls.instances[template]) <= instance:
                cls.instances[template].extend([0] * (instance + 1 - len(cls.instances[template])))
        logger.debug("Containers detected: %r, instances: %r", cls.running, cls.instances)
        for name, stop in destroy:
            try:
                container = Container(name=name)
                template, instance = container.template, container.instance
                cls._destroy_container(
                    PrivilegedContainer(container) if name == "privileged" else container,
                    stop,
                    graceful=False
                )
                instances[template] = max(instances.get(template, 0), int(instance))
            except ValueError:
                logger.warning("Unable to parse container name: %s", name)
        return super(LXCPlatform, cls).maintain()

    @common_patterns.singleton_classproperty
    def available(self):
        return os.path.exists("/var/lxc/root")

    @classmethod
    def _cgroup_prefix(cls, name):
        if common_platform.dist_version() == common_platform.UbuntuRelease.FOCAL:
            return "/lxc.payload.{}".format(name)
        return os.path.join("/", "lxc", name)

    @classmethod
    def _cgroup_name(cls, name):
        return os.path.join(
            cls._cgroup_prefix(name), "sysdefault", "lxc", name, super(LXCPlatform, cls)._cgroup_name("").lstrip("/")
        )

    def _cgroup_id(self):
        return self._container.name

    @common_patterns.singleton_property
    def cgroup(self):
        return self.make_cgroup(self._cgroup_name(self._cgroup_id()))

    def _deprefix_cgroup(self, name):
        if self.cgroup_prefix and name.startswith(self.cgroup_prefix):
            return os.path.join("/", os.path.relpath(name, self.cgroup_prefix))
        else:
            return name

    @property
    def _alias(self):
        return common_platform.get_platform_alias(
            self._cmd.arch or self.running[self._container.name]
        ).replace(".", "_")

    @common_patterns.singleton_property
    def __arch_venvs(self):
        return {_["arch"]: _["path"] for _ in Container.settings.venv}

    @property
    def venv(self):
        for pkg in super(LXCPlatform, self).venv:
            yield pkg
        if system.local_mode():
            venv_rid = getattr(self, "_venv_rid", None)
            if not venv_rid:
                return
            venv_dir = os.path.join(system.SERVICE_USER.home, "venv." + str(venv_rid))
            if not os.path.exists(venv_dir):
                common_fs.untar_archive(
                    self.agentr.resource_sync(venv_rid),
                    self.empty_dir(venv_dir),
                    log=self.logger
                )
        else:
            venv_dir = self.__arch_venvs.get(self._alias)
            if venv_dir:
                yield self.Package("venv", venv_dir, self._container.path(system.SERVICE_USER.home, "venv"))

    @property
    def client(self):
        if system.SERVICE_USER == system.UNPRIVILEGED_USER:
            return
        pkg = next(iter(super(LXCPlatform, self).client), None)
        yield pkg
        yield self.Package(pkg.name, pkg.src, self._container.path(pkg.dst))
        yield self.Package("preexecutor", self.preexecutor_path, self._container.path(self.preexecutor_path))

    @property
    def tasks(self):
        pkg = next(iter(super(LXCPlatform, self).tasks), None)
        if not pkg:
            return
        yield pkg
        # TODO: SANDBOX-5068: Backward-compatibility for tasks with fixed __archive__
        with open(os.devnull, "wb") as devnull:
            sp.call(
                ["lxc-attach", "-n", self._container.name, "--", "/bin/umount", pkg.dst],
                stdout=devnull, stderr=sp.STDOUT, preexec_fn=common_os.User.Privileges().__enter__
            )
        yield self.Package(pkg.name, pkg.src, self._container.path(pkg.dst))

    @property
    def ramdrive(self):
        return super(LXCPlatform, self).ramdrive

    @ramdrive.setter
    def ramdrive(self, value):
        if not self._container:
            return
        self._mount_ramdrive(value, command_prefix=["lxc-attach", "-n", self._container.name, "--"])

    @ramdrive.deleter
    def ramdrive(self):
        ramdrive = self.ramdrive
        if not ramdrive:
            return
        with system.UserPrivileges():
            from sandbox.client.commands import Command
            lxcs = self.running if not Command.last_command_id else filter(None, [getattr(self, "_container")])
            for lxc in lxcs:
                lxc_name = lxc.name if hasattr(lxc, "name") else lxc
                self.logger.debug("Checking mounted RAM drive %r in %r", ramdrive, lxc_name)
                if not self.unmount_ramdrive(
                    ramdrive, ["/usr/bin/lxc-attach", "-n", lxc_name, "--"], logger=self.logger
                ):
                    self._destroy_container(lxc, stop=True)

    def _fs_command(self, cmd):
        # --keep-env: though enabled by default, it is "likely to change in the future"
        return "/usr/bin/lxc-attach --keep-env -n {} -- /usr/bin/sudo -EHu {} {}".format(
            self._container.name, system.UNPRIVILEGED_USER.login, cmd
        )

    @property
    def shell_command(self):
        return self._fs_command(super(LXCPlatform, self)._shell_command)

    @property
    def ps_command(self):
        return self._fs_command(super(LXCPlatform, self)._ps_command)

    @property
    def attach_command(self):
        exe = self.executable
        return (
            "/usr/bin/lxc-attach -n {} -- "
            "{} {}/pydevd_attach_to_process/attach_pydevd.py --port {{port}} --host {{host}} --pid {{pid}}"
        ).format(
            self._container.name, exe, os.path.dirname(os.path.dirname(exe))
        )

    def _unmount_fuse(self):
        super(LXCPlatform, self)._unmount_fuse(
            command_prefix=["/usr/bin/lxc-attach", "-n", self._container.name, "--"]
        )


class PrivilegedLXCPlatform(LXCPlatform):
    SERIALIZABLE_ATTRS = LXCPlatform.SERIALIZABLE_ATTRS + ("_origin",)

    _origin = None

    def __init__(self, cmd):
        super(PrivilegedLXCPlatform, self).__init__(cmd)
        self._origin = self._container

    @property
    def _alias(self):
        return common_platform.get_platform_alias(self._cmd.arch or self.running[self._origin]).replace(".", "_")

    @staticmethod
    def _patch_config(src, dst, patterns):
        """
        Patches given config file saving its backup copy using given patterns. Each matched line will be yielded with
        appropriate pattern object and modified line is expected as yield's result.

        :param src:         Configuration file name to used for patching.
        :param dst:         File name to be used save patched content.
        :param patterns:    A list of compile regular expressions to be searched in configuration file.
        """
        logger.debug("Patching configuration file %r to %r", src, dst)
        with open(src, "r") as ifh:
            with open(dst, "w") as ofh:
                for l in ifh:
                    match = next(((p, l) for p in patterns if p.search(l)), None)
                    if match:
                        l = yield match
                    ofh.write(l)

    @classmethod
    def destroy_container(cls):
        """ Public method to use from :py:mod:`sandbox.devbox.ctl_impl` """
        cls._destroy_container(PrivilegedContainer(Container(), None), True, True)

    @classmethod
    def _destroy_container(
        cls, container, stop=True, check=False, graceful=False,
        keep_template=False, template_lock=None, **kws
    ):
        if container.name != PrivilegedContainer.name:
            return super(PrivilegedLXCPlatform, cls)._destroy_container(
                container, stop, check, graceful, keep_template, template_lock
            )
        if check:
            cnt = next((_ for _ in cls._list_containers() if _.name == container.name), None)
            if not cnt:
                return
            stop = cnt.status == "running"

        rename_delta = kws.get("rename_delta")
        logger.debug("Destroying container %r", container)
        from sandbox.client.pinger import PingSandboxServerThread
        PingSandboxServerThread()._kamikadze_thread.ttl = common_config.Registry().client.disk_op_timeout
        with system.PrivilegedSubprocess(("destroy container", repr(container)), watchdog=300):
            if stop:
                cls._stop_container(container.name)
            try:
                shutil.rmtree(os.path.dirname(container.config))
            except OSError as ex:
                logger.warning("Error dropping container's configuration: %s", ex)
            try:
                cls.logged_command("Unmounting overlayed filesystem", ["/bin/umount", container.root])
            except errors.InfraError as err:
                if not err.stderr or not ("not mounted" in err.stderr or "mountpoint not found" in err.stderr):
                    raise
            except errors.ExecutorFailed:
                pass

            if rename_delta:
                logger.debug("Renaming rootfs delta from %r to %r", container.deltadir, rename_delta)
                if os.path.exists(rename_delta):
                    shutil.rmtree(rename_delta)
                try:
                    os.rename(container.deltadir, rename_delta)
                except OSError as ex:
                    logger.warning("Error dropping container's delta: %s", ex)

            task_id = None
            for dirname in os.listdir(container.settings.rootfs.basedir_privileged):
                try:
                    task_id = int(dirname)
                except (TypeError, ValueError):
                    continue
                cls.empty_dir(
                    os.path.join(container.settings.rootfs.basedir_privileged, dirname),
                    rm_msg="Removing stale directory %r",
                    create=False,
                )
            if not keep_template and container.template:
                cls._remove_template(container, template_lock)

            if not rename_delta and task_id:
                task_workdir = os.path.join(common_config.Registry().client.tasks.data_dir, *ctt.relpath(task_id))
                cls._safe_chown(system.UNPRIVILEGED_USER, task_workdir, recursive=True)

            try:
                cls.empty_dir(container.basedir, create=False)
            except OSError as ex:
                logger.warning("Error dropping container's basedir: %s", ex)
        cls.running.pop(container.name, None)
        logger.debug("Container %r destroyed", container)

    def _create_configs_privileged(self):
        fstab = self._container.fstab
        cfg = self._container.config
        self.empty_dir(os.path.dirname(cfg))

        root = self._container.root
        task_workdir = self.ensure_taskdir(self._cmd.task_id)

        lxc_if, lxc_mac, lxc_ip = LXCNetwork.lxc_iface(self._container.no)
        self._container.hostname, self._container.ip = self._alias.replace("_", "-"), lxc_ip
        with open(cfg, "w") as fh:
            config_contents = "\n".join((
                self.LXC_CONFIG_TEMPLATE,
                LXCNetwork.LXC_CONFIG_TEMPLATE
            )).format(
                hostname=self._container.hostname,
                no=self._container.no,
                rootfs=root,
                fstab_path=fstab,
                iface_name=lxc_if,
                lxc_mtu=LXCNetwork.lxc_mtu,
                lxc_mac=lxc_mac,
                o4=random.randint(0, 255),
                o5=random.randint(0, 255),
                o6=random.randint(0, 255),
                arc_mount=self.LXC_ARC_MOUNT.format(
                    arc_repo=common_config.Registry().devbox.arc_repo,
                    rootfs=root
                ) if system.local_mode() and common_config.Registry().devbox.arc_repo else ""
            )
            fh.write(config_contents)

        with open(fstab, "w") as fh:
            if system.SERVICE_USER == system.UNPRIVILEGED_USER:
                fh.write(self.LXC_FSTAB_SU_HOME.format(rootfs=root, home=system.SERVICE_USER.home))
            if self._container.settings.mount_tmp_dir:
                fh.write(self.LXC_FSTAB_TMP_DIR.format(rootfs=root, tmp=self._container.settings.dirs.tmp))
            arc_vcs_delta = self.empty_dir(os.path.join(task_workdir, ".vcs", "arc", "delta"))
            arc_vcs_work = self.empty_dir(os.path.join(task_workdir, ".vcs", "arc", "work"))
            fh.write(self.LXC_FSTAB_BASE_TEMPLATE.format(
                rootfs=root,
                datadir_mount_mode="ro",
                rundir=common_config.Registry().client.dirs.run,
                logdir=common_config.Registry().client.log.root,
                base_cache=common_config.Registry().client.vcs.dirs.base_cache,
                datadir=common_config.Registry().client.dirs.data,
                skydir=self._skynet_root,
                arc_vcs_fs="overlay",
                arc_vcs_opts=(
                    "rw,relatime,"
                    "lowerdir=/place/sandbox-data/srcdir/arc_vcs,"
                    "upperdir={upperdir},"
                    "workdir={workdir}"
                ).format(upperdir=arc_vcs_delta, workdir=arc_vcs_work),
            ))
            fh.write(self.LXC_FSTAB_PRIVILEGED_TEMPLATE.format(
                rootfs=root, workdir=task_workdir
            ))

    @classmethod
    def _mount_container(cls, container):
        with system.UserPrivileges():
            cls.empty_dir(os.path.dirname(container.deltadir))
        super(PrivilegedLXCPlatform, cls)._mount_container(container)

    def _ensure_container(self):
        real_container = PrivilegedContainer(self._container, self._cmd.task_id)
        reuse = real_container.settings.keep_privileged and PrivilegedContainer.name in self.running
        if not reuse:
            self.logger.debug("Creating container %r (%r).", self._container, self._cmd.arch)
            path = self.agentr.resource_sync(self._container.resource_id, fastbone=False)
            container_meta = self.agentr.resource_meta(self._container.resource_id)
            with self.lock:
                if container_meta["attributes"].get("type") != ctt.ImageType.Group.IMAGE:
                    self._unpack_template(self._container, path)
                else:
                    self._mount_template(self._container, path)
                self._mount_container(real_container)
                with system.UserPrivileges():
                    self._patch_container(real_container.root)
        self._origin, self._container = self._container, real_container

        self.logger.debug(
            "%s %r container (%r) based on %r for task #%r",
            "Reusing" if reuse else "Creating", self._container, self._cmd.arch, self._origin, self._cmd.task_id
        )
        from sandbox.client.pinger import PingSandboxServerThread
        PingSandboxServerThread()._kamikadze_thread.ttl = common_config.Registry().client.idle_time * 100
        with system.UserPrivileges():
            if reuse:
                self._stop_container(self._container.name, True)
            self._create_configs_privileged()
            self.restore_system_files()
            self.cleanup_home()  # `sandboxsdk.svn` looks always into "/home/sandbox"
            if not reuse and not system.local_mode():
                def members(tar):
                    for tarinfo in tar:
                        tarinfo.gid = tarinfo.uid = 0
                        tarinfo.uname = tarinfo.gname = "root"
                        yield tarinfo
                self.logger.debug("Extracting sandbox layout to the root's home.")
                with tarfile.open(common_config.Registry().client.sandbox_home_tarball) as tar:
                    tar.extractall(real_container.path("root"), members=members(tar))
            self._start_container(self._container)
            self.running[self._container.name] = ""

    @common_patterns.singleton_property
    def cgroup(self):
        cgroup_name = super(LXCPlatform, self)._cgroup_name(self._cgroup_id())
        return self.make_cgroup(cgroup_name)

    @common_patterns.singleton_property
    def inner_cgroup(self):
        return self.make_cgroup(self._cgroup_name(self._cgroup_id()))

    def cleanup(self):
        self.logger.info("Performing platform cleanup.")
        with system.UserPrivileges():
            self.resume()
            if self._cmd is not None and self._cmd.token:
                self._kill_executor(self._cmd.token)
            self._clean_proc_debris(cgroups=[self.cgroup, self.inner_cgroup])
        self.maintain()

    @classmethod
    def maintain(cls):
        if not Container.settings.keep_privileged and Container.settings.enabled:
            cls.destroy_container()
        return super(PrivilegedLXCPlatform, cls).maintain()

    def terminate(self):
        with system.UserPrivileges():
            self.resume()
            self._clean_proc_debris(cgroups=[self.cgroup, self.inner_cgroup])
        task_workdir = os.path.join(common_config.Registry().client.tasks.data_dir, *ctt.relpath(self._cmd.task_id))
        if not Container.settings.keep_privileged:
            self._destroy_container(self._container, rename_delta=os.path.join(task_workdir, "rootfs_diff"))
        else:
            self.logger.debug("Keep privileged container for future use.")

        from sandbox.client.pinger import PingSandboxServerThread
        PingSandboxServerThread()._kamikadze_thread.ttl = common_config.Registry().client.disk_op_timeout
        with system.PrivilegedSubprocess(("taking ownership on", repr(task_workdir))):
            self._safe_chown(system.UNPRIVILEGED_USER, task_workdir, recursive=True)

    def on_system_error(self):
        pass

    def spawn(self, executor_args):
        self.set_resolv_conf_arg(executor_args)
        executor_args["container"] = self._container_info()

        with system.UserPrivileges():
            self.set_cgroup_prefix()
            executor_args["cgroup"] = self.inner_cgroup and self._deprefix_cgroup(self.inner_cgroup.name)
            cgroup_name = (self.cgroup and self.cgroup.name) or ""
            self.set_container_limits()

        self._cmd.executor_args = executor_args  # must not be executed under root
        self._cmd.save_state()

        # Execute "prepare" stage under root but with dropped privileges.
        env = os.environ.copy()
        env["HOME"] = system.UNPRIVILEGED_USER.home
        env["USER"] = env["LOGNAME"] = system.UNPRIVILEGED_USER.login
        env["LANG"] = "en_US.UTF8"
        env.pop("PYTHONPATH", None)
        env[common_config.Registry.CONFIG_ENV_VAR] = self.config_path
        return system.TaskLiner(
            common_format.obfuscate_token(self._cmd.token),
            self.logger,
            [super(LXCPlatform, self).preexecutor_path, self._cmd.executor_args],
            env,
            "root",
            cgroup_name,
        ), executor_args

    def _fs_command(self, cmd):
        return "/usr/bin/lxc-attach -n {} -- {}".format(
            self._container.name, cmd
        )
