import os
import time
import errno
import shutil
import signal

from sandbox.common import os as common_os
from sandbox.common import itertools as common_itertools

from sandbox.client import system

from . import base


class OSXPlatform(base.Platform):
    GUI_SESSION_PROCESSES = {
        "/System/Library/CoreServices/AirPlayUIAgent.app/Contents/MacOS/AirPlayUIAgent",
        "/System/Library/CoreServices/AppleIDAuthAgent",
        "/System/Library/CoreServices/Bluetooth Setup Assistant.app/Contents/MacOS/Bluetooth Setup Assistant",
        "/System/Library/CoreServices/cloudpaird",
        "/System/Library/CoreServices/cloudphotosd.app/Contents/MacOS/cloudphotosd",
        "/System/Library/CoreServices/CoreServicesUIAgent.app/Contents/MacOS/CoreServicesUIAgent",
        "/System/Library/CoreServices/diagnostics_agent",
        "/System/Library/CoreServices/Dock.app/Contents/MacOS/Dock",
        "/System/Library/CoreServices/Dock.app/Contents/XPCServices/com.apple.dock.extra.xpc/Contents/MacOS/com.apple.dock.extra",  # noqa
        "/System/Library/CoreServices/EscrowSecurityAlert.app/Contents/MacOS/EscrowSecurityAlert",
        "/System/Library/CoreServices/Finder.app/Contents/MacOS/Finder",
        "/System/Library/CoreServices/FolderActionsDispatcher.app/Contents/MacOS/FolderActionsDispatcher",
        "/System/Library/CoreServices/iconservicesagent",
        "/System/Library/CoreServices/mapspushd",
        "/System/Library/CoreServices/Menu Extras/TextInput.menu/Contents/SharedSupport/TISwitcher.app/Contents/MacOS/TISwitcher",  # noqa
        "/System/Library/CoreServices/NotificationCenter.app/Contents/MacOS/NotificationCenter",
        "/System/Library/CoreServices/pbs",
        "/System/Library/CoreServices/RemoteManagement/ARDAgent.app/Contents/MacOS/ARDAgent",
        "/System/Library/CoreServices/RemoteManagement/ScreensharingAgent.bundle/Contents/MacOS/ScreensharingAgent",
        "/System/Library/CoreServices/SafariSupport.bundle/Contents/MacOS/SafariBookmarksSyncAgent",
        "/System/Library/CoreServices/sharedfilelistd",
        "/System/Library/CoreServices/Siri.app/Contents/MacOS/Siri",
        "/System/Library/CoreServices/SocialPushAgent.app/Contents/MacOS/SocialPushAgent",
        "/System/Library/CoreServices/Software Update.app/Contents/Resources/softwareupdate_notify_agent",
        "/System/Library/CoreServices/Spotlight.app/Contents/MacOS/Spotlight",
        "/System/Library/CoreServices/SystemUIServer.app/Contents/MacOS/SystemUIServer",
        "/System/Library/CoreServices/WiFiAgent.app/Contents/MacOS/WiFiAgent",
        "/System/Library/Frameworks/Accounts.framework/Versions/A/Support/accountsd",
        "/System/Library/Frameworks/AddressBook.framework/Executables/ContactsAccountsService",
        "/System/Library/Frameworks/ApplicationServices.framework/Versions/A/Frameworks/ATS.framework/Versions/A/Support/fontd",  # noqa
        "/System/Library/Frameworks/CoreTelephony.framework/Support/CommCenter",
        "/System/Library/Frameworks/CryptoTokenKit.framework/ctkahp.bundle/Contents/MacOS/ctkahp",
        "/System/Library/Frameworks/CryptoTokenKit.framework/ctkd",
        "/System/Library/Frameworks/InputMethodKit.framework/Versions/A/Resources/imklaunchagent",
        "/System/Library/Frameworks/LocalAuthentication.framework/Support/coreauthd",
        "/System/Library/Frameworks/MediaAccessibility.framework/Versions/A/XPCServices/com.apple.accessibility.mediaaccessibilityd.xpc/Contents/MacOS/com.apple.accessibility.mediaaccessibilityd",  # noqa
        "/System/Library/Frameworks/Metal.framework/Versions/A/XPCServices/MTLCompilerService.xpc/Contents/MacOS/MTLCompilerService",  # noqa
        "/System/Library/Frameworks/Metal.framework/Versions/A/XPCServices/MTLCompilerService.xpc/Contents/MacOS/MTLCompilerService",  # noqa
        "/System/Library/Frameworks/PubSub.framework/Versions/A/Resources/PubSubAgent.app/Contents/MacOS/PubSubAgent",
        "/System/Library/Frameworks/QuickLook.framework/Versions/A/Resources/quicklookd.app/Contents/MacOS/quicklookd",
        "/System/Library/Frameworks/Security.framework/Versions/A/MachServices/SecurityAgent.bundle/Contents/MacOS/SecurityAgent",  # noqa
        "/System/Library/Frameworks/Security.framework/Versions/A/Resources/CloudKeychainProxy.bundle/Contents/MacOS/CloudKeychainProxy",  # noqa
        "/System/Library/Frameworks/Security.framework/Versions/A/Resources/KeychainSyncingOverIDSProxy.bundle/Contents/MacOS/KeychainSyncingOverIDSProxy",  # noqa
        "/System/Library/Image Capture/Support/icdd",
        "/System/Library/PrivateFrameworks/AOSKit.framework/Versions/A/XPCServices/com.apple.iCloudHelper.xpc/Contents/MacOS/com.apple.iCloudHelper",  # noqa
        "/System/Library/PrivateFrameworks/AskPermission.framework/Versions/A/Resources/askpermissiond",
        "/System/Library/PrivateFrameworks/AssistantServices.framework/assistantd",
        "/System/Library/PrivateFrameworks/AuthKit.framework/Versions/A/Support/akd",
        "/System/Library/PrivateFrameworks/CacheDelete.framework/deleted",
        "/System/Library/PrivateFrameworks/CalendarAgent.framework/Executables/CalendarAgent",
        "/System/Library/PrivateFrameworks/CalendarNotification.framework/Versions/A/XPCServices/CalNCService.xpc/Contents/MacOS/CalNCService",  # noqa
        "/System/Library/PrivateFrameworks/CallHistory.framework/Support/CallHistoryPluginHelper",
        "/System/Library/PrivateFrameworks/CallHistory.framework/Support/CallHistorySyncHelper",
        "/System/Library/PrivateFrameworks/CloudDocsDaemon.framework/Versions/A/Support/bird",
        "/System/Library/PrivateFrameworks/CloudKitDaemon.framework/Support/cloudd",
        "/System/Library/PrivateFrameworks/CloudPhotoServices.framework/Versions/A/Frameworks/CloudPhotosConfigurationXPC.framework/Versions/A/XPCServices/com.apple.CloudPhotosConfiguration.xpc/Contents/MacOS/com.apple.CloudPhotosConfiguration",  # noqa
        "/System/Library/PrivateFrameworks/CommerceKit.framework/Versions/A/Resources/LaterAgent.app/Contents/MacOS/LaterAgent",  # noqa
        "/System/Library/PrivateFrameworks/CommerceKit.framework/Versions/A/Resources/storeaccountd",
        "/System/Library/PrivateFrameworks/CommerceKit.framework/Versions/A/Resources/storeassetd",
        "/System/Library/PrivateFrameworks/CommerceKit.framework/Versions/A/Resources/storedownloadd",
        "/System/Library/PrivateFrameworks/CommerceKit.framework/Versions/A/Resources/storelegacy",
        "/System/Library/PrivateFrameworks/CommerceKit.framework/Versions/A/XPCServices/com.apple.CommerceKit.TransactionService.xpc/Contents/MacOS/com.apple.CommerceKit.TransactionService",  # noqa
        "/System/Library/PrivateFrameworks/ContactsAgent.framework/Executables/ContactsAgent",
        "/System/Library/PrivateFrameworks/CoreFollowUp.framework/Versions/A/Support/followupd",
        "/System/Library/PrivateFrameworks/CoreParsec.framework/parsecd",
        "/System/Library/PrivateFrameworks/CoreSuggestions.framework/Versions/A/Support/suggestd",
        "/System/Library/PrivateFrameworks/CoreSuggestions.framework/Versions/A/Support/suggestd",
        "/System/Library/PrivateFrameworks/CoreWLANKit.framework/Versions/A/XPCServices/WiFiProxy.xpc/Contents/MacOS/WiFiProxy",  # noqa
        "/System/Library/PrivateFrameworks/GeoServices.framework/geodMachServiceBridge",
        "/System/Library/PrivateFrameworks/GeoServices.framework/Versions/A/XPCServices/com.apple.geod.xpc/Contents/MacOS/com.apple.geod",  # noqa
        "/System/Library/PrivateFrameworks/HelpData.framework/Versions/A/Resources/helpd",
        "/System/Library/PrivateFrameworks/IDS.framework/identityservicesd.app/Contents/MacOS/identityservicesd",
        "/System/Library/PrivateFrameworks/IMCore.framework/imagent.app/Contents/MacOS/imagent",
        "/System/Library/PrivateFrameworks/IMDPersistence.framework/XPCServices/IMDPersistenceAgent.xpc/Contents/MacOS/IMDPersistenceAgent",  # noqa
        "/System/Library/PrivateFrameworks/IMFoundation.framework/XPCServices/IMRemoteURLConnectionAgent.xpc/Contents/MacOS/IMRemoteURLConnectionAgent",  # noqa
        "/System/Library/PrivateFrameworks/IMFoundation.framework/XPCServices/IMRemoteURLConnectionAgent.xpc/Contents/MacOS/IMRemoteURLConnectionAgent",  # noqa
        "/System/Library/PrivateFrameworks/MediaRemote.framework/Support/mediaremoteagent",
        "/System/Library/PrivateFrameworks/MessagesKit.framework/Versions/A/Resources/soagent.app/Contents/MacOS/soagent",  # noqa
        "/System/Library/PrivateFrameworks/Noticeboard.framework/Versions/A/Resources/nbagent.app/Contents/MacOS/nbagent",  # noqa
        "/System/Library/PrivateFrameworks/PassKitCore.framework/passd",
        "/System/Library/PrivateFrameworks/PhotoAnalysis.framework/Versions/A/Support/photoanalysisd",
        "/System/Library/PrivateFrameworks/PhotoLibraryPrivate.framework/Versions/A/Support/photolibraryd",
        "/System/Library/PrivateFrameworks/QuickLookThumbnailing.framework/Support/com.apple.quicklook.ThumbnailsAgent",  # noqa
        "/System/Library/PrivateFrameworks/StorageManagement.framework/Versions/A/Resources/diskspaced",
        "/System/Library/PrivateFrameworks/SyncedDefaults.framework/Support/syncdefaultsd",
        "/System/Library/PrivateFrameworks/TCC.framework/Versions/A/Resources/tccd",
        "/System/Library/PrivateFrameworks/TelephonyUtilities.framework/callservicesd",
        "/System/Library/PrivateFrameworks/UserActivity.framework/Agents/useractivityd",
        "/System/Library/PrivateFrameworks/ViewBridge.framework/Versions/A/XPCServices/ViewBridgeAuxiliary.xpc/Contents/MacOS/ViewBridgeAuxiliary",  # noqa
        "/usr/libexec/AppSandbox/ContainerRepairAgent",
        "/usr/libexec/fmfd",
        "/usr/libexec/lsd",
        "/usr/libexec/networkserviceproxy",
        "/usr/libexec/nsurlsessiond",
        "/usr/libexec/nsurlstoraged",
        "/usr/libexec/pboard",
        "/usr/libexec/pkd",
        "/usr/libexec/secd",
        "/usr/libexec/secinitd",
        "/usr/libexec/securityd_service",
        "/usr/libexec/sharingd",
        "/usr/libexec/spindump_agent",
        "/usr/libexec/swcd",
        "/usr/libexec/trustd",
        "/usr/libexec/UserEventAgent",
        "/usr/libexec/videosubscriptionsd",
        "/usr/libexec/WiFiVelocityAgent",
        "/usr/sbin/cfprefsd",
        "/usr/sbin/distnoted",
        "/usr/sbin/securityd",
        "/usr/sbin/usernoted",
    }

    ROZETTA_CACHE_PATH = "/var/db/oah"
    ROZETTA_CACHE_TTL = 3 * 24 * 3600  # in seconds

    def _signal_user_processes(self, user, sig, wait=False):
        self.logger.debug("Sending signal '%d' to all processes of user %r", sig, user.login)
        pids = []
        for p in common_os.processes():
            # Skip other users processes
            if p.uid != user.uid:
                continue
            # Skip gui session services
            if p.exe in self.GUI_SESSION_PROCESSES:
                continue

            try:
                self.logger.debug("Send signal '%d' to process #%d: %s", sig, p.pid, p.exe)
                os.kill(p.pid, sig)
                pids.append(p.pid)
            except OSError as e:
                self.logger.error("Failed to signal process #%d: %s", p.pid, e)
        if wait:
            for _ in common_itertools.progressive_yielder(.1, 1, 5):
                for pid in list(pids):
                    try:
                        os.kill(pid, 0)
                    except OSError as ex:
                        if ex.errno == errno.ESRCH:
                            pids.remove(pid)
                if not pids:
                    break
            else:
                self.logger.error("Failed to kill processes: %s", pids)

    def _clean_proc_debris(self):
        if system.local_mode():
            return

        self.logger.debug("Killing all processes of user %r", system.UNPRIVILEGED_USER.login)
        self._signal_user_processes(system.UNPRIVILEGED_USER, signal.SIGKILL, wait=True)

    def send_signal(self, sig):
        if system.local_mode():
            if self._cmd.liner:
                with system.UserPrivileges():
                    os.kill(self._cmd.liner.pid, sig)
        else:
            self._signal_user_processes(system.UNPRIVILEGED_USER, sig)

    @property
    def _ps_command(self):
        return "/bin/ps wwux"

    def cleanup(self):
        self.logger.info("Performing platform cleanup.")

        for _ in range(3):
            try:
                with system.PrivilegedSubprocess(
                    ("platform cleanup", self._cmd is not None and self._cmd.token), check_status=True
                ):
                    if self._cmd is not None and self._cmd.token:
                        self._kill_executor(self._cmd.token)
                    self._clean_proc_debris()
                    self._clean_fs_debris()

                with system.PrivilegedSubprocess(
                    ("platform cleanup (restore home)", self._cmd is not None and self._cmd.token), check_status=True
                ):
                    self._restore_home(system.UNPRIVILEGED_USER.home, job_logger=self.logger)
                break
            except common_os.SubprocessAborted:
                self.logger.exception("Exception in platform cleanup on try %s", _ + 1)

    @classmethod
    def cleanup_rozetta_cache(cls):
        now = time.time()
        cache_it = iter(os.walk(cls.ROZETTA_CACHE_PATH))
        root, dirs, _ = next(cache_it, (None, (), ()))
        for dir_name in dirs:
            cache_it = iter(os.walk(os.path.join(root, dir_name)))
            root, dirs, _ = next(cache_it, (None, (), ()))
            for subdir in dirs:
                full_path = os.path.join(root, subdir)
                stat = os.stat(full_path)
                if now - stat.st_ctime > cls.ROZETTA_CACHE_TTL:
                    shutil.rmtree(full_path)
