from __future__ import absolute_import

import os
import shutil
import struct
import socket
import logging
import weakref
import textwrap
import threading

import datetime as dt
import subprocess as sp

import netaddr
try:
    import porto
except ImportError:
    porto = None

from sandbox.common import os as common_os
from sandbox.common import fs as common_fs
from sandbox.common import hash as common_hash
from sandbox.common import config as common_config
from sandbox.common import format as common_format
from sandbox.common import random as common_random
from sandbox.common import package as common_package
from sandbox.common import patterns as common_patterns

import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc

import sandbox.agentr.client

from sandbox.client import errors, network, system

from . import base, lxc


logger = logging.getLogger(__name__)


class PrivilegedSocket(socket.socket):
    def connect(self, *args, **kws):
        with system.UserPrivileges():
            return super(PrivilegedSocket, self).connect(*args, **kws)


class PortoBind(object):
    def __init__(self, src, dst=None, mode="ro"):
        self.src = src
        self.dst = dst or src
        self.mode = mode

    def __str__(self):
        return "{} {} {}".format(self.src, self.dst, self.mode)


class PortoLayerDefinition(object):
    PORTO_VOLUMES_ROOT = "/var/porto"

    def __init__(self, resource_id):
        self.resource_id = resource_id
        self.path = os.path.join(self.PORTO_VOLUMES_ROOT, self.name, "rootfs")
        self.mounted = os.path.exists(self.path)

    @property
    def name(self):
        return "sbr:{}".format(self.resource_id)

    @property
    def mount_path(self):
        return os.path.dirname(self.path)

    def __repr__(self):
        return "<Layer {}>".format(self.name)

    @classmethod
    def from_string(cls, data):
        _, rid = data.split(":", 1)
        rid = rid.split("/", 1)[0]
        return cls(int(rid))

    def __mount(self, porto_client, image_path, namespace, lock):
        mount_path = self.mount_path
        if self.mounted:
            return porto_client.FindVolume(mount_path)
        with lock:
            try:
                volume = porto_client.FindVolume(mount_path)
            except porto.exceptions.VolumeNotFound:
                if not os.path.exists(mount_path):
                    with common_os.Capabilities(common_os.Capabilities.Cap.Bits.CAP_DAC_OVERRIDE):
                        os.makedirs(mount_path)
                volume = porto_client.CreateVolume(
                    path=mount_path, storage=mount_path,
                    layers=[image_path],
                    backend="squash",
                    read_only="true",
                    private_value=namespace,
                )
            self.mounted = True
            return volume

    def import_layer(self, agentr, porto_client, namespace, logger_, lock):
        logger_.debug("Import porto layer '%s' from resource #%s", self.name, self.resource_id)
        resource_path = agentr.resource_sync(self.resource_id)
        if resource_path.endswith(".squashfs"):
            self.__mount(porto_client, resource_path, namespace, lock)
            return
        try:
            porto_client.FindLayer(self.name)
        except porto.exceptions.LayerNotFound:
            try:
                layer = porto_client.ImportLayer(self.name, resource_path)
            except porto.exceptions.LayerAlreadyExists:
                logger_.debug("Layer '%s' already exists", self.name)
            else:
                layer.SetPrivate(namespace)


class PortoVolumeWrapper(object):
    def __init__(self, volume):
        self.volume = volume

    def path(self, *path):
        """volume.path("/place/foo/bar"") => ${volume.path}/place/foo/bar"""
        return os.path.join(self.volume.path, *[_.lstrip("/") for _ in path])

    @property
    def root(self):
        return self.volume.path


class PortoContainerDefinition(object):
    def __init__(self, name, layers, ip_address=None):
        self.name = name
        self.layers = layers
        self.ip_address = ip_address
        self.privileged = False

        self.container = None
        self.last_usage = None
        self._locked = False

    def match_layers(self, layers):
        if len(self.layers) != len(layers):
            return False
        for l, r in zip(self.layers, layers):
            if l.resource_id != r.resource_id:
                return False
        return True

    def is_locked(self):
        return self._locked

    def lock(self):
        self._locked = True
        self.last_usage = dt.datetime.utcnow()

    def unlock(self):
        self._locked = False
        self.last_usage = dt.datetime.utcnow()


class PortoContainerRegistryTls(threading.local):
    def __init__(self):
        self.connection = None
        self.logger = None


class PortoContainerRegistry(object):
    __metaclass__ = common_patterns.ThreadSafeSingletonMeta

    MAX_UNUSED_CONTAINERS = 0  # Allow to cache up to 8 unused containers
    UNUSED_LAYER_TTL = 2  # days

    def __init__(self):
        self._tls = PortoContainerRegistryTls()
        self._lock = threading.Lock()

        # Currently owned containers
        self._containers = {}  # name -> PortoContainerDefinition
        self._host_ip_addr = None
        self._host_mtn_trunk_addr = None
        self.default_project_id = common_config.Registry().client.porto.network.default_project_id

    @property
    def logger(self):
        return self._tls.logger and self._tls.logger() or logger

    @logger.setter
    def logger(self, job_logger):
        self._tls.logger = weakref.ref(job_logger)

    @property
    def porto_client(self):
        if self._tls.connection is None:
            self._tls.connection = porto.Connection(socket_constructor=PrivilegedSocket)
            self._tls.connection.connect()
        return self._tls.connection

    @property
    def namespace(self):
        if system.local_mode():
            # For local sandbox add client port to allow several installations
            return "sb-{}".format(common_config.Registry().client.port)
        return "sb"

    def destoy_container(self, name):
        try:
            self.porto_client.Destroy(name)
        except porto.exceptions.ContainerDoesNotExist as e:
            self.logger.error("Can't destroy container: %s", e)
        else:
            self.logger.debug("Destroyed container '%s'", name)
        self._containers.pop(name, None)

    def _ensure_namespace_container(self):
        container_name = ""
        for part in self.namespace.split("/"):
            container_name = "{}/{}".format(container_name, part).lstrip("/")
            try:
                container = self.porto_client.Create(container_name)
            except porto.exceptions.ContainerAlreadyExists:
                pass
            else:
                self.logger.debug("Namespace container '%s' doesn't exist, create it", container_name)
                container.Start()

        container = self.porto_client.Find(container_name)
        devices_path = dict(map(lambda _: _.split(": "), container.GetProperty("cgroups").split("; ")))["devices"]
        if os.path.exists(devices_path):
            cap = common_os.Capabilities(
                common_os.Capabilities.Cap.Bits.CAP_DAC_OVERRIDE |
                common_os.Capabilities.Cap.Bits.CAP_SYS_ADMIN
            )
            for dev in ("b 7:* rwm", "b 259:* rwm"):
                with cap, open(os.path.join(devices_path, "devices.allow"), "w") as f:
                    f.write(dev)

    def initialize(self):
        self.logger.info("Loading PortoRegistry")

        self._ensure_namespace_container()
        self.logger.info("Using namespace '%s', look for children", self.namespace)

        for container in self.porto_client.ListContainers():
            if container.GetProperty("private") != self.namespace:
                # Skip unknown containers
                continue

            if container.GetProperty("state") not in ["running", "meta"]:
                self.logger.warning(
                    "Container '%s' is in state '%s', skip it",
                    container.name, container.GetProperty("state")
                )
                continue

            try:
                volume = self.porto_client.FindVolume(container.GetProperty("root"))
            except porto.exceptions.VolumeNotFound as ex:
                self.logger.warning("Can't find volume for container '%s', gonna destroy it\n%s", container.name, ex)
                self.destoy_container(container.name)
                continue

            layers = [
                PortoLayerDefinition.from_string(layer.name)
                for layer in volume.layers
            ]

            ip_address = netaddr.IPAddress(container.GetProperty("ip").split()[1])
            if ip_address.is_private():
                ip_address = None  # this is NAT

            running_time = int(container.GetProperty("time"))
            created = dt.datetime.now() - dt.timedelta(seconds=running_time)
            self.logger.info(
                "Found container '%s', layers: %s, ip: %s, created: %s",
                container.name, layers, ip_address, common_format.dt2str(created),
            )
            container_def = PortoContainerDefinition(container.name, layers, ip_address)
            container_def.container = container
            # We cannot get real last-usage of container here. But creation time seems reasonable.
            container_def.last_usage = dt.datetime.utcnow() - dt.timedelta(seconds=running_time)
            self._containers[container.name] = container_def

        # Get host ip address
        host_nets = sp.check_output(["ip", "-o", "-6", "addr", "sh", "scope", "global"]).splitlines()
        for iface in host_nets:
            iface = iface.split()
            if "vlan" in iface[1] or iface[3].startswith("f"):
                continue

            self._host_ip_addr = netaddr.IPNetwork(iface[3]).ip
            self.logger.debug("Detected host IP address '%s' on dev '%s'", self._host_ip_addr, iface[1])
            break
        else:
            raise Exception("Can't detect host ip address")

        # Get address of "mtn-trunk" vlan for veth network
        if common_config.Registry().client.porto.network.type == ctm.Network.Type.VETH:
            for iface in host_nets:
                iface = iface.split()
                if common_config.Registry().client.porto.network.veth_vlan == iface[1]:
                    self._host_mtn_trunk_addr = netaddr.IPNetwork(iface[3]).ip
                    break
            else:
                raise Exception("Can't find veth_vlan for mtn-trunk network")

    def destroy_namespace(self):
        self.logger.info("Destroy ALL porto containers with namespace '%s'", self.namespace)
        self.destoy_container(self.namespace)

    def remove_unused_layers(self, age=None):
        """
        Remove layers, that hasn't been used for at least `age` days.
        """
        if age is None:
            age = self.UNUSED_LAYER_TTL
        self.logger.info("Remove porto layers, that hasn't been used for %d days", age)
        for layer in self.porto_client.ListLayers():
            if layer.GetPrivate() != self.namespace:
                # Skip unknown layers
                continue

            if layer.last_usage / float(3600 * 24) < age:
                # Layer has been used recently
                continue

            try:
                layer.Remove()
            except porto.exceptions.Busy:
                # Layer is used by some volume, skip it
                pass
            else:
                self.logger.info("Removed layer '%s', last used %ds ago", layer.name, layer.last_usage)

    def import_porto_layers(self, layers, agentr):
        for layer_def in layers:
            layer_def.import_layer(agentr, self.porto_client, self.namespace, self.logger, self._lock)

    @staticmethod
    def _patch_passwd(fname, user, newline):
        tmp_fname = fname + "~"
        marker = user + ":"
        found = False
        with open(fname, "r") as old_fh, open(tmp_fname, "w") as new_fh:
            for line in old_fh:
                if line.startswith(marker):
                    line = newline
                    found = True
                new_fh.write(line.strip() + "\n")
            if not found:
                new_fh.write(newline + "\n")
        os.rename(tmp_fname, fname)

    def _hack_volume(self, v):
        # TODO: This is copypaste from LXCPlatform, refactor later

        # Create directory for service stuff (venv, code, tasks)
        path = v.path("/sandbox")
        self.logger.debug("Creating '%s'", path)
        os.makedirs(path, 0o755)
        os.chown(path, system.SERVICE_USER.uid, system.SERVICE_USER.gid)

        for user in (system.UNPRIVILEGED_USER, system.SERVICE_USER):
            self._patch_passwd(
                v.path("/etc/passwd"),
                user.login, "{}:x:{}:{}:{}:{}:/bin/bash".format(user.login, user.uid, user.gid, user.group, user.home)
            )
        self._patch_passwd(
            v.path("/etc/group"),
            system.SERVICE_USER.group, "{}:x:{}:".format(system.SERVICE_USER.group, system.SERVICE_USER.gid)
        )

        shell_path = v.path("/sandbox/shell.sh")
        self.logger.debug("Create webshell script in %s", shell_path)
        with open(shell_path, 'w') as f:
            f.write('#!/bin/sh\nsudo -u {} bash'.format(system.UNPRIVILEGED_USER.login))
        os.chmod(shell_path, 0o755)

        if not os.path.lexists(v.path("/Berkanavt")):
            os.symlink("/place/berkanavt", v.path("/Berkanavt"))

        origin = lxc.LXCPlatform._skynet_link
        cnt_origin = v.path("/skynet")
        dst = v.path(origin.lstrip(os.path.sep))
        self.logger.debug("Mirroring skynet link pointing to '%s' at '%s'", dst, cnt_origin)
        if os.path.lexists(cnt_origin):
            os.unlink(cnt_origin)
        os.symlink(origin, cnt_origin)

        self.logger.debug("Disabling cron at '%s'", v.root)
        for f in ("/etc/init.d/cron", "/etc/init/cron.conf", "/lib/systemd/system/cron.service"):
            if os.path.exists(v.path(f)):
                os.unlink(v.path(f))

        # this script sets cpu scaling_governor to ondemand, but we want PERFORMANCE
        init_d_ondemand = v.path("/etc/init.d/ondemand")
        if os.path.exists(init_d_ondemand):
            self.logger.debug("Found 'etc/init.d/ondemand' inside container, remove it")
            os.unlink(init_d_ondemand)

        # this script cleans tmp and remounts it as tmpfs
        init_mounted_tmp = v.path("/etc/init/mounted-tmp.conf")
        if common_config.Registry().client.porto.mount_tmp_dir and os.path.exists(init_mounted_tmp):
            self.logger.debug("Mounting host tmp directory is enabled, remove '/etc/init/mounted-tmp.conf'")
            os.unlink(init_mounted_tmp)

    def _create_data_volume(self, vol_path):
        try:
            if not os.path.exists(vol_path):
                os.makedirs(vol_path)
            return self.porto_client.CreateVolume(path=vol_path, storage=vol_path, backend="bind")
        except porto.exceptions.InvalidPath:
            self.logger.error("Can't create Volume with bind backend in %s", vol_path)
            raise

    def _find_volume(self, local_path):
        try:
            return self.porto_client.FindVolume(os.path.abspath(local_path))
        except porto.exceptions.VolumeNotFound:
            return False

    def _nested_layers(self, parent_dir):
        """
        Return all existing volumes with layers in parent_dir
        parent_dir: str
        return: list(porto.api.Volume, )
        """
        volumes_nested = []
        for volume in self.porto_client.ListVolumes():
            if volume.properties.get("layers", "").startswith(parent_dir):
                volumes_nested.append(volume)
        return volumes_nested

    def _safe_delete_volume(self, volume):
        """
        Stop and delete all containers linked with volume and volume.
        volume: porto.api.Volume
        """
        for container in volume.GetContainers():
            try:
                self.logger.debug("Stopping and destroying container %s in order to free volume %s", container, volume)
                container.Stop()
                container.Destroy()
            except porto.exceptions.ContainerDoesNotExist:
                pass
        try:
            volume.Destroy()
        except porto.exceptions.VolumeNotFound:
            pass

    def _get_bind_volume(self, vol):
        if not os.path.isdir(vol):
            self.logger.error("Data volume '%s' should be a directory", vol)
            return None
        volume = self._find_volume(vol)
        if not volume:
            nested_layers = self._nested_layers(vol)
            if nested_layers:
                self.logger.warning(
                    "Found volumes with layers nested in %s: %s",
                    vol,
                    [l.properties.get("layers") for l in nested_layers]
                )
                for l in nested_layers:
                    self._safe_delete_volume(l)
            volume = self._create_data_volume(vol)
        return volume

    def _get_container_binds(self):
        binds = [
            PortoBind(common_config.Registry().client.log.root, mode="rw"),
            PortoBind(common_config.Registry().client.dirs.run, mode="rw"),
            PortoBind(lxc.LXCPlatform._skynet_root, mode="ro"),

            PortoBind("/place/vartmp", mode="rw"),
            PortoBind("/place/coredumps", mode="rw"),
        ]

        if system.local_mode():
            binds.append(PortoBind(system.SERVICE_USER.home, mode="rw"))

        if common_config.Registry().client.porto.mount_tmp_dir:
            binds.append(PortoBind(common_config.Registry().client.porto.dirs.tmp, mode="rw"))

        return ";".join(str(_) for _ in binds)

    def _get_container_hostname(self, container_def):
        return "{}.sandbox".format(container_def.name[len(self.namespace) + 1:])

    def _get_container_etc_hosts(self, container_def):
        etc_hosts = textwrap.dedent("""\
            # Generated by Sandbox client
            # The following block copied from the host
            {host}
            # The following block is individual for the container
            {container}
        """)

        with open("/etc/hosts", "rb") as f:
            host = f.read()

        container = "{}\t{}\n".format(
            container_def.ip_address if container_def.ip_address else self._host_ip_addr,
            self._get_container_hostname(container_def)
        )

        return etc_hosts.format(host=host, container=container)

    def _link_layers(self, container_def, container):
        with self._lock:
            for layer in container_def.layers:
                if layer.mounted:
                    volume = self.porto_client.FindVolume(layer.mount_path)
                    volume.Link(container)
                    # Unlink all non sb containers
                    # This way layer volume will be destroyed after all linked container are destroyed
                    for linked_container in volume.containers:
                        if linked_container.GetProperty("private") != self.namespace:
                            volume.Unlink(linked_container)

    def _create_base_container(self, container_def, privileged_task_id=None):
        # Create volume and container
        # This is a cheap operation as layers are already imported
        with self._lock:
            v = PortoVolumeWrapper(
                self.porto_client.CreateVolume(
                    layers=[(layer.path if layer.mounted else layer.name) for layer in container_def.layers],
                    timeout=60
                )
            )
        c = self.porto_client.Create(container_def.name)
        c.SetProperty("private", self.namespace)

        # Link volume to container and unlink it from root
        # This way volume will be destroyed along with the container
        self._link_layers(container_def, c)
        v.volume.Link(c)
        try:
            v.volume.Unlink()
        except porto.exceptions.VolumeNotLinked as ex:
            self.logger.warning("Failed to unlink volume: %s", ex)

        with system.PrivilegedSubprocess(("prepare base volume", v.root)):
            # Apply some general hacks (ensure users, disable cron, etc)
            self._hack_volume(v)

        # Setup container
        c.SetProperty("virt_mode", "app")
        c.SetProperty("command", "/bin/sleep infinity")
        c.SetProperty("cwd", "/")
        c.SetProperty("root", v.root)

        c.SetProperty("bind", self._get_container_binds())
        c.SetProperty("hostname", self._get_container_hostname(container_def))
        c.SetProperty("etc_hosts", self._get_container_etc_hosts(container_def))

        if common_config.Registry().client.porto.network.type == ctm.Network.Type.L3:
            c.SetProperty("net", "L3 eth0")
            c.SetProperty("ip", "eth0 {}".format(container_def.ip_address))
        elif common_config.Registry().client.porto.network.type == ctm.Network.Type.VETH:
            c.SetProperty("net", "L3 veth")
            c.SetProperty("ip", "veth {}".format(container_def.ip_address))
        elif common_config.Registry().client.porto.network.type == ctm.Network.Type.NAT:
            c.SetProperty("net", "NAT")

        # Fire!
        c.Start()

        # Link data volume aka /place/sandbox-data
        with self._lock:
            data_volume = self._get_bind_volume(common_config.Registry().client.dirs.data)
            try:
                data_volume.Link(c, data_volume.path, read_only=bool(privileged_task_id))
            except porto.exceptions.VolumeAlreadyLinked as ex:
                self.logger.warning("Failed to link volume: %s", ex)
            if privileged_task_id:
                with system.UserPrivileges():
                    workdir = PortoPlatform.ensure_taskdir(privileged_task_id)
                workdir_volume = self.porto_client.CreateVolume(
                    path=workdir, storage=workdir, backend="bind", containers=c.name
                )
                try:
                    workdir_volume.Link(c, workdir_volume.path)
                except porto.exceptions.VolumeAlreadyLinked as ex:
                    self.logger.warning("Failed to link volume: %s", ex)
                with system.UserPrivileges():
                    arc_vcs_storage = PortoPlatform.empty_dir(os.path.join(workdir, ".vcs", "arc"))
                arc_vcs_path = os.path.join(data_volume.path, "srcdir", "arc_vcs")
                self.porto_client.CreateVolume(
                    path=v.path(arc_vcs_path), storage=arc_vcs_storage, layers=[arc_vcs_path], containers=c.name
                )

        return c

    def _find_free_container_nums(self):
        locked_bytes = {
            c.ip_address.value & 0xFF
            for c in self._containers.values()
        }
        for last_byte in range(1, 256):
            if last_byte not in locked_bytes:
                return last_byte
        else:
            raise errors.ExecutorFailed("Can't allocate ip address for container")

    def _gen_ip_address(self, name):
        # Generate ip address based on selected network configuration
        if common_config.Registry().client.porto.network.type == ctm.Network.Type.L3:
            # Find free last byte and generate ip address
            last_byte = self._find_free_container_nums()
            # IP generation:
            # * convert last 4 bytes to 3 bytes via crc24
            # * use container number as last byte
            data = common_hash.crc24(bytearray(struct.pack("!HH", *self._host_ip_addr.words[-2:])))
            hi = data >> 8
            lo = ((data & 0xFF) << 8) + last_byte
            ip_as_number = netaddr.strategy.ipv6.words_to_int(self._host_ip_addr.words[:-2] + (hi, lo))
            ip_address = netaddr.IPAddress(ip_as_number)
            self.logger.info("Allocated ip address %s for '%s'", ip_address, name)
            return ip_address
        elif common_config.Registry().client.porto.network.type == ctm.Network.Type.VETH:
            last_byte = self._find_free_container_nums()
            # Get last 32 bits of ip6 address from untagged network + container num aka last_byte
            data = common_hash.crc24(bytearray(struct.pack("!HH", *self._host_ip_addr.words[-2:])))
            hi = data >> 8
            lo = ((data & 0xFF) << 8) + last_byte
            # Make project id 32bit binary
            bin_prj_id = bin(int(self.default_project_id, 16))[2:].zfill(32)
            # Split it for 2x16 bit "words"
            prj_lo = int(bin_prj_id[:16], 2)
            prj_hi = int(bin_prj_id[16:], 2)
            # Finally squash all together: 64 bits for YNDX + DC + SWITH + PORT network.
            # 32 bit for project id and 32 bits for "uniq" part include container num
            ip_as_number = netaddr.strategy.ipv6.words_to_int(self._host_mtn_trunk_addr.words[:-4] +
                                                              (prj_lo, prj_hi, hi, lo))
            ip_address = netaddr.IPAddress(ip_as_number)
            self.logger.info("Allocated ip address %s for '%s'", ip_address, name)
            return ip_address
        elif common_config.Registry().client.porto.network.type == ctm.Network.Type.NAT:
            return None  # this is NAT

    def _get_name(self):
        while True:
            name = common_random.random_string(4)
            if name not in self._containers:
                return name

    def _allocate_base_container(self, layers, privileged=False):
        if privileged:
            with self._lock:
                name = "{}/{}-p".format(self.namespace, self._get_name())
                ip_address = self._gen_ip_address(name)
                container_def = PortoContainerDefinition(name, layers, ip_address)
                container_def.privileged = True
                self._containers[container_def.name] = container_def

                container_def.lock()
                return container_def

        with self._lock:
            for container_def in self._containers.values():
                if container_def.is_locked():
                    continue
                if not container_def.match_layers(layers):
                    continue
                # Found suitable unused container
                break

            # No suitable containers, create new one.
            else:
                # First remove extra containers
                while (
                    len(self._containers) > common_config.Registry().client.max_job_slots + self.MAX_UNUSED_CONTAINERS
                ):
                    # Find the oldest unlocked container
                    not_locked = filter(lambda c: not c.is_locked(), self._containers.values())
                    if not not_locked:
                        break
                    container_def = min(
                        not_locked,
                        key=lambda c: c.last_usage or dt.datetime.min,
                    )
                    self.logger.info("Going to destroy '%s', reason: too many containers", container_def.name)
                    if container_def.container:
                        # TODO: Destroy porto-containter asynchronously
                        self.destoy_container(container_def.name)
                    else:
                        self.logger.warning("Base container '%s' defined, but doesn't exist", container_def.name)

                # Generate random cool name
                name = "{}/{}".format(self.namespace, self._get_name())

                ip_address = self._gen_ip_address(name)
                container_def = PortoContainerDefinition(name, layers, ip_address)
                self._containers[container_def.name] = container_def

            container_def.lock()
            return container_def

    def get_base_container(self, layers, privileged_task_id=None):
        # Allocate base container
        container_def = self._allocate_base_container(layers, privileged=bool(privileged_task_id))

        if container_def.container is None or privileged_task_id:
            self.logger.debug("Create and run base container '%s'", container_def.name)
            container_def.container = self._create_base_container(container_def, privileged_task_id=privileged_task_id)
        else:
            self.logger.debug("Found suitable base container '%s'", container_def.name)

        return container_def.name

    def lock_base_container(self, name):
        with self._lock:
            if name in self._containers:
                self._containers[name].lock()
            else:
                self.logger.warning("Can't lock container %s, it doesn't exist", name)

    def unlock_base_container(self, name):
        with self._lock:
            if name in self._containers:
                self._containers[name].unlock()


class PortoNetwork(object):
    NAT_FIRST_IPV6 = "fec0::42:1"
    IP6TABLES_RULE = ["POSTROUTING", "-s", "fec0::42:0/120", "-j", "MASQUERADE"]
    PORTOD_CONFIG_NAT_PATH = "/etc/portod.conf.d/20_sandbox-nat.conf"

    @classmethod
    def setup_porto_network(cls):
        if common_config.Registry().client.porto.network.type == ctm.Network.Type.NAT:
            cls._setup_porto_network_nat()
        elif common_config.Registry().client.porto.network.type == ctm.Network.Type.L3:
            cls._setup_porto_network_l3()
        elif common_config.Registry().client.porto.network.type == ctm.Network.Type.VETH:
            pass
        else:
            raise Exception("Unknown porto network type: {}".format(common_config.Registry().client.porto.network.type))

    @classmethod
    def _ensure_portod_nat_config(cls):
        """
        Ensure portod has up-to-date config for NAT networking
        """
        config_value = textwrap.dedent("""\
            network {{
                nat_first_ipv6: "{nat_first_ipv6}"
                nat_count: 255
            }}
        """).format(nat_first_ipv6=cls.NAT_FIRST_IPV6)

        if os.path.exists(cls.PORTOD_CONFIG_NAT_PATH):
            if open(cls.PORTOD_CONFIG_NAT_PATH).read() == config_value:
                # Config exists and up-to-date
                return
            logger.info("Existing portod NAT-config is different from ours")

        logger.info("Write portod NAT-config at %s", cls.PORTOD_CONFIG_NAT_PATH)
        common_fs.make_folder(os.path.dirname(cls.PORTOD_CONFIG_NAT_PATH), log=logger)
        with open(cls.PORTOD_CONFIG_NAT_PATH, "w") as f:
            f.write(config_value)

        logger.info("Reload yandex-porto service to apply configuration changes")
        sp.check_call(["service", "yandex-porto", "reload"])

    @classmethod
    def _setup_porto_network_nat(cls):
        with system.PrivilegedSubprocess("initial network configuration for NAT"):
            network.ensure_ip6tables_rule(cls.IP6TABLES_RULE)
            network.ensure_sysctl_forwarding()
            cls._ensure_portod_nat_config()

    @classmethod
    def _setup_porto_network_l3(cls):
        with system.PrivilegedSubprocess("initial network configuration for L3"):
            network.ensure_sysctl_forwarding()


class PortoPlatformTls(threading.local):
    def __init__(self):
        self.connection = None


class PortoPlatform(base.Platform, base.ResolvConfMixin):
    SERIALIZABLE_ATTRS = base.Platform.SERIALIZABLE_ATTRS + (
        "_container_props",
        "_base_container_name",
    )

    RAMDRIVE_PATH = "/ramdrive"

    executable = "/home/zomb-sandbox/venv/bin/python"
    executor_path = "/home/zomb-sandbox/client/sandbox/bin/executor.py"
    tasks_dir = "/home/zomb-sandbox/tasks"

    _update_tasks_lock = threading.RLock()
    _ramdrive_lock = threading.RLock()

    def __init__(self, cmd):
        super(PortoPlatform, self).__init__(cmd)
        self._container_props = self._cmd.args.pop("container", None)  # None for reset cmd
        self._cmd.arch = self._container_props["alias"] if self._container_props else None

        # Parent container for executor
        self._base_container_name = None

    @common_patterns.singleton_property
    def base_container(self):
        if self._base_container_name is None:
            return None

        try:
            container = PortoContainerRegistry().porto_client.Find(self._base_container_name)
        except porto.exceptions.ContainerDoesNotExist as e:
            raise errors.ExecutorFailed("Can't find container %s: %s", self._base_container_name, e)

        return container

    @common_patterns.singleton_property
    def base_volume(self):
        if self.base_container is None:
            raise errors.ExecutorFailed("Base container '{}' is dead".format(self._base_container_name))

        volume_path = self.base_container.GetProperty("root")
        return PortoVolumeWrapper(PortoContainerRegistry().porto_client.FindVolume(volume_path))

    @property
    def executor_container_name(self):
        return "{}/{}-{}".format(self.base_container.name, self._cmd.task_id, self._cmd.token[:8])

    def send_signal(self, sig):
        PortoContainerRegistry().porto_client.Kill(self.executor_container_name, sig)

    def suspend(self):
        PortoContainerRegistry().porto_client.Pause(self.executor_container_name)

    def resume(self):
        if self._container_props is None:
            return
        try:
            PortoContainerRegistry().porto_client.Resume(self.executor_container_name)
        except (porto.exceptions.ContainerDoesNotExist, porto.exceptions.InvalidState) as ex:
            self.logger.warning("Cannot resume container %s: %s", self.executor_container_name, ex)

    @property
    def shell_command(self):
        if self.base_container is None:
            return None
        return (
            "/usr/sbin/portoctl exec {}/bash command=/sandbox/shell.sh "
            "isolate=false enable_porto=false".format(self.base_container.name)
        )

    @property
    def ps_command(self):
        if self.base_container is None:
            return None

        cmd = "/usr/sbin/portoctl exec {}/ps isolate=false enable_porto=false command='ps wwuf -u {}'"
        return cmd.format(self.base_container.name, system.UNPRIVILEGED_USER.login)

    @property
    def attach_command(self):
        if self._base_container_name is None:
            return super(PortoPlatform, self).attach_command
        exe = self.executable
        command = " ".join([
            exe,
            os.path.join(os.path.dirname(os.path.dirname(exe)), "pydevd_attach_to_process", "attach_pydevd.py"),
            "--port", "{port}", "--host", "{host}", "--pid", "{pid}"
        ])
        return "/usr/sbin/portoctl exec {}/debugger isolate=false enable_porto=false command='{}'".format(
            self.base_container.name, command
        )

    @property
    def ramdrive(self):
        return self.RAMDRIVE_PATH

    @ramdrive.setter
    def ramdrive(self, value):
        if self.base_container is None:
            return
        rd_type, size = value

        if rd_type != ctm.RamDriveType.TMPFS:
            raise errors.ExecutorFailed("Unknown ramdrive type: {}".format(rd_type))

        # Create volume and link it to base container
        with self._ramdrive_lock:
            self.logger.debug(
                "Creating ramdrive volume (type: %s, size: %dM) for %s", rd_type, size, self.base_container.name
            )
            try:
                ramdrive_path = PortoContainerRegistry().porto_client.CreateVolume(
                    path="",  # choose path automatically
                    backend=rd_type,
                    space_limit="{}M".format(size),
                    containers="{} {}".format(self.base_container.name, self.RAMDRIVE_PATH)
                ).path
            except porto.exceptions.Busy as ex:
                self.logger.warning("Failed to create volume: %s", ex)
                ramdrive_path = self.base_volume.path(self.RAMDRIVE_PATH)
        with common_os.Capabilities(common_os.Capabilities.Cap.Bits.CAP_CHOWN):
            os.chown(ramdrive_path, system.UNPRIVILEGED_USER.uid, system.UNPRIVILEGED_USER.gid)
        self.logger.debug("Created ramdrive volume at %s", ramdrive_path)

    @ramdrive.deleter
    def ramdrive(self):
        if self._container_props is None:
            return
        # Find ramdrive volume by mount target
        # TODO: use ListVolumeLinks after contrib/python/porto upgrade in Arcadia
        for link in self.base_container.GetProperty("volumes_linked").split(";"):
            if " " not in link:
                continue
            path, target = link.split(" ", 1)
            if target == self.RAMDRIVE_PATH:
                break
        else:
            # Nothing to delete
            return

        self.logger.debug("Found ramdrive volume at %s, delete it", path)

        # Base container should be the only one linked with the volume,
        # thus unlink will lead to the volume death.
        try:
            PortoContainerRegistry().porto_client.UnlinkVolume(path, container=self.base_container.name)
        except porto.exceptions.VolumeNotFound as ex:
            self.logger.warning("Cannot unlink volume: %s", ex)

    @common_patterns.singleton_property
    def agentr(self):
        return sandbox.agentr.client.Session.service(self.logger)

    def setup_agentr(self, agentr):
        # Provide container name for AgentR
        if self._container_props is not None:
            agentr.lxc = self.executor_container_name
        return super(PortoPlatform, self).setup_agentr(agentr)

    def _replace_package_if_newer(self, src, dst):
        # Check if container has actual version,
        # if not, copy new package to container
        src_version = common_package.directory_version(src)
        dst_version = common_package.directory_version(dst)
        if not dst_version or src_version != dst_version:
            if os.path.exists(dst):
                if os.path.islink(dst):
                    os.unlink(dst)
                if os.path.isfile(dst):
                    os.remove(dst)
                else:
                    shutil.rmtree(dst)
            self.logger.info("Update '%s' %s -> %s", dst, dst_version, src_version)
            if os.path.isfile(src):
                shutil.copy(src, dst)
            else:
                common_fs.copy_dir(src, dst, log=self.logger)

    def _prepare_tasks_res(self, tasks_rid, path):
        image_path = self.agentr.resource_sync(tasks_rid)
        if self._cmd.exec_type == ctt.ImageType.REGULAR_ARCHIVE:
            # Update generic tasks code package if necessary
            # Prevent multiple jobs from updating tasks code simultaneously
            res_meta = self.agentr.resource_meta(tasks_rid)
            with self._update_tasks_lock:
                updater = common_package.PackageUpdater(self.logger)
                res_meta = {"revision": res_meta["attributes"]["commit_revision"]}
                updater.update_package("tasks", res_meta, force=False, pkg_path=image_path)
                # TODO: use squashfs image instead of tarfile
                with system.PrivilegedSubprocess(("Update tasks_res, rid {}".format(tasks_rid), self.base_volume.root)):
                    # Update local tasks_dir from resource
                    self._replace_package_if_newer(common_config.Registry().client.tasks.code_dir, self.tasks_dir)

            # TODO: use squashfs or volume bind to container
            with system.PrivilegedSubprocess(("update tasks", self.base_volume.root)):
                # Update tasks_dir in container
                self._replace_package_if_newer(common_config.Registry().client.tasks.code_dir, path)

        elif self._cmd.exec_type == ctt.ImageType.CUSTOM_ARCHIVE:
            # Just download and extract custom archives
            with system.PrivilegedSubprocess(("untar custom tasks", self.base_volume.root)):
                common_fs.untar_archive(image_path, path, log=self.logger)

        elif self._cmd.exec_type == ctt.ImageType.BINARY:
            # Nothing to do for BINARY executor
            pass

        elif self._cmd.exec_type in ctt.ImageType.Group.IMAGE:
            if not os.path.exists(path):
                with common_os.Capabilities(common_os.Capabilities.Cap.Bits.CAP_DAC_OVERRIDE):
                    os.mkdir(path)

        else:
            raise errors.InvalidJob("Invalid resource to execute task: #{}".format(tasks_rid))

    def _prepare_sandbox_code(self, v):
        # Ensure directories for all sandbox-related runtime
        venv = v.path("/home/zomb-sandbox/venv")
        src = v.path("/home/zomb-sandbox/client")
        preexecutor = v.path("/home/zomb-sandbox/preexecutor")

        if system.local_mode():
            # Download resource without root privileges
            venv_tar = self.agentr.resource_sync(self._container_props["venv_rid"])
            with system.PrivilegedSubprocess(("update venv and client", self.base_volume.root)):
                # Just extract venv directly to container
                common_fs.untar_archive(venv_tar, venv, log=self.logger)
                # For local Sandbox whole home directory will be mounted, use symlink
                if os.path.exists(src):
                    os.unlink(src)
                os.symlink(os.path.dirname(os.environ[base.SANDBOX_DIR]), src)
                if os.path.exists(preexecutor):
                    os.unlink(preexecutor)
                os.symlink(
                    os.path.join(common_config.Registry().common.dirs.data, "preexecutor", "preexecutor"),
                    preexecutor
                )
        else:
            with system.PrivilegedSubprocess(("update venv and client", self.base_volume.root)):
                # Update venv in container
                venv_by_arch = {_["arch"]: _["path"] for _ in common_config.Registry().client.porto.venv}
                venv_dir = venv_by_arch[self._cmd.arch.replace(".", "_")]
                self._replace_package_if_newer(venv_dir, venv)
                # Update client code in container
                self._replace_package_if_newer(os.path.dirname(os.environ[base.SANDBOX_DIR]), src)
                self._replace_package_if_newer(os.path.join(system.SERVICE_USER.home, "preexecutor"), preexecutor)

    def get_container_resolv_conf(self):
        if self.dns_type == ctm.DnsType.LOCAL:
            return "keep"  # Use resolv.conf from container

        return ";".join(
            line for line in self.resolv_conf.splitlines() if line.startswith("nameserver")
        )

    def _prepare(self, tasks_rid):
        porto_registry = PortoContainerRegistry()

        layers = [
            PortoLayerDefinition(rid)
            for rid in self._container_props["layers"]
        ]

        self.logger.info("PortoPlatform.prepare")

        # Check if executor type is valid
        if self._cmd.exec_type == ctt.ImageType.INVALID:
            raise errors.InvalidJob("Invalid resource to execute task: #{}".format(tasks_rid))
        # Check if container properties are valid
        if self._container_props["alias"] is None:
            raise errors.InvalidJob("Job platform is not defined")
        if self._container_props["venv_rid"] is None:
            raise errors.InvalidJob("Job venv is not defined")
        self.logger.debug("Create 'bind' volume %s", common_config.Registry().client.dirs.data)
        porto_registry._get_bind_volume(common_config.Registry().client.dirs.data)
        self.logger.debug("Import porto layers: %s", layers)
        porto_registry.import_porto_layers(layers, self.agentr)
        return porto_registry, layers

    def prepare(self, tasks_rid):
        if tasks_rid is None or self._container_props is None:
            return super(PortoPlatform, self).prepare(tasks_rid)
        if self._cmd.arch is None:
            self._cmd.arch = self._container_props["alias"]
        PortoContainerRegistry().logger = self.logger

        porto_registry, layers = self._prepare(tasks_rid)

        self._base_container_name = porto_registry.get_base_container(layers)
        self.logger.debug(
            "Got base container '%s', volume: '%s'",
            self._base_container_name, self.base_volume.root
        )

        # Update sandbox packages (client code and venv)
        self._prepare_sandbox_code(self.base_volume)

        if not system.local_mode():
            self._prepare_tasks_res(tasks_rid, self.base_volume.path(self.tasks_dir))

            with system.PrivilegedSubprocess(("extracting /home/sandbox", self.base_volume.root)):
                home_sandbox = self.base_volume.path(system.UNPRIVILEGED_USER.home)
                if os.path.exists(home_sandbox):
                    shutil.rmtree(home_sandbox)
                self._restore_home(home_sandbox, job_logger=self.logger)

        # Intentionally don't call `super().prepare`
        # There is nothing useful there, everything was done here

    def _spawn(self, executor_args):
        if not system.local_mode():
            executor_args["tasks_dir"] = self.tasks_dir

        executor_args["container"] = {
            "platform": self._cmd.arch,
            "name": self.executor_container_name,
            "volume": self.base_volume.root,
            "container_type": "porto",
            "resolvconf": self.get_container_resolv_conf(),
            "properties": self._container_limits,
        }

        self._cmd.executor_args = executor_args
        self._cmd.save_state()

        env = os.environ.copy()
        env["HOME"] = system.UNPRIVILEGED_USER.home
        env["USER"] = env["LOGNAME"] = system.UNPRIVILEGED_USER.login
        env["LANG"] = "en_US.UTF8"
        env["container"] = "porto"
        # Cleanup a little bit
        env.pop("LS_COLORS", None)
        env.pop("LSCOLORS", None)
        env.pop("TMPDIR", None)
        env.pop("TEMP", None)
        env.pop("LAUNCHER_CONF", None)
        env[common_config.Registry.CONFIG_ENV_VAR] = self.config_path

        command = "{} {}".format(self.preexecutor_path, self._cmd.executor_args)
        # Escape ";" values as porto uses them as separators for key-value pairs
        env_value = ";".join("{}={}".format(k, v.replace(";", "\;")) for k, v in env.items())  # noqa
        return command, env, env_value

    @property
    def _container_limits(self):
        return [
            "cpu_guarantee={}c".format(self._cmd.args.get("cores") or 0),
            "memory_guarantee={}".format((self._cmd.args.get("ram") or 0) << 20),
            "ulimit=core: unlimited unlimited" + (
                "; nofile: {nolimit} {nolimit}".format(nolimit=system.TASK_NOLIMIT)
                if ctc.Tag.MULTISLOT in common_config.Registry().client.tags else
                ""
            )
        ]

    def spawn(self, executor_args):
        self.logger.info("PortoPlatform.spawn")
        if self._container_props is None:
            return super(PortoPlatform, self).spawn(executor_args)
        command, env, env_value = self._spawn(executor_args)

        # TODO: Run container directly, no need for liner
        return system.TaskLiner(
            common_format.obfuscate_token(self._cmd.token),
            self.logger,
            [
                "/usr/sbin/portoctl", "exec", "-C", self.executor_container_name,
                "isolate=false",
                "weak=false",
                "enable_porto=false",
                "virt_mode=fuse",
                "command={}".format(command),
                "env={}".format(env_value),
                "resolv_conf={}".format(self.get_container_resolv_conf()),
            ] + self._container_limits,
            env,
            "root"
        ), executor_args

    def _cleanup_executor_container(self):
        try:
            container = PortoContainerRegistry().porto_client.Find(self.executor_container_name)
        except porto.exceptions.ContainerDoesNotExist:
            # Executor container is dead as it should be
            return

        self.logger.warning("Executor container '%s' is still alive", self.executor_container_name)
        try:
            container.Destroy()
        except porto.exceptions.ContainerDoesNotExist as e:
            self.logger.error("Can't destroy container: %s", e)

    def _cleanup_base_container(self):
        try:
            base_container = self.base_container  # noqa
        except errors.ExecutorFailed as ex:
            self.logger.warning("Base container '%s' is dead, nothing to cleanup: %s", self._base_container_name, ex)
            return

        if base_container is None:
            self.logger.warning("Base container is not even initialized, nothing to cleanup")
            return

        # Destroy executor container if it still exists
        self._cleanup_executor_container()

        # Delete ramdrive volume if any
        del self.ramdrive

    def _clean_proc_debris(self):
        pass

    def cleanup(self):
        self.logger.info("Porto cleanup()")
        if self._container_props is None:
            super(PortoPlatform, self).cleanup()
            return

        try:
            self._cleanup_base_container()
        except Exception:
            self.logger.error("Failed to cleanup container '%s'", self._base_container_name)
            raise
        finally:
            # Always return base container back to registry
            # TODO: maybe destroy instead?
            PortoContainerRegistry().unlock_base_container(self._base_container_name)

    @classmethod
    def maintain(cls):
        # Try to cleanup unused layers
        PortoContainerRegistry().remove_unused_layers()

    def cancel(self):
        PortoContainerRegistry().destoy_container(self.executor_container_name)


class PrivilegedPortoPlatform(PortoPlatform):

    def prepare(self, tasks_rid):
        porto_registry, layers = self._prepare(tasks_rid)

        self._base_container_name = porto_registry.get_base_container(layers, privileged_task_id=self._cmd.task_id)
        self.logger.debug(
            "Got base container '%s', volume: '%s'",
            self._base_container_name, self.base_volume.root
        )
        # Update sandbox packages (client code and venv)
        self._prepare_sandbox_code(self.base_volume)

        if not system.local_mode():
            self._prepare_tasks_res(tasks_rid, self.base_volume.path(self.tasks_dir))

    def cleanup(self):
        self.logger.info("Porto cleanup()")

        try:
            self._cleanup_base_container()
        except Exception:
            self.logger.error("Failed to cleanup container '%s'", self._base_container_name)
            raise
        finally:
            PortoContainerRegistry().unlock_base_container(self._base_container_name)
            if not common_config.Registry().client.porto.keep_privileged:
                PortoContainerRegistry().destoy_container(self._base_container_name)
            else:
                logging.info("Will keep {} for debug, reason: keep_privileged=true".format(self._base_container_name))

    def terminate(self):
        task_workdir = os.path.join(common_config.Registry().client.tasks.data_dir, *ctt.relpath(self._cmd.task_id))
        self.export_overlayfs_diff(os.path.join(task_workdir, "rootfs_diff.tar"))
        self._cleanup_base_container()
        PortoContainerRegistry().unlock_base_container(self._base_container_name)
        if not common_config.Registry().client.porto.keep_privileged:
            PortoContainerRegistry().destoy_container(self._base_container_name)
        else:
            logging.info("Will keep {} for debug, reason: keep_privileged=true".format(self._base_container_name))

        from sandbox.client.pinger import PingSandboxServerThread
        PingSandboxServerThread()._kamikadze_thread.ttl = common_config.Registry().client.disk_op_timeout
        with common_os.Capabilities(
            common_os.Capabilities.Cap.Bits.CAP_CHOWN +
            common_os.Capabilities.Cap.Bits.CAP_DAC_OVERRIDE +
            common_os.Capabilities.Cap.Bits.CAP_LINUX_IMMUTABLE
        ):
            logger.debug("Taking ownership on %s", repr(task_workdir))
            self._safe_chown(system.UNPRIVILEGED_USER, task_workdir, recursive=True)

    def spawn(self, executor_args):
        self.logger.info("PrivilegedPortoPlatform.spawn")
        command, env, env_value = self._spawn(executor_args)

        # TODO: Run container directly, no need for liner
        return system.TaskLiner(
            common_format.obfuscate_token(self._cmd.token),
            self.logger,
            [self.preexecutor_path, self._cmd.executor_args],
            env,
            "root"
        ), executor_args

    def export_overlayfs_diff(self, export_path):
        # Sometimes layers called Volumes :/
        self.logger.info("Export rootfs of {} to {}".format(self._base_container_name, export_path))
        base_layer = self.base_container.GetProperty("root")
        if not os.path.exists(export_path):
            PortoContainerRegistry().porto_client.ExportLayer(base_layer, export_path)

    @property
    def shell_command(self):
        task_dir = os.path.join(common_config.Registry().client.tasks.data_dir, *ctt.relpath(self._cmd.task_id))
        if self.base_container is None:
            return None
        return (
            "/usr/sbin/portoctl exec {}/bash command=/bin/bash "
            "isolate=false enable_porto=isolate "
            "cwd={}".format(self.base_container.name, task_dir)
        )

    @property
    def ps_command(self):
        if self.base_container is None:
            return None

        cmd = "/usr/sbin/portoctl exec {}/ps isolate=false enable_porto=false command='ps -auxfwww'"
        return cmd.format(self.base_container.name)

    @property
    def attach_command(self):
        exe = self.executable
        command = " ".join([
            exe,
            os.path.join(os.path.dirname(os.path.dirname(exe)), "pydevd_attach_to_process", "attach_pydevd.py"),
            "--port", "{port}", "--host", "{host}", "--pid", "{pid}"
        ])
        return "/usr/sbin/portoctl exec {}/debugger isolate=false enable_porto=isolate command='{}'".format(
            self.base_container.name, command
        )
