# coding: utf-8
from __future__ import absolute_import, unicode_literals

import os
import yaml

from sandbox.common import config as common_config
from sandbox.common import format as common_format

from sandbox.client import errors
from sandbox.common.windows import wsl
from sandbox.common.windows import ramdrive

from . import base
from .. import system


class WSLPlatform(base.Platform):
    HOME = os.path.join(wsl.FS.WINDOWS_DRIVE_MOUNTPOINT, "Users")
    WSL_CONFIG_PATH = os.path.join(HOME, "zomb-sandbox/settings.yaml")
    PREEXECUTOR = os.path.join(HOME, "zomb-sandbox/preexecutor.exe")
    RAMDRIVE_PATH = wsl.FS.platform_path(os.path.join(wsl.FS.WINDOWS_DRIVE_MOUNTPOINT, "RAMDRIVE"))

    @property
    def USER_GROUP(self):
        # TODO: replace method with class constant="Users" SANDBOX-9262
        return "Users" if "437" in wsl._get_codepage() else "Пользователи"

    @classmethod
    def native_path(cls, path):
        return wsl.FS.win_path(path)

    @classmethod
    def create_task_config(cls):
        host_settings = common_config.Registry()
        config = {
            "path_sep": "\\",
            "this": {
                "dc": host_settings.this.dc,
                "id": host_settings.this.id,
                "fqdn": host_settings.this.fqdn,
            },
            "common": {
                "dirs": {
                    "service": wsl.FS.win_path(host_settings.common.dirs.service),
                    "runtime": wsl.FS.win_path(host_settings.common.dirs.runtime),
                },
                "installation": host_settings.common.installation,
            },
            "agentr": {
                "daemon": {
                    "server": {
                        "host": host_settings.agentr.daemon.server.host or "localhost",
                        "port": host_settings.agentr.daemon.server.port,
                    },
                },
                "wsl_rootfs": wsl.FS.root,
            },
            "client": {
                "dirs": {
                    "data": wsl.FS.win_path(host_settings.client.dirs.data),
                },
                "tasks": {
                    "env_dir": wsl.FS.win_path(host_settings.client.tasks.env_dir),
                    "build_cache_dir": wsl.FS.win_path(host_settings.client.tasks.build_cache_dir),
                },
                "vcs": {
                    "dirs": {
                        "base_cache": wsl.FS.win_path(host_settings.client.vcs.dirs.base_cache),
                    },
                },
                "rest_url": host_settings.client.rest_url,
                "tags": host_settings.client.tags,
                "log": {
                    "root": wsl.FS.win_path(host_settings.client.log.root),
                },
                "executor": {
                    "log": {
                        "root": wsl.FS.win_path(host_settings.client.executor.log.root),
                    },
                    "dirs": {
                        "data": wsl.FS.win_path(host_settings.client.dirs.data),
                        "run": wsl.FS.win_path(host_settings.client.executor.dirs.run),
                    },
                },
                "sandbox_user": host_settings.client.sandbox_user,
            },
        }
        with open(cls.WSL_CONFIG_PATH, "w") as f:
            yaml.dump(config, f)

    def set_permissions(self):
        host_settings = common_config.Registry()
        user = host_settings.client.sandbox_user
        if not wsl.User.win_user_exists(user):
            return
        sid = wsl.User.win_user_sid(self.USER_GROUP)
        if not sid:
            return
        sid = "*" + sid
        wsl.FS.acl_grant(host_settings.client.executor.dirs.run, user, "(OI)(CI)(RX,W,D)", reset=True)
        # Tasks dir read for all users if not set already
        if "RX" not in wsl.FS.acl_get_by_user(host_settings.client.tasks.data_dir, self.USER_GROUP):
            wsl.FS.acl_grant(host_settings.client.tasks.data_dir, sid, "(OI)(CI)(RX)", reset=True)

        for path in (
            host_settings.client.tasks.env_dir,
            host_settings.client.vcs.dirs.base_cache,
            host_settings.client.tasks.build_cache_dir,
        ):
            if "(M)" not in wsl.FS.acl_get_by_user(host_settings.client.tasks.data_dir, self.USER_GROUP):
                wsl.FS.acl_grant(path, sid, "(OI)(CI)(RX,W,M)", reset=True)
        wsl.FS.acl_grant(self.WSL_CONFIG_PATH, user, "(R)", reset=True)

    @property
    def _shell_command(self):
        return "/bin/windows_shell"

    def suspend(self):
        for id in wsl.Process.win_user_processes(system.UNPRIVILEGED_USER.login):
            wsl.Process.win_suspend_process(id)

    def resume(self):
        for id in wsl.Process.win_user_processes(system.UNPRIVILEGED_USER.login):
            wsl.Process.win_resume_process(id)

    @property
    def ps_command(self):
        return (
            "{} /V /FI \"USERNAME eq {}\" | iconv -f cp866 -t UTF-8".format(
                wsl.Process.TASKLIST,
                system.UNPRIVILEGED_USER.login
            )
        )

    @property
    def ramdrive(self):
        return wsl.FS.win_path(self.RAMDRIVE_PATH)

    @ramdrive.setter
    def ramdrive(self, value):
        rd_type, size = value
        self.logger.debug("Mount ramdisk image (type: %s, size: %dM, path: %s)", rd_type, size, self.RAMDRIVE_PATH)
        try:
            ramdrive.ImDisk.mount(size, self.RAMDRIVE_PATH, force=True)
        except RuntimeError:
            raise errors.ExecutorFailed("Ramdrive mount failed")
        return self.RAMDRIVE_PATH

    @ramdrive.deleter
    def ramdrive(self):
        self.logger.debug("Umount all ramdrives")
        ramdrive.ImDisk.unmount_all()

    def cleanup(self):
        self.logger.info("Performing platform cleanup.")
        del self.ramdrive
        user = system.UNPRIVILEGED_USER.login
        win_home_dir = wsl.User.win_user_home_path(user)
        self.resume()
        wsl.Process.win_kill_user_procs(user, logger=self.logger)
        wsl.User.win_logout_user(user, logger=self.logger)
        if wsl.User.win_user_sid(user):
            wsl.User.win_remove_user(user, logger=self.logger)
            wsl.User.win_remove_home_dir(user, system.SERVICE_USER.login, win_home_dir, logger=self.logger)
            wsl.User.win_create_user(user, logger=self.logger)
        self.logger.info("Platform cleanup finished.")

    def spawn(self, executor_args):
        self.create_task_config()
        user = system.UNPRIVILEGED_USER.login
        if not wsl.User.win_user_exists(user):
            wsl.User.win_create_user(user, logger=self.logger)
        self.set_permissions()

        self._cmd.executor_args = executor_args
        self._cmd.save_state()

        env = dict(os.environ)
        env["WSLENV"] = "LOGNAME:SANDBOX_CONFIG/w"
        env["SANDBOX_CONFIG"] = wsl.FS.win_path(self.WSL_CONFIG_PATH)
        env["LOGNAME"] = common_config.Registry().client.sandbox_user

        return system.TaskLiner(
            common_format.obfuscate_token(self._cmd.token),
            self.logger,
            [self.PREEXECUTOR, wsl.FS.win_path(self._cmd.executor_args)],
            env,
        ), executor_args

    def sigterm(self, executor_pid):
        wsl.Process.win_kill_process(executor_pid, logger=self.logger)
