import os
import errno
import psutil
import socket
import logging
import platform
import resource
import threading
import itertools as it
import subprocess as sp

import api.procman

from sandbox.common import os as common_os
from sandbox.common import config as common_config
from sandbox.common import context as common_context
from sandbox.common import patterns as common_patterns
from sandbox.common import itertools as common_itertools

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

from . import base, errors

import msgpack


logger = logging.getLogger(__name__)


# Process title base
PROCESS_TITLE = "[sandbox] Client"

# tag with which executor would running by the ProcMan
PROCMAN_TAG = "sandbox_executor"

# User information for executor subprocess
UNPRIVILEGED_USER = common_os.User(None)
# User information for this subprocess
SERVICE_USER = common_os.User(None)

TASK_NOLIMIT = 200000


def setup_users():
    global SERVICE_USER
    global UNPRIVILEGED_USER
    SERVICE_USER, UNPRIVILEGED_USER = common_os.User.service_users


def local_mode():
    """ Returns is this client instance running in local (development) mode or not (production/pre-production). """
    return common_config.Registry().common.installation in ctm.Installation.Group.LOCAL


class UserPrivileges(common_os.User.Privileges):
    """
    The class has the same behaviour as `common.os.User.Privileges` but also it acquires process-wide lock.
    """
    lock = threading.RLock()

    def __enter__(self):
        self.lock.acquire()
        return super(UserPrivileges, self).__enter__()

    def __exit__(self, *args):
        super(UserPrivileges, self).__exit__(*args)
        self.lock.release()


class PrivilegedSubprocess(common_os.Subprocess):
    lock = UserPrivileges.lock

    def __init__(self, title, check_status=False, watchdog=None):
        title = " ".join(map(str, common_itertools.chain(PROCESS_TITLE, "-", title)))
        super(PrivilegedSubprocess, self).__init__(
            title, check_status=check_status, logger=logger, privileged=True, watchdog=watchdog
        )


class TaskProc(base.Serializable):
    """ Client and executor run their jobs (executor/atop processes) via ProcMan """

    __proc = None

    def __find_proc(self, uuid):
        try:
            self.__proc = api.procman.ProcMan().find_by_uuid(uuid)
        except api.procman.ProcessLookupException:
            logger.error("Process with uuid %r is not found", uuid)
            self.__proc = None

    def __init__(self, proc_or_uuid):
        if type(proc_or_uuid).__name__ == "ProcessProxy":
            self.__proc = proc_or_uuid
        else:
            self.__proc = lambda: self.__find_proc(proc_or_uuid)

    def __repr__(self):
        return "<{}: pid={}, uuid={}>".format(self.__class__.__name__, self.pid, self.uuid)

    def __nonzero__(self):
        return bool(self.proc)

    def __getstate__(self):
        return {"uuid": self.uuid}

    def __setstate__(self, state):
        self.__init__(state.get("uuid"))

    @classmethod
    def create(cls, cmd, env=None, user=None, cgroup=None, tags=()):
        if env is None:
            env = dict(os.environ)
        if not user:
            user = UNPRIVILEGED_USER.login
        if cgroup:
            cgroup = cgroup.lstrip("/")
        logger.debug(
            "Executing command %r via procman (user=%r, cgroup=%r, tags=%r).",
            cmd, user, cgroup, tags
        )
        proc = api.procman.ProcMan().create(
            cmd, liner=True, keeprunning=False, env=env, user=user, cgroup=cgroup, tags=tags
        )
        return cls(proc)

    @classmethod
    def find_by_tags(cls, tags):
        return api.procman.ProcMan().find_by_tags(tags)

    @property
    def proc(self):
        if callable(self.__proc):
            self.__proc()
        return self.__proc

    @property
    def stat(self):
        return self.proc.stat() if self.proc else {}

    @property
    def pid(self):
        return self.stat.get("pid")

    @property
    def uuid(self):
        return self.stat.get("uuid", "") if self.proc else ""

    @property
    def poll(self):
        if self.__proc is None:
            # ProcMan doesn't remember the process anymore, but it must have finished at some point
            return api.procman.FINISHED
        else:
            return self.proc.status()


class TaskLiner(base.Serializable):
    """ Client and executor run their jobs (executor/atop processes) via Liner """

    LINER_BIN = "/skynet/python/bin/liner"
    CONNECT_TIMEOUT = 5

    def __init__(self, tag, logger_=None, cmd=None, env=None, user=None, cgroup=None):
        self.__tag = tag
        self.__sock = None
        self.__status = None
        self.__stdout, self.__stderr = [], []
        logger_ = logger_ or logger

        sockfile = self.sockfile
        if cmd:
            if os.path.exists(sockfile):
                raise RuntimeError("Liner '{}' already running.".format(self.__tag))
            logger_.debug("Starting liner binary %r with socket %r.", self.LINER_BIN, sockfile)
            if (
                not cgroup and common_config.Registry().common.installation in ctm.Installation.Group.NONLOCAL and
                "Microsoft" not in platform.platform() and not common_config.Registry().client.porto.enabled
            ):
                # Do not start any liners in own cgroup on public installations, otherwise deploy will kill them
                cgroup = common_os.CGroup("/")
            self.__daemonize([self.LINER_BIN, sockfile], env, user, cgroup)
            logger_.info("Executing command %r via liner (user=%r, cgroup=%r). Env: %s", cmd, user, cgroup, env)
            for _ in common_itertools.progressive_waiter(.01, .3, 15, lambda: os.path.exists(sockfile)):
                logger_.debug("Waiting for socket %r to be created.", sockfile)

        if not os.path.exists(sockfile):
            raise RuntimeError("Socket file '{}' does not exists.".format(sockfile))

        logger_.info("Connecting liner at socket %r.", sockfile)
        with UserPrivileges():
            self.__sock = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
            self.__sock.settimeout(self.CONNECT_TIMEOUT)
            try:
                self.__sock.connect(sockfile)
            except socket.error as ex:
                if ex.errno == errno.ECONNREFUSED:
                    logger_.debug("There is no process that listening socket %r.", sockfile)
                    os.remove(sockfile)
                raise
        self.__send("info")

        self.__reader = self.__reader_gen()
        self.__info = it.ifilter(None, self.__reader).next()
        if not cmd:
            if not self.__info["pid"]:
                self.__send("terminate")
                raise RuntimeError("Liner '{}' has no any child process: {!r}.".format(self.__tag, self.__info))
            logger_.debug("Connected liner %r: %r. Adopting it.", self.__tag, self.__info)
            self.__send("adopt")
        else:
            logger_.debug("Connected liner %r: %r. Spawning a subprocess.", self.__tag, self.__info)
            self.__send(["Executor " + self.__tag] + list(cmd))
            logger_.debug("Waiting for subprocess PID from liner %r", self.__tag)
            self.__info["pid"] = it.ifilter(None, self.__reader).next()
            logger_.info("Subprocess PID is %r for liner %r", self.__info["pid"], self.__tag)

            if (
                common_config.Registry().common.installation in ctm.Installation.Group.NONLOCAL and
                ctc.Tag.PORTOD not in common_config.Registry().client.tags and
                all(
                    tag in common_config.Registry().client.tags
                    for tag in (ctc.Tag.MULTISLOT, ctc.Tag.GENERIC, str(ctc.Tag.Group.LINUX))
                )
            ):
                with UserPrivileges():
                    try:
                        logger.info("Set nofile limits for process %s to %s", self.__info["pid"], TASK_NOLIMIT)
                        psutil._psplatform.prlimit(
                            int(str(self.__info["pid"])), resource.RLIMIT_NOFILE, (TASK_NOLIMIT, TASK_NOLIMIT)
                        )
                    except Exception as ex:
                        logger.error("Can't set nolimit for process %s.", self.__info["pid"], exc_info=ex)

        self.__sock.settimeout(None)
        self.__sock.setblocking(False)

    @staticmethod
    def __daemonize(cmd, env, user, cgroup):
        """
        Daemonize and reset PPID to avoid killing by Samogon.
        To perform this the process will fork and then fork again and start the liner.
        """
        if cgroup:
            if not isinstance(cgroup, common_os.CGroup):
                cgroup = common_os.CGroup(cgroup)
        pid = os.fork()
        if pid:
            return os.waitpid(pid, 0)  # Do not forget to collect zombie process
        with open(os.devnull, "w") as devnull:
            with (common_os.User.Privileges(user) if user else common_context.NullContextmanager()):
                os.setsid()
                p = sp.Popen(cmd, env=env, stdout=devnull, stderr=devnull, close_fds=True)
                if cgroup:
                    with common_os.User.Privileges():
                        cgroup = cgroup.create()
                        cgroup += p.pid
                os._exit(0)

    def __send(self, data):
        for slept, tick in common_itertools.progressive_yielder(0.01, 0.1, 1, False):
            try:
                return self.__sock.send(msgpack.dumps(data))
            except socket.error as ex:
                if ((slept or 0) + tick) >= 1 or ex.errno != errno.EINTR:
                    raise

    def __reader_gen(self):
        self.__unpacker = msgpack.Unpacker()
        while True:
            try:
                buf = self.__sock.recv(0x100)
            except socket.timeout as ex:
                raise errors.InfraError("Error while starting Liner: {}".format(ex))
            except socket.error as ex:
                if ex.errno not in (errno.EWOULDBLOCK, errno.EINTR):
                    raise
                yield None
                continue

            if not buf:
                raise RuntimeError("Liner '{}' died unexpectedly.".format(self.__tag))
            self.__unpacker.feed(buf)
            for obj in self.__unpacker:
                yield obj

    def __repr__(self):
        return "<{}: pid={}, sock={}>".format(self.__class__.__name__, self.pid, self.__tag)

    def __getstate__(self):
        return {"liner": self.__tag}

    def __setstate__(self, state):
        try:
            self.__init__(state.get("liner"))
        except (RuntimeError, socket.error) as ex:
            logger.error("Problem communicating liner tagged as %r: %s", self.__tag, ex)
            # Try to keep object looks like correct, but the underlying process was died
            self.__status = {"status": "UNKNOWN"}
            self.__info = self.__sock = None

    @common_patterns.classproperty
    def active_sockets(cls):
        try:
            for fname in os.listdir(common_config.Registry().client.dirs.liner):
                parts = fname.split(".")
                if len(parts) == 3 and parts[0] == "executor" and parts[2] == "sock":
                    yield parts[1]
        except OSError as ex:
            if ex.errno != errno.ENOENT:
                raise

    @classmethod
    def drop_stale_socket(cls, tag):
        fname = os.path.join(common_config.Registry().client.dirs.liner, ".".join(["executor", tag, "sock"]))
        if os.path.exists(fname):
            try:
                os.unlink(fname)
            except OSError:
                pass

    @common_patterns.singleton_property
    def sockfile(self):
        return os.path.join(common_config.Registry().client.dirs.liner, ".".join(["executor", self.__tag, "sock"]))

    @property
    def stdout(self):
        return "".join(self.__stdout)

    @property
    def stderr(self):
        return "".join(self.__stderr)

    @property
    def info(self):
        return self.__info

    @property
    def pid(self):
        return self.__info and self.__info["pid"]

    @property
    def poll(self):
        if self.__status:
            return self.__status
        while True:
            data = self.__reader.next()
            if data is None:
                return
            if "exited" in data:
                self.__status = data
                self.terminate()
                return
            if "stderr" in data:
                self.__stderr.append(data["stderr"])
            if "stdout" in data:
                self.__stdout.append(data["stdout"])

    def terminate(self):
        if not self.__sock:
            return
        try:
            self.__send("terminatepg")
            self.__sock.close()
        except socket.error:
            pass
        self.__info["pid"] = self.__sock = None
        return True
