"""
Helper functions for working with ABC service
"""

import re
import six

from .. import fs
from .. import auth
from .. import rest
from .. import config as common_config
from .. import patterns
from ..types import misc as ctm

ABC_CHECK_RE = re.compile(r"^[A-Z0-9_-]+$", re.IGNORECASE)
RESOURCE_MANAGER = "742"
QUOTAS_MANAGER = "1553"
PRODUCT_HEAD = "1"


@patterns.ttl_cache(3600, ignore_kws=True)
def _oauth_token(config=None):
    if config is None:
        config = common_config.Registry()
    return fs.read_settings_value_from_file(config.server.auth.oauth.token)


@patterns.ttl_cache(3600, ignore_kws=True)
def abc_service_id(name, config=None):
    """
    Find ABC service id by name (slug).

    :param name: ABC service slug (e.g. `sandbox`)
    :param config: optional config object
    :return: ABC service id or None if not exists
    """
    if not name or not ABC_CHECK_RE.match(name):
        return None
    if config is None:
        config = common_config.Registry()
    abc_api = rest.Client(config.server.services.group_synchronizer.abc_api_url, _oauth_token(config=config))
    resp = abc_api.services["/"].read(slug=name, fields="id")
    results = resp.get("results")
    if not results:
        return None
    return results[0].get("id")


@patterns.ttl_cache(3600, ignore_kws=True)
def abc_service_name(abc_id, config=None):
    """
    Find ABC service name (slug) by id.

    :param abc_id: ABC service id (e.g. 469)
    :param config: optional config object
    :return: ABC service name (slug) or None if not exists
    """
    if abc_id is None:
        return None
    if config is None:
        config = common_config.Registry()
    abc_api = rest.Client(
        config.server.services.group_synchronizer.abc_api_url, _oauth_token(config=config) or auth.NoAuth()
    )
    resp = abc_api.services["/"].read(id=abc_id, fields="slug")
    results = resp.get("results")
    if not results:
        return None
    return results[0].get("slug")


def abc_group_content(name, config=None, id=None):
    """
    Get ABC group content

    :param name: group name:
        "service" - ABC service e.g. "sandbox"
        "service/scope" - ABC service with specific scope e.g. "sandbox/development"
    :param config: optional config object
    :param id: optional service id, is is used instead of the name if set
        (use /<scope> to set scope without service name)
    :return: list of logins
    """
    if config is None:
        config = common_config.Registry()
    abc_api = rest.Client(
        config.server.services.group_synchronizer.abc_api_url, _oauth_token(config=config) or auth.NoAuth()
    )
    query = {"state": "approved", "page_size": 1000, "fields": "person.login"}
    if "/" in name:
        service, scope = name.split("/", 1)
        query["role__scope"] = scope
    else:
        service = name

    if id:
        query["service"] = id
    else:
        query["service__slug"] = service

    logins = []
    cursor = True
    while cursor:
        result = abc_api.services.members["/"].read(query)
        logins.extend(item["person"]["login"] for item in result["results"])
        cursor = result["next"]
        if cursor:
            # noinspection PyUnresolvedReferences
            cursor = six.moves.urllib.parse.parse_qs(six.moves.urllib.parse.urlparse(cursor).query)["cursor"][0]
            query["cursor"] = cursor
    return logins


@patterns.ttl_cache(3600, ignore_kws=True)
def cached_sandbox_group(name, config=None, rest_client=None):
    """
    Get cached Sandbox group from server

    :param name: Sandbox group name
    :param config: optional config object
    :param rest_client: Sandbox REST API client
    :return: Sandbox group
    """
    if config is None:
        config = common_config.Registry()
    if config.common.installation == ctm.Installation.LOCAL:
        return None
    if rest_client is None:
        rest_client = rest.Client(auth=_oauth_token(config=config) or auth.NoAuth())
    try:
        return rest_client.group[name][:]
    except rest.Client.HTTPError as ex:
        if ex.status != rest.requests.codes.NOT_FOUND:
            raise
        return None


@patterns.ttl_cache(3600, ignore_kws=True)
def sandbox_group_to_abc_id(name, config=None, rest_client=None):
    """
    Get ABC service id from Sandbox group

    :param name: Sandbox group name
    :param config: optional config object
    :param rest_client: Sandbox REST API client
    :return: ABC service id or None if not exists
    """
    group = cached_sandbox_group(name, config=config, rest_client=rest_client)
    if not group:
        return None
    abc_name = group["abc"]
    return abc_service_id(abc_name, config=config)


def users_by_service_and_role(abc_service, roles, config=None):
    """
    Get users by ABC service and roles

    :param abc_service: abc service name or id
    :param roles: list of roles id
    :param config: optional config object
    :return: set of users in abc_group with role in roles
    """
    if config is None:
        config = common_config.Registry()
    abc_api = rest.Client(
        config.server.services.group_synchronizer.abc_api_url, _oauth_token(config=config) or auth.NoAuth()
    )
    query = dict(role__in=",".join(roles))
    if str(abc_service).isdigit():
        query["service"] = int(abc_service)
    else:
        query["service__slug"] = str(abc_service)
    items = abc_api.services.members["/"].read(query)["results"]
    return {
        item["person"]["login"] for item in items
    }


@patterns.ttl_cache(3600, ignore_kws=True)
def service_responsibles(abc_service, config=None):
    """
    Get users responsible for the service

    :param abc_service: ABC service name or id
    :param config: optional config object
    :return: set of responsible users in ABC service
    """
    if config is None:
        config = common_config.Registry()
    abc_api = rest.Client(
        config.server.services.group_synchronizer.abc_api_url, _oauth_token(config=config) or auth.NoAuth()
    )
    if str(abc_service).isdigit():
        query = dict(service=int(abc_service))
    else:
        query = dict(service__slug=str(abc_service))
    items = abc_api.services.responsibles["/"].read(query)["results"]
    return {
        item["person"]["login"] for item in items
    }


def bucket_notification_recipient(abc_service):
    return users_by_service_and_role(
        abc_service, [RESOURCE_MANAGER, QUOTAS_MANAGER]
    ) or service_responsibles(abc_service)
