from __future__ import absolute_import

import sys
import inspect
import numbers
import datetime as dt
import itertools as it
import collections
import distutils.util

import six
import aniso8601

from .. import rest
from .. import enum
from .. import format
from .. import patterns
from .. import itertools
from ..types import user as ctu
from ..types import misc as ctm

six.moves.http_client.responses[451] = "Unavailable For Legal Reasons"
six.moves.http_client.LEGAL_REASONS = 451
if six.PY2:
    six.moves.http_client.responses[423] = "Locked"


def _get_order_by(order_by, list_query_map):
    if order_by is None:
        return

    for order in order_by:
        if order[0] in ("+", "-"):
            sign = order[0]
            order = order[1:]
        else:
            sign = "+"

        field_params = list_query_map.get(order, None)
        if field_params:
            mapped = field_params[1]
            if mapped is None:
                raise ValueError("Ordering by field `{}` is not supported".format(order))
        else:
            mapped = order

        yield sign + mapped


def remap_query(query, list_query_map, save_query=False):
    limit = query.pop("limit", 1)
    offset = query.pop("offset", 0)

    if "order" in query:
        order_by = list(_get_order_by(query.pop("order"), list_query_map))
    else:
        order_by = None

    # Map all request parameters to query builder parameters
    query = {list_query_map[k][0]: v for k, v in six.iteritems(query) if v is not None}

    if order_by is not None:
        query["order_by"] = order_by

    if save_query:
        query["limit"] = limit
        query["offset"] = offset

    return query, offset, limit


def filter_query(query, list_query_map):
    if query is None:
        return query
    limit = query.pop("limit", 1)
    offset = query.pop("offset", 0)
    query = {k: v for k, v in six.iteritems(query) if v is not None and k in list_query_map}
    query["limit"] = limit
    query["offset"] = offset
    return query


def is_empty(value):
    return value is None or value is ctm.NotExists


class Scope(enum.Enum):
    QUERY = "query"
    BODY = "body"
    PATH = "path"


class ArrayFormat(enum.Enum):
    CSV = "csv"
    MULTI = "multi"


class IntegerFormat(enum.Enum):
    I16 = "int16"
    I32 = "int32"
    I64 = "int64"


class __DataTypeMeta(type):
    __type__ = None
    __re__ = r"[^/]+"
    __param_name__ = None

    def __call__(cls, *args, **kws):
        frame = inspect.currentframe()
        for _ in six.moves.xrange(kws.pop("__frame_depth", 1)):
            frame = frame.f_back
        if "__names__" not in frame.f_locals:
            frame.f_locals["__names__"] = frame.f_code.co_names
        func = cls.__new__
        code = func.__code__
        names = code.co_varnames[1:code.co_argcount + 1]
        defaults_dict = dict(six.moves.zip(names, func.__defaults__))
        defaults_dict.update(dict(six.moves.zip(names[:len(args)], args)))
        defaults_dict.update(kws)
        required = defaults_dict.get("required")
        description = defaults_dict.get("description")
        assert isinstance(description, (six.string_types, type(None))), "`description` must be a string"
        if description is None and cls.__doc__:
            defaults_dict["description"] = cls.__doc__.strip()
        defaults = tuple(_[1] for _ in sorted(six.iteritems(defaults_dict), key=lambda _: names.index(_[0])))
        return type(
            "__metaclass__",
            (type(cls),),
            dict(
                __bool__=lambda _: required,
                __nonzero__=lambda _: required,
                __getattr__=lambda _, attr: DataType.__getattr__(attr, defaults_dict)
            )
        )(
            cls.__name__,
            (cls,),
            dict(
                # __doc__=description,
                __new__=type(func)(func.__code__, func.__globals__, func.__name__, defaults, func.__closure__)
            )
        )

    def __iter__(cls):
        func = cls.__new__
        code = func.__code__
        names = code.co_varnames[1:code.co_argcount + 1]
        if cls.__param_name__:
            yield "name", cls.__param_name__
        if cls.__type__:
            yield "type", cls.__type__
        for k, v in six.moves.zip(names, func.__defaults__):
            if is_empty(v):
                continue
            if k == "scope":
                k = "in"
            _ = k.split("_")
            k = "".join(it.chain(_[:1], six.moves.map(str.capitalize, _[1:])))
            yield k, v

    def __getattr__(cls, attr, attrs=None):
        if not attrs or attr not in attrs:
            raise AttributeError
        return attrs[attr]


@six.add_metaclass(__DataTypeMeta)
class DataType(object):
    """
    Base data type
    @DynamicAttrs
    """

    __fixed_param_fields__ = {"in", "required", "name", "description"}
    __schema_fields__ = {
        "type", "title", "format", "description", "default", "minimum", "maximum", "required", "enum", "items"
    }

    def __new__(cls, description=None, required=False, scope=Scope.QUERY, default=None):
        pass

    def __call__(self, *args, **kws):
        pass

    @classmethod
    def decode(cls, value):
        if cls and is_empty(value):
            raise ValueError("`{}` is required".format(cls.__param_name__))
        return cls.default if is_empty(value) else value

    encode = decode


class String(DataType):
    __type__ = "string"

    @classmethod
    def decode(cls, value):
        value = super(String, cls).decode(value)
        if value is cls.default:
            return value
        return value if isinstance(value, six.string_types) else str(value)

    encode = decode


class DateTime(DataType):
    """ UTC date in ISO 8601 format """
    __type__ = "string"

    # noinspection PyShadowingBuiltins,PyShadowingNames
    def __new__(cls, description=None, required=False, scope=Scope.QUERY, default=None, format="date-time"):
        pass

    @classmethod
    def decode(cls, value):
        value = super(DateTime, cls).decode(value)
        if value is cls.default:
            return value
        if isinstance(value, dt.datetime):
            return value
        value = value.replace(" ", "T")  # in case there's no proper separator
        return aniso8601.parse_datetime(value).replace(tzinfo=None)

    @classmethod
    def encode(cls, value):
        value = super(DateTime, cls).encode(value)
        if is_empty(value):
            return
        if isinstance(value, six.string_types) and aniso8601.parse_datetime(value):
            return value
        if isinstance(value, int):
            value = dt.datetime.utcfromtimestamp(value)
        elif not isinstance(value, dt.datetime):
            raise ValueError("`{}` must be a datetime object".format(cls.__param_name__))
        return value.isoformat() + "Z"


class DateTimeRange(DataType):
    """ A range of UTC dates in ISO 8601 format separated by `..` to be used in list filter """
    __type__ = "string"

    @classmethod
    def decode(cls, value):
        value = super(DateTimeRange, cls).decode(value)
        if value is cls.default:
            return value
        date_range = str(value).split("..")
        if len(date_range) != 2:
            raise ValueError(
                "`{}` must be a range of UTC dates in ISO 8601 format separated by `..`".format(cls.__param_name__)
            )
        return list(map(lambda _: aniso8601.parse_datetime(_).replace(tzinfo=None), date_range))

    @classmethod
    def encode(cls, value):
        value = super(DateTimeRange, cls).encode(value)
        if is_empty(value):
            return
        if isinstance(value, six.string_types) and cls.decode(value):
            return value
        if len(value) != 2:
            raise ValueError(
                "`{}` must be a pair of two datetime objects".format(cls.__param_name__)
            )
        return "..".join(map(lambda _: _.isoformat() + "Z", value))


class __EnumMeta(type(String)):
    # noinspection PyMethodParameters
    def __iter__(cls):
        # noinspection PyUnresolvedReferences
        return six.moves.map(
            lambda kv: ("enum", sorted(kv[1])) if kv[0] == "values" else kv,
            type(String).__iter__(cls)
        )


@six.add_metaclass(__EnumMeta)
class Enum(String):
    def __new__(cls, description=None, required=False, scope=Scope.QUERY, default=None, values=()):
        pass

    @classmethod
    def decode(cls, value):
        value = super(Enum, cls).decode(value)
        if value is not None and value not in cls.values:
            raise ValueError(
                "`{}` must be one of the {}, passed {}".format(cls.__param_name__, list(cls.values), value)
            )
        return value

    encode = decode


class Integer(DataType):
    __type__ = "integer"
    __re__ = r"\d+"

    # noinspection PyShadowingBuiltins,PyShadowingNames
    def __new__(
        cls, description=None, required=False, scope=Scope.QUERY, default=None,
        format=IntegerFormat.I64, minimum=None, maximum=None
    ):
        pass

    @classmethod
    def decode(cls, value):
        value = super(Integer, cls).decode(value)
        if is_empty(value):
            return value
        value = int(value)
        if cls.minimum is not None and value < cls.minimum or cls.maximum is not None and value > cls.maximum:
            raise ValueError("`{}` must be from {} to {}".format(cls.__param_name__, cls.minimum, cls.maximum))
        return value

    encode = decode


class __IterableTypeMeta(type(DataType)):
    # noinspection PyMethodParameters
    def __iter__(cls):
        # noinspection PyUnresolvedReferences
        for k, v in type(DataType).__iter__(cls):
            if inspect.isclass(v):
                if issubclass(v, InlineSchema):
                    v = dict(v)
                elif issubclass(v, Schema):
                    v = {"$ref": v.ref}
                else:
                    v = dict(
                        six.moves.filter(
                            lambda _:
                                _[0] not in DataType.__fixed_param_fields__ and _[0] in DataType.__schema_fields__,
                            v
                        )
                    )
            yield k, v


@six.add_metaclass(__IterableTypeMeta)
class IterableType(DataType):
    pass


class __ArrayMeta(type(IterableType)):
    # noinspection PyMethodParameters
    def __call__(cls, *args, **kws):
        frame = inspect.currentframe().f_back
        if "__names__" not in frame.f_locals:
            frame.f_locals["__names__"] = frame.f_code.co_names
        cls_ = type(DataType).__call__(cls, *args, **kws)
        assert inspect.isclass(cls_.items) and issubclass(
            cls_.items, DataType
        ), "Item's type must be specified for Array: {!r}".format(cls_.items)
        if args and args[0] is cls_.items:
            cls_ = type(DataType).__call__(cls, args[0](), *args[1:], **kws)
        return cls_


@six.add_metaclass(__ArrayMeta)
class Array(IterableType):
    __type__ = "array"

    # noinspection PyShadowingBuiltins
    def __new__(
        cls, items=None, description=None, required=False, scope=Scope.QUERY, default=None,
        collection_format=ArrayFormat.CSV
    ):
        pass

    @classmethod
    def decode(cls, value):
        value = super(Array, cls).decode(value)
        if value is cls.default:
            return value
        if cls.collection_format == ArrayFormat.CSV and cls.scope != Scope.BODY:
            if not isinstance(value, six.string_types):
                raise ValueError("Expected string in CSV format")
            value = six.moves.filter(None, value.split(","))
        return list(map(cls.items.decode, itertools.chain(value)))

    @classmethod
    def encode(cls, value):
        value = super(Array, cls).encode(value)
        if value is cls.default:
            return value
        return list(map(cls.items.encode, itertools.chain(value)))


class __MapMeta(type(IterableType)):
    # noinspection PyMethodParameters
    def __call__(cls, *args, **kws):
        frame = inspect.currentframe().f_back
        if "__names__" not in frame.f_locals:
            frame.f_locals["__names__"] = frame.f_code.co_names
        cls_ = type(DataType).__call__(cls, *args, **kws)
        assert inspect.isclass(cls_.items) and issubclass(
            cls_.items, DataType
        ), "Item's type must be specified for Map: {!r}".format(cls_.items)
        if args and args[0] is cls_.items:
            cls_ = type(DataType).__call__(cls, args[0](), *args[1:], **kws)
        return cls_


@six.add_metaclass(__MapMeta)
class Map(IterableType):
    __type__ = "object"

    # noinspection PyShadowingBuiltins
    def __new__(
        cls, items=None, description=None, required=False, scope=Scope.QUERY, default=None
    ):
        pass

    @classmethod
    def decode(cls, value):
        value = super(Map, cls).decode(value)
        if value is cls.default:
            return value
        return {k: cls.items.decode(v) for k, v in six.iteritems(value)}

    @classmethod
    def encode(cls, value):
        value = super(Map, cls).encode(value)
        if value is cls.default:
            return value
        return {k: cls.items.encode(v) for k, v in six.iteritems(value)}


class Object(DataType):
    __type__ = "object"

    @classmethod
    def decode(cls, value):
        value = super(Object, cls).decode(value)
        if value is cls.default:
            return value
        if not isinstance(value, dict):
            raise ValueError("`{}` must be a dict: {!r}".format(cls.__param_name__, value))
        return value

    encode = decode


class Id(Integer):
    """ Object identifier """
    # noinspection PyShadowingBuiltins,PyShadowingNames
    def __new__(
        cls, description=None, required=True, scope=Scope.QUERY, default=None,
        format=IntegerFormat.I64, minimum=1, maximum=None
    ):
        pass


class Number(DataType):
    __type__ = "number"

    # noinspection PyShadowingBuiltins
    def __new__(cls, description=None, required=False, scope=Scope.QUERY, default=None):
        pass

    @classmethod
    def decode(cls, value):
        value = super(Number, cls).decode(value)
        if is_empty(value):
            return value
        if isinstance(value, numbers.Number):
            return value
        try:
            value = int(value)
        except (ValueError, TypeError):
            value = float(value)
        return value

    encode = decode


class __SchemaMeta(type(DataType)):
    """ @DynamicAttrs """
    __api__ = None
    __registry__ = None
    __used__ = None

    # noinspection PyMethodParameters
    def __new__(mcs, name, bases, namespace):
        namespace["__ref__"] = "#/definitions/{}".format(name)
        # noinspection PyTypeChecker
        cls = type(DataType).__new__(mcs, name, bases, namespace)
        if bases != (DataType,) and "__new__" not in namespace:
            assert mcs.__registry__ is not None, "Do not use common.api.Schema directly"
            assert name not in mcs.__registry__, "schema duplication: {}".format(name)
            fields = {k: v for base in bases if issubclass(base, Schema) for k, v in six.iteritems(base.__fields__)}
            for k, v in six.iteritems(namespace):
                if inspect.isclass(v) and issubclass(v, DataType):
                    fields[k] = v
                    v.__param_name__ = k
            cls.__fields__ = fields
            mcs.__registry__[name] = cls
        return cls

    # noinspection PyMethodParameters
    def __iter__(cls):
        if type(cls).__api__ is not None and cls is type(cls).__api__.__schema__:
            for k, v in six.iteritems(type(cls).__registry__):
                yield (k, dict(v))
            return
        # noinspection PyUnresolvedReferences
        for k, v in type(DataType).__iter__(cls):
            if k in DataType.__fixed_param_fields__:
                continue
            yield (k, v)
        required = []
        properties = {}
        for k, v in cls.__fields__.items():
            k = k.rstrip("_")
            if v:
                required.append(k)
            if issubclass(v, InlineSchema):
                dv = dict(v)
                properties[k] = dv
                description = dv.get("description")
                if description is not None:
                    properties[k]["description"] = description
            elif issubclass(v, Schema):
                properties[k] = {"$ref": v.ref}
                description = dict(v).get("description")
                if description is not None:
                    properties[k]["description"] = description
            else:
                v = dict(
                    six.moves.filter(
                        lambda _:
                            _[0] not in DataType.__fixed_param_fields__ and _[0] in DataType.__schema_fields__,
                        v
                    )
                )
                properties[k] = v
        yield "properties", properties
        if required:
            yield "required", sorted(required)

    # noinspection PyMethodParameters,PyPropertyDefinition
    @property
    def ref(cls):
        type(cls).__used__.add(cls.__name__)
        return cls.__ref__


@six.add_metaclass(__SchemaMeta)
class Schema(DataType):
    __fields__ = {}
    __ref__ = None

    def __iter__(self):
        for name, field in six.iteritems(self.__fields__):
            value = getattr(self, name, field.default)
            if field.required and is_empty(value):
                raise ValueError("{}.{} is required and has no default value".format(type(self).__name__, name))
            yield name.rstrip("_"), value

    def __getstate__(self):
        return dict(self)

    def __setattr__(self, name, value):
        try:
            field = self.__fields__[name]
        except KeyError:
            raise AttributeError
        object.__setattr__(self, name, value if is_empty(value) else field.encode(value))

    def __getitem__(self, item):
        return getattr(self, item)

    @classmethod
    def decode(cls, value):
        data = super(Schema, cls).decode(value)
        if is_empty(data):
            return data
        if not isinstance(data, dict):
            raise ValueError("`{}` must be a dict (`{}`)".format(cls.__param_name__, cls.__name__))
        data = data.copy()  # prevents argument mutation
        obj = object.__new__(cls)
        for name, field in six.iteritems(obj.__fields__):
            value = data.pop(name.rstrip("_"), field.default)
            if field.required and is_empty(value):
                raise ValueError("{}.{} is required and has no default value".format(cls.__name__, name))
            setattr(obj, name, value if is_empty(value) else field.encode(value))
        return obj

    @classmethod
    def encode(cls, *args, **kws):
        selected_fields = kws.pop("__selected_fields__", [])
        if args:
            if len(args) > 1:
                raise ValueError("Only one positional argument supported and it must be a dict")
            obj = args[0]
            if isinstance(obj, DataType):
                return obj
            kws = dict(obj)
        obj = object.__new__(cls)
        if selected_fields:
            # Avoid of checking attribute in '__fields__'
            object.__setattr__(
                obj, "__fields__",
                {k: v for k, v in six.iteritems(obj.__fields__) if k in selected_fields}
            )
        for name, field in six.iteritems(obj.__fields__):
            value = kws.pop(name, field.default)
            object.__setattr__(obj, name, value if is_empty(value) else field.encode(value))
        if kws:
            raise ValueError("Extra fields for {} given: {}".format(cls.__name__, list(kws.keys())))
        return obj

    @classmethod
    def create(cls, *args, **kws):
        filter_empty = kws.pop("__filter_empty__", False)
        if filter_empty:
            for k, v in list(six.iteritems(kws)):
                if is_empty(v):
                    kws.pop(k)
        return cls.encode(*args, **kws)


class __InlineSchemaMeta(type(Schema)):
    # noinspection PyMethodParameters
    def __call__(cls, *args, **kws):
        kws["__frame_depth"] = 2
        overrides = kws.pop("overrides", None)
        if overrides is not None:
            assert isinstance(overrides, dict), "'overrides' must be dict"
            cls = type(DataType).__call__(cls, *args, **kws)
            cls.__fields__ = dict(cls.__fields__)
            for k, v in six.iteritems(overrides):
                assert inspect.isclass(v) and issubclass(v, DataType),\
                    "Overrides values must be subclass of DataType"
                cls.__fields__[k] = v
                v.__param_name__ = k
            return cls
        return type(DataType).__call__(cls, *args, **kws)

    def __new__(mcs, name, bases, namespace):
        # noinspection PyTypeChecker
        cls = type(DataType).__new__(mcs, name, bases, namespace)
        if bases != (DataType,) and "__new__" not in namespace:
            fields = {
                k: v for base in bases if issubclass(base, InlineSchema) for k, v in six.iteritems(base.__fields__)
            }
            for k, v in six.iteritems(namespace):
                if inspect.isclass(v) and issubclass(v, DataType):
                    fields[k] = v
                    v.__param_name__ = k
            cls.__fields__ = fields
        return cls


@six.add_metaclass(__InlineSchemaMeta)
class InlineSchema(Schema):
    def __new__(cls, description=None, required=False, scope=Scope.QUERY, default=None, overrides=None):
        pass


class Boolean(DataType):
    __type__ = "boolean"

    @classmethod
    def decode(cls, value):
        value = super(Boolean, cls).decode(value)
        if is_empty(value) or isinstance(value, bool):
            return value
        # noinspection PyUnresolvedReferences
        return bool(distutils.util.strtobool(str(value)))

    @classmethod
    def encode(cls, value):
        value = super(Boolean, cls).encode(value)
        if is_empty(value):
            return
        return bool(value)


class __ResponseMeta(type):
    __content__ = rest.Client.JSON
    __schema__ = None
    __http_code__ = None
    __headers__ = None

    def __new__(mcs, name, bases, namespace):
        if bases != (object,):
            http_name = format.ident(name)
            # noinspection PyTypeChecker
            http_code = getattr(six.moves.http_client, http_name, None)
            assert http_code and http_code in six.moves.http_client.responses,\
                "name {!r} does not correspond to any of known HTTP responses".format(name)

            # NB: python3 is using IntEnum for status codes, python2 used ints.
            # py2: https://github.com/python/cpython/blob/2.7/Lib/httplib.py#L111
            # py3: https://github.com/python/cpython/blob/3.10/Lib/http/__init__.py#L34
            # see: https://github.com/benjaminp/six/issues/161
            if hasattr(http_code, "value"):
                http_code = http_code.value
            schemas = []
            headers = {}
            for k, v in six.iteritems(namespace):
                if not inspect.isclass(v):
                    continue
                if issubclass(v, DataType):
                    schemas.append(v)
                elif issubclass(v, Header):
                    headers["-".join(map(lambda _: _.capitalize(), k.split("_")))] = v
            assert len(schemas) < 2, "Cannot define more than one schema for response"
            namespace.update(
                __schema__=(schemas[0] if schemas else None),
                __http_code__=http_code,
                __headers__=headers
            )
        return type.__new__(mcs, name, bases, namespace)

    def __iter__(cls):
        yield "description", (cls.__doc__ or "").strip()
        if cls.__schema__ is not None:
            if issubclass(cls.__schema__, Schema) and not issubclass(cls.__schema__, InlineSchema):
                schema = {"$ref": cls.__schema__.ref}
            else:
                schema = dict(
                    six.moves.filter(
                        lambda _:
                            _[0] not in DataType.__fixed_param_fields__ and _[0] in DataType.__schema_fields__,
                        cls.__schema__
                    )
                )
            yield "schema", schema
        if cls.__headers__:
            yield "headers", {k: dict(v) for k, v in six.iteritems(cls.__headers__)}


@six.add_metaclass(__ResponseMeta)
class Response(object):
    pass


class __RequestMeta(type):
    __content__ = None
    __security__ = ctu.Restriction.AUTHENTICATED
    __allow_ro__ = False
    __parameters__ = ()
    __responses__ = None
    __operation_id__ = None

    def __new__(mcs, name, bases, namespace):
        if bases != (object,):
            parameters = []
            responses = []
            parameters.extend(it.chain.from_iterable(_.__parameters__ for _ in bases))
            own_parameters = []
            for key, value in six.iteritems(namespace):
                if not inspect.isclass(value):
                    continue
                if issubclass(value, DataType):
                    value.__param_name__ = key
                    if issubclass(value, Array):
                        value.items.__param_name__ = "{}[]".format(key)
                    own_parameters.append(value)
                elif issubclass(value, Response):
                    responses.append(value)
            parameters_names = namespace.get("__names__", ())
            own_parameters.sort(key=lambda _: parameters_names.index(_.__param_name__))
            parameters.extend(own_parameters)
            namespace.update(__parameters__=parameters, __responses__=responses)
        return type.__new__(mcs, name, bases, namespace)

    def __iter__(cls):
        if cls.__doc__:
            doc = cls.__doc__.strip().split("\n")
            summary = doc[0]
            description = " ".join(six.moves.filter(None, six.moves.map(str.strip, doc[1:])))
            if summary:
                yield "summary", summary
            if description:
                yield "description", description
        if cls.__operation_id__:
            yield "operationId", cls.__operation_id__
        yield "tags", [cls.__module__.split(".")[-1]]
        parameters = []
        for param in cls.__parameters__:
            if issubclass(param, Schema) and not issubclass(param, InlineSchema):
                ref = param.ref
                # noinspection PyUnresolvedReferences
                param = dict(type(DataType).__iter__(param))
                param["schema"] = {"$ref": ref}
            else:
                if param.scope == Scope.BODY:
                    schema = {}
                    param = dict(param)
                    for k, v in dict(param).items():
                        if k not in DataType.__fixed_param_fields__:
                            v = param.pop(k)
                            if k in DataType.__schema_fields__:
                                schema[k] = v
                    param = {k: v for k, v in six.iteritems(param) if k in DataType.__fixed_param_fields__}
                    if schema:
                        param["schema"] = schema
                else:
                    param = dict(param)
            parameters.append(param)
        yield "parameters", parameters
        responses = {}
        produces = []
        for response in cls.__responses__:
            responses[str(response.__http_code__)] = dict(response)
            if response.__content__ != type(response).__content__:
                produces.append(response.__content__.type)
        yield "responses", responses
        if cls.__content__ != type(cls).__content__:
            yield "consumes", [cls.__content__.type]
        if produces:
            yield "produces", produces
        if cls.__security__ != ctu.Restriction.ANY:
            yield (
                "security",
                [{
                    "oauth": list(map(str.lower, itertools.chain(cls.__security__)))
                }]
            )


@six.add_metaclass(__RequestMeta)
class Request(object):
    pass


class Get(Request):
    __security__ = ctu.Restriction.ANY


class __PostMeta(type(Request)):
    __content__ = rest.Client.JSON


@six.add_metaclass(__PostMeta)
class Post(Request):
    pass


class __PutMeta(type(Request)):
    __content__ = rest.Client.JSON


@six.add_metaclass(__PutMeta)
class Put(Request):
    pass


class Delete(Request):
    pass


class __HeaderMeta(type):
    __item__ = None

    def __call__(cls, item):
        if isinstance(item, six.string_types):
            item = String(item)
        return type(cls.__name__, (cls,), dict(__item__=item))

    def __iter__(cls):
        return six.moves.filter(lambda _: _[0] not in DataType.__fixed_param_fields__, cls.__item__)


@six.add_metaclass(__HeaderMeta)
class Header(object):
    def __new__(cls, item):
        pass


class __ApiMeta(type):
    class __PathMeta(type):
        def __new__(mcs, name, bases, namespace):
            assert bases == (object,) or bases[0].path, "path must be defined"
            if bases == (object,):
                return type.__new__(mcs, name, bases, namespace)
            cls = bases[0]
            operation_prefix = format.ident(name).lower()
            requests = []
            for value in six.itervalues(namespace):
                if not inspect.isclass(value) or not issubclass(value, Request):
                    continue
                value.__operation_id__ = "{}_{}".format(operation_prefix, value.__name__.lower())
                requests.append(value)
                setattr(cls, value.__name__, value)
            cls.requests = requests
            cls.api.__modules__.add(namespace["__module__"])
            return cls

        def __call__(cls, api_or_path):
            # noinspection PyUnresolvedReferences
            api = cls.api
            if api:
                path = api_or_path
                cls.path = path
                paths = api.__paths__
                assert path not in paths, "path duplication: {}".format(path)
                paths[path] = cls
                return cls
            else:
                return type.__new__(type(cls), cls.__name__, cls.__bases__, dict(api=api_or_path))

        def __iter__(cls):
            for request in cls.requests:
                # noinspection PyTypeChecker
                yield request.__name__.lower(), dict(request)

    class __SchemaMeta(type(Schema)):
        # noinspection PyMethodParameters
        def __call__(cls, api):
            if api.__schema__:
                return api.__schema__
            mcs = type(Schema)
            # noinspection PyUnresolvedReferences
            api.__schema__ = type.__new__(
                type(
                    mcs.__name__,
                    (mcs,),
                    dict(
                        __api__=api,
                        __registry__={},
                        __used__=set()
                    )
                ),
                cls.__name__, cls.__bases__, dict(api=api)
            )
            return api.__schema__

    def __new__(mcs, name, bases, namespace):
        if bases == (object,):
            return type.__new__(mcs, name, bases, namespace)
        mcs = type(mcs.__name__, (mcs,), dict(dict=patterns.singleton_property(mcs.dict.fget)))
        cls = type.__new__(mcs, name, bases, namespace)
        cls.__header__ = {k: v for k, v in six.iteritems(namespace) if not k.startswith("_")}
        cls.__paths__ = {}
        cls.__modules__ = set()
        return cls

    @property
    @six.add_metaclass(__PathMeta)
    class Path(object):
        api = None
        path = None
        requests = None

        def __new__(cls, path):
            pass

    @property
    @six.add_metaclass(__SchemaMeta)
    class Schema(Schema):
        def __new__(cls):
            pass

    @patterns.singleton_property
    def dict(cls):
        result = {}
        result.update(cls.__header__)
        # noinspection PyArgumentList
        result.update(
            paths=collections.OrderedDict(
                sorted(((k, dict(v)) for k, v in six.iteritems(cls.__paths__)), key=lambda _: _[0])
            )
        )
        # XXX: need to account for sub schemas
        # noinspection PyTypeChecker
        list(cls.Schema)
        # noinspection PyTypeChecker
        result.update(definitions={k: v for k, v in cls.Schema if k in cls.Schema.__used__})
        result.update(
            tags=sorted(
                map(
                    lambda _: dict(name=_.split(".")[-1], description=(sys.modules[_].__doc__ or "").strip()),
                    cls.__modules__
                ),
                key=lambda x: x["name"],
            )
        )
        return result


@six.add_metaclass(__ApiMeta)
class Api(object):
    __header__ = None
    __paths__ = None
    __schema__ = None
    __modules__ = None

    # just for auto completion
    Path = None
    Schema = Schema
    InlineSchema = InlineSchema
    dict = None
