import abc
import six

from ..types import misc as ctm


@six.add_metaclass(abc.ABCMeta)
class Authentication(object):
    """ Pure abstract class for authentication method. """

    @abc.abstractmethod
    def __str__(self):
        """ The method should return substring for debug log entry. """

    @abc.abstractmethod
    def __iter__(self):
        """ The method should return pairs, which will be passed as headers to the server. """


class NoAuth(Authentication):
    def __str__(self):
        return "none"

    def __iter__(self):
        return iter(())


class Plain(Authentication):
    def __init__(self, auth_token):
        super(Plain, self).__init__()
        self.__auth_token = auth_token

    def __str__(self):
        return "plain"

    def __iter__(self):
        yield ("Authorization", self.__auth_token)


class OAuth(Authentication):
    def __init__(self, token):
        super(OAuth, self).__init__()
        self.__token = token

    @property
    def token(self):
        return self.__token

    def __str__(self):
        return "oauth token"

    def __iter__(self):
        yield (ctm.HTTPHeader.AUTHORIZATION, "OAuth " + self.__token)


class AdminOAuth(Authentication):
    def __init__(self, token):
        super(AdminOAuth, self).__init__()
        self.__token = token

    @property
    def token(self):
        return self.__token

    def __str__(self):
        return "oauth token"

    def __iter__(self):
        yield (ctm.HTTPHeader.ADMIN_AUTHORIZATION, "OAuth " + self.__token)


class GroupOAuth(Authentication):
    def __init__(self, token):
        super(GroupOAuth, self).__init__()
        self.__token = token

    @property
    def token(self):
        return self.__token

    def __str__(self):
        return "group oauth token"

    def __iter__(self):
        yield ("Authorization", "OAuthTeam " + self.__token)


class YandexSession(Authentication):
    COOKIE_NAME = "Session_id"

    def __init__(self, session_id):
        super(YandexSession, self).__init__()
        self.__session_id = session_id

    @property
    def session_id(self):
        return self.__session_id

    def __str__(self):
        return "Yandex session cookie"

    def __iter__(self):
        yield ("Cookie", "{}={}".format(self.COOKIE_NAME, self.__session_id))


class Session(Authentication):
    def __init__(self, user_token, task_token):
        self.__user_token = OAuth(user_token) if user_token else None
        self.__task_token = task_token

    @property
    def user_token(self):
        return self.__user_token.token if self.__user_token else None

    @property
    def task_token(self):
        return self.__task_token

    @task_token.setter
    def task_token(self, value):
        self.__task_token = value

    @task_token.deleter
    def task_token(self):
        self.__task_token = None

    def __str__(self):
        kind = "".join([self.__user_token and "user " or "", self.__task_token and "task " or ""])
        return "".join([kind or "empty ", "session"])

    def __iter__(self):
        if self.__task_token:
            yield "Authorization", "OAuth " + self.__task_token
        elif self.__user_token:
            for _ in self.__user_token:
                yield _


class TVMSession(Authentication):
    def __init__(self, service_ticket, user_ticket=None):
        super(TVMSession, self).__init__()
        self.service_ticket = service_ticket
        self.user_ticket = user_ticket

    def __str__(self):
        return "TVM"

    def __iter__(self):
        yield (ctm.HTTPHeader.SERVICE_TICKET, self.service_ticket)
        if self.user_ticket is not None:
            yield (ctm.HTTPHeader.USER_TICKET, self.user_ticket)
