from __future__ import absolute_import, print_function

import gc
import sys
import uuid
import time
import inspect
import logging
import weakref
import contextlib
import collections

try:
    import collections.abc as collections_abc
except ImportError:
    import collections as collections_abc  # Fallback for Python <= 3.2.

import six

from .. import system


# noinspection PyAttributeOutsideInit
class Timer(collections_abc.MutableMapping):
    """
    Simple context manager which measures time amount spent in its context.
    It also can measure sub-stages, which are available via ``[]`` operator.
    Also, it can automatically reduce the counter on block leave if it will be passed into the constructor.

    Examples (times may differ for you):

    .. code-block:: python

        >>> with Timer() as timer:
        ...  with timer["brushing my teeth"]:
        ...   with timer["squeezing toothpaste"]:
        ...    import time
        ...    time.sleep(1)
        ...
        >>> str(timer)
        '1.004s (brushing my teeth:1.004s/squeezing toothpaste:1.004s)'
        >>> int(timer.secs)
        1
        >>> clock = Timer()
        >>> clock.start()
        <sandbox.common.context.Timer object at 0x105b33ba8>
        >>> time.sleep(1)
        >>> clock.stop()
        <sandbox.common.context.Timer object at 0x105b33ba8>
        >>> int(clock.secs)
        1
    """

    __slots__ = (
        "_started", "_stopped", "_subruns", "_iterations", "_total_time", "_root_timer", "_auto_start_parent", "_stack",
        "_init_tracking_id", "_tracking_id", "_init_countdown", "_countdown",
    ) + (() if six.PY2 else ("__weakref__",))

    def __init__(self, countdown=0, tracking_id=None, root_timer=None, auto_start_parent=False):
        self._init_countdown = countdown
        self._init_tracking_id = tracking_id
        self._root_timer = weakref.ref(self if root_timer is None else root_timer)
        self._auto_start_parent = auto_start_parent
        self.reset()

    def __getstate__(self):
        pass

    @property
    def current(self):
        root_timer = self._root_timer()
        assert root_timer is not None
        if root_timer is not self:
            return root_timer.current
        while len(self._stack) > 1 and not self._stack[-1].running:
            self._stack.pop(-1)
        return self._stack[-1]

    @current.setter
    def current(self, timer):
        root_timer = self._root_timer()
        assert root_timer is not None
        if root_timer is self:
            self._stack.append(timer)
        else:
            root_timer.current = timer

    def start(self):
        if self._started is None or self._stopped is not None:
            self._started = time.time()
        self._stopped = None
        self.current = self
        self._iterations += 1
        return self

    def stop(self):
        self._stopped = time.time()
        self._total_time += self._stopped - self._started
        return self

    def __enter__(self):
        return self.start()

    def __exit__(self, *_):
        self.stop()

    def __str__(self, subrun=False):
        ret = "{:.4g}s:{}".format(self.secs, self._iterations) if self._started else super(Timer, self).__str__()
        if self._subruns:
            ret += " ({})".format(
                "/".join("{}:{}".format(k, v.__str__(subrun=True)) for k, v in six.iteritems(self._subruns))
            )
        if not subrun:
            ret += " [{}]".format(self._tracking_id)
        return ret

    @property
    def running(self):
        return self._started is not None and self._stopped is None

    @property
    def tracking_id(self):
        return self._tracking_id

    @property
    def secs(self):
        if not self._started:
            return None
        elif self._stopped:
            return self._total_time
        return time.time() - self._started

    @property
    def left(self):
        if not self._countdown:
            return None
        self._countdown -= self.secs or 0
        return self._countdown

    @property
    def iterations(self):
        return self._iterations

    def reset(self):
        self._tracking_id = self._init_tracking_id or uuid.uuid4().hex
        self._started = None
        self._stopped = None
        self._iterations = 0
        self._total_time = 0
        self._countdown = self._init_countdown
        self._subruns = collections.OrderedDict()
        if self._root_timer() is self:
            self._stack = [self]

    def __float__(self):
        return self.secs

    def __getitem__(self, item):
        current = self.current
        if not current._started:
            if self._auto_start_parent:
                current.start()
            else:
                raise KeyError("Cannot start subtimer on not started timer object.")
        if item in current._subruns:
            return current._subruns[item]
        ret = current._subruns[item] = type(self)(tracking_id=self._tracking_id, root_timer=self._root_timer())
        return ret

    def __setitem__(self, key, value):
        raise NotImplementedError("Timer does not allow to set items explicitly.")

    def __delitem__(self, item):
        del self._subruns[item]

    def __len__(self):
        return self._subruns.__len__()

    def __iter__(self):
        return self._subruns.__iter__()

    def __contains__(self, item):
        return item in self._subruns


def call_with(context_manager, func, *args, **kws):
    with context_manager:
        return func(*args, **kws)


@contextlib.contextmanager
def disabled_gc():
    """
    Context manager for temporarily disabling Python's garbage collector
    """

    saved_threshold = gc.get_threshold()
    gc.set_threshold(0)
    try:
        yield None
    finally:
        gc.set_threshold(*saved_threshold)


class LoggerLevelChanger(object):
    """
    Context manager for suppressing logger's verbosity
    (messages with importance level less than specified one will be omitted).
    Example:

    .. code-block:: python

        >>> import logging
        >>> logger = logging.getLogger()
        >>> with LoggerLevelChanger(logger, logging.FATAL):
        ...   logger.error("All systems fail and seize")  # will not be logged, as logging.ERROR < logging.FATAL
        ...   logger.fatal("Mayday! Mayday! The ship is slowly sinking")
        ...
        CRITICAL:root:Mayday! Mayday! The ship is slowly sinking
    """

    def __init__(self, logger_to_change, level):
        self.__logger = logger_to_change
        self.__level = level
        self.__previous_level = logger_to_change.level

    def __enter__(self):
        self.__logger.setLevel(self.__level)

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.__logger.setLevel(self.__previous_level)


class NullContextmanager(object):
    """ Context manager which does nothing """

    def __init__(self, *_, **__):
        self.__obj = __.get("enter_obj", self)

    def __enter__(self):
        return self if self.__obj is None else self.__obj

    def __exit__(self, *_, **__):
        pass


# noinspection PyPep8Naming
class skip_if_binary(object):
    """
    Skip a block of code if executed from a binary.
    An instance of this class can act as either context manager or decorator.
    """

    def __init__(self, description, return_value=None):
        self._skip = system.inside_the_binary()
        self._description = description
        self._return_value = return_value

    class Skip(Exception):
        pass

    def __enter__(self):
        if self._skip:
            # https://stackoverflow.com/questions/12594148/skipping-execution-of-with-block
            sys.settrace(lambda *args, **keys: None)
            frame = inspect.currentframe().f_back
            frame.f_trace = self.trace

    def trace(self, frame, event, arg):
        raise self.Skip

    def __exit__(self, exc_type, exc_val, exc_tb):
        if exc_type is self.Skip:
            logging.debug("Binary detected, skipping: %s", self._description)
            return True

    def __call__(self, func):
        @contextlib.wraps(func)
        def inner(*args, **kwds):
            with self:
                return func(*args, **kwds)
            # noinspection PyUnreachableCode
            return self._return_value
        return inner
