# coding: utf-8
from __future__ import absolute_import, print_function, unicode_literals

import inspect
import itertools as it

import six

from .. import itertools as common_itertools


class EnumMeta(type):
    def __new__(mcs, name, bases, namespace):
        cls = type.__new__(mcs, name, bases, namespace)
        if bases == (object,):
            return cls

        new_namespace = {}
        for key, value in six.iteritems(namespace):
            value = mcs.__transform__(namespace, key, value)
            if isinstance(value, type) and issubclass(value, getattr(cls, "Item", ())):
                # noinspection PyTypeChecker
                value = type(key, (value,), {"__module__": None, "__doc__": value.__doc__})
            new_namespace[key] = value
        return type.__new__(mcs, name, bases, new_namespace)

    @staticmethod
    def __transform__(namespace, key, value):
        if value is not None:
            return value
        if namespace.get("__lower_case__"):
            key = key.lower()
        if namespace.get("__no_underscores__"):
            key = key.replace("_", "")
        return key

    def __iter__(cls):
        """
        Returns all known constants.
        """
        for attr, val in cls.iteritems():
            yield val

    def __getitem__(cls, item):
        try:
            return getattr(cls, item)
        except AttributeError:
            raise KeyError("Enumerator '{}.{}' does not contain key {!r}".format(
                cls.__module__, cls.__name__, item
            ))

    def iteritems(cls):
        """
        Returns all known attribute names and their constants.
        """
        for attr, val in (
                ((_, cls.__dict__.get(_)) for _ in cls.__names__)
                if hasattr(cls, "__names__") else
                six.iteritems(cls.__dict__)
        ):
            if attr.isupper() and val is not None:
                yield attr, val

    def val2str(cls, item):
        for attr, val in cls.iteritems():
            if val == item:
                return attr
        raise ValueError("Enumerator '{}.{}' does not contain value {!r}".format(
            cls.__module__, cls.__name__, item
        ))


@six.add_metaclass(EnumMeta)
class Enum(object):
    """
    The base class for enumerations, all members of which declared in UPPERCASE will be
    treated as enumeration element.

    Supports various modifiers such as ``preserve_order()``, which need to be called in a class' body.

    Usage examples:

    .. code-block:: python

        >>> class Fruit(Enum):
        ...   APPLE = None
        ...   PEAR = None
        ...   ORANGE = "plum"
        ...
        >>> Fruit.APPLE
        'APPLE'
        >>> Fruit.ORANGE == "plum"
        True
        >>> "PEAR" in Fruit
        True
        >>> list(Fruit)
        ['plum', 'PEAR', 'APPLE']
        >>> class Cat(Enum):
        ...   Enum.preserve_order()
        ...   Enum.lower_case()
        ...   Enum.no_underscores()
        ...
        ...   RAGDOLL = None
        ...   RUSSIAN_BLUE = None
        ...   CHESHIRE = None
        ...
        >>> Cat.RAGDOLL
        'ragdoll'
        >>> "cheshire" in Cat
        True
        >>> list(Cat)
        ['ragdoll', 'russianblue', 'cheshire']

    """

    class ItemMeta(type):
        def __call__(cls, doc=None):
            return type(cls)(cls.__name__, (cls,), {"__doc__": doc})

        def __eq__(cls, other):
            return str(cls) == str(other)

        def __lt__(cls, other):
            return str(cls) < str(other)

        def __gt__(cls, other):
            return str(cls) > str(other)

        def __le__(cls, other):
            return str(cls) <= str(other)

        def __ne__(cls, other):
            return str(cls) != str(other)

        def __ge__(cls, other):
            return str(cls) >= str(other)

        def __setattr__(cls, *_):
            raise AttributeError("Object is immutable")

        def __delattr__(cls, *_):
            raise AttributeError("Object is immutable")

        def __hash__(cls):
            return hash(cls.__name__)

        def __repr__(cls):
            return six.text_type(cls.__name__)

    @six.add_metaclass(ItemMeta)
    class Item(object):
        pass

    @staticmethod
    def preserve_order():
        """ Enum modifier, preserve order of enum items """
        inspect.currentframe().f_back.f_locals["__names__"] = [
            name for name in inspect.currentframe().f_back.f_code.co_names if name.isupper()
        ]

    @staticmethod
    def lower_case():
        """ Enum modifier, make default values as lowercased items names """
        inspect.currentframe().f_back.f_locals["__lower_case__"] = True

    @staticmethod
    def no_underscores():
        """
        Enum modifier: translate autogenerated names to these without underscores,
        for example, API_CALL -> "APICALL"
        """
        inspect.currentframe().f_back.f_locals["__no_underscores__"] = True


class GroupEnumMeta(EnumMeta):
    def __init__(cls, name, bases, namespace):
        if bases == (object,):
            return
        EnumMeta.__init__(cls, name, bases, namespace)
        for group in six.moves.filter(lambda _: _.isupper(), namespace):
            # noinspection PyCallByClass,PyTypeChecker
            def closure(item_cls):
                # The closure ensures that `item_cls` is bound.
                type.__setattr__(
                    cls,
                    group,
                    type(group, (item_cls,), {
                        "__doc__": item_cls.__doc__,
                        "__module__": None,
                        "__repr__": lambda _, g=group: g,
                        "__setattr__": frozenset.__setattr__,
                        "__enter__": lambda _: setattr(
                            _,
                            "items",
                            set(inspect.currentframe().f_back.f_locals)
                        ),
                        "__exit__": lambda _, *__: type.__setattr__(
                            cls,
                            type(_).__name__,
                            type(type(_).__name__, (item_cls,), {
                                "__doc__": type(_).__doc__,
                                "__module__": None,
                                "__repr__": lambda _, g=type(_).__name__: g,
                                "primary": True
                            })(
                                items=[
                                    item for item in set(inspect.currentframe().f_back.f_locals) - _.items
                                    if item.isupper()
                                ],
                                parent_namespace=inspect.currentframe().f_back.f_locals
                            )
                        )
                    })()
                )
            # noinspection PyUnresolvedReferences
            closure(namespace[group] or cls.Item)

    def __setattr__(cls, group, items):
        if len(getattr(cls, group)):
            raise AttributeError("Group cannot be modified")
        # noinspection PyUnresolvedReferences,PyTypeChecker
        type.__setattr__(
            cls,
            group,
            type(group, (cls.Item,), {
                "__doc__": getattr(cls, group).__doc__,
                "__module__": None,
                "__repr__": lambda _, g=group: g
            })(items=items)
        )

    def __contains__(cls, group):
        if isinstance(group, (six.text_type, six.binary_type)):
            group = getattr(cls, group, None)
        return group is not None and group in iter(cls)


@six.add_metaclass(GroupEnumMeta)
class GroupEnum(object):
    """
    Enumeration for groups.
    Automagically fills out the primary groups if items are defined in scope of group definition:

    .. code-block:: python

        class Item(Enum):
            class Group(GroupEnum):
                # primary groups
                GROUP1 = None
                GROUP2 = None

                # secondary groups
                GROUP3 = None

            with Group.GROUP1:
                ITEM1 = None
                ITEM2 = None

            with Group.GROUP2:
                ITEM3 = None
                ITEM4 = None

            ITEM5 = None

        Item.Group.GROUP3 = (Item.ITEM1, Item.ITEM3, Item.ITEM4)
    """

    class ItemMeta(type):
        def __call__(cls, doc=None, items=(), parent_namespace=None):
            if parent_namespace is not None:
                def is_item(item):
                    from_parent = parent_namespace.get(item)
                    return inspect.isclass(from_parent) and issubclass(from_parent, Enum.Item)

                items = [
                    item if is_item(item) else Enum.__transform__(parent_namespace, item, parent_namespace.get(item))
                    for item in items
                ]

            if not doc:
                instance = super(cls, cls).__new__(cls, items)
                instance.__init__(items)
                return instance
            ret = type(cls)(cls.__name__, (cls,), {"__doc__": doc})
            return ret

    @six.add_metaclass(ItemMeta)
    class Item(frozenset):
        # noinspection PyPep8Naming

        primary = False

        def __hash__(self):
            return hash(str(self))

        def __eq__(self, other):
            return str(self) == str(other)

        def __ne__(self, other):
            return str(self) != str(other)

        def __setattr__(self, *_):
            raise AttributeError("Object is immutable")

        def __delattr__(self, *_):
            raise AttributeError("Object is immutable")

        def __add__(self, other):
            # It used to be implemented as `frozenset.__or__(self, other)`,
            # but in Python 3 it does not return an instance of `Item` anymore.
            cls_name = str("+").join((type(self).__name__, type(other).__name__))
            cls = type(type(self))(
                cls_name, (type(self),), {"__doc__": self.__doc__, "__repr__": lambda _: cls_name}
            )
            items = set(self) | set(other)
            instance = super(cls, cls).__new__(cls, items)
            instance.__init__(items)
            return instance

        __or__ = __add__

        def expand(self):
            return GroupEnum.expand(self)

    @classmethod
    def expand(cls, *items):
        """
        Expand list of groups and items to list of items

        :param items: list of groups and/or items or single group or item
        :return: list of items
        """

        # noinspection PyTypeChecker
        groups = {str(_): _ for _ in iter(cls)}
        return set(it.chain.from_iterable(
            list(groups.get(str(_), [_])) for _ in common_itertools.chain(*items)
        ))

    @classmethod
    def collapse(cls, *items):
        """
        Collapse items from list into appropriate groups

        :param items: list of groups and/or items
        :return: list of items and groups
        """

        items = set(common_itertools.chain(*items))
        # noinspection PyTypeChecker
        for group in iter(cls):
            if not group.primary:
                continue
            group_items = set(group)
            if group_items <= items:
                items -= group_items
                items.add(str(group))
        return items
