"""
Sandbox's exceptions
"""


class SandboxException(Exception):
    """ Base class for all exceptions used in sandbox """


class TemporaryError(SandboxException):
    """ Temporary error """


class UpdateConflict(SandboxException):
    """ Possible race condition on database entry update. """


class ReleaseError(SandboxException):
    """ Error when releasing task """


class IncorrectStatus(SandboxException):
    """ Incorrect task status """


class AuthorizationError(SandboxException):
    """ User not authorized """


class TaskAccessError(SandboxException):
    """ User doesn't have permissions for some actions with task """


class TaskError(SandboxException):
    """ Task error, shall switch task to status EXCEPTION """


class TaskFailure(SandboxException):
    """ Task error, shall switch task to status FAILURE """


class TaskStop(SandboxException):
    """ Ask runtime to switch current task into the "STOPPED" state. """


class DataSizeError(SandboxException):
    """ Error when data size exceeds limit """


class SubprocessError(TaskError):
    """ Error when running subprocess from task """


class TaskContextError(TaskError):
    """ Error in operation with task context """


class Wait(SandboxException):
    """ Base class for wait exceptions """


class WaitTask(Wait):
    """ Wait for task(s) """


class WaitTime(Wait):
    """ Wait for time is out """


class NothingToWait(SandboxException):
    """ Raises if there are no tasks to wait """


class TriggerAlreadyExists(SandboxException):
    """ Raises if such trigger already registered """


class ViewError(SandboxException):
    """ Error when generating output to http request """


class TaskNotEnqueued(TaskError):
    """ Raises if newly created task is not enqueued """


class UnknownTaskType(TaskError):
    """ Task type not registered """


class UnknownResourceType(TaskError):
    """ Resource type not registered """


class InvalidResource(TaskError):
    """ Resource is invalid """


class VaultError(TaskFailure):
    """ Error in operation with Vault """


class VaultNotFound(VaultError):
    """ Vault item not found """


class VaultAmbiguity(VaultError):
    """ More than one items found """


class VaultNotAllowed(VaultError):
    """ Access to the vault item not allowed """


class SemaphoreError(TaskError):
    """ Error while operating with semaphore """


class TaskNotFound(TaskError):
    """ Required task does not exist """


class ResourceNotFound(TaskError):
    """ Required resource does not exist """


class InvalidRESTCall(TaskError):
    """ REST calls are forbidden in this context """


class SandboxEnvironmentError(TaskError):
    """ Error while preparing environment """


class ResourceDuplicationError(TaskError):
    """ Registered resource class is duplicate of existing resource class """
