# coding: utf-8

from __future__ import print_function
from __future__ import absolute_import

import os
import sys
import gzip
import mmap
import stat
import uuid
import errno
import array
import ctypes
import shutil
import logging
import platform
import itertools
import threading
import contextlib
import subprocess

import six
from six.moves.urllib import parse as urlparse

from .. import os as common_os
from .. import enum
from .. import config
from .. import context
from .. import errors
from .. import system
from .. import threading as cth
from ..types import task as ctt


logger = logging.getLogger(__name__)


def get_dir_size(path):
    """
        Получить размер директории (сколько места она занимает).
        Работает через du

        :param path: путь до папки
        :return: размер директории, в случае ошибки возвращает 0
        :rtype: int
    """
    path = os.path.abspath(path)
    try:
        p = subprocess.Popen(('du', '-s', '-k', path), stdout=subprocess.PIPE)
        size = int(p.stdout.readlines()[0].split('\t')[0])
        return size
    except Exception as error:
        logger.error("Can't get size of folder {0}. Error: {1}".format(path, error))
        return 0


def allocate_file(path, size, force_seek=False):
    """
    Create a file of specified size

        :param path: path to file
        :param size: size in bytes
        :param force_seek: always use seek/write method
        :return: True if successful
        :rtype: bool
    """
    path = os.path.abspath(path)

    try:
        # use fallocate for linux; use seek/write for OS X/Windows
        if not force_seek and sys.platform.startswith('linux'):
            subprocess.call(('fallocate', '-l', str(size), path))
        else:
            with open(path, 'w') as out:
                out.seek(size - 1)
                out.write('\0')
        return True
    except Exception as error:
        logger.error("Unable to allocate {0}. Error: {1}".format(path, error))
        return False


def chmod_for_path(path, mode, recursively=True):
    """
        Установить права для пути

        :param path: путь, для котого попробовать установить права
        :param mode: какие права установить

        :return: tuple (stdout, stderr)
        :rtype: tuple
    """
    cmd = ['chmod']
    if recursively:
        cmd.append('-R')
    cmd.append(mode)
    cmd.append("'{0}'".format(path))

    cmd = ' '.join(cmd)

    logger.info("Execute: {0}".format(cmd))
    return subprocess.Popen(cmd, shell=True).communicate()


def fetch_file_via_http(remote_path, local_path, timeout=None, log=None):
    """
    wget analog
    """
    if not log:
        log = logger
    log.info('Try to download "{0}" via HTTP with timeout "{1}"'.format(
        remote_path, timeout))
    import urllib2
    try:
        resp = urllib2.urlopen(remote_path, None, timeout)
    except Exception as ex:
        raise RuntimeError("Download remote file '{}' failed with: '{}'".format(remote_path, str(ex)))
    with open(local_path, 'wb') as f:
        f.write(resp.read())
    log.info('File saved as "{0}"'.format(local_path))


def untar_archive(archive_path, destination, log=None):
    """
    tar xzf replacement
    """
    if not log:
        log = logger
    log.info("Unarchive tar file '%s' to '%s'", archive_path, destination)
    make_folder(destination, log=log)

    p = subprocess.Popen(
        ["tar", "-xf", archive_path, "-C", destination],
        stdout=subprocess.PIPE,
        stderr=subprocess.PIPE,
    )
    stdout, stderr = p.communicate()

    if stdout:
        log.warning("tar command stdout: %s", stdout.strip())
    if stderr:
        log.error("tar command stderr: %s", stderr.strip())

    if p.returncode:
        raise RuntimeError(
            "tar command failed with code {}.\nstdout: {}\nstderr:{}".format(
                p.returncode, stdout.strip(), stderr.strip()
            )
        )

    log.info("Archive extracted successfully as %r", destination)


def copy_dir(src, dst, log=None):
    if not log:
        log = logger
    log.debug("Copy directory: %s -> %s", src, dst)
    shutil.copytree(src, dst, symlinks=True)


def _windows_symlink(source, link_name):
    csl = ctypes.windll.kernel32.CreateSymbolicLinkW
    csl.argtypes = (ctypes.c_wchar_p, ctypes.c_wchar_p, ctypes.c_uint32)
    csl.restype = ctypes.c_ubyte
    flags = 1 if os.path.isdir(source) else 0
    if csl(link_name, source, flags) == 0:
        raise ctypes.WinError()


def make_symlink(source, link_name, force=False, log=None):
    csl = _windows_symlink if platform.system() == "Windows" else os.symlink
    if not log:
        log = logger
    if force and os.path.islink(link_name):
        temp_link = os.path.join(os.path.dirname(link_name), uuid.uuid4().hex)
        log.info('Recreate symlink "{0}" -> "{1}"'.format(link_name, source))
        csl(source, temp_link)
        os.rename(temp_link, link_name)
    else:
        log.info('Create symlink "{0}" -> "{1}"'.format(link_name, source))
        csl(source, link_name)


def remove_path_safely(path, log=None):
    if not log:
        log = logger
    try:
        return remove_path(path, log=log)
    except Exception as ex:
        log.error(str(ex))


def get_task_dir(task_id, data_dir=None):
    if data_dir is None:
        data_dir = config.Registry().client.tasks.data_dir
    data_dir = os.path.abspath(data_dir)
    return os.path.abspath(os.path.join(data_dir, *ctt.relpath(task_id)))


def create_task_dir(task_id, data_dir=None):
    """
    Creates working directory for given task ID. Directory will be created with mode 0755.

    :param task_id: task identifier
    :param data_dir: the common ancestor for all tasks' working directories; defaults to {runtime_directory}/tasks
    :return: absolute path to the created directory.
    """

    if data_dir is None:
        data_dir = config.Registry().client.tasks.data_dir

    data_dir = os.path.abspath(data_dir)
    task_dir = get_task_dir(task_id, data_dir)
    if os.path.exists(task_dir):
        chmod_for_path(task_dir, 'a+w')
        return task_dir
    else:
        cache_dir_two = os.path.dirname(task_dir)
        cache_dir_one = os.path.dirname(cache_dir_two)
        cache_tasks_dir = os.path.dirname(cache_dir_one)
        try:
            if not os.path.exists(cache_dir_two):
                if not os.path.exists(cache_dir_one):
                    if not os.path.samefile(cache_tasks_dir, data_dir):
                        raise errors.TemporaryError(
                            'Incorrect cache tasks dir {0} for task path {1}.'
                            'Correct tasks dir from settings is {2}'.format(cache_tasks_dir, task_dir, data_dir)
                        )
                    if not os.path.exists(data_dir):
                        os.makedirs(data_dir, 0o755)
                    else:
                        chmod_for_path(data_dir, '0755', recursively=False)
                    os.makedirs(cache_dir_one, 0o755)
                else:
                    chmod_for_path(cache_dir_one, '0755', recursively=False)
                os.makedirs(cache_dir_two, 0o755)
            else:
                chmod_for_path(cache_dir_two, '0755', recursively=False)
            os.makedirs(task_dir)
        finally:
            chmod_for_path(data_dir, '0555', recursively=False)
            chmod_for_path(cache_dir_one, '0555', recursively=False)
            chmod_for_path(cache_dir_two, '0555', recursively=False)
        return task_dir


def tail(fh, n):
    """
    Reads `n` last lines from the end of file specified.

    :param fh:  File object to read from.
    :param n:   Amount of lines to read.
    :return:    Array of strings of length less or equal to `n`.
    """
    fh.seek(0, os.SEEK_END)
    size = fh.tell()
    offset, left, ret = size, b"", []
    while offset > 0 and len(ret) < n:
        at, prev = offset, offset
        last_offset, offset = offset, (max((offset // mmap.ALLOCATIONGRANULARITY - 1), 0) * mmap.ALLOCATIONGRANULARITY)

        with contextlib.closing(
            mmap.mmap(fh.fileno(), last_offset - offset, access=mmap.ACCESS_READ, offset=offset)
        ) as m:
            while at > 0 and len(ret) < n:
                at = m.rfind(b"\n", 0, prev)
                if at > 0:
                    ret.append(six.ensure_text(m[at + 1:prev] + left))
                    left = b""
                else:
                    left = m[:prev]
                prev = at
    if len(ret) < n and left:
        ret.append(six.ensure_text(left))

    return reversed(ret)


def cut_log(path, outpath, offset=0, inode=None):
    """
    Copy a part of the log, starting from `offset` bytes, to another file.

    If `offset` is larger than file size or `inode` is provided and doesn't match the log creation date,
    retrieve previous (uncompressed) log

    :param path: path to source file
    :param outpath: path to target file
    :param offset: position in the file starting from which to copy
    :param inode: if present, compare with source file inode; if they're different,

      assume the log has been rotated

    :return: True on success
    """

    if not os.path.exists(path):
        return False

    fstat = os.stat(path)

    logging.debug("[cut_log] input: inode=%s, offset=%s", inode, offset)
    logging.debug("[cut_log] fstat: inode=%s, st_size=%s", fstat.st_ino, fstat.st_size)

    if (inode is not None and inode != fstat.st_ino) or offset > fstat.st_size:
        cut_log(path + '.1', outpath, offset)
        offset = 0

    with open(path, 'r') as fin:
        fin.seek(offset)
        with open(outpath, 'a') as fout:
            shutil.copyfileobj(fin, fout)
    return True


def check_resource_data(resource_path):
    """
    Checks resource data consistency.
    :param resource_path: Path to the resource

    :return: Resource size in KiB.
    """

    if isinstance(resource_path, six.text_type):
        resource_path = resource_path.encode("utf-8")
    if not os.path.exists(resource_path):
        raise ValueError("path {!r} does not exist".format(resource_path))
    elif os.path.islink(resource_path):
        raise ValueError("resource cannot be a symbolic link")
    elif not os.path.isdir(resource_path):
        return os.path.getsize(resource_path) >> 10

    resource_path = os.path.join(os.path.realpath(resource_path), '')
    # checking for symbolic links in resource directory and calculating total size
    dir_size = 0
    has_files = False
    for root_path, dirs, files in os.walk(resource_path):
        for path in itertools.chain(dirs, files):
            full_path = os.path.join(root_path, path)
            f_stat = os.lstat(full_path)
            if not stat.S_ISDIR(f_stat.st_mode):
                dir_size += f_stat.st_size
                has_files = True
            real_path = os.path.realpath(full_path)
            # next line ensures that the check passes if the symbolic link
            # points to the root directory of the resource, i.e: link -> '.'
            real_path = os.path.join(real_path, '')
            if not real_path.startswith(resource_path):
                raise ValueError(
                    "directory {!r} contains symbolic link {!r} pointing to {!r}".format(
                        resource_path, full_path, real_path
                    )
                )
    if not has_files:
        raise ValueError(
            "directory {!r} does not contain any files".format(resource_path)
        )
    return dir_size >> 10


def remove_path(path, log=None):
    """
        Удалить путь @path
        Если путь является симлинком, просто удаляется симлинк

        :param path: путь до папки или файла
    """
    if not log:
        log = logger
    path = os.path.abspath(str(path))
    if os.path.exists(path) or os.path.lexists(path):
        if os.path.islink(path) or not os.path.isdir(path):
            log.info("Remove file, socket or symlink %s.", path)
            os.remove(path)
        else:
            log.info("Remove directory %s.", path)
            shutil.rmtree(path)
    else:
        log.info("Path does not exist %s, cannot remove it.", path)


def make_folder(path, delete_content=False, log=None):
    """
        Подготовить папку. Если папка не существует, она будет создана.
        Если был передан путь до файла, за основу берётся его директория
        (файл может быть удалён, если @delete_content равно True)

        :param path: путь до папки
        :param delete_content: если равен True и папка существует, пробуем удалить содержимое
            (по умолчанию не удаляем).
        :param log: logger
        :return: абсолютный путь до подготовленной папки
    """
    if not log:
        log = logger
    path = os.path.abspath(str(path))
    if delete_content:
        log.info("Delete folder contents %s", path)
        remove_path(path)
    if not os.path.exists(path):
        log.info("Make folder %s", path)
        os.makedirs(path, 0o755)
    return path


def get_unique_file_name(folder, file_name, limit=10000):
    """
        Возвращает путь до несущестующего файла в папке folder для имени file_name
        Если folder/file_name уже существует, ищется незанятое имя путём добавления _n
        где n - натуральное число

        :param folder: путь до папки, относительно которой будет происходить проверка пути
        :param file_name: основная часть названия файла
        :param limit: максимальное кол-во файлов
        :return: путь в виде строки
    """
    file_path = os.path.join(folder, file_name)
    if not os.path.exists(file_path):
        return file_path

    name_and_extension = file_name.rsplit(".", 1)
    name_template = name_and_extension[0]
    for i in six.moves.xrange(1, limit):
        name_and_extension[0] = "{}_{}".format(name_template, i)
        file_path = os.path.join(folder, ".".join(name_and_extension))
        if not os.path.exists(file_path):
            return file_path
    raise errors.TaskFailure("To many files: {}, {}".format(limit, file_name))


def recursive_remove_immutable(path):
    if platform.uname()[0] != "Linux":
        raise RuntimeError("Change attrs over ioctl work only on linux")
    else:
        import fcntl

    FS_IOC_GETFLAGS = 0x80086601
    FS_IOC_SETFLAGS = 0x40086602
    FS_IMMUTABLE_FL = 0x010
    ATTR_e = array.array("L", [524288])
    fd = None

    # Same as chattr -R -i {path}
    for root, dirs, files in os.walk(path):
        for p in itertools.chain(files, dirs):
            p = os.path.join(root, p)
            try:
                fd = os.open(p, os.O_RDONLY | os.O_NONBLOCK)
                im_flag = array.array("L", [0])
                fcntl.ioctl(fd, FS_IOC_GETFLAGS, im_flag, True)
                if bool(im_flag[0] & FS_IMMUTABLE_FL):
                    # Found immutable file/dir
                    fcntl.ioctl(fd, FS_IOC_SETFLAGS, ATTR_e, True)
            except (IOError, OSError) as ex:
                logger.warning("Unable to chattr %s: %s", p, str(ex))
            finally:
                if fd is not None:
                    try:
                        os.close(fd)
                    except (IOError, OSError):
                        pass


class WorkDir(object):
    """
        Context manager for working directory.
        Temporarily change current working directory in execution context.
    """

    def __init__(self, working_dir):
        self.__working_dir = str(working_dir)

    def __enter__(self):
        self.__cwd = os.getcwd()
        os.chdir(self.__working_dir)

    def __exit__(self, exc_type, exc_val, exc_tb):
        os.chdir(self.__cwd)


class FSJournal(object):
    """
    Singleton class which represents some kind of thread-safe filesystem journal.
    Used to allow concurrent resources synchronization.
    """
    # Process-wide operation lock.
    lock = threading.RLock()
    # Instance object.
    _instance = None

    class Directory(object):
        """
        Hashable (sub-)directory object with recursive state compilation.
        """
        class State(enum.Enum):
            """
            Directory states enumeration.
            """
            REMOVE = 0
            RONLY = 1
            KEEP = 2

        def __init__(self, name, path, state=None):
            self.name = name
            self.path = path
            self._state = state if state is not None else self.State.REMOVE
            self.children = {}

        def __eq__(self, other):
            return self.name == other.name

        def __hash__(self):
            return hash(self.name)

        @property
        def state(self):
            with FSJournal.lock:
                max_state = max(self.State)
                state = self._state
                nodes = [self]
                while state < max_state and nodes:
                    nodes = list(itertools.chain.from_iterable(n.children.itervalues() for n in nodes))
                    if nodes:
                        state = max(state, max(n._state for n in nodes))
                return state

        @state.setter
        def state(self, value):
            with FSJournal.lock:
                self._state = value

        def _mkrelto(self, name):
            if name and name[0] == os.path.sep:
                if not name.startswith(self.path):
                    raise ValueError("Subdirectory '%s' is not relative to '%s'", name, self.path)
                else:
                    name = name[len(self.path):]
            return (name[1:] if name and name[0] == os.path.sep else name).strip()

        def is_relto(self, other):
            return other.path.startswith(self.path)

        def adddir(self, name, state=None):
            name = self._mkrelto(name)
            with FSJournal.lock:
                parent = self
                path = os.path.split(name) if name else []
                for p in path:
                    if not p:
                        continue
                    child = parent.children.get(p)
                    if not child:
                        child = parent.children[p] = self.__class__(p, os.path.join(parent.path, p))
                    parent = child
                # Set given state only for the leaf
                if state is not None and state > parent._state:
                    parent._state = state
                return parent

        def mkdir(self, name='', state=None):
            name = self._mkrelto(name)
            with FSJournal.lock:
                path = os.path.join(self.path, name) if name else self.path
                try:
                    os.makedirs(path)
                except OSError as ex:
                    if ex.args[0] != errno.EEXIST:
                        raise
                    mode = os.stat(path).st_mode
                    if mode & stat.S_IWUSR != stat.S_IWUSR:
                        os.chmod(path, mode | stat.S_IWUSR)
                return self.adddir(name, state)

        def tree(self, printer, indent=''):
            printer('{0}{1}: {2} [{3}]'.format(
                indent, self.name, self.State.val2str(self.state), self.State.val2str(self._state)
            ))
            for n in self.children.itervalues():
                n.tree(printer, indent=indent + '\t')

    def __new__(cls, *args, **kwargs):
        with cls.lock:
            if not cls._instance:
                cls._instance = super(FSJournal, cls).__new__(cls, *args, **kwargs)
            return cls._instance

    def __init__(self):
        self._roots = {}
        self.__removed_files_cache = []
        self.log = logging.getLogger('fsjournal')
        self.log.info('Initializing the journal.')
        # Now replace constructor method to avoid double initialization.
        self.__class__.__init__ = super(FSJournal, self).__init__

    def mkroot(self, id, state=None, maker=None, args=tuple(), kwargs={}):
        if state is None:
            state = self.Directory.State.RONLY
        with self.lock:
            dr = self._roots.get(id)
            if not dr:
                path = maker(*args, **kwargs)
                if path[0] != os.path.sep:
                    raise ValueError("Registering root #%s with relative path '%s'.", id, path)
                self.log.info("[%s] Registered root #%s at '%s'", cth.name(), id, path)
                dr = self._roots[id] = []
            else:
                path = dr[0].path
                self.log.debug("[%s] Adding reference #%d to root #%s", cth.name(), len(dr), id)

            dr.append(self.Directory(os.path.basename(path), path, state))
            return dr[-1]

    def _skybone_files_expirer(self):
        if not self.__removed_files_cache:
            return
        # Expire file to remove in skybone's database.
        self.log.debug("Updating skybone's database about %d file records", len(self.__removed_files_cache))
        try:
            with open(os.devnull, 'wb') as null:
                subprocess.check_call([
                    "/skynet/tools/skybone-ctl", "query",
                    "; ".join(
                        "UPDATE file SET chktime = 0 WHERE path GLOB '{}*'".format(
                            _.replace('\\', '\\\\').replace("'", "\'")
                        ) for _ in self.__removed_files_cache
                    )
                ], stdout=null, stderr=null)
        except Exception as ex:
            self.log.warning("Error updating skybone's database: %s", ex)
        self.__removed_files_cache = []

    def commit(self, id, force=False):
        with self.lock:
            nodes = self._roots.get(id)
            if not nodes:
                self.log.info("[%s] Root #%s is committed already.", cth.name(), id)
                return
            max_state = max(root.state for root in nodes) if not force else self.Directory.State.REMOVE
            if max_state == self.Directory.State.KEEP:
                self.log.info(
                    "[%s] Postpone root #%s at '%s' commit. Currently holding %d references.",
                    cth.name(), id, nodes[0].path, len(nodes)
                )
                return

            self.log.info(
                "[%s] Committing%s %d references for root #%s at '%s'",
                cth.name(), " forcedly" if force else "", len(nodes), id, nodes[0].path
            )
            chmods, rmtrees = [], []
            while nodes:
                children = []
                for n in nodes:
                    state = n.state
                    if state == self.Directory.State.REMOVE:
                        if not any(c.is_relto(n) for c in rmtrees):
                            rmtrees.append(n)
                        n.children = {}
                        continue
                    if state == self.Directory.State.RONLY and not any(c.is_relto(n) for c in chmods):
                        chmods.append(n)
                    children += list(n.children.values())
                nodes = children

            for n in rmtrees:
                try:
                    isdir = os.path.isdir(n.path) and not os.path.islink(n.path)
                    self.log.debug("[%s] Remove %s at '%s'", cth.name(), "tree" if isdir else "file", n.path)
                    (shutil.rmtree if isdir else os.unlink)(n.path)
                except OSError as ex:
                    if ex.args[0] != errno.ENOENT:
                        self.log.warn("Unable to remove tree at '%s': %s", n.path, str(ex))

                self.__removed_files_cache.append(n.path)
                if len(self.__removed_files_cache) > 100:
                    self._skybone_files_expirer()

            for n in chmods:
                self.log.debug("[%s] Readonly tree at '%s'", cth.name(), n.path)
                chmod_for_path(n.path, 'a-w')

            self._roots.pop(id)

    def tree(self, printer=print, indent=''):
        for dr in self._roots.itervalues():
            dr[0].tree(printer, indent)

    @property
    def roots(self):
        return self._roots.iterkeys()

    def clear(self, rollback=False):
        with self.lock:
            self.log.info("[%s] %s the journal.", cth.name(), "Rollback" if rollback else "Clearing")
            for i in self._roots.keys() if not rollback else []:
                self.commit(i, force=True)
            self._roots = {}
            self._skybone_files_expirer()


def read_file(path, force_from_fs=False):
    """
    Read from file system or, when called from binary, from in-memory key-value storage --
    see RESOURCE() macro at https://wiki.yandex-team.ru/devtools/commandsandvars

    For real files, both relative and absolute paths are allowed.

    :param path: file path, or a key under which it is stored
    :param force_from_fs: read from filesystem instead of resource storage, fail on file absence
    """

    path = str(path)
    from_fs = force_from_fs or not system.inside_the_binary()
    if from_fs:
        with open(path, "r") as fileobj:
            return fileobj.read()

    from library.python import resource

    data = resource.find(path)
    if data is not None:
        return data

    raise IOError(errno.ENOENT, "No such file stored as resource: {!r}".format(path))


def to_path(uri):
    """
    Strip protocol from a path

    >>>to_path("file://~/oauth.token")
    "~/oauth.token"
    >>>to_path("file:///home/zomb-sandbox/oauth.token")
    "/home/zomb-sandbox/oauth.token"
    >>>to_path("/home/robot-sandbox/oauth.token")
    "/home/robot-sandbox/oauth.token"

    :param uri: a path (not necessarily protocol-prefixed)
    """

    parsed = urlparse.urlparse(uri)
    return parsed.netloc + parsed.path


def read_settings_value_from_file(value, ignore_file_existence=False, binary=False):
    """
    Return a setting's value from file, if it starts with file://, otherwise return the value itself

    :param value: value of path to file that contains value
    :param ignore_file_existence: if True then return None if file does not exist
    :param binary: open file for reading in binary mode

    :return: settings value or None
    :rtype: str or None
    """

    if isinstance(value, six.string_types) and value.startswith("file://"):
        path = to_path(value)
        path = os.path.expanduser(path)
        if ignore_file_existence and not os.path.exists(path):
            return
        with open(path, "rb" if binary else "r") as f:
            value = f.read().strip()
    return value


def gzip_file(in_path, out_path, remove_source=False):
    with open(in_path, 'rb') as orig_file:
        with gzip.open(out_path, 'wb') as zipped_file:
            zipped_file.writelines(orig_file)
    if remove_source:
        os.remove(in_path)


def cleanup(path, ignore_path=None, remove_upper_dir=True, logger=logger):
    """
    Recursively clean task's executing directory, with exception of certain paths.

    :param path: directory to cleanup in
    :type path: str
    :param ignore_path: an iterable with absolute paths to ignore
    :type ignore_path: iterable
    :param remove_upper_dir: remove upper directory
    :type remove_upper_dir: bool
    :param logger: logger
    """

    def _cleanup(path_, ignore_path_, remove_upper_dir_):
        """
        XXX: `shutil.rmtree()` doesn't check for symbolic links:
            http://svn.python.org/projects/python/trunk/Lib/shutil.py, and
            `os.walk()` is inconvenient to use due to file name generation order
        """

        if path_ in ignore_path_:
            return 0
        if os.path.islink(path_):
            os.unlink(path_)
            return 1
        if not os.path.exists(path_):
            return 0
        try:
            if not os.path.isdir(path_):
                os.unlink(path_)
                return 1
            else:
                # recursive call
                n = 0
                for name in os.listdir(path_):
                    n += _cleanup(os.path.join(path_, name), ignore_path_, True)
                # remove empty directory after it's cleared
                if not os.listdir(path_) and remove_upper_dir_:
                    os.rmdir(path_)
                return n
        except OSError as e:
            logger.exception(e)
        return 0
    if ignore_path is None:
        ignore_path = set()
    n = _cleanup(path, ignore_path, remove_upper_dir)
    return n


def _check_task_files(task_id, task_resources, admin_privileges=False):
    # TODO: remove after SANDBOX-9317
    extra, missing, ok = set(), set(), set()  # Extra files, missing resources, ok resources
    tpath = os.path.join(config.Registry().client.tasks.data_dir, *ctt.relpath(task_id))

    exists = os.path.isdir(tpath)
    stack = [""]
    privileges = (
        common_os.Capabilities(common_os.Capabilities.Cap.Bits.CAP_DAC_READ_SEARCH)
        if admin_privileges else
        context.NullContextmanager()
    )

    with privileges:
        while exists and stack:
            nxt = []
            for dname in stack:
                abspath = os.path.join(tpath, dname) if dname else tpath
                for fname in os.listdir(abspath):
                    relname = os.path.join(dname, fname) if dname else fname
                    rid = task_resources.pop(relname, None)
                    if rid:
                        ok.add(rid)
                        continue

                    isdir = os.path.isdir(os.path.join(abspath, fname))
                    if isdir and any(fn.startswith(relname) for fn in task_resources):
                        nxt.append(relname)
                    else:
                        extra.add((task_id, relname))
            stack = nxt

        missing.update(six.itervalues(task_resources))
    return extra, missing, ok
