#!/usr/bin/env python
import io
import os
import subprocess
import sys
import tarfile
import typing

import click
import jinja2

import library.python.resource as rs


@click.group()
def main():
    pass


@main.command()
@click.argument("prefix", type=click.STRING)
@click.argument("output", type=click.STRING)
def extract(prefix, output):
    all_files = load_all()
    for file_path, payload in all_files.items():
        if os.path.dirname(file_path) != prefix:
            continue
        with open(os.path.join(output, os.path.basename(file_path)), "wb") as f:
            f.write(payload)


def load_all():
    tar_archive = io.BytesIO()
    data = rs.find("code_archive.tar.gz")
    assert data is not None
    tar_archive.write(data)
    tar_archive.flush()
    tar_archive.seek(0)

    out = dict()
    with tarfile.open(fileobj=tar_archive, mode='r:gz') as tar:
        for item in tar.getmembers():
            out[item.name] = tar.extractfile(item).read()
    return out


INC_FILE_TEMPLATE = jinja2.Template("""
SET(
    out_files
{% for item in file_list %}
    {{item}}
{%- endfor %}
)

RUN_PROGRAM(
    {{lazy_gen}}
    extract
    {{extract_folder}}
    ${BINDIR}
    OUT ${out_files}
    CWD ${BINDIR}
)

PEERDIR(
    ${GOSTD}/context
    ${GOSTD}/encoding/json
    ${GOSTD}/fmt
    ${GOSTD}/io
    ${GOSTD}/net/http
    ${GOSTD}/reflect
    ${GOSTD}/strconv
    ${GOSTD}/time

    vendor/github.com/go-openapi/errors
    vendor/github.com/go-openapi/runtime
    vendor/github.com/go-openapi/runtime/client
    vendor/github.com/go-openapi/strfmt
    vendor/github.com/go-openapi/swag
    vendor/github.com/go-openapi/validate
    vendor/github.com/golang/mock/gomock
{{ models_peerdir }}
)

""")


@main.command()
@click.argument("output", type=click.STRING)
def update_library(output):
    root = os.path.abspath(output)
    arc_root = subprocess.check_output("ya dump root", shell=True).decode()
    tool_path = os.path.relpath(os.path.dirname(sys.executable), arc_root)
    files = load_all().keys()

    folder_to_files = dict()  # type: typing.Dict[str, typing.List[str]]
    for item in files:
        dirname = os.path.dirname(item)
        os.makedirs(os.path.join(root, dirname), exist_ok=True)
        folder_to_files.setdefault(dirname, [])
        folder_to_files[dirname].append(os.path.basename(item))

    for folder, file_list in folder_to_files.items():
        with open(os.path.join(root, folder, "ya.make.inc"), "w") as f:
            if folder == "models":
                models_peerdir = ""
            else:
                models_peerdir = "\n    {}".format(
                    os.path.relpath(os.path.join(root, "models"), arc_root)
                )
            inc_body = INC_FILE_TEMPLATE.render(
                file_list=sorted(file_list),
                lazy_gen=tool_path,
                extract_folder=folder,
                models_peerdir=models_peerdir,
            )
            f.write(inc_body)


if __name__ == '__main__':
    main()
