#!/usr/bin/env python3
import argparse
import dataclasses
import json
import os
import shlex
import subprocess
import sys
import tarfile
import tempfile

from sandbox.web import api as sandbox_api


@dataclasses.dataclass(frozen=True)
class Parameters:
    build_root: str
    build_output: str
    go_package: str
    swagger_tool: str
    mockgen_tool: str
    code_archive: str
    debug: bool

    @property
    def output_path(self) -> str:
        return self.build_output


# Consts
CLIENT_PACKAGE = "clients"
MODELS_PACKAGE = "models"


def parse_args(args):
    parser = argparse.ArgumentParser()
    # parser.add_argument("--build-root", required=True)
    build_root = os.getcwd()

    parser.add_argument("--build-output", required=True)
    parser.add_argument("--go-package", required=True)
    parser.add_argument("--swagger-tool", required=True)
    parser.add_argument("--mockgen-tool", required=True)
    parser.add_argument("--code-archive", required=True)
    parser.add_argument("--debug", default=False, action="store_true")
    parsed = parser.parse_args(args)
    if parsed.debug:
        print(f"Launched: {shlex.join(sys.argv)}  [CWD: {build_root}]", file=sys.stderr)
    build_output = parsed.build_output
    if os.path.isabs(parsed.build_output):
        assert os.path.commonprefix([parsed.build_output, build_root]) == build_root
        build_output = os.path.relpath(parsed.build_output, build_root)

    swagger_tool = parsed.swagger_tool
    if not os.path.isabs(swagger_tool):
        swagger_tool = os.path.join(build_root, swagger_tool)
    assert os.access(os.path.abspath(swagger_tool), os.F_OK | os.X_OK)

    mockgen_tool = parsed.mockgen_tool
    if not os.path.isabs(mockgen_tool):
        mockgen_tool = os.path.join(build_root, mockgen_tool)
    assert os.access(os.path.abspath(mockgen_tool), os.F_OK | os.X_OK)

    return Parameters(
        build_root=build_root,
        build_output=build_output,
        go_package=parsed.go_package,
        swagger_tool=os.path.abspath(swagger_tool),
        mockgen_tool=os.path.abspath(mockgen_tool),
        code_archive=parsed.code_archive,
        debug=parsed.debug,
    )


def patch_schema(schema):
    for item in ["consumes", "produces"]:
        assert len(schema[item]) == 1
        schema[item] = ["application/json"]

    # NB: fix client default values
    assert len(schema["schemes"]) == 1
    schema["schemes"] = ["https"]

    assert schema["host"]
    schema["host"] = "sandbox.yandex-team.ru"

    patch_schema_inner(schema, "$")
    # raise RuntimeError


def patch_schema_inner(schema, path):
    if path == "$.definitions.Task":
        # NB: buggy swagger schema: array and object merged to array
        # https://a.yandex-team.ru/review/1880743/files/3#file-0-77498799:L252
        # https://a.yandex-team.ru/review/1880743/files/3#file-0-77498799:L274
        # https://a.yandex-team.ru/review/1880743/files/3#file-0-77498799:R234
        assert "children" in schema.get("properties", {})
        schema["properties"]["children"] = {
            "type": "object",
        }
    if isinstance(schema, list):
        for idx, item in enumerate(schema):
            patch_schema_inner(item, path + f"[{idx}]")
    elif isinstance(schema, dict):
        if set(schema.keys()) == {"type", "items"} and schema.get("type") == "object":
            schema["additionalProperties"] = schema["items"]
            del schema["items"]
        if set(schema.keys()) == {"properties"}:
            schema["type"] = "object"

        for key, value in schema.items():
            patch_schema_inner(value, path + f".{key}")


def do_generate(schema: dict, parameters: Parameters):
    with tempfile.NamedTemporaryFile(mode="w") as f:
        json.dump(schema, f, sort_keys=True, indent=4)
        f.flush()

        cmd = [
            parameters.swagger_tool,
            "generate",
            "client",
            "-f", f.name,
            "-c", CLIENT_PACKAGE,
            "-m", MODELS_PACKAGE,
            "-t", parameters.output_path,
        ]
        if parameters.debug:
            print(f"Spawn: {shlex.join(cmd)}", file=sys.stderr)
        swagger_out = os.path.join(parameters.output_path, "swagger.out")
        swagger_err = os.path.join(parameters.output_path, "swagger.err")
        try:
            with open(swagger_out, "w") as out:
                with open(swagger_err, "w") as err:
                    subprocess.check_call(
                        cmd,
                        stdout=out,
                        stderr=err,
                    )
        except Exception:
            with open(swagger_out) as out, open(swagger_err) as err:
                for name, stream in [["STDOUT", out], ["STDERR", err]]:
                    lines = stream.readlines()
                    for line in lines[-20:]:
                        print(f"{name}: {line}", file=sys.stderr)
            raise
    if parameters.debug:
        subprocess.check_call(f"/usr/bin/tree {parameters.output_path} || true ", shell=True, stdout=sys.stderr)
    # Generating mocks
    clients_root = os.path.join(parameters.output_path, "clients")
    for root, dirs, files in os.walk(clients_root):
        if root == clients_root:
            continue
        client_files = [fn for fn in files if fn.endswith("_client.go")]
        if not client_files:
            continue
        for client_file in client_files:
            src_file = os.path.join(root, client_file)
            dst_file = os.path.join(root, "mock_" + client_file)
            cmd = [
                parameters.mockgen_tool,
                f"-source={src_file}",
                f"-destination={dst_file}",
                f"-package={os.path.basename(root)}",
            ]
            if parameters.debug:
                print(f"Spawn: {shlex.join(cmd)}", file=sys.stderr)
            mockgen_out = os.path.join(root, f"{client_file}.mock.stdout")
            mockgen_err = os.path.join(root, f"{client_file}.mock.stderr")
            try:
                with open(mockgen_out, "w") as out:
                    with open(mockgen_err, "w") as err:
                        subprocess.check_call(
                            cmd,
                            stdout=out,
                            stderr=err,
                        )
            except Exception:
                with open(mockgen_out) as out, open(mockgen_err) as err:
                    for name, stream in [["STDOUT", out], ["STDERR", err]]:
                        lines = stream.readlines()
                        for line in lines[-20:]:
                            print(f"{name}: {line}", file=sys.stderr)
                raise

    if parameters.debug:
        subprocess.check_call(f"/usr/bin/tree {parameters.output_path} || true ", shell=True, stdout=sys.stderr)


def generate(schema, parameters: Parameters):
    go_mod_file = os.path.join(parameters.output_path, "go.mod")
    if parameters.debug:
        print(f"Go mod file: {go_mod_file}", file=sys.stderr)
    with open(go_mod_file, "w") as f:
        f.write(f"module a.yandex-team.ru/{parameters.go_package}\n")
        f.write("go 1.12\n")
    do_generate(schema, parameters)
    os.unlink(go_mod_file)


def make_code_tar(parameters: Parameters):
    with tarfile.open(os.path.join(parameters.output_path, parameters.code_archive), "w:gz") as tar_handle:
        for item in [CLIENT_PACKAGE, MODELS_PACKAGE]:
            for root, _, files in os.walk(os.path.join(parameters.output_path, item)):
                file_root = os.path.relpath(root, parameters.output_path)
                for file in files:
                    if not file.endswith(".go"):
                        continue
                    tar_handle.add(os.path.join(root, file), arcname=os.path.join(file_root, file))


def main():
    parameters = parse_args(sys.argv[1:])
    schema = sandbox_api.v1.Api.dict
    patch_schema(schema)
    generate(schema, parameters)
    make_code_tar(parameters)


if __name__ == '__main__':
    main()
