from __future__ import absolute_import, unicode_literals

import time
import functools as ft
from xml.etree import ElementTree

import six
import requests

from .. import format as common_format


class HostInfo(object):
    """
    Class which provides interface to yandex-team services to get info about host
    (currently uses Bot and Conductor).

    Example:

    .. code-block:: python

        >>> common.hardware.HostInfo.has_ssd("sandbox-server01.search.yandex.net")
        True
    """

    BOT_API_URL = "http://bot.yandex-team.ru/api"
    CONDUCTOR_API_URL = "http://c.yandex-team.ru/api"
    RETRY_ATTEMPTS = 3

    class RetryableError(Exception):
        """Raise this exception when conductor response code is not OK to retry request"""

    @classmethod
    def call_api(cls, url):
        """
        Make request to API

        :param url: url to call API method
        :return: content of API response
        """
        resp = requests.get(url, timeout=5)
        if resp.status_code == requests.codes.ok:
            return resp.text

        raise cls.RetryableError("API responses with code: {}".format(resp.status_code))

    @classmethod
    def _exceptions_handler(cls, func, default_value, *args, **kwargs):

        @ft.wraps(func)
        def wrapper():
            for _ in range(cls.RETRY_ATTEMPTS):
                try:
                    return func(*args, **kwargs)
                except cls.RetryableError:
                    time.sleep(0.1)
                except requests.RequestException:
                    return default_value
            return default_value

        return wrapper

    @classmethod
    def _dc(cls, hostname):
        url = "{base}/hosts/{host}".format(base=cls.CONDUCTOR_API_URL, host=hostname)
        content = cls.call_api(url)
        # XML parser does not support the `unicode` input
        tree = ElementTree.fromstring(content.encode("utf-8"))
        return tree.find("item").find("datacenter").text

    @classmethod
    def dc(cls, hostname):
        """
        Get host data center by specified hostname

        :param hostname: string with hostname
        :return: host's data center, "unk" in case of failure or lack of information
        :rtype: str
        """
        return cls._exceptions_handler(cls._dc, "unk", hostname)() or "unk"

    @classmethod
    def _has_ssd(cls, hostname):
        url = "{base}/consistof.php?name={host}".format(base=cls.BOT_API_URL, host=hostname)
        content = cls.call_api(url)
        # assume, that if golem response contains `/ssd/`, then ssd is on board
        ssd_in_response = '/ssd/' in content.lower()
        return ssd_in_response

    @classmethod
    def has_ssd(cls, hostname):
        """
        Check that solid state drive (ssd) is available on specified hostname

        :param hostname: string with hostname
        :return: flag that ssd is available
        :rtype: bool
        """
        return cls._exceptions_handler(cls._has_ssd, False, hostname)()

    @classmethod
    def neighbours(cls, this, hosts):
        hosts = common_format.brace_expansion(hosts)
        fqdns = {_: _.split(":")[0] for _ in hosts}
        dc = {_: cls.dc(_)[:3] for _ in six.itervalues(fqdns)}
        hosts.sort(key=lambda _: 0 if fqdns[_] == this.fqdn else (1 if dc[fqdns[_]] == this.dc else 2))
        return hosts
