from __future__ import absolute_import, division, print_function, unicode_literals

import six

import time
import collections
import itertools as it

abc = getattr(collections, "abc", collections)


def progressive_yielder(tick, max_tick, max_wait, sleep_first=True, sleep_func=None):
    """
    Generator which yields passed amount of time progressively.
    Very useful for retries with exponential backoff, like:

    .. code-block:: python

        for _, _ in progressive_yielder(1, 10, 180, sleep_first=False):
            try:
                perform_a_tricky_action()
            except TemporaryError:
                pass

    In the above example, an action will be being attempted for 3 minutes straight,
    and delay between retries will gradually increase from 1 to 10 seconds.

    :param tick:        Initial tick amount (in seconds, fractional).
    :param max_tick:    Maximum available tick amount in seconds.
    :param max_wait:    Maximum totally allowed wait amount in seconds.
    :param sleep_first: Sleep before first iteration.
    :param sleep_func:  sleep function, by default is time.sleep
    :return:            Total amount of wait in seconds.
    """

    sleep_func = sleep_func or time.sleep
    slept, started = tick if sleep_first else None, time.time()
    while (slept or 0) < max_wait:
        if slept is not None:
            sleep_func(min(tick, max_wait - slept))
        yield slept, tick
        slept = time.time() - started
        tick = min(max_tick, (tick * 55 / 34) if tick else .001)


def progressive_waiter(tick, max_tick, max_wait, checker, sleep_first=True, inverse=False, sleep_func=None):
    """
    Wait for given event progressively.

    :param tick:        Initial tick amount (in seconds, fractional).
    :param max_tick:    Maximum available tick amount in seconds.
    :param max_wait:    Maximum totally allowed wait amount in seconds.
    :param checker:     Callable which should return `True` in case of event has been occurred.
    :param sleep_first: Sleep before first iteration.
    :param inverse:     Inverse condition.
    :param sleep_func:  sleep function, by default is time.sleep
    :return:            Tuple of last checker value and total amount of wait in seconds.
    """

    ret, slept = None, 0
    for slept, _ in progressive_yielder(tick, max_tick, max_wait, sleep_first=sleep_first, sleep_func=sleep_func):
        ret = checker()
        if bool(ret) ^ inverse:
            break
    return ret, slept


def chain(*args, **kwargs):
    """
    Convert object(s) of almost any type to a generator.

    Useful if you don't want to construct a fake iterable if a function expects one:

    .. code-block:: python

        >>> def f(iterable):
        ...    for elem in chain(iterable):
        ...      print(elem)
        ...
        >>> f([1])
        1
        >>> f(1)  # could have crashed if ``f()`` didn't use ``chain()``
        1

    :param bool recurse: Recursively chain nested iterables
    """
    recurse = kwargs.get("recurse")  # move this parameter to arguments after moving to python3
    for obj in args:
        if _chainable(obj):
            iter_obj = _chain(obj) if recurse else obj
            for _ in iter_obj:
                yield _
        else:
            yield obj


def _chain(iterable):
    for obj in iterable:
        if _chainable(obj):
            for _ in _chain(obj):
                yield _
        else:
            yield obj


def _chainable(obj):
    return isinstance(obj, abc.Iterable) and not isinstance(obj, (six.text_type, six.binary_type))


def as_list(*args):
    """
    Safe method to convert object(s) of almost any type to a list.
    Removes None elements.
    """
    return [x for x in chain(*args) if x is not None]


def grouper(iterable, n):
    """ Make an iterator that returns list-chunks from the `iterable`.
    The chunks have length `n`, except maybe the remainder.

    >>> list(grouper([1, 2, 3, 4, 5, 6, 7], 3))
    [[1, 2, 3], [4, 5, 6], [7]]
    """

    itt = iter(iterable)
    while True:
        batch = list(it.islice(itt, n))
        if not batch:
            break
        yield batch


def grouper_longest(iterable, n, fillvalue=None):
    """ Make an iterator that returns tuple-chunks from the `iterable`.
    All chunks have length `n`; the remainder is padded to the length with `fillvalue`.

    >>> list(grouper_longest("ABCDEFG", 3, "x"))
    [("A", "B", "C"), ("D", "E", "F"), ("G", "x", "x")]
    """

    args = [iter(iterable)] * n
    return six.moves.zip_longest(*args, fillvalue=fillvalue)


def chunker(data, size):
    """ The simplest chunker - yields `size`-length chunks of the given input sequence. """
    i = 0
    while True:
        chunk = data[i:i + size]
        if not chunk:
            break
        yield chunk
        i += size


def merge_dicts(d1, d2):
    """ Merge two dicts recursively. """
    for d in (d2, d1):
        if not isinstance(d, dict):
            return d
    result = {k: merge_dicts(d1[k], v) if isinstance(d1.get(k), dict) else v for k, v in six.iteritems(d2)}
    for k, v in six.iteritems(d1):
        if k not in d2:
            result[k] = v
    return result


def count(iterable):
    """
    The method is COPIED from `cardinality` third-party package.
    Count the number of items that `iterable` yields.
    Equivalent to the expression
    ::
      len(iterable)
    - but it also works for iterables that do not support ``len()``.
      >>> import cardinality
      >>> cardinality.count([1, 2, 3])
      3
      >>> cardinality.count(i for i in range(500))
      500
      >>> def gen():
      ...     yield 'hello'
      ...     yield 'world'
      >>> cardinality.count(gen())
      2
    """
    if hasattr(iterable, '__len__'):
        return len(iterable)

    d = collections.deque(enumerate(iterable, 1), maxlen=1)
    return d[0][0] if d else 0
