package ru.yandex.sandbox.client;

import java.nio.charset.StandardCharsets;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import javax.servlet.http.HttpServletRequest;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.introspect.JacksonAnnotationIntrospector;
import com.squareup.okhttp.OkHttpClient;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import retrofit.RequestInterceptor;
import retrofit.RestAdapter;
import retrofit.RetrofitError;
import retrofit.client.OkClient;
import retrofit.client.Response;
import retrofit.mime.TypedInput;

import ru.yandex.qatools.yclient.AddAuthHeadersInterceptor;
import ru.yandex.qatools.yclient.RetrofitUtils;
import ru.yandex.sandbox.client.error.SandboxFakeException;
import ru.yandex.sandbox.client.retrofit.JacksonConverterKeepingResponse;
import ru.yandex.sandbox.client.retrofit.ResponseConversionException;


/**
 * @author albazh
 */
public class SandboxClientFactory {

    private static final Logger logger = LoggerFactory.getLogger(SandboxClientFactory.class);

    public final static String SANDBOX_API_ENDPOINT = "https://sandbox.yandex-team.ru/api/v1.0";

    public static RestAdapter.Builder builder(long timeoutSec, String oauthToken, int retries) {
        return builder(timeoutSec, oauthToken, SandboxClientFactory.SANDBOX_API_ENDPOINT, retries);
    }

    public static RestAdapter.Builder builder(long timeoutSec, String oauthToken) {
        return builder(timeoutSec, oauthToken, 0);
    }

    public static RestAdapter.Builder builder(long timeoutSec, String oauthToken, String endpoint, int retries) {
        final OkHttpClient okClient = buildOkHttpClient(timeoutSec, retries);
        ObjectMapper mapper = buildObjectMapper();
        return builder(okClient, mapper, oauthToken, endpoint);
    }

    /**
     * This is copy of {@link RetrofitUtils#builder(String, long, HttpServletRequest, String, String, RequestInterceptor)}
     * With replaced Converter ({@link JacksonConverterKeepingResponse})
     */
    public static RestAdapter.Builder builder(long timeoutSec, String oauthToken, String endpoint) {
        return builder(timeoutSec, oauthToken, endpoint, 0);
    }

    public static RestAdapter.Builder builder(OkHttpClient okClient, ObjectMapper mapper, String oauthToken,
            String endpoint
    ) {
        return new RestAdapter.Builder()
                .setClient(new OkClient(okClient))
                .setRequestInterceptor(new AddAuthHeadersInterceptor(null, oauthToken))
                .setConverter(new JacksonConverterKeepingResponse(mapper))
                .setEndpoint(endpoint)
                .setErrorHandler(SandboxClientFactory::handleError);
    }

    private static ObjectMapper buildObjectMapper() {
        return new ObjectMapper()
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false)
                .setSerializationInclusion(JsonInclude.Include.NON_NULL)
                .setAnnotationIntrospector(new JacksonAnnotationIntrospector());
    }

    private static OkHttpClient buildOkHttpClient(long timeoutSec, int retries) {
        OkHttpClient okClient = new OkHttpClient();
        okClient.setConnectTimeout(timeoutSec, TimeUnit.SECONDS);
        okClient.setReadTimeout(timeoutSec, TimeUnit.SECONDS);
        okClient.setWriteTimeout(timeoutSec, TimeUnit.SECONDS);
        if (retries > 0) {
            okClient.interceptors().add(new SandboxUnavailableInterceptor(retries));
        }
        return okClient;
    }

    private static Throwable handleError(RetrofitError cause) {
        Response response = cause.getResponse();
        if (response == null) {
            throw new RuntimeException("Response is empty", cause);
        }

        int statusCode = response.getStatus();
        if (statusCode < 200 || statusCode >= 300) {
            throw new RuntimeException("Sandbox returned response with code " + statusCode
                    + ". Body: " + responseBodyToString(response).orElse(""),
                    cause
            );
        }

        String responseBody;
        if (cause.getCause() instanceof ResponseConversionException) {
            ResponseConversionException parseException = ((ResponseConversionException) cause.getCause());
            responseBody = parseException.getResponseBytes()
                    .map(String::new)
                    .orElse("");
        } else {
            responseBody = responseBodyToString(response).orElse("");
        }

        // statusCode is successful, so we throw SandboxFakeException
        SandboxFakeException resultException = new SandboxFakeException(responseBody, cause);
        logger.warn("Status code is successful, but error is happened", resultException);
        throw resultException;
    }

    private static Optional<String> responseBodyToString(Response response) {
        Optional<String> body = Optional.empty();
        try {
            TypedInput responseBody = response.getBody();
            if (responseBody != null) {
                body = Optional.of(new String(IOUtils.toByteArray(responseBody.in()), StandardCharsets.UTF_8));
            }
        } catch (Exception e) {
            logger.warn("", e);
        }
        return body;
    }

}
