package ru.yandex.sandbox.client;

import java.util.List;

import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.sandbox.client.api.SandboxBatchResponse;
import ru.yandex.sandbox.client.api.SandboxTask;
import ru.yandex.sandbox.client.api.SandboxUpdateTask;
import ru.yandex.sandbox.client.error.SandboxFakeException;

import static java.util.Collections.singletonList;

/**
 * Class was created based on https://a.yandex-team.ru/arc/trunk/arcadia/testenv/core/engine/checks.py?rev=3354892#L647
 */
public class SandboxService {

    private static final Logger logger = LoggerFactory.getLogger(SandboxService.class);

    private final SandboxClient client;

    public SandboxService(SandboxClient client) {
        this.client = client;
    }

    public String create(SandboxTask task, boolean start) throws SandboxFakeException {
        SandboxTask createdTask = createTask(task);

        if (start) {
            startTask(Long.parseLong(createdTask.getId()));
        }

        return createdTask.getId();
    }

    public SandboxTask createTask(SandboxTask task) throws SandboxFakeException {
        SandboxTask response = client.createTask(task);
        logger.info("Task created, id: {}", response.getId());
        writeToLogIfDebugEnabled("Task created. ", response);
        return response;
    }

    public SandboxUpdateTask updateTask(String taskId, SandboxUpdateTask updateTask) {
        SandboxUpdateTask response = client.updateTask(updateTask, taskId);
        logger.info("Task updated, id: {}", taskId);
        writeToLogIfDebugEnabled("Task updated. ", response);
        return response;
    }

    public SandboxBatchResponse startTask(long taskId) throws SandboxFakeException {
        List<SandboxBatchResponse> response = client.startTasks(singletonList(taskId), "start");
        logger.info("Task started, id: {}", taskId);
        writeToLogIfDebugEnabled("Task started. ", response);
        return getFirstOrThrow(response);
    }

    public SandboxBatchResponse stopTask(long taskId) {
        List<SandboxBatchResponse> response = client.startTasks(singletonList(taskId), "stop");
        logger.info("Task stopped, id: {}", taskId);
        writeToLogIfDebugEnabled("Task stopped. ", response);
        return getFirstOrThrow(response);
    }

    private static void writeToLogIfDebugEnabled(String msg, Object obj) {
        if (logger.isDebugEnabled()) {

            String description;
            if (obj == null) {
                description = "null";
            } else if (obj instanceof List) {
                ToStringBuilder sb = new ToStringBuilder(obj, ToStringStyle.SHORT_PREFIX_STYLE);
                //noinspection unchecked
                ((List) obj).stream()
                        .map(item -> ReflectionToStringBuilder.toString(item, ToStringStyle.SHORT_PREFIX_STYLE))
                        .forEach(sb::append);
                description = sb.toString();
            } else {
                description = ReflectionToStringBuilder.toString(obj, ToStringStyle.SHORT_PREFIX_STYLE);
            }
            logger.debug(msg + description);
        }
    }

    private static SandboxBatchResponse getFirstOrThrow(List<SandboxBatchResponse> response) {
        if (response.isEmpty()) {
            throw new SandboxFakeException("List is empty");
        }
        return response.get(0);
    }

}
