from __future__ import absolute_import


class Deferred(object):
    def __init__(self, builder, value_repr="<>"):
        self.__builder = builder
        self.__called = False
        self.__repr = "<{}:{}>".format(type(self).__name__, value_repr)
        self.__value = None

    def __call__(self):
        if not self.__called:
            self.__value = self.__builder()
            self.__called = True
        return self.__value

    def __repr__(self):
        return self.__repr


class DeferredProxy(object):
    def __init__(self, builder):
        object.__setattr__(self, "__builder__", builder)
        object.__setattr__(self, "__real_object__", None)

    def __init_real_object__(self):
        real_object = object.__getattribute__(self, "__real_object__")
        if real_object is None:
            real_object = object.__getattribute__(self, "__builder__")()
            object.__setattr__(self, "__real_object__", real_object)
        return real_object

    def __getattribute__(self, name):
        real_object = object.__getattribute__(self, "__init_real_object__")()
        return getattr(real_object, name)

    def __setattr__(self, name, value):
        real_object = object.__getattribute__(self, "__init_real_object__")()
        setattr(real_object, name, value)

    def __getitem__(self, item):
        real_object = object.__getattribute__(self, "__init_real_object__")()
        return real_object[item]

    def __setitem__(self, item, value):
        real_object = object.__getattribute__(self, "__init_real_object__")()
        real_object[item] = value

    def __iter__(self):
        real_object = object.__getattribute__(self, "__init_real_object__")()
        return iter(real_object)

    def __contains__(self, item):
        real_object = object.__getattribute__(self, "__init_real_object__")()
        return item in real_object


class LazyLoader(object):
    """
    Module lazy loader. Replaces `self` on first `__getattr__()` method call with the imported module.
    It is a good tool to resolve cyclic dependencies - just load a module with this loader at some
    of loop's participant.
    """

    def __init__(self, module, symbol=None):
        """
        Default constructor.
        :param module:  A string with module's name to be lazy loaded.
        :param symbol:  Optional symbol name to be looked inside the module specified.
        """
        self.module, self.symbol = module, symbol

    def __getattr__(self, item):
        module = __import__('importlib').import_module(self.module)
        self = getattr(module, self.symbol) if self.symbol else module
        return getattr(self, item)

    def __call__(self, *args, **kwargs):
        module = __import__('importlib').import_module(self.module)
        self = getattr(module, self.symbol) if self.symbol else module
        return self(*args, **kwargs)
