from __future__ import absolute_import, unicode_literals

import io
import os
import json
import math
import random
import inspect
import tarfile
import hashlib
import logging
import binascii
import mimetypes
import itertools as it
import threading as th
import xml.parsers.expat
import xml.etree.ElementTree

import six
import boto3
import requests
import botocore.config
import boto3.exceptions
import boto3.s3.transfer
import botocore.exceptions
import requests.packages.urllib3.util.connection

try:
    # noinspection PyUnresolvedReferences
    import infra.skyboned.api
except ImportError:
    pass

try:
    # appropriate contrib in Arcaria broken for Windows
    import magic
except ImportError:
    magic = None

from six.moves import urllib

from .. import tvm
from .. import rest
from .. import config
from .. import format
from .. import context
from .. import patterns
from .. import itertools
from ..types import misc as ctm
from ..types import resource as ctr

from . import schema
from . import compression


class MimeTypes(object):
    MAGIC_SIZE = 2048
    JSON = "application/json"
    OCTET_STREAM = "application/octet-stream"
    DEFAULT_TYPE = OCTET_STREAM

    @classmethod
    def get_type(cls, file_name, data=None, logger=None):
        mime_type = mimetypes.guess_type(file_name or "")[0]
        if mime_type is None and magic is not None:
            try:
                if data is None:
                    mime_type = magic.from_file(file_name, mime=True)
                else:
                    mime_type = magic.from_buffer(data, mime=True)
            except magic.MagicException as ex:
                if logger is not None:
                    logger.warning("Cannot detect MIME type of file %s: %s", file_name, ex)
        # use "" instead of DEFAULT_TYPE to save disk space
        return "" if not mime_type or mime_type == cls.DEFAULT_TYPE else mime_type


class TarMetaCollector(object):
    def __init__(self, logger=None, hash_calculator_factory=None):
        self.metadata = []
        self.__offset = 0
        self.__buf = []
        self.__buf_size = 0
        self.__data_blocks = 0
        self.__data_size = 0
        self.__longname = None
        self.__longlink = None
        self.__last_header = None
        self.__dirs = set()
        self._canceled = False
        self.__magic_buffer = None
        self.__hash_calculator = None
        self._logger = logger
        self._hash_calculator_factory = hash_calculator_factory

    def chunker(self):
        if self.__buf_size >= tarfile.BLOCKSIZE:
            full_data = b"".join(self.__buf)
            self.__buf = []

            idx = 0
            while idx + tarfile.BLOCKSIZE <= self.__buf_size:
                yield full_data[idx: idx + tarfile.BLOCKSIZE]
                idx += tarfile.BLOCKSIZE
            if idx != self.__buf_size:
                self.__buf.append(full_data[idx:self.__buf_size])
            self.__buf_size -= idx

    def _cancel(self, error):
        self._canceled = True
        self.metadata = [
            schema.MDSFileMeta.create(
                type=ctr.FileType.ERROR,
                path=error,
            )
        ]

    def update_meta(self, header):
        if header.isdir():
            dirpath, dirnames, filenames = header.name.rstrip(os.sep), [], []
        else:
            dirpath, filename = (
                os.path.split(header.name)
                if os.sep in header.name else
                ("", header.name)
            )
            dirnames, filenames = [], [filename]

        dirname = dirpath
        if dirname and dirname not in self.__dirs:
            self.metadata.append(schema.MDSFileMeta.create(
                type=ctr.FileType.DIR,
                path=dirname,
            ))
            self.__dirs.add(dirname)
        for dirname in dirnames:
            dirname = os.path.join(dirpath, dirname)
            self.metadata.append(schema.MDSFileMeta.create(
                type=ctr.FileType.DIR,
                path=dirname,
            ))
            self.__dirs.add(dirname)
        for filename in filenames:
            filepath = os.path.join(dirpath, filename)
            filepath_rel = filepath
            link_name = None

            file_size = header.size
            executable = bool(header.mode & 0o111)
            if header.issym():
                link_name = header.linkname

            if link_name is not None:
                self.metadata.append(schema.MDSFileMeta.create(
                    type=ctr.FileType.SYMLINK,
                    path=filepath_rel,
                    symlink=link_name,
                ))
                continue
            elif file_size == 0:
                self.metadata.append(schema.MDSFileMeta.create(
                    type=ctr.FileType.TOUCH,
                    path=filepath_rel,
                    executable=executable,
                ))
                continue

            filemeta = schema.MDSFileMeta.create(
                type=ctr.FileType.FILE,
                path=filepath_rel,
                size=file_size,
                offset=header.offset_data,
            )
            if executable:
                filemeta.executable = True
            self.metadata.append(filemeta)

    def process(self, data):
        if data == b"" or self._canceled:
            return
        self.__buf.append(data)
        self.__buf_size += len(data)

        try:
            for chunk in self.chunker():
                if not self.__data_blocks:
                    if isinstance(self.__longname, list):
                        self.__longname = six.ensure_str(
                            b"".join(self.__longname)[:self.__last_header.size].rstrip(b"\0")
                        )
                    if isinstance(self.__longlink, list):
                        self.__longlink = six.ensure_str(
                            b"".join(self.__longlink)[:self.__last_header.size].rstrip(b"\0")
                        )
                    header = tar_header(chunk)
                    if header.type == tarfile.GNUTYPE_SPARSE:
                        self._cancel("Cannot browse tar with sparse files")
                        return
                    header.offset = self.__offset

                    if self.__longname is not None:
                        header.name = self.__longname
                        header.offset = self.__last_header.offset

                    if self.__longlink is not None:
                        header.linkname = self.__longlink
                        header.offset = self.__last_header.offset

                    self.__last_header = header

                    header.offset_data = self.__offset + tarfile.BLOCKSIZE
                    self.__offset += (
                        ((header.size + tarfile.BLOCKSIZE - 1) // tarfile.BLOCKSIZE + 1) * tarfile.BLOCKSIZE
                    )
                    self.__data_blocks = (header.size + tarfile.BLOCKSIZE - 1) // tarfile.BLOCKSIZE
                    self.__data_size = header.size

                    if header.type == tarfile.GNUTYPE_LONGNAME:
                        self.__longname = []
                        continue
                    elif header.type == tarfile.GNUTYPE_LONGLINK:
                        self.__longlink = []
                        continue
                    else:
                        self.__longname = None
                        self.__longlink = None

                    if header.isfile() or header.isdir() or header.issym():
                        self.update_meta(header)
                        if header.isfile():
                            self.__magic_buffer = io.BytesIO()
                            if self._hash_calculator_factory is not None and header.size:
                                self.__hash_calculator = self._hash_calculator_factory()
                else:
                    if self.__last_header.type == tarfile.GNUTYPE_LONGNAME:
                        self.__longname.append(chunk)
                    elif self.__last_header.type == tarfile.GNUTYPE_LONGLINK:
                        self.__longlink.append(chunk)
                    else:
                        if self.__magic_buffer is not None:
                            if self.__magic_buffer.tell() < MimeTypes.MAGIC_SIZE:
                                self.__magic_buffer.write(chunk)
                            if self.__data_blocks == 1:
                                last_meta = self.metadata[-1]
                                mime_type = MimeTypes.get_type(
                                    last_meta.path, self.__magic_buffer.getvalue()[:last_meta.size],
                                    logger=self._logger
                                )
                                last_meta.mime = mime_type
                                self.__magic_buffer = None
                        if self.__hash_calculator is not None:
                            self.__hash_calculator.update(chunk[:min(self.__data_size, len(chunk))])
                            if self.__data_blocks == 1:
                                self.__hash_calculator.stop()
                                last_meta = self.metadata[-1]
                                last_meta.md5 = self.__hash_calculator.md5
                                last_meta.sha1_blocks = self.__hash_calculator.sha1_blocks
                                self.__hash_calculator = None
                    self.__data_blocks -= 1
                    self.__data_size -= len(chunk)
        except tarfile.EOFHeaderError:
            # End of tar archive, do nothing
            pass
        except tarfile.TarError as ex:
            self._cancel(str(ex))


class TgzMetaCollector(TarMetaCollector):
    def __init__(self, index=None, logger=None):
        self.__index = compression.gzip.Index() if index is None else index
        self.__offsets = set()
        super(TgzMetaCollector, self).__init__(logger=logger)

    @property
    def index(self):
        return self.__index

    @property
    def offsets(self):
        return self.__offsets

    def update_meta(self, header):
        super(TgzMetaCollector, self).update_meta(header)
        if not self.metadata:
            return
        offset = self.metadata[-1].offset
        if offset:
            self.__offsets.add(offset)

    def process(self, data):
        if data == b"" or self._canceled:
            return
        try:
            decompressed_data = self.__index.build(data)
        except compression.gzip.ZException as ex:
            self._cancel(str(ex))
            return
        return super(TgzMetaCollector, self).process(decompressed_data)


class HashCalculator(object):
    SHA1_BLOCK_SIZE = 4 << 20  # fixed block size from Skynet Skybone

    def __init__(self, path=None):
        super(HashCalculator, self).__init__()
        self.__file = path and open(path, "rb")
        self.__md5 = hashlib.md5()
        self.__sha1_blocks = [hashlib.sha1()]
        self.__last_sha1_block_size = 0
        self.__queue = None
        self.__thread = None
        self.start()

    @property
    def md5(self):
        return self.__md5.hexdigest()

    @property
    def sha1_blocks(self):
        return (_.hexdigest() for _ in self.__sha1_blocks)

    def start(self):
        self.__queue = six.moves.queue.Queue()
        self.__thread = th.Thread(target=self.run)
        self.__thread.start()

    def stop(self):
        if self.__thread and self.__thread.is_alive():
            self.__queue.put(None)
            self.__thread.join()

    def run(self):
        try:
            while True:
                data = self.__file.read(self.SHA1_BLOCK_SIZE) if self.__file else self.__queue.get()
                if not data:
                    break
                self._update(data)
        finally:
            if self.__file:
                self.__file.close()

    def _update(self, data):
        data = six.ensure_binary(data)
        self.__md5.update(data)
        for chunk in itertools.chunker(data, self.SHA1_BLOCK_SIZE):
            if self.__last_sha1_block_size == self.SHA1_BLOCK_SIZE:
                self.__sha1_blocks.append(hashlib.sha1())
                self.__last_sha1_block_size = 0
            self.__last_sha1_block_size += len(chunk)
            if self.__last_sha1_block_size > self.SHA1_BLOCK_SIZE:
                self.__last_sha1_block_size -= self.SHA1_BLOCK_SIZE
                self.__sha1_blocks[-1].update(chunk[:self.SHA1_BLOCK_SIZE - self.__last_sha1_block_size])
                self.__sha1_blocks.append(hashlib.sha1())
                self.__sha1_blocks[-1].update(chunk[self.SHA1_BLOCK_SIZE - self.__last_sha1_block_size:])
            else:
                self.__sha1_blocks[-1].update(chunk)

    def update(self, data):
        if data != b"":
            self.__queue.put(data)


def file_chunker(fileobj, chunk_size):
    while True:
        data = fileobj.read(chunk_size)
        if not data:
            break
        yield data


def tar_header(data):
    if six.PY2:
        header = tarfile.TarInfo.frombuf(data)
    else:
        header = tarfile.TarInfo.frombuf(data, "utf-8", "ignore")
    return header


class ChunkedFileObject(io.RawIOBase):
    """
    File-like object with chunked data source (generator) as backend
    """

    def __init__(
        self, chunker, size, handlers=None, hash_calculator_factory=HashCalculator, file_name=None, logger=None
    ):
        super(ChunkedFileObject, self).__init__()
        self.__chunker = chunker
        self.__remained = size
        self.__chunk = b""
        self.__offset = 0
        self.__pos = 0
        self.__hash_calculator = hash_calculator_factory and hash_calculator_factory()
        self.__handlers = handlers or []
        self.__mime_type = None
        self.__tar_name = None
        self.__file_name = file_name
        self.__logger = logger

    def __len__(self):
        return self.__remained

    @property
    def hashes(self):
        return self.__hash_calculator

    @property
    def handlers(self):
        return self.__handlers

    def __pre_read(self, size):
        buffer = io.BytesIO(self.__chunk)
        buffer.seek(0, 2)
        while buffer.tell() < size:
            try:
                buffer.write(next(self.__chunker))
            except StopIteration:
                break
        self.__chunk = buffer.getvalue()
        return self.__chunk[:size]

    @property
    def mime_type(self):
        if self.__mime_type is not None:
            return self.__mime_type
        assert self.__offset == 0, "Cannot determine MIME type by offset {}".format(self.__offset)
        max_magic_size = min(MimeTypes.MAGIC_SIZE, self.__remained)
        data = self.__pre_read(max_magic_size)
        self.__mime_type = MimeTypes.get_type(self.__file_name, data, logger=self.__logger)
        return self.__mime_type

    @property
    def tar_name(self):
        if self.__tar_name is not None:
            return self.__tar_name
        assert self.__offset == 0, "Cannot determine tar name by offset {}".format(self.__offset)
        max_buffer_size = min(tarfile.BLOCKSIZE * 2, self.__remained)
        data = self.__pre_read(max_buffer_size)
        try:
            header = tar_header(data[:tarfile.BLOCKSIZE])
        except tarfile.InvalidHeaderError:
            self.__tar_name = ""
        else:
            if header.type == tarfile.GNUTYPE_LONGNAME:
                self.__tar_name = data[tarfile.BLOCKSIZE:tarfile.BLOCKSIZE + header.size].strip(b"\x00/").decode()
            else:
                self.__tar_name = header.name
        self.__tar_name = self.__tar_name.split("/", 1)[0]
        return self.__tar_name

    def tell(self):
        return self.__pos

    def readable(self):
        return True

    def read(self, n=-1):
        if self.__remained <= 0:
            return b""
        if n < 0:
            n = self.__remained
        else:
            n = min(n, self.__remained)
        data = [self.__chunk[self.__offset:min(self.__offset + n, len(self.__chunk))]]
        self.__offset += n
        data_size = len(data[0])
        while data_size < n:
            try:
                self.__chunk = next(self.__chunker)
            except StopIteration:
                break
            chunk_size = len(self.__chunk)
            data_size += chunk_size
            if data_size > n:
                self.__offset = n - data_size + chunk_size
                chunk = self.__chunk[:self.__offset]
            else:
                chunk, self.__chunk, self.__offset = self.__chunk, b"", 0
            data.append(chunk)
        data = b"".join(data)
        self.__remained -= len(data)
        self.__pos += len(data)
        if self.__hash_calculator:
            self.__hash_calculator.update(data)
        for handler in self.__handlers:
            handler.process(data)
        return data


class TARReader(object):
    """
    Read tar stream and yields pairs (<header>, <file-like object>) for files and (<header>, None) for others
    """

    def __init__(self, fileobj, logger, handlers=None, hashes_calculator_factory=HashCalculator):
        self._fileobj = fileobj
        self._logger = logger
        self._handlers = handlers or []
        self._hashes_calculator_factory = hashes_calculator_factory

    def __iter__(self):
        chunker = file_chunker(self._fileobj, tarfile.BLOCKSIZE)
        longname = None
        longlink = None
        long_offset = None
        offset = 0
        while True:
            try:
                header = tar_header(next(chunker))
                header.offset = offset
                header.offset_data = offset + tarfile.BLOCKSIZE
                if long_offset:
                    header.offset = long_offset
                    long_offset = None
                offset += ((header.size + tarfile.BLOCKSIZE - 1) // tarfile.BLOCKSIZE + 1) * tarfile.BLOCKSIZE

                if header.type == tarfile.GNUTYPE_LONGNAME:
                    longname = six.ensure_str(b"".join(
                        self._fileobj.read(tarfile.BLOCKSIZE)
                        for _ in range(header.size // tarfile.BLOCKSIZE + 1)
                    )[:header.size].rstrip(b"\0"))
                    long_offset = header.offset
                    continue
                elif header.type == tarfile.GNUTYPE_LONGLINK:
                    longlink = six.ensure_str(b"".join(
                        self._fileobj.read(tarfile.BLOCKSIZE)
                        for _ in range(header.size // tarfile.BLOCKSIZE + 1)
                    )[:header.size].rstrip(b"\0"))
                    long_offset = header.offset
                    continue
                elif not (header.isfile() or header.isdir() or header.issym()):
                    if self._logger:
                        self._logger.warning("'%s' has unsupported header type: %s", header.name, header.type)
                    for _ in range(header.size // tarfile.BLOCKSIZE + 1):
                        self._fileobj.read(tarfile.BLOCKSIZE)
                    longname = None
                    longlink = None
                    continue
                if longname is not None:
                    header.name = longname
                    longname = None
                if longlink is not None:
                    header.linkname = longlink
                    longlink = None
            except tarfile.EOFHeaderError:
                break
            yield header, (
                ChunkedFileObject(
                    chunker, header.size, handlers=self._handlers,
                    hash_calculator_factory=self._hashes_calculator_factory,
                    file_name=header.name, logger=self._logger
                )
                if header.isfile() and header.size else
                None
            )


class TARBuilder(io.RawIOBase):
    """
    Build TAR stream from directory
    """
    def __init__(self, path, buf_size=tarfile.BLOCKSIZE * 32):
        super(TARBuilder, self).__init__()
        self.__path = path
        self.__buf_size = buf_size
        self.__builder = self.__build()
        self.__tar = tarfile.open(mode="w|", fileobj=io.BytesIO())
        self.__buffer = bytearray()

    def __walk(self):
        base_name = os.path.basename(self.__path)
        offset = len(self.__path) - len(base_name)
        yield base_name, self.__path, None
        for root, dirs, files in os.walk(self.__path):
            for dir_name in dirs:
                full_path = os.path.join(root, dir_name)
                yield full_path[offset:], full_path, None
            for file_name in files:
                full_path = os.path.join(root, file_name)
                file_obj = open(full_path, "rb") if not os.path.islink(full_path) else None
                yield full_path[offset:], full_path, file_obj
                if file_obj is not None:
                    file_obj.close()

    def __build(self):
        for file_name, file_path, file_obj in self.__walk():
            tar_info = self.__tar.gettarinfo(arcname=file_name, name=file_path, fileobj=file_obj)
            if tar_info.type == tarfile.LNKTYPE:
                tar_info.type = tarfile.REGTYPE
                tar_info.linkname = ""
                tar_info.size = os.fstat(file_obj.fileno()).st_size
            yield tar_info.tobuf(format=tarfile.GNU_FORMAT)
            if file_obj is not None:
                total_size = 0
                blocks, remainder = divmod(tar_info.size, self.__buf_size)
                for _ in six.moves.range(blocks):
                    buf = file_obj.read(self.__buf_size)
                    if len(buf) < self.__buf_size:
                        raise MDS.UnexpectedEndOfFile("Unexpected end of file {}".format(file_name))
                    yield buf
                    total_size += len(buf)
                if remainder != 0:
                    buf = file_obj.read(remainder)
                    if len(buf) < remainder:
                        raise MDS.UnexpectedEndOfFile("Unexpected end of file {}".format(file_name))
                    yield buf
                    total_size += len(buf)
                remainder = total_size % tarfile.BLOCKSIZE
                if remainder > 0:
                    yield tarfile.NUL * (tarfile.BLOCKSIZE - remainder)

    def readable(self):
        return True

    def read(self, n=-1):
        if n == 0:
            return b""
        while n < 0 or len(self.__buffer) <= n:
            data = next(self.__builder, None)
            if data is None:
                break
            self.__buffer.extend(data)
        if len(self.__buffer) == 0:
            return b""
        if n > 0:
            result = self.__buffer[:n]
            del self.__buffer[:n]
        else:
            result = self.__buffer[:]
            del self.__buffer[:]
        return bytes(result)


def mds_link(key, namespace=None, mds_settings=None):
    """
    Make MDS url for download

    :param key: MDS key
    :param namespace: MDS namespace, if not defined, Sandbox's namespace will be used
    :param mds_settings: MDS specific config section
    :return: MDS url
    """
    if mds_settings is None:
        mds_settings = config.Registry().common.mds
    namespace = namespace or ctr.DEFAULT_MDS_NAMESPACE
    return "{}/get-{}/{}".format(mds_settings.dl.url, namespace, urllib.parse.quote(key))


def s3_link(key, namespace=None, mds_settings=None):
    """
    Make S3 url for download

    :param key: S3 key
    :param namespace: S3 bucket
    :param mds_settings: MDS specific config section
    :return: MDS url
    """
    if mds_settings is None:
        mds_settings = config.Registry().common.mds
    namespace = namespace or ctr.DEFAULT_S3_BUCKET
    return "{}/{}/{}".format(mds_settings.s3.url, namespace, urllib.parse.quote(key))


def _rest_client(url, timeout=None):
    logger = logging.getLogger("rest_client")
    rest_client = rest.Client(url, total_wait=timeout, logger=logger) >> rest.Client.PLAINTEXT
    object.__setattr__(
        rest_client, "RETRYABLE_CODES", rest_client.RETRYABLE_CODES + (requests.codes.INTERNAL_SERVER_ERROR,)
    )
    return rest_client


@six.add_metaclass(patterns.SingletonMeta)
class MDS(object):
    """
    Class encapsulating methods for working with MDS
    """

    class Exception(Exception):
        pass

    class RBTorrentError(Exception):
        pass

    class RBTorrentMismatch(RBTorrentError):
        pass

    class RBTorrentRegistrationError(RBTorrentError):
        pass

    class RBTorrentRemoveError(RBTorrentError):
        pass

    class RBTorrentForbidden(RBTorrentError):
        pass

    class FatalError(Exception):
        pass

    class ZeroSize(FatalError):
        pass

    class InsufficientSpace(Exception):
        __muted__ = True

    class InsufficientBucketSpace(Exception):
        __muted__ = True

    class TooManyRequests(Exception):
        __muted__ = True

    class UnexpectedEndOfFile(Exception):
        pass

    # Minimal upload speed in bytes per second to calculate upload timeout - 10MiB per second.
    UPLOAD_MIN_SPEED = 10 << 20

    DEFAULT_NAMESPACE = ctr.DEFAULT_MDS_NAMESPACE

    DEFAULT_RBTORRENT_REGISTRATION_TIMEOUT = 600  # in seconds

    _download_link = staticmethod(mds_link)

    @staticmethod
    def _auth_header(service, token, logger):
        if token is not None:
            return {"Authorization": "Basic " + token}
        try:
            return {ctm.HTTPHeader.SERVICE_TICKET: tvm.TVM.get_service_ticket([service])[service]} if service else {}
        except tvm.TVM.Error as ex:
            logger.error("Got error from tvmtool: %s", ex)
            return {}

    def upload(
        self, path, mds_name, tvm_service=None, namespace=None, resource_id=None,
        ttl=None, size=None, timeout=None, logger=None, mds_settings=None, token=None
    ):
        """
        Upload file to MDS

        :param path: path to file or file object to upload
        :param mds_name: name used to form MDS url
        :param tvm_service: TVM service for MDS
        :param namespace: MDS namespace, if not defined, Sandbox's namespace will be used
        :param resource_id: resource id for MDS metadata
        :param ttl: TTL in days
        :param size: file size in bytes, if not defined, will be got from file system
        :param timeout: timeout in seconds, if not defined, will be calculated from file size
        :param logger: custom logger, if not defined, logging will be used
        :param mds_settings: MDS specific config section
        :param token: token used to authenticate on MDS site (for backward compatibility)
        :return: MDS key
        """
        if mds_settings is None:
            mds_settings = config.Registry().common.mds
        logger = logger or logging
        namespace = namespace or self.DEFAULT_NAMESPACE
        if resource_id is not None:
            if mds_name:
                mds_name = "{}/{}".format(resource_id, mds_name)
            else:
                mds_name = str(resource_id)
        is_file_object = not isinstance(path, six.string_types)
        assert not is_file_object or size is not None, "size is required if file object passed"
        try:
            if timeout is None:
                if size is None:
                    size = os.path.getsize(path)
                timeout = max(60, int(math.ceil(float(size) / self.UPLOAD_MIN_SPEED)))
            if not size:
                raise self.ZeroSize("File with zero size cannot be uploaded to MDS: {}".format(path))
            url = "{}/upload-{}/{}".format(mds_settings.up.url, namespace, mds_name)
            logger.debug("Uploading data of size %s via URL %s", format.size2str(size), url)
            with (context.NullContextmanager(enter_obj=path) if is_file_object else open(path, "rb")) as fh:
                params = {}
                if ttl:
                    params["expire"] = "{}d".format(ttl)

                headers = {ctm.HTTPHeader.CONTENT_LENGTH: str(size)}
                headers.update(self._auth_header(tvm_service, token, logger))
                resp = requests.post(
                    url,
                    data=fh,
                    params=params,
                    headers=headers,
                    timeout=timeout
                )
            if resp.status_code == requests.codes.FORBIDDEN:
                try:
                    key = xml.etree.ElementTree.fromstring(resp.content).find("key").text
                    logger.info("Data already uploaded with key %s", key)
                except (AttributeError, KeyError, xml.etree.ElementTree.ParseError, xml.parsers.expat.ExpatError) as ex:
                    raise self.Exception(
                        "Data upload failed - unable to parse FORBIDDEN response: {}. Data is: {}".format(
                            ex, resp.content
                        )
                    )
            elif resp.status_code != requests.codes.OK:
                raise self.Exception(
                    "Data upload failed - service respond code {}: {}".format(resp.status_code, resp.reason)
                )
            else:
                try:
                    key = xml.etree.ElementTree.fromstring(resp.content).attrib["key"]
                except (AttributeError, KeyError, xml.etree.ElementTree.ParseError, xml.parsers.expat.ExpatError) as ex:
                    raise self.Exception(
                        "Data upload failed - unable to parse response: {}. Data is: {}".format(ex, resp.content)
                    )
            return key
        finally:
            if isinstance(path, ChunkedFileObject):
                path.hashes.stop()

    def remove_torrent(self, skynet_id, timeout=None, logger=None, mds_settings=None):
        logger = logger or logging
        if mds_settings is None:
            mds_settings = config.Registry().common.mds

        rest_client = _rest_client(mds_settings.rb.url, timeout=timeout)
        logger.debug("Removing torrent %s from MDS", skynet_id)
        try:
            resp = rest_client.remove.read(rbtorrent_id=skynet_id)
        except rest_client.HTTPError as ex:
            raise self.RBTorrentRemoveError("Remove torrent failed: {}".format(ex))
        return resp


class S3(MDS):
    """
    Class encapsulating methods for working with MDS S3
    """
    PROXY_CACHE_TTL = 5  # in seconds
    PROXY_REQUEST_TIMEOUT = 60  # in seconds
    PROXY_PORT = 4080
    DEFAULT_NAMESPACE = ctr.DEFAULT_S3_BUCKET
    DEFAULT_SKYBONED_ANNOUNCE_TIMEOUT = 300  # in seconds
    DEFAULT_MULTIPART_CHUNK_SIZE = 8 << 20  # in bytes
    MAX_MULTIPART_CHUNKS = 10000
    MAX_MULTIPART_CHUNK_SIZE = 104 << 20  # for max resource size: ((104 << 20) * 10000) >> 30 ~ 1TiB
    FILE_CHUNK_SIZE = 8192  # in bytes

    EMPTY_MD5 = hashlib.md5().hexdigest()

    ERRORS_MAPPING = {
        "TooManyRequests": MDS.TooManyRequests,
        "ServiceTotalSizeQuotaExceeded": MDS.InsufficientSpace,
        "An error occurred (429)": MDS.TooManyRequests,
        "BucketMaxSizeExceeded": MDS.InsufficientBucketSpace,
    }

    _download_link = staticmethod(s3_link)

    def __init__(self):
        super(S3, self).__init__()
        self._tls = th.local()

        # TODO SANDBOX-8957 remove
        self.skyboned_enabled = config.Registry().common.installation not in ctm.Installation.Group.LOCAL

    @classmethod
    @patterns.ttl_cache(PROXY_CACHE_TTL)
    def __proxy_fqdn(cls, s3_url):
        return six.ensure_str(_rest_client(s3_url, timeout=cls.PROXY_REQUEST_TIMEOUT)["hostname"].read())

    def s3_client(self, mds_settings=None):
        if mds_settings is None:
            mds_settings = config.Registry().common.mds
        client_kws = dict(endpoint_url=mds_settings.s3.url)
        boto_conf = botocore.config.Config(
            retries={"max_attempts": mds_settings.s3.retry_attempts, "mode": mds_settings.s3.retry_mode}
        )
        if mds_settings.s3.use_proxy:
            boto_conf.proxies = dict(http="{}:{}".format(self.__proxy_fqdn(mds_settings.s3.url), self.PROXY_PORT))
        client_kws.update(config=boto_conf)
        if not hasattr(self._tls, "s3_session"):
            self._tls.s3_session = boto3.session.Session()
        return self._tls.s3_session.client("s3", **client_kws)

    def _multipart_chunk_size(self, size):
        if size is None:
            return self.MAX_MULTIPART_CHUNK_SIZE
        chunk_size = max(
            (size + self.MAX_MULTIPART_CHUNKS - 1) // self.MAX_MULTIPART_CHUNKS,
            self.DEFAULT_MULTIPART_CHUNK_SIZE
        )
        return (
            (chunk_size + self.DEFAULT_MULTIPART_CHUNK_SIZE - 1) //
            self.DEFAULT_MULTIPART_CHUNK_SIZE *
            self.DEFAULT_MULTIPART_CHUNK_SIZE
        )

    def upload(
        self, path_or_obj, s3_name, namespace=None, resource_id=None, size=None, logger=None, mds_settings=None,
        content_type=None, **_
    ):
        """
        Upload file to MDS S3

        :param path_or_obj: path to file or file object to upload
        :param s3_name: name used to form S3 key
        :param namespace: S3 bucket
        :param resource_id: resource id for MDS metadata
        :param size: file size in bytes, if not defined, will be got from file system
        :param logger: custom logger, if not defined, logging will be used
        :param mds_settings: MDS specific config section
        :param content_type: MIME type
        :return: S3 key, uploaded size
        """
        logger = logger or logging
        namespace = namespace or self.DEFAULT_NAMESPACE
        s3_key = s3_name
        if resource_id is not None:
            if s3_name:
                s3_key = "{}/{}".format(resource_id, s3_name)
            else:
                s3_key = str(resource_id)
        is_file_object = not isinstance(path_or_obj, six.string_types)
        try:
            if size is None and not is_file_object:
                size = os.path.getsize(path_or_obj)
            if size == 0:
                raise self.ZeroSize("File with zero size cannot be uploaded to MDS: {}".format(path_or_obj))
            try:
                extra_args = {}
                content_type_log_str = ""
                if content_type is not None:
                    extra_args["ContentType"] = content_type
                    content_type_log_str = " as {}".format(content_type)
                s3 = self.s3_client(mds_settings)
                str_size = "undefined" if size is None else format.size2str(size)
                if is_file_object:
                    logger.debug(
                        "Uploading data with key %s of size %s to S3 bucket %s%s",
                        s3_key, str_size, namespace, content_type_log_str
                    )
                    upload_method = s3.upload_fileobj
                else:
                    logger.debug(
                        "Uploading file %s with key %s of size %s to S3 bucket %s%s",
                        path_or_obj, s3_key, str_size, namespace, content_type_log_str
                    )
                    upload_method = s3.upload_file
                chunk_size = self._multipart_chunk_size(size)
                transfer_config = boto3.s3.transfer.TransferConfig(use_threads=False, multipart_chunksize=chunk_size)
                uploaded_size = [0]
                upload_method(
                    path_or_obj, namespace, s3_key,
                    Config=transfer_config, ExtraArgs=extra_args,
                    Callback=lambda s: uploaded_size.__setitem__(0, uploaded_size[0] + s),
                )
            except (boto3.exceptions.S3UploadFailedError, botocore.exceptions.ClientError) as ex:
                ex_str = str(ex)
                for err_code, exception in self.ERRORS_MAPPING.items():
                    if err_code in ex_str:
                        raise exception("Failed to upload resource #{} ({}) to MDS: {}".format(
                            resource_id, namespace, ex
                        ))
                raise self.Exception(str(ex) or str(type(ex)))
            except botocore.exceptions.BotoCoreError as ex:
                raise self.Exception(str(ex) or str(type(ex)))
        finally:
            if isinstance(path_or_obj, ChunkedFileObject):
                path_or_obj.hashes.stop()
        return s3_key, uploaded_size[0]

    def upload_file(
        self, path_or_obj, s3_name, namespace=None, resource_id=None,
        size=None, executable=None, logger=None, mds_settings=None,
    ):
        """
        Upload file to MDS S3

        :param path_or_obj: path to file or file object to upload
        :param s3_name: name used to form S3 key
        :param namespace: S3 bucket
        :param resource_id: resource id for MDS metadata
        :param size: file size in bytes, if not defined, will be got from file system
        :param executable: whether file is executable
        :param logger: custom logger, if not defined, logging will be used
        :param mds_settings: MDS specific config section
        :rtype (str, list)
        :return: S3 key of uploaded file and the metadata
        """
        check = isinstance(path_or_obj, six.string_types) or executable is not None
        assert check, "executable is required if file object passed"
        hashes = self._get_hashes(path_or_obj)
        if isinstance(path_or_obj, ChunkedFileObject):
            mime_type = path_or_obj.mime_type
        elif isinstance(path_or_obj, six.string_types):
            mime_type = MimeTypes.get_type(path_or_obj, logger=logger)
        else:
            assert size is not None, "size is required if file object passed"
            path_or_obj = ChunkedFileObject(
                file_chunker(path_or_obj, self.FILE_CHUNK_SIZE), size, file_name=s3_name, logger=logger
            )
            mime_type = path_or_obj.mime_type
        mds_key, uploaded_size = self.upload(
            path_or_obj, s3_name, namespace=namespace, resource_id=resource_id,
            size=size, logger=logger, mds_settings=mds_settings, content_type=mime_type
        )
        metadata = schema.MDSFileMeta.create(
            key=mds_key,
            type=ctr.FileType.FILE,
            path=s3_name,
            executable=executable if executable is not None else os.access(path_or_obj, os.X_OK),
            size=uploaded_size,
            mime=mime_type,
        )
        if hashes is not None:
            metadata.md5 = hashes.md5
            metadata.sha1_blocks = hashes.sha1_blocks
        return mds_key, [metadata]

    def uploaded_keys(self, bucket, resource_id, mds_settings=None, s3_client=None, retry_timeout=20):
        prefix = "{}/".format(resource_id)
        cont_token = ""
        ex = None
        while cont_token is not None:
            for _ in itertools.progressive_yielder(1, 2, retry_timeout, sleep_first=False):
                ex = None
                try:
                    s3 = self.s3_client(mds_settings) if s3_client is None else s3_client
                    result = s3.list_objects_v2(
                        Bucket=bucket or self.DEFAULT_NAMESPACE, Prefix=prefix, ContinuationToken=cont_token
                    )
                    cont_token = result.get("NextContinuationToken")
                    for item in result.get("Contents", ()):
                        yield item["Key"]
                    break
                except botocore.exceptions.ProxyConnectionError as _ex:
                    ex = _ex
            if ex:
                raise ex

    @staticmethod
    def _mds_name(resource_id, path):
        filename = path.rsplit("/", 1)[-1]
        prefix = hashlib.md5(six.ensure_binary("{}/{}".format(resource_id, path))).hexdigest()
        return "{}/{}".format(prefix, filename)

    def upload_directory(
        self, path_or_obj, namespace=None, resource_id=None,
        logger=None, mds_settings=None, hash_calculator_factory=HashCalculator, tar_dir=False,
    ):
        """
        Upload directory to MDS S3

        :param path_or_obj: path to file or file object (tar) to upload
        :param namespace: S3 bucket
        :param resource_id: resource id for MDS metadata
        :param logger: custom logger, if not defined, logging will be used
        :param mds_settings: MDS specific config section
        :param hash_calculator_factory: factory for hashes (md5, sha1 blocks) calculator
        :param tar_dir: True for directory uploaded as a single TAR
        :rtype (str, list)
        :return: MDS key of uploaded files' metadata and the metadata
        """
        if tar_dir:
            if isinstance(path_or_obj, six.string_types):
                path_or_obj = TARBuilder(path_or_obj)
            return self.upload_tar(
                path_or_obj, None, namespace=namespace, resource_id=resource_id, logger=logger,
                mds_settings=mds_settings, hash_calculator_factory=hash_calculator_factory, tar_dir=True
            )

        uploaded_keys = set(self.uploaded_keys(namespace, resource_id, mds_settings))
        metadata = []
        is_file_object = not isinstance(path_or_obj, six.string_types)
        if is_file_object:
            root_mds_name = path_or_obj.tar_name if isinstance(path_or_obj, ChunkedFileObject) else None
            path_offset = 0
        else:
            root_path, root_mds_name = os.path.split(path_or_obj)
            path_offset = len(root_path) + (1 if root_path else 0)
        tar_header = None
        dirs = set()
        for item in (
            TARReader(path_or_obj, logger, hashes_calculator_factory=hash_calculator_factory)
            if is_file_object else
            os.walk(path_or_obj)
        ):
            if is_file_object:
                tar_header, fileobj = item
                if root_mds_name is None:
                    root_mds_name = tar_header.name.split("/", 1)[0]
                if tar_header.isdir():
                    dirpath, dirnames, filenames = tar_header.name, [], []
                else:
                    dirpath, filename = (
                        os.path.split(tar_header.name)
                        if os.sep in tar_header.name else
                        ("", tar_header.name)
                    )
                    dirnames, filenames = [], [filename]
            else:
                fileobj = None
                # noinspection PyTupleAssignmentBalance
                dirpath, dirnames, filenames = item
            dirname = dirpath[path_offset:]
            if dirname and dirname not in dirs:
                metadata.append(schema.MDSFileMeta.create(
                    type=ctr.FileType.DIR,
                    path=dirname,
                ))
                dirs.add(dirname)
            for dirname in dirnames:
                fullpath = os.path.join(dirpath, dirname)
                dirname = os.path.join(dirpath[path_offset:], dirname)
                link_name = None
                if is_file_object:
                    if tar_header.issym():
                        link_name = tar_header.linkname
                else:
                    if os.path.islink(fullpath):
                        link_name = os.readlink(fullpath)
                if link_name is not None:
                    metadata.append(schema.MDSFileMeta.create(
                        type=ctr.FileType.SYMLINK,
                        path=dirname,
                        symlink=link_name,
                        executable=os.access(dirname, os.X_OK),
                    ))
                else:
                    metadata.append(schema.MDSFileMeta.create(
                        type=ctr.FileType.DIR,
                        path=dirname,
                    ))
                    dirs.add(dirname)
            for filename in filenames:
                filepath = os.path.join(dirpath, filename)
                filepath_rel = filepath[path_offset:]
                link_name = None
                file_size = 0
                executable = False

                if is_file_object:
                    file_size = tar_header.size
                    executable = bool(tar_header.mode & 0o111)
                    if tar_header.issym():
                        link_name = tar_header.linkname
                else:
                    if os.path.islink(filepath):
                        link_name = os.readlink(filepath)
                    else:
                        file_size = os.path.getsize(filepath)
                        executable = os.access(filepath, os.X_OK)

                if link_name is not None:
                    metadata.append(schema.MDSFileMeta.create(
                        type=ctr.FileType.SYMLINK,
                        path=filepath_rel,
                        symlink=link_name,
                    ))
                    continue
                elif file_size == 0:
                    metadata.append(schema.MDSFileMeta.create(
                        type=ctr.FileType.TOUCH,
                        path=filepath_rel,
                        executable=executable,
                    ))
                    continue

                mds_name = self._mds_name(resource_id, filepath_rel)
                mds_key = "{}/{}".format(resource_id, mds_name)
                if is_file_object:
                    file_ = fileobj
                    mime_type = fileobj.mime_type
                else:
                    file_ = filepath
                    mime_type = MimeTypes.get_type(filepath, logger=logger)
                if mds_key not in uploaded_keys:
                    hashes = self._get_hashes(file_)
                    mds_key, _ = self.upload(
                        file_,
                        mds_name,
                        namespace=namespace, resource_id=resource_id, size=file_size,
                        logger=logger, mds_settings=mds_settings, content_type=mime_type
                    )
                    filemeta = schema.MDSFileMeta.create(
                        key=mds_key,
                        type=ctr.FileType.FILE,
                        path=filepath_rel,
                        size=file_size,
                        mime=mime_type,
                    )
                    if executable:
                        filemeta.executable = True
                    if hashes is not None:
                        filemeta.md5 = hashes.md5
                        filemeta.sha1_blocks = hashes.sha1_blocks
                else:
                    hashes = self._get_hashes(file_)
                    filemeta = schema.MDSFileMeta.create(
                        key=mds_key,
                        type=ctr.FileType.FILE,
                        path=filepath_rel,
                        size=file_size,
                        mime=mime_type,
                    )
                    if executable:
                        filemeta.executable = True
                    if hashes is not None:
                        filemeta.md5 = hashes.md5
                        filemeta.sha1_blocks = hashes.sha1_blocks
                    uploaded_keys.remove(mds_key)
                metadata.append(filemeta)

        if uploaded_keys:
            self._delete_keys(list(uploaded_keys), namespace, mds_settings, logger)

        metafile = io.BytesIO(six.ensure_binary(json.dumps(metadata, cls=rest.Client.CustomEncoder)))
        mds_key, _ = self.upload(
            metafile, root_mds_name,
            namespace=namespace, resource_id=resource_id, size=len(metafile.getvalue()),
            logger=logger, mds_settings=mds_settings, content_type=MimeTypes.JSON
        )
        return mds_key, metadata

    @staticmethod
    def _get_hashes(path_or_obj):
        if isinstance(path_or_obj, six.string_types):
            hashes = HashCalculator(path_or_obj)
        elif isinstance(path_or_obj, ChunkedFileObject):
            hashes = path_or_obj.hashes
        else:
            hashes = None
        return hashes

    def upload_tar(
        self, file_obj, s3_name, namespace=None, resource_id=None, size=None,
        compression_type=compression.base.CompressionType.TAR,
        logger=None, mds_settings=None, hash_calculator_factory=HashCalculator, executable=False, tar_dir=False
    ):
        """
        Upload tar to S3

        :param file_obj: file object (tar) to upload
        :param s3_name: name used to form S3 url
        :param namespace: MDS namespace, if not defined, Sandbox's namespace will be used
        :param resource_id: resource id for MDS metadata
        :param size: file size in bytes, if not defined, will be got from file system
        :param compression_type: compression type, 1 - without compression, 2 - gzip
        :param logger: custom logger, if not defined, logging will be used
        :param mds_settings: MDS specific config section
        :param hash_calculator_factory: factory for hashes (md5, sha1 blocks) calculator
        :param executable: executable bit
        :param tar_dir: True for directory uploaded as a single TAR
        :rtype (str, list)
        :return: S3 key of uploaded files' metadata and the metadata
        """
        if not isinstance(file_obj, ChunkedFileObject):
            if compression_type == compression.base.CompressionType.TGZ:
                collector = TgzMetaCollector(logger=logger)
            else:
                collector = TarMetaCollector(
                    logger=logger, hash_calculator_factory=hash_calculator_factory if tar_dir else None
                )
            file_obj = ChunkedFileObject(
                file_chunker(file_obj, tarfile.BLOCKSIZE), size or float("inf"), [collector], hash_calculator_factory,
                logger=logger
            )
        else:
            collector = file_obj.handlers[0]
        if s3_name is None:
            s3_name = file_obj.tar_name + ".tar"
        hashes = self._get_hashes(file_obj)

        tar_mds_key, uploaded_size = self.upload(
            file_obj, s3_name, namespace=namespace, resource_id=resource_id,
            size=size, logger=logger, mds_settings=mds_settings, content_type=file_obj.mime_type
        )

        tar_metadata = schema.MDSFileMeta.create(
            key=tar_mds_key,
            type=ctr.FileType.TARDIR if tar_dir else ctr.FileType.FILE,
            path=s3_name,
            executable=executable,
            size=uploaded_size,
            mime=file_obj.mime_type,
        )
        if hashes is not None:
            tar_metadata.md5 = hashes.md5
            tar_metadata.sha1_blocks = hashes.sha1_blocks

        metadata = [
            tar_metadata,
            # Root directory
            schema.MDSFileMeta.create(
                type=ctr.FileType.DIR,
                path="",
            )
        ]

        metadata.extend(collector.metadata)

        metafile = None
        try:
            metafile = io.BytesIO(six.ensure_binary(json.dumps(metadata, cls=rest.Client.CustomEncoder)))
        except UnicodeDecodeError as ex:
            if tar_dir:
                raise
            if logger is not None:
                logger.warning("Cannot upload metadata for resource #%s: %s", resource_id, ex)
        if metafile is not None:
            self.upload(
                metafile, None, namespace=namespace, resource_id=resource_id, size=len(metafile.getvalue()),
                logger=logger, mds_settings=mds_settings, content_type=MimeTypes.JSON
            )
            if compression_type and compression_type != compression.base.CompressionType.TAR:
                index_dump = io.BytesIO(collector.index.dump(collector.offsets))
                self.upload(
                    index_dump, s3_name + ".index", namespace=namespace, resource_id=resource_id,
                    size=len(index_dump.getvalue()), logger=logger, mds_settings=mds_settings,
                    content_type=MimeTypes.OCTET_STREAM
                )
        return tar_mds_key, metadata if tar_dir else [tar_metadata]

    def _delete_keys(self, keys, namespace, mds_settings, logger, **_):
        s3 = self.s3_client(mds_settings)
        logger.debug("Removing %s key(s) from S3 bucket %s", len(keys), namespace)
        for chunk in itertools.chunker(keys, 1000):
            resp = s3.delete_objects(Bucket=namespace, Delete=dict(Objects=[dict(Key=key) for key in chunk]))
            errors = resp.get("Errors")
            if errors:
                raise self.Exception(
                    "Removing from S3 failed: {}".format(errors)
                )
        return True

    def delete(self, key, multifile, namespace=None, logger=None, mds_settings=None):
        """
        Remove files from MDS S3

        :param key: S3 key
        :param multifile: whether the key points to multi file resource
        :param namespace: S3 bucket
        :param logger: custom logger, if not defined, logging will be used
        :param mds_settings: MDS specific config section
        :return: whether deletion is success
        """
        logger = logger or logging
        if mds_settings is None:
            mds_settings = config.Registry().common.mds
        namespace = namespace or self.DEFAULT_NAMESPACE
        try:
            if multifile:
                url = self._download_link(key, namespace=namespace)
                logger.debug("Getting metadata from %s", url)
                resp = requests.get(url)
                if resp.status_code == requests.codes.NOT_FOUND:
                    logger.warning("File with S3 key '%s' in bucket '%s' is already deleted", key, namespace)
                    return False
                elif resp.status_code != requests.codes.OK:
                    raise self.Exception(
                        "Getting from S3 failed - service respond code {}: {}".format(resp.status_code, resp.reason)
                    )
                metadata = [schema.MDSFileMeta.create(__filter_empty__=True, **i) for i in json.loads(resp.content)]
                keys = list(map(
                    urllib.parse.quote,
                    it.chain((i.key for i in metadata if i.type == ctr.FileType.FILE), [key])
                ))
            else:
                keys = [urllib.parse.quote(six.ensure_binary(_)) for _ in itertools.chain(key)]
                splitted_key = key.split("/", 1) if isinstance(key, six.string_types) else []
                if len(splitted_key) == 2:
                    meta_key = splitted_key[0]
                    for sub_key in (meta_key, key + ".index"):
                        url = self._download_link(sub_key, namespace=namespace)
                        resp = requests.head(url)
                        if resp.status_code == requests.codes.OK:
                            keys.append(urllib.parse.quote(sub_key))
            return self._delete_keys(keys, namespace, mds_settings, logger)
        except botocore.exceptions.BotoCoreError as ex:
            raise self.Exception(ex)

    def copy(self, src_bucket, dst_bucket, resource_id, logger=None, mds_settings=None, overwrite=False):
        """
        Copy resource files from one S3 bucket to another

        :param src_bucket: source bucket
        :param dst_bucket: destination bucket
        :param resource_id: resource id to copy
        :param logger: custom logger, if not defined, logging will be used
        :param mds_settings: MDS specific config section
        :param overwrite: overwrite already existed keys in destination bucket
        :return: whether copy is success
        """
        logger = logger or logging
        if mds_settings is None:
            mds_settings = config.Registry().common.mds
        logger.debug("Copying resource #%s from %s to %s", resource_id, src_bucket, dst_bucket)
        try:
            s3 = self.s3_client(mds_settings)
            uploaded_keys = set(self.uploaded_keys(src_bucket, resource_id, s3_client=s3))
            if not overwrite:
                uploaded_keys -= set(self.uploaded_keys(dst_bucket, resource_id, s3_client=s3))
            for key in uploaded_keys:
                s3.copy({"Bucket": src_bucket, "Key": key}, dst_bucket, key)
        except botocore.exceptions.BotoCoreError as ex:
            raise self.Exception(ex)

    @staticmethod
    def __make_skyboned_links(item, namespace, mds_settings, tar_dir_item):
        if tar_dir_item is not None:
            link = s3_link(tar_dir_item.key, namespace=namespace, mds_settings=mds_settings)
            links = {link: {"Range": "bytes={}-{}".format(item.offset, item.offset + item.size - 1)}}
        else:
            link = s3_link(item.key, namespace=namespace, mds_settings=mds_settings)
            links = [link]
        return links

    def prepare_skyboned_metadata(self, metadata, namespace=None, mds_settings=None):
        """
        Create metadata for register torrent in SkyboneD

        :param metadata: MDS metadata
        :param namespace: MDS namespace, if not defined, Sandbox's namespace will be used
        :param mds_settings: MDS specific config section
        :return: metadata
        """
        namespace = namespace or self.DEFAULT_NAMESPACE
        torrent_items = {}
        hashes = {}
        links = {}
        tar_dir_item = None
        if metadata[0].type == ctr.FileType.TARDIR:
            tar_dir_item = metadata[0]
            metadata = metadata[2:]
        for item in metadata:
            if item.type == ctr.FileType.FILE:
                torrent_items[six.ensure_str(item.path)] = dict(
                    type=ctr.FileType.FILE,
                    md5=item.md5,
                    executable=item.executable,
                    size=item.size,
                )
                hashes[item.md5] = b"".join(binascii.unhexlify(six.ensure_binary(_)) for _ in item.sha1_blocks)
                links[item.md5] = self.__make_skyboned_links(item, namespace, mds_settings, tar_dir_item)
            elif item.type == ctr.FileType.TOUCH:
                torrent_items[six.ensure_str(item.path)] = dict(
                    type=ctr.FileType.FILE,
                    md5=self.EMPTY_MD5,
                    executable=item.executable,
                    size=0,
                )
            elif item.type == ctr.FileType.SYMLINK:
                torrent_items[six.ensure_str(item.path)] = dict(
                    type=ctr.FileType.SYMLINK,
                    target=six.ensure_binary(item.symlink),
                )
            elif item.type == ctr.FileType.DIR:
                torrent_items[six.ensure_str(item.path)] = dict(
                    type=ctr.FileType.DIR,
                )
        return torrent_items, links, hashes

    @staticmethod
    def _make_source_id(resource_id):
        client_id = tvm.TVM.get_source_client_id()
        return "{}:{}".format(client_id, resource_id) if client_id else None

    @staticmethod
    def _skyboned_resource_has_outdated_format(ex):
        resp = ex.response
        return resp.status_code == 400 and resp.content.decode() == "Current resource has outdated format"

    def skyboned_add(
        self, key_or_metadata, resource_id, namespace=None, tvm_ticket=None, logger=None,
        mds_settings=None, share_fullpath=False, timeout=DEFAULT_SKYBONED_ANNOUNCE_TIMEOUT
    ):
        logger = logger or logging
        if mds_settings is None:
            mds_settings = config.Registry().common.mds
        namespace = namespace or self.DEFAULT_NAMESPACE
        if isinstance(key_or_metadata, six.string_types):
            url = s3_link(key_or_metadata, namespace=namespace)
            logger.debug("Getting metadata from %s", url)
            resp = requests.get(url)
            if resp.status_code != requests.codes.OK:
                raise self.Exception(
                    "Getting from MDS failed - service respond code {}: {}".format(resp.status_code, resp.reason)
                )
            key_or_metadata = json.loads(resp.content)
        metadata = [schema.MDSFileMeta.create(__filter_empty__=True, **dict(i)) for i in key_or_metadata]

        # FIXME: [SANDBOX-7368]
        if not share_fullpath:
            metadata_ = metadata[2:] if metadata[0].type == ctr.FileType.TARDIR else metadata
            common_prefix = os.path.dirname(os.path.commonprefix([item.path for item in metadata_]) + "/")
            base_dir = os.path.basename(common_prefix)
            for item in metadata_:
                item.path = item.path.replace(common_prefix, base_dir, 1)

        items, links, hashes = self.prepare_skyboned_metadata(metadata, namespace=namespace, mds_settings=mds_settings)
        if tvm_ticket is None:
            tvm_service = mds_settings.skyboned.tvm_service
            tvm_ticket = tvm.TVM.get_service_ticket([tvm_service])[tvm_service]
        logger.debug("Announcing MDS file(s) in SkyboneD")
        ex = None
        initial_retry_interval = random.uniform(3, 5)
        for _ in itertools.progressive_yielder(initial_retry_interval, 30, timeout, sleep_first=False):
            try:
                return infra.skyboned.api.skyboned_add_resource(
                    items, hashes, links, mds_settings.skyboned.servers, tvm_ticket,
                    source_id=self._make_source_id(resource_id)
                )[0]
            except infra.skyboned.api.ServiceUnavailable as ex_:
                ex = ex_
            except (requests.HTTPError, requests.ConnectionError) as ex:
                if ex.response.status_code == 403:
                    raise self.RBTorrentForbidden(ex)
                # XXX: special case for resources announced in old format
                elif self._skyboned_resource_has_outdated_format(ex):
                    uid, _ = infra.skyboned.api._generate_resource(items, hashes)
                    return "rbtorrent:{}".format(uid)
                elif ex.response.status_code == 400:
                    raise self.RBTorrentRegistrationError(
                        "Failed to register torrent: {}".format(ex.response.content.decode())
                    )
            except Exception as ex:
                logger.exception("Error occurred while announcing in SkyboneD")
                raise self.RBTorrentRegistrationError("Failed to register torrent: {}".format(ex))
        else:
            raise self.RBTorrentRegistrationError("Failed to register torrent: {}".format(ex))

    def skyboned_remove(
        self, skynet_id, resource_id, tvm_ticket=None, logger=None, mds_settings=None,
        timeout=DEFAULT_SKYBONED_ANNOUNCE_TIMEOUT
    ):
        logger = logger or logging
        if mds_settings is None:
            mds_settings = config.Registry().common.mds
        logger.debug("Removing torrent %s from SkyboneD", skynet_id)
        if tvm_ticket is None:
            tvm_service = mds_settings.skyboned.tvm_service
            tvm_ticket = tvm.TVM.get_service_ticket([tvm_service])[tvm_service]
        ex = None
        for _ in itertools.progressive_yielder(1, 10, timeout, sleep_first=False):
            try:
                return infra.skyboned.api.skyboned_remove_resource(
                    skynet_id, mds_settings.skyboned.servers, tvm_ticket,
                    source_id=self._make_source_id(resource_id)
                )
            except infra.skyboned.api.ServiceUnavailable as ex_:
                ex = ex_
            except requests.HTTPError as ex:
                if ex.response.status_code == 403:
                    raise self.RBTorrentForbidden(ex)
                elif not self._skyboned_resource_has_outdated_format(ex):
                    raise self.RBTorrentRemoveError(
                        "Failed to remove torrent: {}".format(ex.response.content.decode())
                    )
            except Exception as ex:
                logger.exception("Error occurred while removing from SkyboneD")
                raise self.RBTorrentRemoveError("Failed to remove torrent: {}".format(ex))
        else:
            raise self.RBTorrentRemoveError("Failed to remove torrent: {}".format(ex))

    @classmethod
    @patterns.ttl_cache(300, ignore_kws=True)
    def bucket_stats(cls, bucket, mds_settings=None, logger=None, retry_timeout=60):
        if mds_settings is None:
            mds_settings = config.Registry().common.mds
        if config.Registry().common.installation == ctm.Installation.LOCAL:
            return None
        s3_idm = rest.Client(base_url=mds_settings.s3.idm.url, logger=logger)
        stats = None
        ex = None
        for _ in itertools.progressive_yielder(1, 10, retry_timeout, sleep_first=False):
            ex = None
            try:
                stats = s3_idm.stats.buckets[bucket].read()
                break
            except rest.Client.HTTPError as ex_:
                if ex_.status != requests.codes.NOT_FOUND:
                    raise
                stats = None
                break
            except (rest.Client.TimeoutExceeded, requests.ReadTimeout, requests.ConnectionError) as ex_:
                ex = ex_
                if logger is not None:
                    logger.warning("Reading bucket stats for %s: %s", bucket, ex_)
        if ex is not None:
            raise
        return stats

    @classmethod
    def check_bucket(cls, abc_id=None, bucket=None, mds_settings=None, logger=None):
        """
        Check existence of bucket for the ABC service

        :param abc_id: ABC service id
        :param bucket: MDS-S3 bucket name
        :param mds_settings: MDS specific config section
        :return: (bucket, bucket_stats), bucket is None if not exists or default bucket
        """
        if abc_id:
            bucket = "sandbox-{}".format(abc_id)
        bucket_stats = cls.bucket_stats(bucket or cls.DEFAULT_NAMESPACE, mds_settings=mds_settings, logger=logger)
        if bucket_stats is None:
            bucket, bucket_stats = None, cls.bucket_stats(
                cls.DEFAULT_NAMESPACE, mds_settings=mds_settings, logger=logger
            )
        return bucket, bucket_stats

    @staticmethod
    def set_s3_logging_level(level):
        boto3.set_stream_logger("boto3", level)
        boto3.set_stream_logger("botocore", level)
        boto3.set_stream_logger("s3transfer", level)
        boto3.set_stream_logger("urllib3.util.retry", level)

    @staticmethod
    def get_metadata_error(metadata):
        if len(metadata) > 2 and metadata[2]["type"] == ctr.FileType.ERROR:
            return metadata[2]["path"]


for name, exception_class in inspect.getmembers(MDS, lambda m: inspect.isclass(m) and issubclass(m, Exception)):
    globals()[name] = exception_class
