from __future__ import absolute_import, unicode_literals

import io
import tarfile

from ... import enum

import six


class CompressionType(enum.Enum):
    NONE = 0
    TAR = 1
    TGZ = 2


class IndexMeta(type):
    __registry__ = {}

    def __new__(mcs, name, bases, namespace):
        cls = super(IndexMeta, mcs).__new__(mcs, name, bases, namespace)
        if bases != (object,):
            comp_type = namespace.get("compression_type")
            assert comp_type is not None, "compression_type must be defined"
            assert comp_type not in mcs.__registry__, "compression type {} already registered".format(
                CompressionType.val2str(comp_type)
            )
            mcs.__registry__[comp_type] = cls
        return cls


class Index(six.with_metaclass(IndexMeta, object)):
    compression_type = None

    @classmethod
    def load(cls, data):
        comp_type = bytearray(data[:1])[0]
        index_cls = cls.__registry__.get(comp_type)
        assert index_cls is not None, "unknown compression type {}".format(comp_type)
        return index_cls.load(data)

    def dump(self):
        buf = io.BytesIO()
        buf.write(bytes(bytearray([self.compression_type])))
        return buf


def archive_type(path):
    lower_path = path.lower()
    if not any(lower_path.endswith(ext) for ext in (".tar", ".tgz", ".tar.gz")):
        return None

    try:
        tarfile.open(path, mode="r:gz")
        return CompressionType.TGZ
    except tarfile.ReadError:
        pass

    try:
        tarfile.open(path, mode="r:")
        return CompressionType.TAR
    except tarfile.ReadError:
        pass
