from __future__ import absolute_import

import os
import sys
import ctypes
import getpass
import collections
import subprocess as sp

from .. import config as common_config
from .. import context as common_context
from .. import patterns as common_patterns


# noinspection PyInitNewSignature,PyArgumentList
class User(collections.namedtuple("User", ("login", "group", "uid", "gid", "home"))):
    """ User login, group name and home directory location type. """

    SERVICE_USER_ENV = "SANDBOX_USER"
    Users = collections.namedtuple("Users", ("service", "unprivileged"))

    # noinspection PyMethodParameters
    @common_patterns.singleton_classproperty
    def has_root(cls):
        """ Determines the current process is run under root privileges or not. """
        if sys.platform == "win32":
            return bool(ctypes.windll.shell32.IsUserAnAdmin())
        return os.getuid() == 0

    @common_patterns.singleton_classproperty
    def can_root(self):
        """ Determines whether root privileges is available for the current process or not. """
        if sys.platform == "win32":
            return False
        with open(os.devnull, "w") as devnull:
            return not sp.call(["sudo", "-n", "/bin/true"], stderr=devnull)

    def __new__(cls, login=None, group=None):
        if not login:
            return super(User, cls).__new__(cls, None, None, None, None, None)
        else:
            if sys.platform == "win32":
                return super(User, cls).__new__(
                    cls, login, group or login, None, None, os.path.realpath(os.path.expanduser("~" + login))
                )
            import pwd
            import grp
            pw, gid = pwd.getpwnam(login), None
            try:
                if group:
                    gid = grp.getgrnam(group).gr_gid
                else:
                    group = grp.getgrgid(pw.pw_gid).gr_name
            except KeyError:
                pass
            return super(User, cls).__new__(
                cls,
                login,
                group or login,
                pw.pw_uid,
                gid or pw.pw_gid,
                os.path.realpath(os.path.expanduser("~" + login))
            )

    # noinspection PyMethodParameters
    @common_patterns.classproperty
    def service_users(cls):
        """ Returns a pair of service and unprivileged users. """
        return cls.get_service_users(common_config.Registry())

    @classmethod
    def get_service_users(cls, settings):
        if cls.has_root and os.environ.get(cls.SERVICE_USER_ENV):
            # Local root mode
            return cls.Users(*(User(os.environ[cls.SERVICE_USER_ENV]),) * 2)
        elif settings.client.sandbox_user:
            # Production mode
            return cls.Users(
                User("zomb-sandbox", "sandbox"),
                User(settings.client.sandbox_user)
            )
        else:
            # Local unprivileged mode
            return cls.Users(*(User(getpass.getuser()),) * 2)

    Privileges = common_context.NullContextmanager


# Fills in and remember availability of root privileges at module import time.
if User.has_root and sys.platform != "win32":
    # noinspection PyPackageRequirements,PyUnresolvedReferences
    import kernel.util.sys.user

    # noinspection PyUnresolvedReferences
    User.Privileges = kernel.util.sys.user.UserPrivileges


__all__ = ["User"]
