from __future__ import absolute_import, unicode_literals

import os
import abc
import copy
import time
import threading
import collections

import six


# noinspection PyPep8Naming
class classproperty(object):
    """
    Decorator which makes a method a class' property instead
    (it's like ``@classmethod`` and ``@property`` decorators together)
    """

    def __init__(self, getter):
        self.getter = getter

    def __get__(self, obj, objtype):
        return self.getter(objtype)


# noinspection PyPep8Naming
class singleton_classproperty(classproperty):
    """
    Thread-safe decorator for calculating a class' property exactly once
    """

    __none = type(str("None"), (object,), {})()
    __lock = threading.RLock()
    __value = __none

    def __get__(self, obj, objtype):
        if self.__value is not self.__none:
            return self.__value
        with self.__lock:
            value = self.__value
            if value is self.__none:
                value = self.__value = self.getter(objtype)
            return value


def is_classproperty(cls, attrname):
    """
    Test if class attribute named `attrname` is a property (i.e. a descriptor with `__get__` method)
    """

    for class_ in cls.__mro__:
        if attrname in class_.__dict__:
            descriptor = class_.__dict__[attrname]
            if hasattr(descriptor, "__get__"):
                return descriptor
            return None
    return None


class SingletonMeta(abc.ABCMeta):
    """
    Metaclass which forces a class to always return the exact same object as its instance.
    Example:

    .. code-block:: python

        >>> class TrueNorth(object):
        ...   __metaclass__ = SingletonMeta
        ...
        >>> TrueNorth() is TrueNorth()
        True
    """

    def __call__(cls, *args, **kws):
        instance = cls.instance
        if instance is None:
            instance = cls.instance = super(cls, cls).__new__(cls, *args, **kws)
            instance.__init__(*args, **kws)
        return instance

    @property
    def tag(cls):
        return "_{}_{}__instance".format(cls.__module__, cls.__name__)

    @property
    def instance(cls):
        return getattr(cls, cls.tag, None)

    @instance.setter
    def instance(cls, instance):
        if cls.instance is not None:
            raise AttributeError("Cannot set instance because it already exists")
        setattr(cls, cls.tag, instance)

    @instance.deleter
    def instance(cls):
        setattr(cls, cls.tag, None)


class ThreadSafeSingletonMeta(SingletonMeta):
    """
    An extended, thread-safe version of ``SingletonMeta``.
    Use it if you need the same object to appear on class instantiation across multiple threads.
    """

    lock = threading.Lock()

    def __call__(cls, *args, **kws):
        instance = cls.instance
        if instance is None:
            with cls.lock:
                instance = cls.instance
                if instance is None:
                    instance = super(cls, cls).__new__(cls, *args, **kws)
                    instance.__init__(*args, **kws)
                    cls.instance = instance
        return instance


def ttl_cache(ttl, external_cache=None, ignore_kws=False):
    """
    Decorator which caches function's return value for ``ttl`` seconds for every set of parameters separately.
    In case you need cache for an infinite amount of time, use ``singleton`` decorator.
    All the cache records are stored in the function's ``cache`` attribute

    :param ttl: amount of time to cache call result for, seconds
    :param external_cache: dict used as cache
    :param ignore_kws: if True, ignore keyword arguments for key
    """

    def wrapper(f):
        if external_cache is not None:
            cache = external_cache
        elif getattr(f, "cache", None) is None:
            cache = f.cache = {}
        else:
            cache = f.cache

        def func(*args, **kw):
            kwargs = tuple(sorted(kw.items())) if not ignore_kws and kw else ()
            key = (args, kwargs)
            v, t = cache.get(key, (None, None))
            if t and t > time.time():
                return v

            v = f(*args, **kw)
            t = time.time() + ttl
            cache[key] = (v, t)
            return v
        return func

    return wrapper


def singleton(f):
    """
    Same as ``ttl_cache``, but store the result forever.
    Useful if you need to call something exactly once for a given set of arguments.
    """

    return ttl_cache(float("inf"))(f)


def namedlist(name, fields):
    fields = tuple(fields.split() if isinstance(fields, (six.text_type, six.binary_type)) else fields)
    namespace = {
        f: property(lambda s, i=i: s[i], lambda s, v, i=i: s.__setitem__(i, v))
        for i, f in enumerate(fields)
    }
    namespace["__slots__"] = ()
    namespace["_fields"] = fields
    namespace["__repr__"] = lambda self: "{}({})".format(
        name, ", ".join(six.moves.map(lambda _: "=".join((_[0], repr(_[1]))), six.moves.zip(fields, self)))
    )
    return type(
        str(name),
        (list,),
        namespace
    )


class ApiMeta(type):
    __apis__ = {}

    def __iter__(cls):
        return six.iteritems(cls.__apis__)

    def __getitem__(cls, name):
        return cls.__apis__[name]

    @staticmethod
    def register(method):
        assert isinstance(method, (staticmethod, classmethod))
        method.__func__.__isapimethod__ = True
        return method


@six.add_metaclass(ApiMeta)
class Api(type):
    """
    Metaclass for registering of APIs, for example to use via Synchrophazotron
    """

    def __new__(mcs, name, bases, namespace):
        cls = super(Api, mcs).__new__(mcs, name, bases, namespace)
        api_methods = set(
            name
            for name, value in six.iteritems(namespace)
            if (
                isinstance(value, (staticmethod, classmethod)) and
                getattr(value.__func__, "__isapimethod__", False)
            )
        )
        for base in bases:
            for method_name in getattr(base, "__apimethods__", set()):
                value = getattr(cls, method_name, None)
                if getattr(value, "__isapimethod__", False):
                    api_methods.add(method_name)
        cls.__apimethods__ = frozenset(api_methods)
        ApiMeta.__apis__[name] = cls
        return cls

    @property
    def api_methods(cls):
        return cls.__apimethods__


# noinspection PyPep8Naming
class singleton_property(property):
    """
    Thread-safe decorator to be used as @property, which will be calculated and cached on first access.
    """

    __none = type(str("None"), (object,), {})()
    __lock = {os.getpid(): threading.RLock()}
    __value = __none

    @classproperty
    def _lock(self):
        return self.__lock.setdefault(os.getpid(), threading.RLock())

    def __get__(self, obj, objtype=None):
        if obj is None:
            return super(singleton_property, self).__get__(obj, objtype)
        return (self.__cls_get if isinstance(obj, type) else self.__obj_get)(obj, objtype)

    def __delete__(self, obj):
        if isinstance(obj, type):
            self.__value = self.__none
        else:
            try:
                del obj.__dict__[self.fget.__name__]
            except KeyError:
                pass

    def __obj_get(self, obj, objtype):
        name = self.fget.__name__
        ret = obj.__dict__.get(name, self.__none)
        if ret is not self.__none:
            return ret
        with self._lock:
            ret = obj.__dict__.get(name, self.__none)
            if ret is not self.__none:
                return ret
            return obj.__dict__.setdefault(name, super(singleton_property, self).__get__(obj, objtype))

    def __cls_get(self, obj, objtype):
        if self.__value is not self.__none:
            return self.__value
        with self._lock:
            value = self.__value
            if value is self.__none:
                value = self.__value = super(singleton_property, self).__get__(obj, objtype)
            return value


class ThreadLocalMeta(type):
    """Metaclass to keep a single class instance per thread. """

    _local = threading.local()

    def __call__(cls, *args, **kwargs):
        key = hex(hash((cls.__name__,) + tuple(args) + tuple(sorted(kwargs.items()))))
        try:
            return getattr(ThreadLocalMeta._local, key)
        except AttributeError:
            instance = super(ThreadLocalMeta, cls).__call__(*args, **kwargs)
            setattr(ThreadLocalMeta._local, key, instance)
            return instance


class Abstract(object):
    """
    The class is designed as simple slotted data entry storage with an ability to fill in
    slots available via keyword arguments in its constructor. In case of no value provided
    to the constructor, it will be fetched out from `__defs__` array (which should be of
    the same length as `__slots__`).

    Class stolen from `skynet.kernel.base.entity`
    """
    __slots__ = []
    __defs__ = []
    __copier__ = copy.copy

    def __init__(self, *args, **kwargs):
        for i, attr in enumerate(self.__slots__):
            setattr(self, attr, self._value(attr, i, args, kwargs))

    def _value(self, attr, i, args, kwargs):
        _def = self.__defs__[i]
        if i < len(args):
            val = args[i]
        elif attr in kwargs:
            val = kwargs[attr]
        else:
            # noinspection PyUnresolvedReferences
            val = self.__copier__.__func__(_def)
            _def = None  # Avoid double checking
        return val if _def is None or isinstance(val, _def.__class__) else _def.__class__(val)

    def __repr__(self):
        return self.__class__.__name__ + repr(dict(iter(self)))

    def __iter__(self):
        for attr in self.__slots__:
            yield attr, getattr(self, attr)

    def __eq__(self, other):
        return list(self) == list(other)

    def itervalues(self):
        for attr in self.__slots__:
            yield getattr(self, attr)

    def copy(self):
        return self.__class__(*(v for _, v in self))


class NamedTupleMeta(type):
    # Attributes prohibited to set to nametuple
    _PROHIBITED = (
        "__new__", "__init__", "__getnewargs__", "_fields", "_field_defaults", "_field_types", "_make",
        "_replace", "_asdict", "_source"
    )
    # Attributes ignored from the NamedTuple class
    _SPECIAL = ("__module__", "__name__", "__qualname__", "__annotations__", "__slots__", "__defs__")

    def __new__(mcs, name, bases, namespace):  # type -> tuple
        if bases == (object,):
            return super(NamedTupleMeta, mcs).__new__(mcs, name, bases, namespace)

        slots = namespace.get("__slots__")
        assert slots is not None, "`__slots__` must be defined"

        nm_tpl = collections.namedtuple(name, slots)

        defs = namespace.get("__defs__")

        if defs is not None:
            assert len(defs) == len(slots), (
                "When present, `__defs__` must have the same length as `__slots__`"
            )

            # Since defaults are defined in global scope, they should not be mutable.
            # While hashability does not imply immutablity, it is still a good proxy.
            for default in defs:
                hash(default)

            nm_tpl.__new__.__defaults__ = tuple(defs)

        # Update from user namespace without overriding special namedtuple attributes
        for key in namespace:
            if key in mcs._PROHIBITED:
                raise AttributeError("Cannot overwrite NamedTuple attribute " + key)
            elif key not in mcs._SPECIAL and key not in nm_tpl._fields:
                setattr(nm_tpl, key, namespace[key])

        return nm_tpl


@six.add_metaclass(NamedTupleMeta)
class NamedTuple(object):
    """
    The class defines a namedtuple with the given default values. The usage is similar to `Abstract`:

    .. code-block:: python

        class Data(NamedTuple):
            __slots__ = ("foo", "bar")
            __defs__ = (False, 42)

    Unlike `Abstract`, the default values can only be hashable. They can also be omitted,
    in which case the class acts as an ordinary namedtuple.
    """


class RWLock(object):
    __metaclass__ = abc.ABCMeta

    reader = abc.abstractproperty()
    writer = abc.abstractproperty()

    @abc.abstractmethod
    def acquire_read(self):
        """ Acquire the lock in shared mode. """

    @abc.abstractmethod
    def acquire_write(self):
        """ Acquire the lock in exclusive mode. """

    @abc.abstractmethod
    def release(self):
        """ Release the lock. """
