# coding: utf-8

from __future__ import absolute_import, unicode_literals

import logging

import six

import platform as pl
import subprocess as sp

from .. import enum
from .. import patterns
from ..types import misc as ctm
from ..types import client as ctc
from ..windows import wsl

if six.PY3:
    import distro
else:
    distro = None

PLATFORM_ALIASES = {
    "linux_ubuntu_20.04_focal": (
        "Linux-4.4.7-1.ng-x86_64-with-Ubuntu-20.04-focal".lower(),
        "Linux-4.19.119-30.2-x86_64-with-Ubuntu-20.04-focal".lower(),
    ),
    "linux_ubuntu_18.04_bionic": (
        "Linux-4.4.7-1.ng-x86_64-with-Ubuntu-18.04-bionic".lower(),
    ),
    "linux_ubuntu_16.04_xenial": (
        "Linux-4.4.7-1.ng-x86_64-with-Ubuntu-16.04-xenial".lower(),
    ),
    "linux_ubuntu_14.04_trusty": (
        "linux-3.13.0-35-generic-x86_64-with-ubuntu-14.04-trusty".lower(),
    ),
    "linux_ubuntu_12.04_precise": (
        "Linux.3.2.0-38-generic".lower(),
        "Linux.3.2.0-48-generic".lower(),
        "Linux.3.2.0-51-generic".lower(),
        "Linux.3.2.0-43-generic".lower(),
        "Linux-3.2.0-51-generic-x86_64-with-Ubuntu-12.04-precise".lower(),
        "Linux-3.2.0-52-generic-x86_64-with-Ubuntu-12.04-precise".lower(),
        "Linux-3.2.0-48-generic-x86_64-with-Ubuntu-12.04-precise".lower(),
        "Linux-3.2.0-53-generic-x86_64-with-Ubuntu-12.04-precise".lower(),
        "Linux-3.2.0-43-generic-x86_64-with-Ubuntu-12.04-precise".lower(),
    ),
    "linux_ubuntu_10.04_lucid": (
        "Linux.2.6.32-42-server".lower(),
        "Linux.2.6.32-38-server".lower(),
        "Linux.2.6.32-33-server".lower(),
        "Linux-2.6.32-38-server-x86_64-with-Ubuntu-10.04-lucid".lower(),
        "Linux-3.10.9-10yapatch10-x86_64-with-Ubuntu-10.04-lucid".lower(),
        "Linux-2.6.32-42-server-x86_64-with-Ubuntu-10.04-lucid".lower(),
        "Linux-3.2.0-27-server-x86_64-with-Ubuntu-10.04-lucid".lower(),
    ),
    "osx_10.9_mavericks": (
        "Darwin-13.4.0-x86_64-i386-64bit".lower(),
    ),
    "osx_10.10_yosemite": (
        "Darwin-14.0.0-x86_64-i386-64bit".lower(),
        "Darwin-14.3.0-x86_64-i386-64bit".lower(),
        "Darwin-14.4.0-x86_64-i386-64bit".lower(),
        "Darwin-14.5.0-x86_64-i386-64bit".lower(),
    ),
    "osx_10.11_el_capitan": (
        "Darwin-15.3.0-x86_64-i386-64bit".lower(),
    ),
    "osx_10.12_sierra": (
        "Darwin-16.5.0-x86_64-i386-64bit".lower(),
        "Darwin-16.7.0-x86_64-i386-64bit".lower(),
    ),
    "osx_10.13_high_sierra": (
        "Darwin-17.4.0-x86_64-i386-64bit".lower(),
        "Darwin-17.7.0-x86_64-i386-64bit".lower(),
    ),
    "osx_10.14_mojave": (
        "Darwin-18.5.0-x86_64-i386-64bit".lower(),
        "Darwin-18.6.0-x86_64-i386-64bit".lower(),
        "Darwin-18.7.0-x86_64-i386-64bit".lower(),
    ),
    "osx_10.15_catalina": (
        "Darwin-19.4.0-x86_64-i386-64bit".lower(),
        "Darwin-19.5.0-x86_64-i386-64bit".lower(),
        "Darwin-19.6.0-x86_64-i386-64bit".lower(),
    ),
    "osx_10.16_big_sur": (
        "Darwin-20.2.0-x86_64-i386-64bit".lower(),
        "Darwin-20.3.0-x86_64-i386-64bit".lower(),
        "Darwin-20.4.0-x86_64-i386-64bit".lower(),
        "Darwin-20.5.0-x86_64-i386-64bit".lower(),
        "Darwin-20.6.0-x86_64-i386-64bit".lower(),
        "Darwin-20.2.0-arm64-arm-64bit".lower(),
        "Darwin-20.3.0-arm64-arm-64bit".lower(),
        "Darwin-20.6.0-arm64-arm-64bit".lower(),
    ),
    "osx_12_monterey": (
        "Darwin-21.1.0-x86_64-i386-64bit".lower(),
        "Darwin-21.2.0-x86_64-i386-64bit".lower(),
        "Darwin-21.3.0-x86_64-i386-64bit".lower(),
        "Darwin-21.4.0-x86_64-i386-64bit".lower(),
        "Darwin-21.1.0-arm64-arm-64bit".lower(),
        "Darwin-21.2.0-arm64-arm-64bit".lower(),
        "Darwin-21.3.0-arm64-arm-64bit".lower(),
        "Darwin-21.4.0-arm64-arm-64bit".lower(),
    ),
    "linux_rhel_santiago": (
        "Linux-3.10.48-8-x86_64-with-redhat-6.5-Santiago".lower(),
        "Linux-3.10.53-14-x86_64-with-redhat-6.5-Santiago".lower(),
    ),
    "freebsd9": (
        "FreeBSD.9.0-STABLE".lower(),
        "FreeBSD-9.0-STABLE-amd64-64bit-ELF".lower(),
        "FreeBSD-9.1-RELEASE-amd64-64bit-ELF".lower(),
    ),
    "freebsd8": (
        "FreeBSD.8.2-STABLE".lower(),
        "FreeBSD-8.2-STABLE-amd64-64bit-ELF".lower(),
    ),
    "linux_ubuntu_14.04_trusty_aarch64": (
        "Linux-3.13.0-48-generic-aarch64-with-Ubuntu-14.04-trusty".lower(),
    )
}

# Map from platform alias to pair of binary platform key and version
# Binary from any platform can run on platforms with matching key and greater or equal versions
PLATFORM_ALIAS_BINARY = {
    "freebsd8": ("freebsd8", 0),  # Didn't check whether freebsd 8 and freebsd 9 are binary compatible.
    "freebsd9": ("freebsd9", 0),  # Just mark them incompatible.

    "linux_rhel_santiago": ("rhel", 0),

    "linux_ubuntu_10.04_lucid": ("ubuntu", 10),
    "linux_ubuntu_12.04_precise": ("ubuntu", 12),
    "linux_ubuntu_14.04_trusty": ("ubuntu", 14),
    "linux_ubuntu_16.04_xenial": ("ubuntu", 16),
    "linux_ubuntu_18.04_bionic": ("ubuntu", 18),
    "linux_ubuntu_20.04_focal": ("ubuntu", 20),

    "osx_10.9_mavericks": ("osx", 1009),
    "osx_10.10_yosemite": ("osx", 1010),
    "osx_10.11_el_capitan": ("osx", 1011),
    "osx_10.12_sierra": ("osx", 1012),
    "osx_10.13_high_sierra": ("osx", 1013),
    "osx_10.14_mojave": ("osx", 1014),
    "osx_10.15_catalina": ("osx", 1015),
    "osx_10.16_big_sur": ("osx", 1016),
    "osx_12_monterey": ("osx", 1017),
    "win_nt": ("win32", 10),
    "windows": ("win32", 10),
}

PLATFORM_TO_TAG = {
    "linux_ubuntu_10.04_lucid": ctc.Tag.LINUX_LUCID,
    "linux_ubuntu_12.04_precise": ctc.Tag.LINUX_PRECISE,
    "linux_ubuntu_14.04_trusty": ctc.Tag.LINUX_TRUSTY,
    "linux_ubuntu_16.04_xenial": ctc.Tag.LINUX_XENIAL,
    "linux_ubuntu_18.04_bionic": ctc.Tag.LINUX_BIONIC,
    "linux_ubuntu_20.04_focal": ctc.Tag.LINUX_FOCAL,
    "linux": ctc.Tag.Group.LINUX,

    "osx_10.9_mavericks": ctc.Tag.OSX_MAVERICKS,
    "osx_10.10_yosemite": ctc.Tag.OSX_YOSEMITE,
    "osx_10.11_el_capitan": ctc.Tag.OSX_EL_CAPITAN,
    "osx_10.12_sierra": ctc.Tag.OSX_SIERRA,
    "osx_10.13_high_sierra": ctc.Tag.OSX_HIGH_SIERRA,
    "osx_10.14_mojave": ctc.Tag.OSX_MOJAVE,
    "osx_10.15_catalina": ctc.Tag.OSX_CATALINA,
    "osx_10.16_big_sur": ctc.Tag.OSX_BIG_SUR,
    "osx_12_monterey": ctc.Tag.OSX_MONTEREY,
    "osx": ctc.Tag.Group.OSX,
    "osx_arm": ctc.Tag.Group.OSX & ctc.Tag.M1,

    "win_nt": ctc.Tag.WINDOWS,
}

TAG_TO_PLATFORM = {str(t): p for p, t in six.iteritems(PLATFORM_TO_TAG) if p != "osx_arm"}

SUB_PLATFORMS = {
    TAG_TO_PLATFORM[g]: [TAG_TO_PLATFORM[t] for t in g]
    for g in (ctc.Tag.Group.LINUX, ctc.Tag.Group.OSX, ctc.Tag.WINDOWS)
}

PLATFORM_TAGS = set(TAG_TO_PLATFORM)

DEFAULT_PLATFORM = "linux_ubuntu_12.04_precise"

LXC_PLATFORMS = tuple(sorted(
    (platform for platform, tag in six.iteritems(PLATFORM_TO_TAG) if str(tag) in ctc.Tag.Group.LINUX),
    key=lambda p: "" if p == DEFAULT_PLATFORM else p
))


@patterns.singleton
def get_platform_alias(platform):
    """
        Получить алиас совместимых платформ для указанной платформы

        :param platform: название платформы
        :return: название алиаса в виде строки; название платформы, если алиаса найти не удалось
        :rtype: str
    """
    if not platform:
        return str("")
    platform = str(platform).lower()
    for platform_alias, platforms_list in six.iteritems(PLATFORM_ALIASES):
        if platform in platforms_list:
            return str(platform_alias)

    # Catch marginal platforms
    platform_alias = platform
    if "windows" in platform or "win_nt" in platform or "win32" in platform:
        platform_alias = "win_nt"

    if "osx_arm" in platform:
        return "osx_arm"

    if "aarch64" in platform:
        if "trusty" in platform:
            platform_alias = "linux_ubuntu_14.04_trusty_aarch64"
    else:
        if "focal" in platform:
            platform_alias = "linux_ubuntu_20.04_focal"
        elif "bionic" in platform:
            platform_alias = "linux_ubuntu_18.04_bionic"
        elif "xenial" in platform:
            platform_alias = "linux_ubuntu_16.04_xenial"
        elif "trusty" in platform:
            platform_alias = "linux_ubuntu_14.04_trusty"
        elif ("precise" in platform) or ("wheezy" in platform):
            platform_alias = "linux_ubuntu_12.04_precise"
        elif ("lucid" in platform) or ("squeeze" in platform):
            platform_alias = "linux_ubuntu_10.04_lucid"
        elif "rhel" in platform or "redhat" in platform:
            if "santiago" in platform:
                platform_alias = "linux_rhel_santiago"
            else:
                platform_alias = "linux_rhel"
        elif "cygwin_nt-" in platform:
            platform_alias = "cygwin_" + platform[10:13]

    return str(platform_alias)


def compare_platforms(platform, another_platform):
    """
        Сравнить платформы с учётом алиасов совместимости.

        :param platform: значение архитектуры для проверки
        :param another_platform: с какой платформой сравнивать.
        :return: True, если платформы совместимы; False в противном случае
        :rtype: bool
    """
    if platform == another_platform:
        return True
    return get_platform_alias(platform) == get_platform_alias(another_platform)


def is_binary_compatible(binary_platform, host_platform):
    """
        Сравнить платформу исполняемого файла и хоста, на котором предполагается исполнение
        :param binary_platform: платформа исполняемого файла
        :param host_platform: платформа хоста
        :return: True, если исполняемый файл может быть запущен на указанной платформе
        :rtype: bool
    """
    if compare_platforms(host_platform, binary_platform):
        return True
    try:
        host_key_version = PLATFORM_ALIAS_BINARY.get(get_platform_alias(host_platform))
        binary_key_version = PLATFORM_ALIAS_BINARY.get(get_platform_alias(binary_platform))

        if host_key_version and binary_key_version:
            return host_key_version[0] == binary_key_version[0] and host_key_version[1] >= binary_key_version[1]
    except Exception:
        logging.getLogger(__name__).exception('Failed to compare platforms')
        return False


@patterns.singleton
def get_arch_from_platform(platform):
    """
    Returns OS family name for given platform string.

    :param platform: string to be examined
    :return: one of "freebsd", "linux", "osx", "cygwin", "win_nt", "any"
    """
    if not platform:
        return ctm.OSFamily.ANY
    if platform in ctm.OSFamily:
        return platform

    if "aarch64" in platform:
        return ctm.OSFamily.LINUX_ARM

    alias = get_platform_alias(platform)
    if "freebsd" in alias:
        return ctm.OSFamily.FREEBSD
    elif "linux" in alias:
        return ctm.OSFamily.LINUX
    elif "osx" in alias or "darwin" in alias:
        if "osx_arm" in alias or "arm64" in alias:
            return ctm.OSFamily.OSX_ARM
        return ctm.OSFamily.OSX
    elif "cygwin" in alias:
        return ctm.OSFamily.CYGWIN
    elif "win_nt" in alias:
        return ctm.OSFamily.WIN_NT
    raise ValueError("Unknown platform {!r}".format(platform))


def on_windows():
    return "Windows" == pl.uname()[0]


def on_wsl():
    return "Linux" == pl.uname()[0] and "Microsoft" in str(pl.uname())


def on_linux():
    return "Linux" == pl.uname()[0] and "Microsoft" not in str(pl.uname())


def on_osx():
    return "Darwin" == pl.uname()[0]


@patterns.singleton
def platform(aliased=0, terse=0):
    if six.PY2 or pl.system() not in ("Linux",):
        return pl.platform(aliased, terse)
    import distro
    system, node, release, version, machine, processor = pl.uname()
    if machine == processor:
        processor = ""
    if aliased:
        system, release, version = pl.system_alias(system, release, version)

    distname, distversion, distid = distro.linux_distribution("")
    if distname and not terse:
        return pl._platform(system, release, machine, processor, "with", distname, distversion, distid)
    return pl.platform(aliased, terse)


@patterns.singleton
def dist_version():
    def _get_win_ver():
        for line in sp.check_output([wsl.User.cmd, "/C", "systeminfo"]).splitlines():
            if line.startswith("OS Version"):
                return line.split(":")[1].strip()
            return "Unknown windows"

    if on_linux():
        if distro:
            return distro.version()
        return pl.linux_distribution()[1]
    if on_osx():
        if six.PY2:
            return sp.check_output(["sw_vers", "-productVersion"])
        return pl.mac_ver()[0]
    if on_wsl():
        return _get_win_ver()
    if on_windows():
        if six.PY2:
            return _get_win_ver()
        return pl.win32_ver()[1]


class UbuntuRelease(enum.Enum):
    LUCID = "10.04"
    PRECISE = "12.04"
    TRUSTY = "14.04"
    XENIAL = "16.04"
    BIONIC = "18.04"
    FOCAL = "20.04"
    JAMMY = "22.04"
