# coding: utf-8

import io
import time
import cProfile
import marshal
import zlib
import base64


class Profiler(object):
    """
    A wrapper around `cProfile.Profile` class.

    Direct usage:

    .. code-block:: python

        from common.profiler import Profiler
        profiler = Profiler('test')
        profiler.enable()
        # do something
        profiler.disable()
        profiler.dump_to_file()

    Decorator usage:

    .. code-block:: python

        from common.profiler import Profiler

        @Profiler.profiler(log_path="some_path")
        def test():
            pass

    How to analyze profiles:
        - `~sandbox/scripts/dev/read_profiler.py /var/tmp/sandbox_profiler_test.log`
        - `~sandbox/scripts/dev/read_profiler.py <custom path to log>`
    """

    LOG_FILE_NAME = '/var/tmp/sandbox_profiler_{0}.log'

    @classmethod
    def profiler(cls, log_path=None):
        def dec(func):
            def wrapper(*args, **kwargs):
                profiler = cls(name=func.__name__, log_path=log_path)
                profiler.enable()
                try:
                    return func(*args, **kwargs)
                finally:
                    profiler.disable()
                    profiler.dump_to_file()
            return wrapper
        return dec

    def __init__(self, name=None, log_path=None, sort_field=-1):
        self.name = name or 'default'
        self.log_file_name = log_path or self.LOG_FILE_NAME.format(name)
        self.profiler = cProfile.Profile()
        self.sort_field = sort_field

    def enable(self):
        self.profiler.enable()

    def disable(self):
        self.profiler.disable()

    def dump_to_str(self):
        self.profiler.create_stats()
        import pstats
        buf = io.BytesIO()
        pstats.Stats(self.profiler, stream=buf).strip_dirs().sort_stats(self.sort_field).print_stats()
        return buf.getvalue()

    def dump_to_file(self, file_name=None):
        file_name = self.log_file_name if not file_name else file_name
        self.profiler.create_stats()
        profiler_dump = marshal.dumps(self.profiler.stats)
        try:
            profiler_compressed = zlib.compress(profiler_dump)
        except zlib.error:
            profiler_compressed = profiler_dump
        profiler_data = base64.encodestring(profiler_compressed).replace('\n', '')
        with open(file_name, 'ab') as f:
            f.write('{0}\t{1}\t{2}\n'.format(time.time(), self.name, profiler_data))

    def dump_to_console(self, sort=-1):
        self.profiler.create_stats()
        self.profiler.print_stats(sort=sort)
