from __future__ import absolute_import

import os
import re
import sys
import inspect
import logging
import importlib
import subprocess
import collections

import six

from .. import config
from .. import system
from .. import patterns
from ..types import misc as ctm


TYPES = {}


logger = logging.getLogger(__name__)

TaskTypeLocation = collections.namedtuple("TaskType", ("package", "cls", "revision"))

_MODULE_NAME_PATTERN = re.compile("[a-zA-Z][_a-zA-Z0-9]*")
_OWNERS_PATTERN = re.compile("OWNER\((?P<owners>[\-\ \n_:a-zA-Z0-9]+)\)")
_PREFIX_PATTERN = re.compile("^(rb|g):")

# Task discovering is allowed for modules with this suffix. Even for those not in sandbox.projects namespace.
SBTASK_SUFFIX = "_sbtask"


def load_task_classes(module):
    from sandbox import sdk2

    if isinstance(module, six.string_types):
        module = importlib.import_module(module)
    if not module:
        return
    cls = getattr(module, "__Task__", None)
    if cls and cls.type:
        yield cls

    if six.PY2:
        from sandbox.sandboxsdk import task as sdk1_task
        for _, sym in inspect.getmembers(module, inspect.isclass):
            if sym is not cls and issubclass(sym, sdk1_task.SandboxTask) and sym.type:
                yield sym

    for _, sym in inspect.getmembers(module, inspect.isclass):
        if issubclass(sym, sdk2.Task) and sym.name in sdk2.Task:
            yield sym


def _get_package_revision(package):
    dir_path = None
    if package and hasattr(package, "__file__"):
        dir_path = os.path.dirname(package.__file__)
    elif isinstance(package, six.string_types):
        dir_path = package
    if dir_path:
        rf = os.path.join(dir_path, ".revision")
        if os.path.exists(rf):
            try:
                with open(rf) as f:
                    return int(f.read().strip())
            except ValueError:
                pass
    return None


@patterns.singleton
def tasks_dir():
    """
    :return: path to sandbox code directory (more generally, that which contains "projects" directory)
    """

    from sandbox import projects
    return os.path.realpath(os.path.join(os.path.dirname(inspect.getfile(projects)), os.pardir))


def _default_import_wrapper(f, mod, *args, **kwargs):
    return f(mod, *args, **kwargs)


def py3_sources_binary_path():
    if not system.inside_the_binary():
        if config.Registry().common.installation in ctm.Installation.Group.LOCAL:
            return config.Registry().common.py3_sources_binary
    return None


def py3_modules(root_dir, py3_modules_parser):
    try:
        if py3_modules_parser is None:
            return set()
        p = six.ensure_str(subprocess.check_output([py3_modules_parser, "-r", root_dir])).strip()
        return set(p.split(","))
    except Exception:
        logger.exception("Can't find py3 modules.")
        return set()


def _import_modules_inside_binary(import_wrapper=_default_import_wrapper, registry_modules=()):
    import __res as resource
    init_suffix = ".__init__"
    modules = []
    for name in resource.iter_py_modules():
        if name.endswith(init_suffix):
            name = name[:-len(init_suffix)]
        if (
            name.startswith("projects.") or name.startswith("sandbox.projects.") or
            name.endswith(SBTASK_SUFFIX) or name in registry_modules
        ):
            modules.append(import_wrapper(importlib.import_module, name))
    return modules


def _import_modules_on_fs(
    raise_exceptions, target_dir, import_wrapper=_default_import_wrapper, py3_modules_parser=None
):
    if config.Registry().common.installation == ctm.Installation.TEST:
        # TODO: This actually a dirtiest hack to speedup tests and should be relocated to `conftest.py` somehow.
        im = importlib.import_module
        modules = [
            im("sandbox.sdk2.tests"),
            im("sandbox.projects.sandbox.reshare_resource"),
            im("sandbox.projects.sandbox.restore_resource"),
            im("sandbox.projects.sandbox.test_task"),
            im("sandbox.projects.sandbox.test_task_2"),
            im("sandbox.projects.sandbox.test_task_21"),
            im("sandbox.projects.sandbox.unit_test_task"),
            im("sandbox.projects.sandbox.unit_test_task_2"),
            im("sandbox.projects.sandbox.backup_resource_2"),
            im("sandbox.projects.sandbox.cleanup_2"),
        ]
    else:
        modules = [
            import_wrapper(_import_module, pkg, raise_exceptions=raise_exceptions)
            for pkg in get_packages(target_dir, py3_modules_parser=py3_modules_parser)
        ]

    return modules


def load_project_types(
    raise_exceptions=False, reuse=False, force_from_fs=False, import_wrapper=_default_import_wrapper,
    py3_modules_parser=None
):
    """
    Import all tasks' code and store modules and their respective information in sandbox.projects.TYPES dictionary.
    When called from binary, use built-in sandbox/projects target for inspection

    :param raise_exceptions: fail if any exceptions during import occur
    :param reuse: return cached tasks modules from sandbox.projects.TYPES, if possible
    :param force_from_fs: load tasks from the filesystem, even when inside binary
    :param import_wrapper: wrapper for import function
    """

    import_from_fs = force_from_fs or not system.inside_the_binary()

    if import_from_fs:
        if reuse:
            source_load = "sandbox.projects.TYPES"
        else:
            source_load = tasks_dir()
    else:
        if reuse:
            source_load = "sandbox.common.projects_handler.TYPES"
        else:
            source_load = "built-in sandbox.projects module and modules from macros"

    message = "Trying to load tasks code from {}".format(source_load)
    logger.debug(message)

    try:
        from sandbox import projects
        types = getattr(projects, "TYPES", None)
        if not types:
            types = projects.TYPES = {}
        revision = getattr(projects, "__revision__", 0)
    except ImportError:
        if import_from_fs:
            if raise_exceptions:
                raise
            logger.warning("Cannot import projects")
            return
        else:
            types = TYPES
            revision = None

    if reuse and types:
        return types

    if import_from_fs:
        modules = _import_modules_on_fs(
            raise_exceptions, tasks_dir(), import_wrapper=import_wrapper,
            py3_modules_parser=py3_modules_parser
        )
    else:
        registry_modules = set()
        import library
        for var_name, var_value in library.python.resource.iteritems():
            if var_name.startswith("SANDBOX_TASK_REGISTRY."):
                registry_modules.add(six.ensure_str(var_value))

        modules = _import_modules_inside_binary(import_wrapper=import_wrapper, registry_modules=registry_modules)

    try:
        from sandbox.projects.tests.sdk1_tasks import is_new_sdk1_task
    except ImportError:
        def is_new_sdk1_task(_):
            return False

    types.clear()
    black_list = []
    dunder_types = {}

    for m in filter(None, modules):
        revision = _get_package_revision(m)
        for cls in load_task_classes(m):
            if not (cls and cls.type):
                continue
            is_dunder = getattr(m, "__Task__", None) == cls
            if is_dunder:
                dunder_types[cls.type] = cls
            if cls.__module__ == m.__name__ or is_dunder:
                if is_new_sdk1_task(cls):
                    if cls.type not in black_list:
                        black_list.append(cls.type)
                else:
                    types[cls.type] = TaskTypeLocation(m.__name__, cls, revision)

    logger.info("Projects code r%s loaded. Modules inspected: %d", revision, len(modules))
    if black_list:
        logger.warning("Blacklisted task types: %s", " ".join(black_list))

    if import_from_fs:
        projects.BLACK_LIST = black_list
        projects.DUNDER_TYPES = dunder_types
    return types


def _is_package(root_dir, modules):
    if not all(map(_MODULE_NAME_PATTERN.match, modules)):
        return False
    for ind in range(len(modules)):
        init_file_path = os.path.join(root_dir, *(modules[:ind + 1] + ["__init__.py"]))
        if not os.path.isfile(init_file_path):
            return False
    return True


def get_packages(root_dir, py3_modules_parser=None):
    packages = []

    if py3_modules_parser is None:
        py3_modules_parser = py3_sources_binary_path()

    # This function is also called from a binary, whose working directory is arcadia,
    # so it may accidentally import modules from both namespaces. Well, now it may not!
    def appropriate_package(split_package_name):
        if system.inside_the_binary():
            return split_package_name[0:2] == ["sandbox", "projects"]
        return split_package_name[0] == "projects"
    py3_sources = py3_modules(root_dir, py3_modules_parser)
    last_dir = None

    for dir_path, _, _ in os.walk(root_dir):
        if dir_path == root_dir:
            continue
        modules = os.path.relpath(dir_path, root_dir).split(os.sep)
        if modules and appropriate_package(modules) and _is_package(root_dir, modules):
            package_name = ".".join(modules)
            if last_dir is not None and package_name.startswith(last_dir):
                continue
            if package_name in py3_sources:
                last_dir = package_name
                continue
            if package_name not in py3_sources:
                packages.append(".".join(modules))
    return packages


def _import_module(name, raise_exceptions=False):
    """
    Safely import a module.

    :return:    Module object in case the module has been loaded successfully and `None` otherwise.
    """
    try:
        return importlib.import_module(name)
    except Exception as e:
        if raise_exceptions:
            type_, value, tb = sys.exc_info()
            six.reraise(type_, type_("Exception during \"{}\" import: {}".format(name, value)), tb)
        else:
            logger.error("Module {!r} import FAILURE".format(name))
            logger.exception(e)
            return None


def current_projects_revision():
    """
    Get task code version from projects

    :return: version
    :rtype: str
    """
    try:
        from sandbox import projects
        return str(projects.__revision__)
    except Exception as error:
        logger.error("Can't load projects version. Error: {0}".format(error))
        return "cannot get tasks version"


def task_type_relative_path(typename):
    from sandbox import projects
    if typename not in projects.TYPES:
        return None
    fullpath = sys.modules[projects.TYPES[typename].cls.__module__].__file__
    if fullpath.endswith('.pyc'):
        fullpath = fullpath[:-1]
    return os.path.relpath(fullpath, os.path.dirname(os.path.dirname(projects.__file__)))
