from __future__ import absolute_import

import re
import time
import uuid
import errno
import random
import socket
import logging
import datetime
import threading
import collections
import xml.parsers.expat

import six
from six.moves import http_client as httplib
from six.moves import xmlrpc_client as xmlrpclib

from .. import auth as common_auth
from .. import format
from .. import patterns
from ..types import misc as ctm

_ILLEGAL_UNICHARS = [0xd800, 0xdbff, 0xdc00, 0xdfff, 0xd800, 0xdbff, 0xdc00, 0xdfff, 0xd800, 0xdbff, 0xdc00, 0xdfff]
_ILLEGAL_XML_CHARS_RE = re.compile(
    u"([\u0000-\u0008\u000b-\u000c\u000e-\u001f\ufffe-\uffff])"
    u"|([{}-{}][^{}-{}])|([^{}-{}][{}-{}])|([{}-{}]$)|(^[{}-{}])".format(*map(six.unichr, _ILLEGAL_UNICHARS))
)


# (DEPRECATED) Backward compatibility
Authentication = common_auth.Authentication
NoAuth = common_auth.NoAuth
Plain = common_auth.Plain
OAuth = common_auth.OAuth
YandexSession = common_auth.YandexSession
Session = common_auth.Session
brace_expansion = format.brace_expansion


def convert_unicode_to_safe_xml(data):
    if isinstance(data, six.string_types) and data:
        data = _ILLEGAL_XML_CHARS_RE.sub("?", data)
    return data


def safe_xmlrpc_dict(ctx):
    """
        Filter keys and values from dict which don't serialize to xmlrpc
        @param ctx:
        @return:
    """
    res = {}
    for k, v in ctx.items():
        try:
            xmlrpclib.dumps((dict([(k, v)]),), allow_none=True)
        except Exception:
            pass
        else:
            res[k] = convert_unicode_to_safe_xml(v)
    return res


def safe_xmlrpc_list(ctx):
    return [convert_unicode_to_safe_xml(item) for item in ctx]


class TimeoutTransport(xmlrpclib.Transport):
    # Limit in seconds to establish a new connection.
    CONNECT_TIMEOUT = 5

    def __init__(self, logger, auth, ssl=True):
        xmlrpclib.Transport.__init__(self)
        self.timeout = 60
        self.ua = None
        self.rid = None
        self.hosts = None
        self.component = None
        self.logger = logger
        self.request_body = None
        self.auth = auth
        self.ssl = ssl

    def reset(self):
        self.logger.debug("Resetting the connection.")
        self._connection = None
        if self.hosts:
            self.hosts.rotate(-1)

    @property
    def host(self):
        return self.hosts[0] if self.hosts else None

    def make_connection(self, host):
        # return an existing connection if possible.  This allows HTTP/1.1 keep-alive.
        if not self.hosts:
            self.hosts = collections.deque(brace_expansion(map(str.strip, host.split(" "))))
            random.shuffle(self.hosts)
        if self._connection and host == self._connection[0]:
            return self._connection[1]

        host = self.host
        # create a HTTPS connection object from a host descriptor
        chost, self._extra_headers, x509 = self.get_host_info(host)
        # connect to the remote host with special timeout
        if self.ssl:
            conn = httplib.HTTPSConnection(chost, timeout=self.CONNECT_TIMEOUT, **(x509 or {}))
        else:
            conn = httplib.HTTPConnection(chost, timeout=self.CONNECT_TIMEOUT)
        conn.connect()
        conn.sock.settimeout(self.timeout)
        # store the host argument along with the connection object
        self._connection = host, conn
        return conn

    def send_request(self, connection, handler, request_body):
        self.request_body = request_body
        xmlrpclib.Transport.send_request(self, connection, handler, request_body)
        for k, v in (
                (ctm.HTTPHeader.USER_AGENT, self.ua),
                (ctm.HTTPHeader.REQUEST_ID, self.rid),
                (ctm.HTTPHeader.COMPONENT, self.component),
        ) + tuple(self.auth):
            if v is not None:
                connection.putheader(k, v)


class ReliableServerProxy(object):
    """
    XMLRPC client wrapper, which can retry network and database errors.
    Instances of the class has no multi-threading support, so
    each thread should use its own instance of the class.
    """

    DEFAULT_INTERVAL = 5    # Initial wait period between retries in seconds.
    DEFAULT_TIMEOUT = 60    # Initial call timeout in seconds.
    MAX_INTERVAL = 300      # Maximum wait period between retries in seconds.
    MAX_TIMEOUT = 180       # Maximum call timeout in seconds.
    DEFAULT_URL = "https://sandbox.yandex-team.ru/sandbox/xmlrpc"

    class TimeoutExceeded(Exception):
        """ A error class to be raised in case of maximum amount of wait time exceeded. """

    class SessionExpired(BaseException):
        """ Raises if XMLRPC session is expired """

    # xmlrpclib.Fault errorCodes
    class ErrorCodes:
        ERROR = 1  # will raise to user code
        RETRYABLE_ERROR = -1  # will retry till OK
        FATAL_ERROR = -2  # fatal error, will raise to user code as successor of the BaseException

    HTTP_NOT_RETRYABLE_ERRORS = (
        httplib.FORBIDDEN,
        httplib.UNAUTHORIZED,
        httplib.BAD_REQUEST,
        httplib.REQUEST_ENTITY_TOO_LARGE,
    )

    # Global authentication object case, used for external authentication information providers.
    _external_auth = None
    # Globally defined Sandbox component name to identify internal requests.
    _default_component = None

    def __init__(
        self,
        url=None, verbose=False, auth=None, logger=None, total_wait=None, oauth_token=None,
        transport=None, ua=None, component=None,
    ):
        """
        Constructor.

        :param url:            URL of XMLRPC server to connect. Supports space-separated servers list and bash brackets
                               expanding. Will use balancer address if not provided.
        :param verbose:        Verbose mode - prints HTTP requests and responses to STDERR.
        :param auth:           Authentication object to be used for authentication on remote server.
                               Also, in case of the object is not an instance of :class:`Authentication`,
                               it will be treated as OAuth token.
        :param logger:         Logger object to use. Uses root logger if not provided.
        :param total_wait:     Maximum total wait time in seconds. Restricts maximum time spend to single RPC call,
                               including the time of call itself taken and also call re-tries.
        :param oauth_token:    OAuth token to be passed via "Authorization" header.
                               **Deprecated**. Use `auth` parameter instead.
        :param transport:      Not used. Keep for backward compatibility.
        :param ua:             User Agent name to be sent with requests
        :param component:      Sandbox component name. For internal use only.
        """
        self.__interval = self.DEFAULT_INTERVAL
        self.__timeout = self.DEFAULT_TIMEOUT
        self.url = url or self.DEFAULT_URL
        self.logger = logger or logging.getLogger(__name__)

        if oauth_token or (auth and not isinstance(auth, Authentication)):
            auth = OAuth(oauth_token or auth)
        elif not auth:
            auth = self.__class__._external_auth or NoAuth()

        self.transport = TimeoutTransport(self.logger.getChild("transport"), auth, self.url.lower().startswith("https"))
        self.transport.ua, self.transport.component = ua, component or self._default_component
        self.logger.debug(
            "[0x%x] XMLRPC client instance for '%s' created in thread '%s' with %s authorization.",
            id(self), self.url, threading.current_thread().ident, auth
        )
        self.total_wait = total_wait
        self.remote = xmlrpclib.ServerProxy(self.url, allow_none=True, transport=self.transport, verbose=verbose)

    @property
    def interval(self):
        return self.__interval

    @property
    def timeout(self):
        return self.__timeout

    def _update_interval(self):
        self.__interval = min(self.__interval * 55 / 34, self.MAX_INTERVAL)
        return self.__interval

    def _update_timeout(self):
        self.__timeout = min(self.__timeout * 55 / 34, self.MAX_TIMEOUT)
        return self.__timeout

    def reset(self):
        self.__interval = self.DEFAULT_INTERVAL
        self.__timeout = self.DEFAULT_TIMEOUT

    def __getattr__(self, method):
        def wrapper(*args):
            self.reset()
            self.transport.rid = uuid.uuid4().hex
            self.transport.timeout = min(self.timeout, self.total_wait if self.total_wait else self.timeout)

            spent = 0
            started = time.time()
            while spent < self.total_wait if self.total_wait else True:
                try:
                    self.logger.debug(
                        "[0x%x] Remote method '%s' call (id '%s', tout %s, srv: '%s')",
                        id(self), method,
                        self.transport.rid, self.transport.timeout, self.transport.host or self.url
                    )
                    return getattr(self.remote, method)(*args)
                except (xml.parsers.expat.ExpatError, TypeError):
                    raise
                except Exception as ex:
                    self.transport.reset()
                    self.logger.error(
                        "[0x%x] Remote method '%s' call error (id '%s', tout %s, int %s, srv: '%s'): %s",
                        id(self), method, self.transport.rid, self.timeout, self.interval,
                        self.transport.host or self.url, ex
                    )
                    if isinstance(ex, xmlrpclib.Fault) and ex.faultCode != self.ErrorCodes.RETRYABLE_ERROR:
                        # re-raise all except RETRYABLE_ERROR and authorization errors.
                        raise self.SessionExpired(ex) if ex.faultCode == self.ErrorCodes.FATAL_ERROR else ex
                    if isinstance(ex, xmlrpclib.ProtocolError):
                        if ex.errcode == httplib.GONE:
                            raise self.SessionExpired(ex)
                        elif ex.errcode in self.HTTP_NOT_RETRYABLE_ERRORS:
                            if ex.errcode == httplib.BAD_REQUEST:
                                self.logger.debug(
                                    "[0x%x] Error parsing request on server side: %s",
                                    id(self), self.transport.request_body
                                )
                                ex.errmsg = ": ".join((ex.errmsg, ex.headers.get("X-Error-Message", "")))
                            raise

                    if not (
                        (isinstance(ex, xmlrpclib.ProtocolError) and ex.errcode == httplib.BAD_GATEWAY) or
                        (
                            isinstance(ex, socket.error) and
                            ex.errno in (errno.ECONNREFUSED, errno.EHOSTDOWN, errno.ENETUNREACH)
                        )

                    ) or len(self.transport.hosts) == 1:
                        tick = self._update_interval()
                        time.sleep(min(tick, self.total_wait - spent) if self.total_wait else tick)
                        self.transport.timeout = self._update_timeout()
                spent = time.time() - started

            raise self.TimeoutExceeded(
                "Error calling method '{}' at '{}' - no response given after {!s}.".format(
                    method, self.url, datetime.timedelta(seconds=spent)
                )
            )

        setattr(self, method, wrapper)
        return wrapper


@six.add_metaclass(patterns.ThreadLocalMeta)
class ThreadLocalCachableServerProxy(ReliableServerProxy):
    """
    To eliminate a problem of thread safety and also provide a single method to obtain a server proxy object,
    the class using special meta, which ensures one instance per thread.
    """
