from __future__ import absolute_import, unicode_literals

import re
import abc
import copy
import json
import uuid
import time
import errno
import random
import socket
import logging
import warnings
import datetime
import threading
import contextlib
import collections

import requests
import requests.auth
import requests.structures
import requests.packages.urllib3.exceptions

import six
from six.moves import http_client as httplib
from six.moves.urllib import parse as urlparse

from .. import auth as common_auth
from .. import format as common_format
from .. import patterns
from .. import itertools as common_itertools
from ..types import misc as ctm


def patch_session_timeout(session):
    import requests.adapters
    import requests.packages.urllib3.connection
    import requests.packages.urllib3.poolmanager

    class HTTPConnection(requests.packages.urllib3.connection.HTTPConnection):
        def _new_conn(self):
            timeout, self.timeout = self.timeout, Client.CONNECT_TIMEOUT
            try:
                conn = super(HTTPConnection, self)._new_conn()
                conn.settimeout(timeout)
                return conn
            finally:
                self.timeout = timeout

    class HTTPSConnection(requests.packages.urllib3.connection.HTTPSConnection):
        def _new_conn(self):
            timeout, self.timeout = self.timeout, Client.CONNECT_TIMEOUT
            try:
                conn = super(HTTPSConnection, self)._new_conn()
                conn.settimeout(timeout)
                return conn
            finally:
                self.timeout = timeout

    class HTTPConnectionPool(requests.packages.urllib3.HTTPConnectionPool):
        ConnectionCls = HTTPConnection

    class HTTPSConnectionPool(requests.packages.urllib3.HTTPSConnectionPool):
        ConnectionCls = HTTPSConnection

    class PoolManager(requests.packages.urllib3.poolmanager.PoolManager):
        SCHEME2CLASS = {"http": HTTPConnectionPool, "https": HTTPSConnectionPool}

        def _new_pool(self, scheme, host, port):
            pool_cls = self.SCHEME2CLASS[scheme]
            kwargs = self.connection_pool_kw
            if scheme == 'http':
                kwargs = kwargs.copy()
                for kw in requests.packages.urllib3.poolmanager.SSL_KEYWORDS:
                    kwargs.pop(kw, None)

            return pool_cls(host, port, **kwargs)

    class HTTPAdapter(requests.adapters.HTTPAdapter):
        def init_poolmanager(self, connections, maxsize, block=requests.adapters.DEFAULT_POOLBLOCK, **pool_kwargs):
            self._pool_connections = connections
            self._pool_maxsize = maxsize
            self._pool_block = block
            self.poolmanager = PoolManager(num_pools=connections, maxsize=maxsize, block=block, **pool_kwargs)

    session.mount('http://', HTTPAdapter())
    session.mount('https://', HTTPAdapter())


class Path(object):
    """ Pythonish representation of the URL path """

    DELIMITER = "/"

    def __init__(self, rest, path="", input_mode=None, output_mode=None):
        self.__rest = rest
        self.__path = path
        self.__input = input_mode or Client.JSON()
        self.__output = output_mode or Client.JSON()

    def __str__(self):
        return self.__path or self.DELIMITER

    def __lshift__(self, other):
        own = self.__input
        if callable(other):
            other = other()
        if not isinstance(other, Client.Modifiers):
            raise ValueError("Modifier is not a subclass of `Client.Modifiers`")
        hdrs_jar = own.headers
        if not other.type:
            own.headers = other.headers
            other, own = own, other
            other.headers.update(hdrs_jar)
        else:
            other.headers = own.headers
        return self.__class__(self.__rest, self.__path, other, self.__output)

    def __rshift__(self, other):
        own = self.__output
        if callable(other):
            other = other()
        if not isinstance(other, Client.Modifiers):
            raise ValueError("Modifier is not a subclass of `Client.Modifiers`")
        hdrs_jar = own.headers
        if not other.type:
            own.headers = other.headers
            other, own = own, other
            other.headers.update(hdrs_jar)
        else:
            other.headers = own.headers
        return self.__class__(self.__rest, self.__path, self.__input, other)

    @staticmethod
    def __slice2params(slice_):
        params = {}
        if slice_.start is not None:
            params["offset"] = slice_.start
        if slice_.stop is not None:
            params["limit"] = slice_.stop
        if slice_.step is not None:
            params["order"] = slice_.step
        return params

    def __getitem__(self, item):
        params, slice_ = None, None
        if isinstance(item, tuple):
            params, slice_ = item
            if not (isinstance(params, (dict, type(Ellipsis))) and isinstance(slice_, slice)):
                raise ValueError("Parameters must be of type (dict or ..., slice)")
            params.update(self.__slice2params(slice_))
        elif isinstance(item, dict):
            params = item
        elif isinstance(item, slice):
            params = self.__slice2params(item)
        if params is not None or item == Ellipsis:
            return self.read(params)
        return Path(self.__rest, self.DELIMITER.join((self.__path, str(item))), self.__input, self.__output)

    def __getattr__(self, item):
        if item.isupper() and hasattr(Client, item):
            return getattr(Client, item)
        return Path(self.__rest, self.DELIMITER.join((self.__path, str(item))), self.__input, self.__output)

    def __setattr__(self, item, value):
        if item.startswith("_"):
            return super(Path, self).__setattr__(item, value)
        self[item].update(value)

    def __setitem__(self, item, value):
        self[item].update(value)

    def create(self, data=None, params=None, **kws):
        if data is not None:
            if params is None:
                params = kws
        elif params is None:
            data = kws
        return self.__rest.create(str(self), data, params, self.__input, self.__output)

    def __call__(self, params=None, **kws):
        return self.create(params, **kws)

    def read(self, params=None, **kws):
        if params is None:
            params = kws
        return self.__rest.read(str(self), params, self.__input, self.__output)

    def delete(self, data=None, params=None, **kws):
        if data is not None:
            if params is None:
                params = kws
        elif params is None:
            data = kws
        self.__rest.delete(str(self), data, params, self.__input, self.__output)

    def __delitem__(self, item):
        self[item].delete()

    __delattr__ = __delitem__

    def update(self, data=None, params=None, **kws):
        if data is not None:
            if params is None:
                params = kws
        elif params is None:
            data = kws
        return self.__rest.update(str(self), data, params, self.__input, self.__output)


@contextlib.contextmanager
def _urllib3_warning_suppress(*wrn_cls_names):
    with warnings.catch_warnings():
        for cls_name in wrn_cls_names:
            cls = getattr(requests.packages.urllib3.exceptions, cls_name, None)
            if cls is not None:
                warnings.simplefilter("ignore", cls)
        yield


def _urllib3_logging_suppress(called=[]):
    """
    Silence urllib3's logging which floods debug.log, as it's unnecessary (common.rest.Client does the thing)
    :param called: fake parameter to only execute meaningful body once
    """

    if called:
        return
    blacklist = {"urllib3.connectionpool"}
    for name, logger in logging.Logger.manager.loggerDict.items():
        if name not in blacklist:
            continue
        if not isinstance(logger, logging.PlaceHolder):
            for handler in logger.handlers[:]:
                logger.removeHandler(handler)
        logger.disabled = 1
    called.append(True)


class Client(object):
    """
    REST API client, which can retry network and database errors.
    Instances of the class has no multi-threading support, so
    each thread should use its own instance of the class.

    Usage examples:

    .. code-block:: python

        response_headers = sandbox.HEADERS()
        sandbox = Client() << sandbox.HEADERS({"X-Header": "Value"}) >> response_headers

        # GET /<base path>/resource:
        sandbox.resource[:]
        sandbox.resource[...]
        sandbox.resource.read()

        # GET /<base path>/resource?limit=10:
        sandbox.resource[:10]
        sandbox.resource.read(limit=10)
        sandbox.resource.read({"limit": 10})

        # GET /<base path>/resource?limit=10&offset=20:
        sandbox.resource[20:10]
        sandbox.resource.read(offset=20, limit=10)
        sandbox.resource.read({"offset": 20, "limit": 10})

        # GET /<base path>/resource?limit=10&offset=20&order=-id:
        sandbox.resource[20:10:"-id"]
        sandbox.resource.read(offset=20, limit=10, order="-id")
        sandbox.resource.read({"offset": 20, "limit": 10, "order": "-id"})

        # GET /<base path>/resource?type=OTHER_RESOURCE&limit=10&order=state:
        sandbox.resource[{"type": "OTHER_RESOURCE"}, : 10: "state"]
        sandbox.resource.read(type="OTHER_RESOURCE", limit=10, order="state")
        sandbox.resource.read({"type": "OTHER_RESOURCE", "limit": 10, "order": "state"})

        # GET /<base path>/resource/12345:
        sandbox.resource[12345][:]
        sandbox.resource[12345][...]
        sandbox.resource[12345].read()

        # GET /<base path>/resource/12345/attribute:
        sandbox.resource[12345].attribute[:]
        sandbox.resource[12345].attribute[...]
        sandbox.resource[12345].attribute.read()

        # POST /<base path>/resource:
        sandbox.resource(**fields)
        sandbox.resource({<fields>})
        sandbox.resource.create(**fields)
        sandbox.resource.create({<fields>})

        # PUT /<base path>/resource/12345:
        sandbox.resource[12345] = {<fields>}
        sandbox.resource[12345].update(**fields)
        sandbox.resource[12345].update({<fields>})

        # WARNING: PUT endpoints in Sandbox API normally *replace*
        # the whole resource rather than partially update it as one
        # would expect from a method called `update()`.

        # DELETE /<base path>/resource/12345/attribute/attr1:
        del sandbox.resource[12345].attribute.attr1
        del sandbox.resource[12345].attribute["attr1"]
        sandbox.resource[12345].attribute["attr1"].delete()
    """

    DEFAULT_INTERVAL = 5  # Initial wait period between retries in seconds.
    DEFAULT_TIMEOUT = 60  # Initial call timeout in seconds.
    DEFAULT_BASE_URL = "https://sandbox.yandex-team.ru/api/v1.0"
    CONNECT_TIMEOUT = 5  # Maximum wait time in seconds to establish a new connection.
    MAX_INTERVAL = 300  # Maximum wait period between retries in seconds.
    MAX_TIMEOUT = 180  # Maximum call timeout in seconds.
    DEFAULT_EXP_INTERVAL_FACTOR = 55 / 34.0  # Interval factor coefficient
    DEFAULT_EXP_TIMEOUT_FACTOR = 55 / 34.0  # Timeout factor coefficient

    # status code for invalid user response
    USER_DISMISSED = 451
    # The user has sent too many requests in a given amount of time
    TOO_MANY_REQUESTS = requests.codes.TOO_MANY_REQUESTS
    # tuple of HTTP response codes to retry requests for
    RETRYABLE_CODES = (
        TOO_MANY_REQUESTS,
        requests.codes.REQUEST_TIMEOUT,
        requests.codes.BAD_GATEWAY,
        requests.codes.SERVICE_UNAVAILABLE,
        requests.codes.GATEWAY_TIMEOUT,
    )

    # The object will be returned by `__call__` operator in case of server respond HTTP 205 Reset Content
    RESET = type(str("RESET"), (object,), {})
    # The object will be returned by `__call__` operator in case of server respond HTTP 204 No Content
    NO_CONTENT = type(str("NO_CONTENT"), (object,), {})

    # Global authentication object case, used for external authentication information providers.
    _external_auth = None
    # Globally defined Sandbox component name to identify internal requests.
    _default_component = None

    class Request(patterns.Abstract):
        """
        A structure which represents HTTP request. On every REST call, it is fed into a request_callback() function.

        - `id`: a request identifier, X-Request-Id header
        - `method`: HTTP method name, like "HEAD" or "GET"
        - `path`: HTTP path relative to Sandbox root, like /api/v1.0/task/1234567. No query string
        - `duration`: request duration in milliseconds (floating point number)
        - `params`: a dictionary with query parameters ("params") and request payload ("data")
        - `response`: `requests.Response`, which contains various information like response code and such
        - `attempt`: number of current retry attempt (begins by 0)
        - `attempt_duration`: attempt duration in seconds
        """

        __slots__ = (
            "id",
            "method",
            "path",
            "duration",
            "params",
            "response",
            "attempt",
            "attempt_duration"
        )
        __defs__ = (None,) * 8

    _request_callback = None

    @property
    def request_callback(self):
        return type(self)._request_callback

    @request_callback.setter
    def request_callback(self, value):
        assert type(self)._request_callback is None, "Request callback may only be set once"
        type(self)._request_callback = value

    class CustomEncoder(json.JSONEncoder):
        def default(self, o):
            if hasattr(o, "__getstate__"):
                return o.__getstate__()
            return super(Client.CustomEncoder, self).default(o)

    class Auth(requests.auth.AuthBase):
        def __init__(self, auth):
            self.auth = auth
            super(Client.Auth, self).__init__()

        def __call__(self, r):
            r.headers.update({k: v for k, v in self.auth})
            return r

    class Modifiers(object):
        """ Request and/or response data representation modifiers. """
        __metaclass__ = abc.ABCMeta

        class HeadersJar(object):
            def __init__(self, default=None, custom=None):
                self.request = requests.structures.CaseInsensitiveDict(default)
                if custom:
                    self.request.update(custom)
                self.response = requests.structures.CaseInsensitiveDict()

            def update(self, other):
                self.request.update(other.request)
                self.response.update(other.response)
                return self

        def __init__(self, headers=None):
            self.headers = self.HeadersJar({"Content-Type": self.type}, headers)

        @abc.abstractproperty
        def type(self):
            """ Content type MIME name. """

        @staticmethod
        def request(data):
            """ Request data preprocessor. """
            return data

        def response(self, data):
            """ Response data getter. """
            self.headers.response = data.headers
            if data.status_code == requests.codes.NO_CONTENT:
                return Client.NO_CONTENT
            if data.status_code == requests.codes.RESET_CONTENT:
                return Client.RESET
            return data.content

    class JSON(Modifiers):
        type = "application/json"

        @staticmethod
        def request(data):
            return json.dumps(data, cls=Client.CustomEncoder)

        def response(self, data):
            self.headers.response = data.headers
            if data.status_code == requests.codes.NO_CONTENT:
                return Client.NO_CONTENT
            if data.status_code == requests.codes.RESET_CONTENT:
                return Client.RESET
            return data.json() if data.content else None

    class PLAINTEXT(Modifiers):
        type = "text/plain"

    class BINARY(Modifiers):
        type = "application/octet-stream"

    class HEADERS(Modifiers):
        """ The class is designed to act as request and response headers collector. """
        type = None

        def __setitem__(self, key, value):
            self.headers.request[key] = value

        def __getitem__(self, key):
            return self.headers.response[key]

        def __contains__(self, item):
            return item in self.headers.response

        def __repr__(self):
            return repr(self.headers.response)

    class TimeoutExceeded(Exception):
        """ A error class to be raised in case of maximum amount of wait time exceeded """

    class SessionExpired(BaseException):
        """ Raises if REST API session is expired """

    class HTTPError(requests.HTTPError):
        """ HTTP error while requesting server """
        def __new__(cls, ex):
            """ Creates a new object based on object of :class:`request.HTTPError` instance. """
            ex = copy.copy(ex)
            ex.__class__ = cls
            return ex

        def __init__(self, *args, **kwargs):
            self.args = tuple(self.args + (self.response.text,))

        def __str__(self):
            return "{}: {}".format(self.args[:-1], self.response.text.encode("utf8"))

        @property
        def status(self):
            return self.response.status_code

    def __init__(
        self, base_url=None, auth=None, logger=None, total_wait=None, ua=None, component=None, debug=False, batch=False,
        version=None,
        min_timeout=None, max_timeout=None, exp_timeout_factor=DEFAULT_EXP_TIMEOUT_FACTOR,
        min_interval=None, max_interval=None, exp_interval_factor=DEFAULT_EXP_INTERVAL_FACTOR,
    ):
        """
        Constructor.

        :param base_url:            Base URL of REST API server to connect.
                                    Supports space-separated servers list and bash brackets.
        :param auth:                Authorization object, an instance of :class:`common.auth.Authorization`
        :param logger:              Logger object to use. Uses root logger if not provided.
        :param total_wait:          Maximum total wait time in seconds. Restricts maximum time spend to single RPC call,
                                    including the time of call itself taken and also call re-tries,
                                    if 0 then do not retry request.
        :param ua:                  User Agent identification string to be passed with the request
        :param component:           Sandbox component name. For internal use only.
        :param debug:               Log occured exceptions' types and arguments
        :param batch:               Enable batch mode via API v2 generic batch endpoint.
        :param version:             Sandbox API version.
        :param min_timeout:         Minimal timeout for requests
        :param max_timeout:         Maximal timeout for requests
        :param exp_timeout_factor:  For timeout increasing: timeout_(i) = timeout_(i-1) * this_parameter
        :param min_interval:        Minimal interval for requests
        :param max_interval:        Maximal interval for requests
        :param exp_interval_factor: For interval increasing: interval_(i) = interval_(i-1) * this_parameter
        """

        # noinspection PyBroadException
        try:
            from .. import config
            settings = config.Registry()
            settings.root()
        except Exception:
            settings = None

        if base_url is None and settings:
            base_url = settings.client.rest_url

        self.__min_interval_custom = min_interval
        self.__max_interval_custom = max_interval
        self.__exp_interval_factor = exp_interval_factor
        self.__interval = self.__min_interval

        self.__min_timeout_custom = min_timeout
        self.__max_timeout_custom = max_timeout
        self.__exp_timeout_factor = exp_timeout_factor
        self.__timeout = self.__min_timeout

        parsed_base_url = urlparse.urlparse(base_url or self.DEFAULT_BASE_URL)
        if version is not None:
            version = str(version)
            parsed_base_url = parsed_base_url._replace(path="/api/v{}".format(version))
        self.__parsed_base_url = parsed_base_url

        self.__hosts = collections.deque(
            common_format.brace_expansion([p.strip() for p in self.__parsed_base_url.netloc.split(" ")])
        )
        random.shuffle(self.__hosts)

        if auth and not isinstance(auth, common_auth.Authentication):
            auth = common_auth.OAuth(auth)
        else:
            auth = auth or self.__class__._external_auth or common_auth.NoAuth()

        self.__root = Path(self)
        self.__session = requests.Session()
        self.__session.auth = self.Auth(auth)

        self.__batch = batch
        self._batch_futures = []

        requests_version = _requests_version()
        if requests_version:
            v = [_.isdigit() and int(_) or 0 for _ in requests_version.split(".")]
            self.__legacy_requests = len(v) < 3 or v[0] < 2 or v[1] < 10
            if self.__legacy_requests and v[0] == 2:
                patch_session_timeout(self.__session)
        else:
            self.__legacy_requests = False

        _urllib3_logging_suppress()
        self.logger = logger or logging.getLogger(__name__)
        self.logger.debug(
            "REST API client instance created in thread '%s' with %s authorization%s.",
            threading.current_thread().ident, auth, " with legacy requests module" if self.__legacy_requests else ""
        )
        self.debug = debug  # KORUM: FIXME: Is only to debug "Interrupted system call" in AgentR

        self.total_wait = total_wait

        if settings and settings.common.installation == ctm.Installation.TEST:
            self.total_wait = self.__min_timeout_custom = self.__timeout = 30

        self.ua = ua or (settings.this.id if settings else socket.getfqdn())
        self.component = component or self._default_component

    # There is legacy code that changes class constants in `Client` to change default retry intervals and/or timeouts.
    # The following four properties are only needed to preserve this behaviour.
    # TODO (SANDBOX-7339): remove these properties after we stop changing the constants at runtime.

    @property
    def task_token(self):
        if not isinstance(self.__session.auth.auth, common_auth.Session):
            return
        return self.__session.auth.auth.task_token

    @property
    def __min_interval(self):
        return self.__min_interval_custom or self.DEFAULT_INTERVAL

    @property
    def __max_interval(self):
        return self.__max_interval_custom or self.MAX_INTERVAL

    @property
    def __min_timeout(self):
        return self.__min_timeout_custom or self.DEFAULT_TIMEOUT

    @property
    def __max_timeout(self):
        return self.__max_timeout_custom or self.MAX_TIMEOUT

    def __getitem__(self, item):
        return getattr(self.__root, item)

    __getattr__ = __getitem__

    def __lshift__(self, other):
        return self.__root.__lshift__(other)

    def __rshift__(self, other):
        return self.__root.__rshift__(other)

    def copy(self, **kwargs):
        """ Creates a new full (deep) copy instance of self. """
        args = self._init_args
        args.update(kwargs)
        return self.__class__(**args)

    @property
    def _init_args(self):
        """
        Return keyword arguments that are enough to fully recreate the `Client` instance.
        :return: dict
        """
        return dict(
            base_url=urlparse.urlunparse(self.__parsed_base_url),
            auth=self.__session.auth.auth,
            logger=self.logger,
            total_wait=self.total_wait,
            debug=self.debug,
            batch=self.__batch,
        )

    def create(self, path, data, params=None, input_mode=JSON, output_mode=JSON):
        resp = self._request(
            self.__session.post, path,
            {"data": input_mode.request(data), "params": normalize_params(params)},
            input_mode.headers.request
        )
        return self.__output(resp, output_mode)

    def read(self, path, params=None, input_mode=JSON, output_mode=JSON):
        resp = self._request(self.__session.get, path, {"params": normalize_params(params)}, input_mode.headers.request)
        return self.__output(resp, output_mode)

    def update(self, path, data, params=None, input_mode=JSON, output_mode=JSON):
        resp = self._request(
            self.__session.put, path,
            {"data": input_mode.request(data), "params": normalize_params(params)},
            input_mode.headers.request
        )
        return self.__output(resp, output_mode)

    def patch(self, path, data, params=None, input_mode=JSON, output_mode=JSON):
        resp = self._request(
            self.__session.patch, path,
            {"data": input_mode.request(data), "params": normalize_params(params)},
            input_mode.headers.request
        )
        return self.__output(resp, output_mode)

    def delete(self, path, data=None, params=None, input_mode=JSON, output_mode=JSON):
        resp = self._request(
            self.__session.delete, path,
            {"data": input_mode.request(data), "params": normalize_params(params)},
            input_mode.headers.request,
        )
        return self.__output(resp, output_mode)

    def __output(self, resp, output_mode):
        if self.__batch:
            self._batch_futures.append(resp)
            return resp
        return output_mode.response(resp)

    @property
    def interval(self):
        tick = self.__interval
        self.__interval = min(int(self.__interval * self.__exp_interval_factor), self.__max_interval)
        return tick

    @property
    def timeout(self):
        tick = self.__timeout
        self.__timeout = min(int(self.__timeout * self.__exp_timeout_factor), self.__max_timeout)
        return tick

    @property
    def host(self):
        return self.__hosts[0]

    def reset(self):
        self.__interval = self.__min_interval
        self.__timeout = self.__min_timeout

    def _request(self, method, path, params=None, headers=None):
        self.reset()
        request_id = uuid.uuid4().hex

        timeout = self.timeout
        if self.total_wait:
            timeout = min(timeout, self.total_wait)

        if not self.__legacy_requests:
            timeout = (self.CONNECT_TIMEOUT, timeout)
        method_name = method.__name__.upper()
        if params:
            params_params = params.get("params") or {}
            for name, value in six.iteritems(params_params):
                if isinstance(value, dict):
                    params_params[name] = json.dumps(value)
                elif hasattr(value, "__iter__"):
                    params_params[name] = list(value)
        else:
            params = {}

        spent = 0
        started = time.time()
        last_error_code = None
        ret = None
        current_attempt_number = 0

        while spent <= self.total_wait if self.total_wait is not None else True:
            try:
                current_attempt_started = time.time()
                if headers is None:
                    headers = {}
                if request_id:
                    headers[ctm.HTTPHeader.REQUEST_ID] = request_id
                headers[ctm.HTTPHeader.TRUSTED_CLIENT] = "1"
                headers[ctm.HTTPHeader.USER_AGENT] = self.ua
                headers[ctm.HTTPHeader.TOTAL_DURATION] = str(int(spent * 1000))
                if last_error_code is not None:
                    headers[ctm.HTTPHeader.RETRY_REASON] = str(last_error_code)

                if self.component:
                    headers[ctm.HTTPHeader.COMPONENT] = self.component
                if timeout:
                    headers[ctm.HTTPHeader.REQUEST_TIMEOUT] = str(timeout[1]) if isinstance(timeout, tuple) else timeout

                full_path = "{}{}".format(self.__parsed_base_url.path, re.sub("/+", "/", path))
                parsed_url = self.__parsed_base_url._replace(netloc=self.host, path=full_path)

                if self.__batch:
                    return BatchFuture(
                        method=method_name,
                        parsed_url=parsed_url,
                        params=params.get("params", ctm.NotExists),
                        data=params.get("data", ctm.NotExists),
                        headers=headers,
                    )

                url = urlparse.urlunparse(parsed_url)
                query_string_expanded = sorted((params.get("params") or {}).items())
                self.logger.debug(
                    "REST request [%s] %s %s, timeout %s, query %r",
                    request_id, method_name, url, self.__timeout, urlparse.urlencode(query_string_expanded, doseq=True)
                )
                with _urllib3_warning_suppress("InsecurePlatformWarning", "SNIMissingWarning"):
                    ret = method(url, timeout=timeout, headers=headers, allow_redirects=True, **params)
                self.logger.debug(
                    "REST request [%s] finished at %r after %.3fs (HTTP code %d)",
                    request_id, ret.headers.get(ctm.HTTPHeader.BACKEND_NODE), time.time() - started, ret.status_code
                )
                ret.raise_for_status()
                return ret

            except EnvironmentError as ex:
                self.logger.warning(
                    "REST request [%s]: failed after %.3fs: '%s'", request_id, time.time() - started, ex
                )
                do_sleep = True
                if self.debug:
                    self.logger.warning("Handle exception on REST API call: %r%r", type(ex), ex.args)

                if isinstance(ex, requests.ConnectionError):
                    if ex.args and isinstance(ex.args[0], requests.packages.urllib3.exceptions.MaxRetryError):
                        # Do not sleep on connection timeout
                        do_sleep = False
                    else:
                        try:
                            do_sleep = not (
                                isinstance(ex.args[0].args[1], socket.error) and
                                ex.args[0].args[1].errno in (errno.ECONNREFUSED, errno.EHOSTDOWN, errno.ENETUNREACH)
                            )
                        except IndexError:
                            pass
                elif isinstance(ex, requests.HTTPError):
                    if ex.response.status_code == requests.codes.GONE:
                        raise self.SessionExpired(ex)
                    elif ex.response.status_code not in self.RETRYABLE_CODES:
                        raise self.HTTPError(ex)
                    last_error_code = ex.response.status_code

                if do_sleep:
                    tick = self.interval
                    time.sleep(max(0, min(tick, self.total_wait - spent)) if self.total_wait is not None else tick)
                    timeout = self.timeout if self.__legacy_requests else (self.CONNECT_TIMEOUT, self.timeout)

            except BaseException as ex:
                if self.debug:
                    self.logger.warning("Unhandled exception on REST API call: %r%r", type(ex), ex.args)
                raise

            finally:
                current_time = time.time()
                spent = current_time - started
                spent_by_attempt = current_time - current_attempt_started
                # If batch mode is on, the request will be performed once Batch().__exit__() is called,
                # so we're not losing anything here
                if not self.__batch and self.request_callback is not None:
                    payload = self.Request(
                        id=request_id,
                        method=method_name,
                        path=parsed_url.path,
                        duration=int(spent * 1000),
                        params=params,
                        response=ret,
                        attempt_duration=spent_by_attempt,
                        attempt=current_attempt_number,
                    )
                    self.request_callback(payload)
                current_attempt_number += 1

        raise self.TimeoutExceeded(
            "Error requesting method '{}' for path '{}' - no response given after {!s}.".format(
                method_name, path, datetime.timedelta(seconds=spent)
            )
        )


class Batch(object):

    def __init__(self, client, fail_fast=False, retry=True):
        """
        The wrapper provides a context manager for sending generic batch requests via an existing `Client` instance.

        client = Client()
        with Batch(client, fail_fast=True) as batch:
            f1 = batch.task[42][:]
            f2 = batch.client["sandbox-client"][:]

        result = f1.result()
        exc = f2.exception()

        :param client: an instance of `Client`
        :param fail_fast: if True, cancel all further subrequests after the first failure
        :param retry: if True, retry subrequests failed with retriable statuses
        """
        self._client = client
        self._fail_fast = bool(fail_fast)
        self._retry = retry

    @patterns.singleton_property
    def _batch_client(self):
        kwargs = self._client._init_args
        kwargs.update(batch=True)
        return Client(**kwargs)

    def __enter__(self):
        return self._batch_client

    def __exit__(self, exc_type, exc_val, exc_tb):
        client = self._client.copy(version=2, batch=False)
        futures = self._batch_client._batch_futures

        try:
            tick = Client.DEFAULT_INTERVAL
            max_tick = Client.MAX_INTERVAL
            total_wait = float("inf") if client.total_wait is None else client.total_wait
            yielder = common_itertools.progressive_yielder(tick, max_tick, total_wait, sleep_func=time.sleep)

            while True:
                failed = []
                is_redirect = False

                batch_response = client.batch.update(
                    data=[f.request for f in futures],
                    params={"fail_fast": self._fail_fast}
                )
                assert len(batch_response) == len(futures)

                for response, future in zip(batch_response, futures):
                    future.set_result(response)

                    if 301 <= response["status"] < 400 and ctm.HTTPHeader.LOCATION in response["headers"]:
                        is_redirect = True
                        future.resolve_redirect()

                    if (self._retry and response["status"] in Client.RETRYABLE_CODES) or is_redirect:
                        failed.append(future)

                if not failed:
                    break

                client.logger.debug("Going to retry %d sub-requests", len(failed))

                if not is_redirect:
                    try:
                        next(yielder)
                    except StopIteration:
                        break

                futures = failed

        finally:
            self._batch_client._batch_futures[:] = []


class BatchFuture(object):

    class NotReadyError(Exception):
        pass

    def __init__(self, method, parsed_url, params=ctm.NotExists, data=ctm.NotExists, headers=ctm.NotExists):
        self.__method = method
        self.__parsed_url = parsed_url
        self.__params = params
        if data is not ctm.NotExists:
            data = json.loads(data)
        self.__data = data
        self.__headers = dict(headers)
        self.__result = ctm.NotExists

    def __repr__(self):
        return (
            "<{cls}: {method} {path}>"
            .format(cls=self.__class__.__name__, method=self.__method, path=self.__parsed_url.path)
        )

    @property
    def request(self):
        """
        Return a dictionary with the subrequest data for generic batch endpoint
        """
        req = {
            "method": self.__method,
            "path": self.__parsed_url.path
        }
        if self.__params is not ctm.NotExists:
            req["params"] = self.__params
        if self.__data is not ctm.NotExists:
            req["data"] = self.__data
        if self.__headers is not ctm.NotExists:
            req["headers"] = self.__headers
        return req

    def set_result(self, response):
        self.__result = response

    def result(self):
        """
        Return the subrequest response of raise an exception. The behaviour is identical to non-batch Client requests.

        :raises `Client.HTTPError` if the subrequest is unsuccessful.
        :return decoded response object.
        """
        exc = self.exception()
        if exc:
            raise exc
        return self.__result["response"]

    def raw_result(self):
        """
        Return an unprocessed result of the subrequest from the batch endpoint.

        :return: dict with `status`, `response`, and `headers` keys.
        """
        self.__raise_if_not_ready()
        return self.__result

    def __raise_if_not_ready(self):
        if self.__result is ctm.NotExists:
            raise self.NotReadyError(
                "The request has not been completed: `{method} {path}`"
                .format(method=self.__method, path=self.__parsed_url.path)
            )

    def resolve_redirect(self):
        """
        Get new URL and HTTP method from the redirection response and replace corresponding request attributes.
        """
        self.__raise_if_not_ready()

        # We let `requests` resolve redirect method for us depending on the status code.
        # For that, prepare fake request and response objects with minimally necessary attributes.
        req = requests.Request(method=self.__method)
        resp = requests.Response()
        resp.status_code = self.__result["status"]
        requests.sessions.SessionRedirectMixin().rebuild_method(req, resp)

        self.__method = req.method  # resolved redirect method
        if self.__method == "GET":
            self.__data = ctm.NotExists
        self.__parsed_url = urlparse.urlparse(self.__result["headers"][ctm.HTTPHeader.LOCATION])

    def exception(self):
        """
        Return the exception raised by the batch subrequest. `None` is returned for successful subrequests.

        :return `Client.HTTPError` exception, or `None`.
        :raises BatchFuture.NotReadyError: batch request has not been completed.
        """
        self.__raise_if_not_ready()
        if self.__result["status"] >= 400:
            resp = requests.Response()
            resp.status_code = self.__result["status"]
            resp.url = urlparse.urlunparse(self.__parsed_url)
            resp._content = json.dumps(self.__result["response"]).encode("utf-8")
            resp.reason = httplib.responses.get(self.__result["status"], "").upper() or None
            try:
                resp.raise_for_status()
            except Exception as exc:
                exc.response = resp
                return Client.HTTPError(exc)


@six.add_metaclass(patterns.ThreadLocalMeta)
class ThreadLocalCachableClient(Client):
    """
    To eliminate a problem of thread safety and also provide a single method to obtain a server proxy object,
    the class using special meta, which ensures one instance per thread.
    """


class DispatchedClientMeta(type):
    __default_client = ThreadLocalCachableClient
    __local = threading.local()

    @property
    def __clients(cls):
        try:
            clients = cls.__local.clients
        except AttributeError:
            clients = cls.__local.clients = []
        return clients

    def __call__(cls, *args, **kwargs):
        return (
            cls.__clients[-1](*args, **kwargs)
            if cls.__clients else
            cls.__default_client(*args, **kwargs)
        )

    def __enter__(cls):
        return lambda client: cls.__clients.append(client)

    def __exit__(cls, *_):
        try:
            cls.__clients.pop()
        except IndexError:
            pass


@six.add_metaclass(DispatchedClientMeta)
class DispatchedClient(object):
    """
    Used to dynamically switch actual REST API client class.
    Usage:

    .. code-block:: python

        with DispatchedClient as dispatch:
            dispatch(RealClientClass)
            # some code using DispatchedClient as REST API client
            ...

    @DynamicAttrs
    """


def normalize_params(params):
    """
    Convert query string params into a dictionary of primitive types.

    :param params: Optional[dict]:
    :return: dict
    """

    if not params:
        return {}

    params = params.copy()

    for name, value in six.iteritems(params):
        if isinstance(value, dict):
            params[name] = json.dumps(value)

    query_string = requests.PreparedRequest._encode_params(params)
    return urlparse.parse_qs(query_string, keep_blank_values=True)


def _requests_version():
    try:
        version = getattr(requests, "__version__", None)
        if not isinstance(version, (six.text_type, six.binary_type)):
            # For some weird environments where `requests.__version__` is resolved to a module, see SANDBOX-7398
            version = getattr(version, "__version__", None)
        return version
    except Exception:
        pass
    return None
