from __future__ import absolute_import, print_function

import os
import re
import sys
import platform
import functools
import subprocess
import collections

from .. import patterns
from .. import platform as common_platform

DiskSpace = collections.namedtuple("DiskSpace", ("total", "free"))


def inside_the_binary():
    return getattr(sys, "is_standalone_binary", False)


def skip_if_binary(function):
    """
    Decorator runs function if not running in binary
    """
    @functools.wraps(function)
    def wrapped(*args, **kwargs):
        if inside_the_binary():
            return None
        return function(*args, **kwargs)
    return wrapped


def _get_command_stdout_first_line(cmd):
    """
    Run a command, wait until it's finished and get the first line of its output

    :param cmd: a list of arguments
    :return: command's first line from STDOUT
    """

    p = subprocess.Popen(cmd, stdout=subprocess.PIPE)
    p.wait()
    return p.stdout.readlines()[0].strip()


class CPU(object):
    UNKNOWN_MODEL = "unknown"
    INTEL_NOISY_WORDS = ["Intel", "Xeon", "CPU", "Core", "(R)", "(TM)"]

    @staticmethod
    def raw_cpu_model():
        cpu_model = CPU.UNKNOWN_MODEL
        arch = platform.uname()[0].lower()
        if arch == "freebsd":
            cpu_model = _get_command_stdout_first_line(["/sbin/sysctl", "hw.model"])
            cpu_model = cpu_model.replace("hw.model:", "").strip()

        elif arch == "linux" or arch.startswith("cygwin"):
            with open("/proc/cpuinfo") as f:
                model_name_line = next((line for line in f.readlines() if line.startswith("model name")), None)
                if model_name_line:
                    splited = model_name_line.split(":", 1)
                    if len(splited) > 1:
                        cpu_model = splited[1]
                    cpu_model = cpu_model.strip()

        elif arch == "darwin":
            cpu_model = _get_command_stdout_first_line(["/usr/sbin/sysctl", "-n", "machdep.cpu.brand_string"])

        return cpu_model

    @staticmethod
    def parse_cpu_model(cpu_model):
        if cpu_model == "5148":
            cpu_model = "E5148"

        if "QEMU" in cpu_model:
            g = re.search("CPU version *([^ ]*)", cpu_model)
            if g:
                cpu_model = "QEMU-" + g.group(1)

        if "AMD" in cpu_model:
            g = re.search("Processor *([^ ]*)", cpu_model)
            if g:
                cpu_model = "AMD" + g.group(1)

        if "Intel" in cpu_model:
            for word in CPU.INTEL_NOISY_WORDS:
                cpu_model = cpu_model.replace(word, "")
            cpu_model = cpu_model.split("@")[0].strip()

        return cpu_model

    @patterns.classproperty
    def model(cls):
        return cls.parse_cpu_model(cls.raw_cpu_model()) or cls.UNKNOWN_MODEL


def _get_physmem():
    """ Host RAM amount in bytes """
    if platform.system().startswith("CYGWIN"):
        with open("/proc/meminfo", "r") as fh:
            for line in fh:
                if line.startswith("MemTotal:"):
                    return int(line.split()[1]) << 10
            return None
    else:
        if not inside_the_binary():
            import pkg_resources
            pkg_resources.require("psutil")
        import psutil
        return psutil.virtual_memory().total


def get_sysparams():
    """
    Retrieve information about a host

    :return: a dict with the following fields:

        :platform: host platform, a result of `platform.platform()` call
        :fqdn: the host's fully qualified domain name (for example, `sandbox-server01.search.yandex.net`)
        :arch: OS family (linux, freebsd, etc)
        :os: a dict with the following fields:

            :version: OS version
            :name: OS name

        :cpu: CPU model or `unknown`
        :ncpu: CPU cores number
        :physmem: a string containing psysical memory amount, bytes. It's usually
            a multiple of `1 << 20` but is always slightly less than the nominal amount.
            For example, if the host has a memory of size of 64Gb, `physmem` would contain
            `str(64399 * 1024 * 1024)` (expected would be `str(65536 * 1024 * 1024)`
        :fileserver: fileserver URL
        :tasks_dir: local path to Sandbox tasks code directory
    """

    from .. import config
    settings = config.Registry()

    uname_info = platform.uname()
    sys_params = {
        "platform": common_platform.platform(),
        "fqdn": settings.this.fqdn,
        "arch": settings.this.system.family,
        "os": {
            "version": uname_info[2],
            "name": settings.this.system.family,
        },
        "cpu": CPU.model,
        "ncpu": settings.this.cpu.cores,
        "physmem": _get_physmem(),
        "fileserver": "http://{node}:{port}/".format(
            node=settings.this.fqdn,
            port=settings.client.fileserver.port,
        ),
        "tasks_dir": settings.client.tasks.data_dir,
    }
    return sys_params


def get_disk_space(mount_point):
    """
    Calculate total and free disk space in bytes on a mount point (may be disk partition, external device, etc, etc).

    :param mount_point: target to read disk stats from
    :return: a tuple of total and free space
    :rtype: DiskSpace
    """

    st = os.statvfs(mount_point)
    return DiskSpace(
        st.f_blocks * st.f_frsize,
        st.f_bavail * st.f_frsize
    )
