import logging
import requests

from . import enum
from . import context


class Method(enum.Enum):
    SEND_MESSAGE = "sendMessage"
    SEND_STICKER = "sendSticker"
    GET_UPDATES = "getUpdates"
    GET_ME = "getMe"
    GET_CHAT = "getChat"
    DELETE_MESSAGE = "deleteMessage"
    PIN_CHAT_MESSAGE = "pinChatMessage"
    GET_CHAT_ADMINISTRATORS = "getChatAdministrators"
    GET_CHAT_MEMBER = "getChatMember"


class ParseMode(enum.Enum):
    MARKDOWN = "Markdown"
    HTML = "HTML"


class Sticker(enum.Enum):
    """ Different stickers' constants """
    PIKACHU = "CAADAgAD2QADMuNMAAHQYfBFIedKBwI"


class TelegramError(Exception):
    """ An exception just to wrap internal failures with """


class TelegramBot(object):
    """ Main interface to send telegram notifications from sandbox """
    BASE_URL = "https://api.telegram.org/bot"
    REQUEST_TIMEOUT = 30  # 30s should be enough, eh?

    def __init__(self, bot_token, logger=logging):
        self.__token = bot_token
        self.__logger = logger

    def __method(self, http_method, api_method, *args, **kwargs):
        """
            :return `requests.response` parsed into JSON or raise TelegramError
        """
        try:
            method = getattr(requests, http_method)
            if method is None:
                raise TelegramError("Invalid HTTP method requested")

            kwargs.update(timeout=self.REQUEST_TIMEOUT)
            # Change logging level because of security reason:
            # `requests` logs full url with token inside it on DEBUG level.
            with context.LoggerLevelChanger(logging.getLogger(requests.__name__), logging.INFO):
                self.__logger.debug("%s %s: %s, %s", http_method.upper(), api_method, args, kwargs)
                response = method("{}{}/{}".format(self.BASE_URL, self.__token, api_method), *args, **kwargs)
                self.__logger.debug("Response: %s", response.text)
            return response.json()

        except Exception as exc:
            raise TelegramError(exc)

    def _get(self, api_method, *args, **kwargs):
        return self.__method("get", api_method, *args, **kwargs)

    def _post(self, api_method, *args, **kwargs):
        return self.__method("post", api_method, *args, **kwargs)

    def send_sticker(self, chat_id, sticker, **kwargs):
        params = {
            "chat_id": chat_id,
            "sticker": sticker,
        }
        params.update(kwargs)
        return self._get(Method.SEND_STICKER, params=params)

    def send_message(self, chat_id, text, parse_mode="", **kwargs):
        params = {
            "chat_id": chat_id,
            "text": text,
            "parse_mode": parse_mode,
        }
        params.update(kwargs)
        return self._post(Method.SEND_MESSAGE, data=params)

    def delete_message(self, chat_id, message_id):
        params = {
            "chat_id": chat_id,
            "message_id": message_id,
        }
        return self._post(Method.DELETE_MESSAGE, params=params)

    def get_updates(self, offset=None, **kwargs):
        params = {"offset": offset} if offset else {}
        params.update(kwargs)
        return self._get(Method.GET_UPDATES, params=params)

    def get_me(self):
        return self._get(Method.GET_ME)

    def get_chat(self, chat_id):
        params = {"chat_id": chat_id}
        return self._get(Method.GET_CHAT, params=params)

    def get_chat_administrators(self, chat_id):
        params = {"chat_id": chat_id}
        return self._get(Method.GET_CHAT_ADMINISTRATORS, params=params)

    def pin_message(self, chat_id, message_id, disable_notification=True):
        """
            Return value example (string is wrapped for better readability):
            {
                "ok":false,
                "error_code":400,
                "description":"Bad Request: not enough rights to pin a message"
            }
        """
        params = {
            "chat_id": chat_id,
            "message_id": message_id,
            "disable_notification": disable_notification,
        }
        return self._get(Method.PIN_CHAT_MESSAGE, params=params)

    def get_chat_member(self, chat_id, user_id):
        params = {"chat_id": chat_id, "user_id": user_id}
        return self._get(Method.GET_CHAT_MEMBER, params=params)
