"""
This file's content is mostly coming from a patch, which is suggested by Sebastian Noack
in a scope of Python's bug #8800: http://bugs.python.org/issue8800
"""

from __future__ import absolute_import

import gc
import os
import sys
import time
import logging
import threading as th
import traceback

import six
import gevent

from .. import patterns

# Maximum allowed timeout, which will not raise `OverflowError` exception.
TIMEOUT_MAX = 2 << 30


def ident():
    """ Shortcut for current thread ident. """
    cth = th.current_thread()
    return None if not cth else cth.ident


def name():
    """ Shortcut for current thread name. """
    cth = th.current_thread()
    return None if not cth else cth.name


def dump_threads(logger=None):
    logger = logger or logging
    for tid, frame in six.iteritems(sys._current_frames()):
        # noinspection PyProtectedMember
        t = th._active.get(tid)
        logger.info(
            "%saemonized thread %s\nFrame:\n%s\n---\n",
            "D" if t.daemon else "Non-d",
            t,
            "".join(traceback.format_stack(frame)),
        )


def dump_greenlet_threads(logger=None):
    logger = logger or logging
    logger.info("Dumping current greenlet stacks.")
    for obj in gc.get_objects():
        if obj and isinstance(obj, gevent.greenlet.greenlet):
            logger.info("%r:\n%s", obj, "".join(map(lambda x: "\t" + x, traceback.format_stack(obj.gr_frame))))


class Condition(th._Condition if six.PY2 else th.Condition):
    def wait_for(self, predicate, timeout=None):
        """Wait until a condition evaluates to True.

        predicate should be a callable which result will be interpreted as a
        boolean value.  A timeout may be provided giving the maximum time to
        wait.

        """
        endtime = None
        waittime = timeout
        result = predicate()
        while not result:
            if waittime is not None:
                if endtime is None:
                    endtime = time.time() + waittime
                else:
                    waittime = endtime - time.time()
                    if waittime <= 0:
                        break
            self.wait(waittime)
            result = predicate()
        return result


# noinspection PyAbstractClass
class _RWLockBase(patterns.RWLock):
    """
    A Reader/Writer lock, that can be acquired either in reader or writer mode.
    In reader mode, the lock may be held by any number of threads.
    In writer mode, only one thread can hold the lock.
    A RWLock is reentrant in a limited fashion:  A thread holding the lock
    can always get another read lock.  And a thread holding an write lock
    can get another lock (read or write.)
    But in general, a thread holding a read lock cannot recursively acquire an
    write lock.
    Any acquire_read() or acquire_write must be matched with a release().
    """

    # Proxy classes that do either read or write locking
    class _ReaderLock(object):
        def __init__(self, lock):
            self.lock = lock
            super(_RWLockBase._ReaderLock, self).__init__()

        @staticmethod
        def _timeout(blocking, timeout):
            # A few sanity checks to satisfy the unittests.
            if timeout < 0 and timeout != -1:
                raise ValueError("invalid timeout")
            if timeout > TIMEOUT_MAX:
                raise OverflowError
            if blocking:
                return timeout if timeout >= 0 else None
            if timeout > 0:
                raise ValueError("can't specify a timeout when non-blocking")
            return 0

        def acquire(self, blocking=True, timeout=-1):
            return self.lock.acquire_read(self._timeout(blocking, timeout))

        def release(self):
            self.lock.release()

        def __enter__(self):
            self.acquire()

        def __exit__(self, exc, val, tb):
            self.release()

        def _is_owned(self):
            raise TypeError("a reader lock cannot be used with a Condition")

    class _WriterLock(_ReaderLock):
        def acquire(self, blocking=True, timeout=-1):
            return self.lock.acquire_write(self._timeout(blocking, timeout))

        def _is_owned(self):
            return self.lock._is_owned()

        def _release_save(self):
            return self.lock._release_save()

        def _acquire_restore(self, arg):
            return self.lock._acquire_restore(arg)

    @property
    def reader(self):
        """ Return a proxy object that acquires and releases the lock in read mode. """
        return self._ReaderLock(self)

    @property
    def writer(self):
        """ Return a proxy object that acquires and releases the lock in write mode. """
        return self._WriterLock(self)


class RWLock(_RWLockBase):
    """ FIFO reader-writer recursive lock. """

    def __init__(self):
        self.cond = Condition()
        self.state = 0    # positive is shared count, negative exclusive count
        self.owning = []  # threads will be few, so a list is not inefficient
        super(RWLock, self).__init__()

    def acquire_read(self, timeout=None):
        """ Acquire the lock in shared mode. """
        with self.cond:
            return self.cond.wait_for(self._acquire_read, timeout)

    def acquire_write(self, timeout=None):
        """ Acquire the lock in exclusive mode. """
        with self.cond:
            return self.cond.wait_for(self._acquire_write, timeout)

    def release(self):
        """ Release the lock. """
        with self.cond:
            me = ident()
            try:
                self.owning.remove(me)
            except ValueError:
                raise RuntimeError("cannot release an un-acquired lock")
            if self.state > 0:
                self.state -= 1
            else:
                self.state += 1
            if self.state == 0:
                self.cond.notify_all()

    def _acquire_write(self):
        # we can only take the write lock if no one is there, or we already hold the lock
        me = ident()
        if self.state == 0 or (self.state < 0 and me in self.owning):
            self.state -= 1
            self.owning.append(me)
            return True
        if self.state > 0 and me in self.owning:
            raise RuntimeError("cannot upgrade RWLock from read to write")
        return False

    def _acquire_read(self):
        if self.state < 0:
            # lock is in write mode.  See if it is ours and we can recurse
            return self._acquire_write()
        self.state += 1
        self.owning.append(ident())
        return True

    # Interface for condition variable.  Must hold an exclusive lock since the
    # condition variable's state may be protected by the lock
    def _is_owned(self):
        return self.state < 0 and ident() in self.owning

    def _release_save(self):
        # In a exclusively locked state, get the recursion level and free the lock
        with self.cond:
            if ident() not in self.owning:
                raise RuntimeError("cannot release an un-acquired lock")
            r = self.owning
            self.owning = []
            self.state = 0
            self.cond.notify_all()
            return r

    def _acquire_restore(self, x):
        # Reclaim the exclusive lock at the old recursion level
        self.acquire_write()
        with self.cond:
            self.owning = x
            self.state = -len(x)


class FairRWLock(RWLock):
    """ A specialization giving writer priority. """

    def __init__(self):
        super(FairRWLock, self).__init__()
        self.waiting = 0

    def acquire_write(self, timeout=None):
        """ Acquire the lock in exclusive mode. """
        with self.cond:
            self.waiting += 1
            try:
                return self.cond.wait_for(self._acquire_write, timeout)
            finally:
                self.waiting -= 1

    def _acquire_read(self):
        if self.state < 0:
            # lock is in write mode.  See if it is ours and we can recurse.
            return self._acquire_write()

        # Implement "exclusive bias" giving exclusive lock priority.
        me = ident()
        if not self.waiting:
            ok = True  # no exclusive acquires waiting.
        else:
            # Recursion must have the highest priority
            ok = me in self.owning

        if ok:
            self.state += 1
            self.owning.append(me)
        return ok


class FLockMeta(type):
    def __new__(mcs, name, bases, namespace):
        if not namespace.pop("__platform_implementation__", None) and bases != (object,):
            if sys.platform == "win32":
                bases = (FLockWindows,)
            else:
                bases = (FLockPosix,)
        return super(FLockMeta, mcs).__new__(mcs, name, bases, namespace)


@six.add_metaclass(FLockMeta)
class _FLock(object):
    def __init__(self, fname):
        super(_FLock, self).__init__()
        self._fname = fname

    def acquire(self):
        return NotImplemented

    def release(self):
        return NotImplemented

    def __enter__(self):
        return self.acquire()

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.release()


class FLockWindows(_FLock):
    __platform_implementation__ = True

    def __init__(self, fname):
        super(FLockWindows, self).__init__(fname)
        self.__fd = None
        self.__msvcrt = __import__("msvcrt")
        self.__LOCKED_BYTES = 1
        self.__lock_fname = fname + '.lock_'

    def acquire(self, blocking=True):
        """
        Acquire a lock on the file.

        :param blocking:    if True, block until the file is released; raise `IOError` in py2 and `OSerror` in py3.
        """
        self.__fd = os.open(self.__lock_fname, os.O_RDONLY | os.O_CREAT, 0o666)
        op = self.__msvcrt.LK_NBLCK
        while True:
            try:
                self.__msvcrt.locking(self.__fd, op, self.__LOCKED_BYTES)
            except (IOError, OSError):
                if blocking:
                    time.sleep(.5)
                else:
                    raise
            return self

    def release(self):
        self.__msvcrt.locking(self.__fd, self.__msvcrt.LK_UNLCK, self.__LOCKED_BYTES)
        try:
            os.close(self.__fd)
            os.remove(self.__lock_fname)
        except Exception:
            pass
        return self


class FLockPosix(_FLock):
    __platform_implementation__ = True

    def __init__(self, fname):
        super(FLockPosix, self).__init__(fname)
        self.__fd = None
        self.__fcntl = __import__("fcntl")

    def acquire(self, blocking=True):
        """
        Acquire a lock on the file.

        :param blocking:    if True, block until the file is released; raise `IOError` otherwise.
        """
        self.__fd = os.open(self._fname, os.O_RDONLY | os.O_CREAT, 0o666)
        op = self.__fcntl.LOCK_EX if blocking else self.__fcntl.LOCK_EX | self.__fcntl.LOCK_NB
        try:
            self.__fcntl.flock(self.__fd, op)
        except Exception:
            os.close(self.__fd)
            raise
        return self

    def release(self):
        self.__fcntl.flock(self.__fd, self.__fcntl.LOCK_UN)
        try:
            os.close(self.__fd)
        except Exception:
            pass
        return self


class FLock(_FLock):
    pass


class RFLock(FLock):
    """ Recursive file lock. """

    _locks = {}

    class RLock(object):

        def __init__(self):
            self.lock = th.RLock()
            self.count = 0

        def acquire(self):
            self.lock.acquire()
            self.count += 1
            return self

        def release(self):
            self.count -= 1
            self.lock.release()
            return self

    def __init__(self, fname):
        super(RFLock, self).__init__(fname)
        self.__lock = self._locks.setdefault(fname, self.RLock())

    def acquire(self, blocking=True):
        self.__lock.acquire()
        if self.__lock.count == 1:
            try:
                super(RFLock, self).acquire(blocking)
            except Exception:
                self.__lock.release()
                raise
        return self

    def release(self):
        if self.__lock.count == 1:
            super(RFLock, self).release()
        self.__lock.release()
        return self


class KamikadzeThread(th.Thread):
    def __init__(self, ttl, logger=None, name=None):
        super(KamikadzeThread, self).__init__(name=name or "Kamikadze thread")
        self.logger = logger.getChild("kamikadze") if logger else logging
        self.deadline = time.time() + ttl
        self.stopping = th.Event()

    @property
    def ttl(self):
        return self.deadline - time.time()

    @ttl.setter
    def ttl(self, value):
        self.logger.debug("Setting TTL to %ds.", value)
        self.deadline = time.time() + value

    def stop(self):
        self.stopping.set()
        return self

    def run(self):
        now = time.time()
        self.logger.info("Thread started.")
        while self.deadline > now:
            if self.stopping.wait(self.deadline - now):
                self.logger.info("Thread stopped.")
                return
            now = time.time()
        self.logger.warning("TTL expired. Killing self.")
        dump_threads(self.logger)
        os._exit(42)


def daemon(target, *args, **kwargs):
    """ Starts given target in a daemon thread. """
    t = th.Thread(target=target, args=args, kwargs=kwargs)
    t.daemon = True
    t.start()
    return t
