import logging

import six

from .. import fs
from .. import auth as cauth
from .. import rest
from .. import config
from .. import patterns
from ..types import misc as ctm

logger = logging.getLogger(__name__)


class TVM(object):
    """
    Wrapper for tvmtool API.
    """

    TIMEOUT = 5  # seconds

    class Error(Exception):
        pass

    @patterns.classproperty  # cannot be cached due to mutable header storage
    def __client(cls):
        settings = config.Registry().common.tvm
        token = fs.read_settings_value_from_file(settings.access_token)
        return rest.Client(
            "http://localhost:{}/tvm".format(settings.port),
            auth=cauth.Plain(token),
            total_wait=cls.TIMEOUT
        )

    @classmethod
    def __req(cls, endpoint, params=None, headers=None):
        params = params or {}
        try:
            path = cls.__client[endpoint]
            if headers:
                path <<= rest.Client.HEADERS(headers)
            return path.read(params)

        except rest.Client.HTTPError as exc:
            if exc.status == six.moves.http_client.FORBIDDEN:
                logger.debug(u"Validation error: %s", exc.response.text)
                raise cls.Error(exc.response.json()["error"])

            logger.error("TVMTool error", exc_info=True)
            raise cls.Error(exc.response.text)

        except rest.Client.TimeoutExceeded as exc:
            raise cls.Error("Timeout: {}".format(exc.message))

    @classmethod
    def get_service_ticket(cls, dsts):
        """
        Fetch service tickets for TVM2 requests from Sandbox to other services.

        :param dsts: list of destination aliases (as configured in tvmtool)
        :type dsts: List[str]
        :return: service tickets
        :rtype: Dict[str, str]
        """
        assert dsts, "At least one destination is required"

        resp = cls.__req("tickets", {"dsts": ",".join(dsts), "src": "sandbox"})
        if not resp:
            raise cls.Error("tvmtool returned no keys")

        tickets = {}

        for alias, data in six.iteritems(resp):
            if "error" in data:
                raise cls.Error(data["error"])
            tickets[alias] = data["ticket"]

        return tickets

    @classmethod
    def check_service_ticket(cls, ticket, dst="sandbox"):
        """
        Check service ticket to a given destination, raise `TVM.Error` if invalid.

        :param ticket: service ticket
        :type ticket: str
        :param dst: alias of the destination service configured in tvmtool config
        :type dst: str
        """
        return cls.__req("checksrv", params={"dst": dst}, headers={ctm.HTTPHeader.SERVICE_TICKET: ticket})

    @classmethod
    def check_user_ticket(cls, ticket):
        """
        Check user ticket, raise `TVM.Error` if invalid.

        :param ticket: service ticket
        :type ticket: str
        """
        return cls.__req("checkusr", headers={ctm.HTTPHeader.USER_TICKET: ticket})

    @classmethod
    def get_source_client_id(cls):
        """
        Returns source TVM client_id based on the installation.
        None for tests/local installation.
        """
        tvm_apps = config.Registry().common.tvm.apps
        if config.Registry().common.installation == ctm.Installation.PRODUCTION:
            return tvm_apps.sandbox_production
        elif config.Registry().common.installation == ctm.Installation.PRE_PRODUCTION:
            return tvm_apps.sandbox_testing
        return None


class TVMClient(TVM):
    @classmethod
    def get_service_tickets(cls, client_id):
        return cls.get_service_ticket([client_id])
