from __future__ import absolute_import, unicode_literals

import ast
import inspect
import collections
import operator as op
import itertools as it

import six

from .. import enum

CUSTOM_TAG_PREFIX = "CUSTOM_"
USER_TAG_PREFIX = "USER_"


class ServiceTokens(enum.Enum):
    """ Tokens for service calls """
    SERVICE_TOKEN = "0" * 32
    TASKBOX_TOKEN = "0" * 31 + "1"


class RejectionReason(enum.Enum):
    SESSION_EXPIRED = None
    HOST_REJECTED = None
    HOST_REJECTED_NO_REASON = None
    RESTART_LIMIT_EXCEEDED = None
    ASSIGNED_TIMEOUT = None


class Command(enum.Enum):
    """ Client commands """

    CLEAR = None
    DELETE = None
    EXECUTE = None
    EXECUTE_PRIVILEGED = None
    EXECUTE_PRIVILEGED_CONTAINER = None
    IDLE = None
    RELEASE = None
    STOP = None
    SUSPEND = None
    TERMINATE = None


class ReloadCommand(enum.Enum):
    """ Client reload commands in order of importance (processing by client) """
    enum.Enum.preserve_order()
    enum.Enum.lower_case()

    SHUTDOWN = None
    RESET = None
    REBOOT = None
    POWEROFF = None
    RESTART = "reload"
    CLEANUP = None


class RemovableResources(enum.Enum):
    """ A kind of client's cache removable resources """
    enum.Enum.preserve_order()
    enum.Enum.lower_case()

    DELETED = None
    REPLICATED = None
    EXTRA = None


class ServiceActivity(enum.Enum):
    enum.Enum.lower_case()

    NONE = None
    CLEANUP = None
    RELOAD = None
    MAINTAIN = None


class ClientAvailability(enum.Enum):
    enum.Enum.lower_case()

    ALIVE = None
    DEAD = None
    UNKNOWN = None


class TagMeta(enum.EnumMeta):

    def __new__(mcs, name, bases, namespace):
        cls = enum.EnumMeta.__new__(mcs, name, bases, namespace)

        group_defaults = set()
        positive_defaults = set()
        negative_defaults = set()
        for group in cls.Group:
            if not group.primary:
                continue
            group_positive_defaults = set()
            group_negative_defaults = set()
            for item_name in group:
                item = getattr(cls, item_name)
                if item.group_default is True:
                    group_positive_defaults.add(item_name)
                elif item.group_default is False:
                    group_negative_defaults.add(item_name)
                if item.default is True:
                    positive_defaults.add(item_name)
                elif item.default is False:
                    negative_defaults.add(item_name)
            subgroups = cls.Group.subgroups.get(str(group), frozenset())
            for subgroup_name in subgroups:
                subgroup = getattr(cls.Group, subgroup_name)
                if subgroup.default is True:
                    group_positive_defaults.add(subgroup_name)
                elif subgroup.default is False:
                    group_negative_defaults.add(subgroup_name)
            if group_positive_defaults or group_negative_defaults:
                frozenset.__setattr__(group, "subgroups", subgroups)
                frozenset.__setattr__(group, "positive_defaults", frozenset(group_positive_defaults))
                frozenset.__setattr__(group, "negative_defaults", frozenset(group_negative_defaults))
                group_defaults.add(group)
        type.__setattr__(cls.Group, "defaults", frozenset(group_defaults))
        type.__setattr__(cls, "positive_defaults", frozenset(positive_defaults))
        type.__setattr__(cls, "negative_defaults", frozenset(negative_defaults))

        return cls

    def __getattr__(cls, name):
        if (
            not name.startswith(CUSTOM_TAG_PREFIX) and name != CUSTOM_TAG_PREFIX and
            not name.startswith(USER_TAG_PREFIX) and name != USER_TAG_PREFIX or not name.isupper()
        ):
            raise AttributeError("Tag with name {!r} is not defined".format(name))
        return type(name, (cls.Item,), {})


@six.add_metaclass(TagMeta)
class Tag(enum.Enum):
    """
    Client tags

    @DynamicAttrs
    """

    enum.Enum.preserve_order()

    class QueryMeta(type):
        def __call__(cls, dnf):
            if isinstance(dnf, six.string_types):
                # noinspection PyUnresolvedReferences
                return cls.cast(dnf)
            return type.__call__(cls, dnf)

    @six.add_metaclass(QueryMeta)
    class Query(object):
        NOT = "~"
        ALLOWED_NODES = frozenset((
            ast.Expression, ast.BinOp, ast.UnaryOp, ast.Name, ast.Load, ast.BitOr, ast.BitAnd, ast.Invert
        ))

        __predicates_cache = {}

        class TagDictProxy(object):
            __slots__ = ()

            def __getitem__(self, item):
                try:
                    return getattr(Tag, item, None) or getattr(Tag.Group, item)
                except AttributeError:
                    raise KeyError

        def __init__(self, dnf):
            self.__dnf = tuple(sorted(set(six.moves.map(lambda _: tuple(sorted(set(_))), dnf))))
            self.__repr = str(dnf)
            if isinstance(dnf, (Query, Tag.Item, Tag.Group.Item)):
                self.__prio = dnf.prio
            elif len(self.__dnf) > 1:
                self.__prio = -2
            elif len(self.__dnf) == 1 and len(self.__dnf[0]) > 1:
                self.__prio = -1
            else:
                self.__prio = 0

        def __repr__(self):
            return self.__repr

        def __eq__(self, other):
            return self.__dnf == other.__dnf

        def __ne__(self, other):
            return self.__dnf != other.__dnf

        def __or__(self, item):
            item_query = item.query
            query = type(self)(set(self) | set(item_query))
            query.__prio = -2
            if query.__dnf == self.__dnf:
                query.__repr = self.__repr
            elif query.__dnf == item_query.__dnf:
                query.__repr = item_query.__repr
            else:
                query.__repr = " | ".join((self.__repr, item_query.__repr))
            return query

        def __and__(self, item):
            item_query = item.query
            query = type(self)(six.moves.map(it.chain.from_iterable, it.product(list(self), list(item_query))))
            query.__prio = -1
            if query.__dnf == self.__dnf:
                query.__repr = self.__repr
            elif query.__dnf == item_query.__dnf:
                query.__repr = item_query.__repr
            else:
                query.__repr = " & ".join(
                    map(lambda _: "({})".format(_.__repr) if _.__prio < query.__prio else _.__repr, (self, item_query))
                )
            return query

        def __invert__(self):
            query = type(self)(
                it.product(*map(lambda _: six.moves.map(op.inv, _) if isinstance(_, tuple) else ~_, self))
            )
            query.__prio = 0
            query.__repr = "~{}".format("({})".format(self.__repr) if self.__prio < query.__prio else self.__repr)
            return query

        def __iter__(self):
            return iter(self.__dnf)

        def __getstate__(self):
            return repr(self)

        def __setstate__(self, state):
            self.__dnf = self.cast(state).__dnf
            self.__repr = str(state)
            self.__prio = 0 if len(self.__dnf) < 2 else -2

        def __nonzero__(self):
            return bool(self.__dnf)

        __bool__ = __nonzero__

        @property
        def prio(self):
            return self.__prio

        @property
        def query(self):
            return self

        def check(self):
            if not self.__dnf:
                raise ValueError("Query expression is identically false")

            singles = set(it.chain.from_iterable(six.moves.filter(lambda _: len(_) == 1, self.__dnf)))
            for item in self.__dnf:
                if set(six.moves.map(op.inv, item)) <= singles:
                    raise ValueError("Query expression is identically true")

        @classmethod
        def __decode__(cls, src, raise_errors=False):
            # noinspection PyBroadException
            try:
                # noinspection PyTypeChecker
                return eval(str(src), {}, cls.TagDictProxy()).query
            except Exception:
                if raise_errors:
                    raise ValueError("invalid client tag expression: {}".format(src))

        @classmethod
        def cast(cls, src):
            src = src.upper()
            try:
                tree = ast.parse(src, mode="eval")
            except SyntaxError:
                raise ValueError("syntax error in client_tag expression: {}".format(src))

            tags = cls.TagDictProxy()

            for node in ast.walk(tree):
                node_type = type(node)

                if node_type is ast.Name:
                    try:
                        tags.__getitem__(node.id)
                    except KeyError:
                        raise ValueError("undefined client tag: {}".format(node.id))

                elif node_type not in cls.ALLOWED_NODES:
                    raise ValueError("invalid token in client_tag expression: {}".format(node.__class__.__name__))

            return cls.__decode__(src, raise_errors=True)

        @classmethod
        def predicates(cls, tags, ignore_defaults=False):
            cached = cls.__predicates_cache.get(tags)
            if cached:
                return cached
            if tags:
                result = list(six.moves.map(
                    lambda i: six.moves.reduce(
                        lambda r, t: r[str(t).startswith(cls.NOT)].add(str(t).replace(cls.NOT, "")) or r,
                        i,
                        (set(), set())
                    ),
                    cls(tags)
                ))
            else:
                result = [(set(), set())]
            for p, n in result:
                n.add(str(Tag.Group.SERVICE))
                if ignore_defaults or any(six.moves.filter(lambda t: t in Tag.Group.CUSTOM, p)):
                    continue
                for group in Tag.Group.defaults:
                    all_group_items = set(group) | group.subgroups
                    if all_group_items & p:
                        continue
                    p.update(group.positive_defaults)
                    n.update(group.negative_defaults)
                if not p & Tag.negative_defaults:
                    n.update(Tag.negative_defaults)
                if not n & Tag.positive_defaults:
                    p.update(Tag.positive_defaults)
            cls.__predicates_cache[tags] = result
            return result

    class ItemMeta(enum.Enum.ItemMeta):
        def __call__(cls, doc=None, group_default=None, default=None):
            return type(cls)(cls.__name__, (cls,), {"__doc__": doc, "group_default": group_default, "default": default})

        @property
        def query(cls):
            return Tag.Query(cls)

        @property
        def prio(self):
            return 0

        def __iter__(cls):
            yield cls

        def __or__(cls, item):
            return cls.query | item.query

        def __and__(cls, item):
            return cls.query & item.query

        def __invert__(cls):
            value = str(cls)
            return type(
                str(value[len(Tag.Query.NOT):] if value.startswith(Tag.Query.NOT) else Tag.Query.NOT + value),
                (cls,),
                {}
            )

    @six.add_metaclass(ItemMeta)
    class Item(enum.Enum.Item):
        default = None

        # noinspection PyUnusedLocal
        def __init__(self, doc=None, group_default=None, default=None):
            """ for autocompletion in IDE """

    class Group(enum.GroupEnum):
        """ Groups of tags """
        defaults = frozenset()
        subgroups = {}

        class Subgroups(object):
            def __init__(self, super_group):
                parent_locals = inspect.currentframe().f_back.f_locals
                self.__groups = {
                    value: name
                    for name, value in six.iteritems(parent_locals)
                    if name.isupper()
                }
                self.__super_group_name = self.__groups[super_group]
                self.__subgroups = parent_locals["subgroups"]

            def __enter__(self):
                self.__items = set(inspect.currentframe().f_back.f_locals)

            def __exit__(self, *_):
                items = filter(
                    lambda _: _.isupper(),
                    set(inspect.currentframe().f_back.f_locals) - self.__items
                )
                self.__subgroups[self.__super_group_name] = frozenset(items)

        class ItemMeta(enum.GroupEnum.ItemMeta):
            def __call__(cls, *args, **kws):
                default = kws.pop("default", None)
                if default is not None:
                    cls = type(cls)(cls.__name__, cls.__mro__[1:], {"default": default})
                return enum.GroupEnum.ItemMeta.__call__(cls, *args, **kws)

        @six.add_metaclass(ItemMeta)
        class Item(enum.GroupEnum.Item):

            positive_default = frozenset()
            negative_default = frozenset()
            subgroups = frozenset()
            default = None

            @property
            def query(self):
                return Tag.Query(type(str(self), (Tag.Item,), {}))

            @property
            def prio(self):
                return 0

            def __or__(self, item):
                return self.query | item.query

            def __and__(self, item):
                return self.query & item.query

            def __invert__(self):
                return ~self.query

            # noinspection PyUnusedLocal,PyMissingConstructor
            def __init__(self, doc=None, default=None):
                """ for autocompletion in IDE """

        class CustomItem(Item):
            def __contains__(self, item):
                return str(item).startswith(CUSTOM_TAG_PREFIX)

            def __len__(self):
                return 42

        class UserItem(Item):
            def __contains__(self, item):
                return str(item).startswith(USER_TAG_PREFIX)

            def __len__(self):
                return 42

        # primary groups
        PURPOSE = Item("Purpose tags")
        VIRTUAL = Item("Virtualization types")
        DENSITY = Item("Density of the executing tasks")
        OS = Item("OS")
        CPU = Item("Processors")
        DISK = Item("Disk types")
        NET = Item("IP families")
        SERVICE = Item("Service tags")
        FEATURES = Item("Service features")
        VCS = Item("Tags related to version control system (Arcadia, mostly)")
        DC = Item("Data center tags")

        # secondary groups
        with Subgroups(OS):
            LINUX = Item("Linux versions")
            OSX = Item("OS X versions", default=False)

        with Subgroups(CPU):
            INTEL = Item("Intel processor")
            AMD = Item("AMD processor")

        with Subgroups(DENSITY):
            MULTISLOTS = Item("Multislot tags", default=False)

        CUSTOM = CustomItem("Custom tags")
        USER = UserItem("User tags")

    @classmethod
    def filter(cls, tags, include_groups=True):
        tags = set(six.moves.filter(
            lambda t: t in cls or str(t).startswith(CUSTOM_TAG_PREFIX) or str(t).startswith(USER_TAG_PREFIX), tags
        ))
        # noinspection PyTypeChecker
        return set(
            it.chain(
                tags,
                six.moves.map(str, six.moves.filter(lambda g: any(six.moves.map(lambda t: t in g, tags)), cls.Group))
            )
            if include_groups else
            tags
        )

    with Group.PURPOSE:
        # Please don't use these tags if you are not sure that you know what you are doing.
        GENERIC = Item("generic sandbox clients - default tag", group_default=True)
        SERVER = Item("sandbox-server* hosts")
        STORAGE = Item("sandbox-storage* hosts")
        POSTEXECUTE = Item("Hosts where transient statuses (STOPPING, RELEASING) are executing")
        RTY = Item("sandbox-rty* hosts")
        BROWSER = Item("hosts for Browser testing")
        YABS = Item("hosts for YABS testing")
        UKROP = Item("sandbox-ukrop* hosts for release brunches build")
        SKYNET = Item("hosts for skynet build")
        PORTO = Item("hosts with Porto")
        WS = Item("wsXX-* hosts for Report testing")
        OXYGEN = Item("sandbox-oxygen* hosts")
        COCAINE = Item("hosts for needs of Cocaine")
        ANTISPAM = Item("sandbox-antispam* hosts")
        MARKET = Item("Hosts for market")
        ACCEPTANCE = Item("hosts performing the acceptance of the Sandbox")
        OTT = Item("Media services dedicated hosts for fast video converting tasks")
        VOID = Item("Special tag to define empty set of hosts")
        SDC = Item("Dedicated hosts for self-driving cars project (SANDBOX-6698)")
        VERTICALS = Item("Dedicated hosts for Verticals project (SANDBOX-7092)")
        MOBILE_MONOREPO = Item("Dedicated Mac OS hosts for mobile monorepo (SANDBOX-8956)")

    with Group.VIRTUAL:
        LXC = Item("hosts where tasks are executing in LXC-container (multios clients, privileged tasks, etc...)")
        PORTOD = Item("hosts where tasks are executing in Porto containers", group_default=False)
        OPENSTACK = Item("hosts in openstack")

    with Group.DENSITY:
        MULTISLOT = Item("Multi slot host (1 CPU, 8 GiB)", group_default=False)
        CORES1 = Item("Multi slot host (1 CPU, 8 GiB)")
        CORES4 = Item("Fat multi slot host (4 CPU, 16 GiB)")
        CORES8 = Item("Fat multi slot host (8 CPU, 32 GiB)")
        CORES12 = Item("CPU with 12 cores")
        CORES16 = Item("CPU with 16 cores")
        CORES24 = Item("CPU with 24 cores")
        CORES32 = Item("CPU with 32 cores")
        CORES56 = Item("CPU with 56 cores")
        CORES64 = Item("CPU with 64 cores")
        CORES80 = Item("CPU with 80 cores")

    with Group.OS:
        LINUX_LUCID = Item("Linux Ubuntu 10.04")
        LINUX_PRECISE = Item("Linux Ubuntu 12.04")
        LINUX_TRUSTY = Item("Linux Ubuntu 14.04")
        LINUX_XENIAL = Item("Linux Ubuntu 16.04")
        LINUX_BIONIC = Item("Linux Ubuntu 18.04")
        LINUX_FOCAL = Item("Linux Ubuntu 20.04")
        OSX_MAVERICKS = Item("OS X 10.9")
        OSX_YOSEMITE = Item("OS X 10.10")
        OSX_EL_CAPITAN = Item("OS X 10.11")
        OSX_SIERRA = Item("OS X 10.12")
        OSX_HIGH_SIERRA = Item("OS X 10.13")
        OSX_MOJAVE = Item("OS X 10.14")
        OSX_CATALINA = Item("OS X 10.15")
        OSX_BIG_SUR = Item("OS X 10.16")
        OSX_MONTEREY = Item("OS X 12")
        CYGWIN = Item("Windows/Cygwin platform (deprecated)")
        FREEBSD = Item("FreeBSD (deprecated)")
        WINDOWS = Item("Windows host with linux subsystem", group_default=False)

    with Group.CPU:
        INTEL_E5_2683 = Item()
        INTEL_E5_2683V4 = Item()

        INTEL_E5_2650 = Item()
        INTEL_E5_2650V2 = Item()

        INTEL_E5_2660 = Item()
        INTEL_E5_2660V1 = Item()
        INTEL_E5_2660V4 = Item()

        INTEL_E5_2667 = Item()
        INTEL_E5_2667V2 = Item()
        INTEL_E5_2667V4 = Item()

        INTEL_X5675 = Item()
        INTEL_GOLD_6230 = Item()
        INTEL_GOLD_6230R = Item()
        INTEL_GOLD_6338 = Item()
        INTEL_E5645 = Item()
        INTEL_E312XX = Item()

        INTEL_3720QM = Item()
        INTEL_4278U = Item()
        INTEL_4578U = Item()
        INTEL_8700B = Item()

        AMD6176 = Item()
        ARM = Item()
        KVM = Item()
        M1 = Item(group_default=False)

    with Group.DISK:
        HDD = Item("Client has HDD")
        SSD = Item("Client has SSD")

    with Group.NET:
        IPV4 = Item("Client has IPv4-address")
        IPV6 = Item("Client has IPv6-address")

    with Group.FEATURES:
        NEW_LAYOUT = Item("New resources location layout")
        SQUASHFS_TASKS_IMAGE = Item("SANDBOX-5068: Experimental feature - use SquashFS tasks image")
        DYNAMIC_SLOTS = Item("Host with dynamic slots")

    with Group.VCS:
        ARCADIA_HG = Item("Has actual Arcadia Hg cache")

    with Group.SERVICE:
        # Don't even think about using these tags
        NEW = Item("New client")
        MAINTENANCE = Item("Client is on maintenance")

    with Group.DC:
        IVA = Item()
        SAS = Item()
        MYT = Item()
        VLA = Item()
        MAN = Item()
        UNK = Item()


Tag.Group.LINUX = [getattr(Tag, _) for _ in Tag.Group.OS if _.startswith("LINUX")]
Tag.Group.OSX = [getattr(Tag, _) for _ in Tag.Group.OS if _.startswith("OSX")]
Tag.Group.INTEL = [getattr(Tag, _) for _ in Tag.Group.CPU if _.startswith("INTEL")]
Tag.Group.AMD = [getattr(Tag, _) for _ in Tag.Group.CPU if _.startswith("AMD")]
Tag.Group.MULTISLOTS = (Tag.CORES1, Tag.CORES4, Tag.CORES8)

Query = Tag.Query


class DiskStatus(enum.Enum):
    """ Disk space status """
    enum.Enum.lower_case()

    OK = None
    WARNING = None
    CRITICAL = None


#: Container meta-information:
#:     platform alias, Python executable path, rootfs path on the host system, container name,
#:     template ID, template instance number, container number on the host system, IP address
# noinspection PyAbstractClassRequirements
Container = collections.namedtuple("Container", (
    "platform", "executable", "rootfs", "name",
    "template", "instance", "no", "ip", "container_type",
    "resolvconf", "properties",
))


class ContainerPlatforms(enum.Enum):
    LUCID = "linux_ubuntu_10.04_lucid"
    PRECISE = "linux_ubuntu_12.04_precise"
    TRUSTY = "linux_ubuntu_14.04_trusty"
    XENIAL = "linux_ubuntu_16.04_xenial"
    BIONIC = "linux_ubuntu_18.04_bionic"
    FOCAL = "linux_ubuntu_20.04_focal"
