from __future__ import absolute_import, unicode_literals

import platform
import collections

from .. import enum


class Network(object):
    """ Various network-related commonly used constants """

    class Family(enum.Enum):
        """
        These constants represent the address (and protocol) families to be used commonly
        by different clients and server to report and check client's networking abilities.
        """

        V4 = 4
        V6 = 6

    class Type(enum.Enum):
        """
        Network schemas for containers (related: `bin.client.LXCNetwork`, `bin.client.PortoNetwork`)
        """

        L3 = None
        NAT = None
        MACVLAN = None
        VETH = None


class OSFamily(enum.Enum):
    """ Supported operating system families. """
    enum.Enum.lower_case()

    class Group(enum.GroupEnum):
        # platforms are build by deploy_binary_tasks
        BUILDABLE = None
        # all MacOs platforms, Intel and ARM
        OSX = None

    with Group.BUILDABLE:
        LINUX = None
        WIN_NT = None
        with Group.OSX:
            OSX = None
            OSX_ARM = None

    ANY = None
    FREEBSD = None
    LINUX_ARM = None
    CYGWIN = None

    @classmethod
    def default(cls):
        return cls.LINUX

    @classmethod
    def from_system_name(cls, name=None):
        if name is None:
            name = platform.system().lower()
        if name == "linux":
            return cls.LINUX
        elif name == "freebsd":
            return cls.FREEBSD
        elif name == "darwin":
            return cls.OSX
        elif name.startswith("cygwin"):
            return cls.CYGWIN
        elif name.startswith("windows"):
            return cls.WIN_NT


class StorageDeviceType(enum.Enum):
    """ Storage device types available on clients. """
    enum.Enum.lower_case()

    HDD = None
    SSD = None


class RamDriveType(enum.Enum):
    """ Supported RAM drive types. """
    enum.Enum.lower_case()

    TMPFS = None


class FilesystemType(enum.Enum):
    """ Known filesystems. Name values correspond to the names in /proc/filesystems """
    enum.Enum.lower_case()

    SQUASHFS = None
    TMPFS = None
    OVERLAY = None
    BIND = None  # BIND is not FS type. Bind act in lower (kernel) level than FS.


class FileType(enum.Enum):
    """ File types """
    enum.Enum.lower_case()

    FILE = None
    DIRECTORY = None


class CpuManufacturer(enum.Enum):
    enum.Enum.lower_case()

    INTEL = None
    AMD = None


class CpuArchitecture(enum.Enum):
    enum.Enum.lower_case()

    I386 = None
    X86_64 = None
    IA64 = None


class Installation(enum.Enum):
    """ Installation type """

    class Group(enum.GroupEnum):
        LOCAL = None
        NONLOCAL = None

    with Group.LOCAL:
        LOCAL = None
        TEST = None

    with Group.NONLOCAL:
        PRE_PRODUCTION = None
        PRODUCTION = None


class OperationMode(enum.Enum):
    """ Operation mode """
    enum.Enum.lower_case()

    NORMAL = None
    READ_ONLY = None
    PRE_READ_ONLY = None


class DnsType(enum.Enum):
    """ Type of resolving config """
    enum.Enum.lower_case()

    DEFAULT = None  # system default nameservers
    DNS64 = None  # [d]ns64-cache as nameserver
    LOCAL = None  # use `/etc/resolv.conf.local` config if it exists


class DiskUsageType(enum.Enum):
    """ Type of task disk usage file """
    PEAK = None  # disk usage when task exceeds its disk requirements
    FINAL = None  # disk usage when task had finished


class MongoState(enum.Enum):
    """ Mongo DB process state """

    STARTUP = None
    PRIMARY = None
    SECONDARY = None
    RECOVERING = None
    STARTUP2 = None
    UNKNOWN = None
    ARBITER = None
    DOWN = None
    ROLLBACK = None
    REMOVED = None


class Upload(object):
    """ Upload state. """

    #: Filename template to cache OAuth token.
    TOKEN_CACHE_FILENAME_TMPL = "~/.sandbox/{}.oauth"
    #: Current upload protocol version
    VERSION = 2

    class Stream(enum.Enum):
        """ Type of upload stream. """

        PLAIN = 1   # Plain tarball will be used for upload
        ZIPPED = 1  # The upload tarball stream will be zipped additionally

    class State(object):
        """ Base type for state reporting type. """
        __slots__ = ()

        def __repr__(self):
            return "".join((
                "<", self.__class__.__name__, ">",
                "(", ", ".join(repr(getattr(self, _)) for _ in self.__slots__), ")"
            ))

    class Check(State):
        """ Files check stage. Calculates total files amount and total size. """
        __slots__ = ("amount", "size")

        def __init__(self):
            self.amount = 0
            self.size = 0

    class Prepare(State):
        """ Prepare stage. Provides task and resource IDs. """
        __slots__ = ("task_id", "resource_id")

        def __init__(self):
            self.task_id = None
            self.resource_id = None

    class DataTransfer(State):
        """ Data transfer stage. Provides amount of data in bytes, which has been transferred, and SHA1 of it. """
        __slots__ = ("done", "total", "checksum")

        def __init__(self):
            self.done = 0
            self.total = 0
            self.checksum = None

    class Share(State):
        """ Postprocessing stage. Provides information about data sharing. """
        __slots__ = ("task_state", "skynet_id", "md5sum", "meta")

        def __init__(self):
            self.task_state = None
            self.skynet_id = None
            self.md5sum = None
            self.meta = None

    class ShareResource(Share):
        """ Postprocessing stage. Provides information about data sharing. """
        __slots__ = ("resource_state", "skynet_id", "md5sum", "meta")

        # noinspection PyMissingConstructor
        def __init__(self):
            self.resource_state = None
            self.skynet_id = None
            self.md5sum = None
            self.meta = None


class BatchResultStatus(enum.Enum):
    SUCCESS = None
    WARNING = None
    ERROR = None


#: Special value to return in case of key not found in structure.
NotExists = type(
    str("NotExists"), (str,),
    dict(__nonzero__=lambda _: False, __bool__=lambda _: False, __eq__=lambda self, o: self is o)
)("NotExists")

#: RAM drive requirement specification structure - type (always `TMPFS`), size in megabytes,
#: path (should be set to `None` - it will be provided at task's excution time).
RamDrive = collections.namedtuple("RamDrive", ("type", "size", "path"))


class Component(enum.Enum):
    """ Sandbox component names enumeration. """
    enum.Enum.lower_case()

    TESTS = None
    AGENTR = None
    CLIENT = None
    FILESERVER = None
    EXECUTOR = None
    PROXY = None
    SERVICE = None
    SERVICEQ = None
    SERVER = None
    SERVICEAPI = None
    TASKBOX = None
    STEP = None


class HTTPHeader(enum.Enum):
    """ Service-specific HTTP header names. """
    # Request (user agent-provided) headers
    RETRY_AFTER = "Retry-After"             #: Standard header to wait before retry the request
    ACCEPT_ENCODING = "Accept-Encoding"     #: Standard header with preferred content codings
    ACCEPT_RANGES = "Accept-Ranges"         #: Standard header for partial content transfer. Used for resumed uploads
    ACCEPT = "Accept"                       #: Standard header, indicates what content type(s) acceptable in response
    RANGE = "Range"                        #: Data content range for partial content transfer. Used for resumed download
    CONTENT_LENGTH = "Content-Length"       #: Data content length
    TRANSFER_ENCODING = "Transfer-Encoding"  #: Request transfer encoding header
    RESUME_AT = "X-Resume-At"               #: Offset to resume upload stream at. Used internally for resumed uploads
    USER_AGENT = "User-Agent"               #: Typically node ID send with this field
    REQUEST_ID = "X-Request-Id"             #: API request ID to track it over server logs
    SUBREQUEST_ID = "X-Sub-Request-Id"      #: API request ID to track it over server logs
    READ_PREFERENCE = "X-Read-Preference"   #: Database read preference to be forced for the request
    LINKS = "X-Links"                       #: Force to return resource's data sources links
    NO_LINKS = "X-No-Links"                 #: Request to return no resource's data sources links
    TOTAL_DURATION = "X-Total-Duration"     #: Total request time in milliseconds, including retries (client-side)
    RETRY_REASON = "X-Retry-Reason"         #: Last retryable error code received by the client
    TOUCH_RESOURCE = "X-Touch-Resource"     #: Request to "touch" resource together with GET request
    IGNORE_STATE = "X-Ignore-State"         #: Request to ignore broken resource state on getting its data via proxy
    COMPONENT = "X-Sandbox-Component"       #: Sandbox component name. Used for internal API requests only
    FORWARDED_SCHEME = "X-Scheme"           #: Forwarded request scheme (HTTP/HTTPS). Used by service balancer
    FORWARDED_PATH = "X-Forwarded-Path"     #: Forwarded resource path. Used for proxy-to-fileserver communication
    FORWARDED_USER = "X-Forwarded-User"     #: Forwarded user name. Used for superuser-authenticated requests
    MULTIFILE = "X-Multifile"               #: Uploaded data contain a directory
    TAR = "X-Tar"                           #: Uploaded data is tar archive
    TOTAL_SIZE = "X-Total-Size"             #: Total payload size, in bytes
    MDS_REDIRECT = "X-MDS-Redirect"         #: Do redirect to MDS for single file resources if available
    #: Asks for response with object data on update request instead of HTTP 204 "No Content"
    USER_TICKET = "X-Ya-User-Ticket"        #: TVM2 user ticket.
    SERVICE_TICKET = "X-Ya-Service-Ticket"  #: TVM2 service ticket.
    WANT_UPDATED_DATA = "X-Request-Updated-Data"
    REAL_IP = "X-Real-Ip"                   #: Real IP of the request initiator (request passed through proxy)
    FORWARDED_FOR = "X-Forwarded-For"       #: Chain of IPs the request was processed by, started with initiator's IP
    REQUEST_TARGET = "X-Request-Target"     #: Contains url of request source host
    REQUEST_SOURCE = "X-Request-Source"     #: Source type (see 'sandbox.common.types.task.RequestSource')
    REQUEST_TIMEOUT = "X-Request-Timeout"   #: Timeout for request
    TRUSTED_CLIENT = "X-Trusted-Client"     #: Mark request from trusted client, for internal use only
    ADMIN_AUTHORIZATION = "X-Admin-Authorization"
    #: Used for accounting api quota to user with token in this header
    AUTHORIZATION = "Authorization"         #: Token for authorization
    PROFILER = "X-Profiler"                 #: Enable profiler (value must be from -1 to 2 to choose sort type)
    RESOURCE_META = "X-Resource-Meta"       #: Return meta information about resources
    HOST = "Host"                           #: Server host name

    # Response (server-provided) headers
    CONTENT_RANGE = "Content-Range"         #: Data content range for partial content transfer. Used for resumed upload
    CONTENT_DISPOSITION = "Content-Disposition"  #: Header with information about how to process the response payload
    CONTENT_TYPE = "Content-Type"           #: Standard content type header
    CONTENT_ENCODING = "Content-Encoding"   #: Standard content encoding header
    LOCATION = "Location"                   #: Standard redirect location header
    LAST_MODIFIED = "Last-Modified"         #: Standard last modified time header
    TASKS_REVISION = "X-Tasks-Revision"     #: Server-side loaded tasks archive revision
    TASK_ID = "X-Sandbox-Task-Id"           #: Sandbox task ID
    RESOURCE_ID = "X-Sandbox-Resource-Id"   #: Sandbox resource ID
    DATA_SOURCE = "X-Data-Source"           #: Sandbox resource data provider host ID
    INT_REQUEST_ID = "X-Backend-Request-Id"  #: Internally generated request ID. Used on proxy to track requests
    BACKEND_NODE = "X-Backend-Server"       #: Sandbox backend node ID which processed the request
    SERVICE_MODE = "X-Service-Mode"         #: Current Sandbox service mode (see `OperationMode`)
    REQ_DURATION = "X-Request-Duration"     #: Request total duration
    REQ_METRICS = "X-Request-Metrics"       #: Request execution metrics
    DB_LOCKED_BY = "X-Database-Locked-By"   #: Backend node ID which performing database upgrade at the moment
    CURRENT_USER = "X-Current-User"         #: User name which is performing current API request (response)
    RESET_SESSION = "X-Need-Reset-Session"  #: Request to user agent (browser) to reset Yandex session
    REQUEST_MEASURES = "X-Request-Measures"     #: Measures for request in legacy web server for internal use only
    API_QUOTA_CONSUMPTION = "X-Api-Consumption-Milliseconds"  #: API quota consumption in milliseconds
    API_QUOTA = "X-Api-Quota-Milliseconds"  #: API quota in milliseconds


class RequestMethod(enum.Enum):
    """ HTTP request method enumeration """

    GET = None
    PUT = None
    POST = None
    HEAD = None
    DELETE = None
    OPTIONS = None


class EmailHeader(enum.Enum):
    """ Email headers. """

    FROM = "From"
    TO = "To"
    SUBJECT = "Subject"
    REPLY_TO = "Reply-To"
    SANDBOX_HOST = "X-SANDBOX-HOST"
    SANDBOX_HEADER = "X-SANDBOX-HEADER"
    SANDBOX_NOTIFICATION_ID = "X-SANDBOX-NOTIFICATION-ID"
    SANDBOX_TASK_TYPE = "X-SANDBOX-TASK-TYPE"
    SANDBOX_TASK_OWNER = "X-SANDBOX-TASK-OWNER"
    SANDBOX_TASK_AUTHOR = "X-SANDBOX-TASK-AUTHOR"
    SANDBOX_SCHEDULER_ID = "X-SANDBOX-SCHEDULER-ID"
    SANDBOX_SCHEDULER_TYPE = "X-SANDBOX-SCHEDULER-TYPE"


class TaskboxBackQuery(enum.Enum):
    """
    Used for make back queries to the server initiating current query to TaskBox
    """
    RESOURCES = None
    CREATE_RESOURCES = None
    TASKS = None
    API_REQUEST = None


class JugglerCheckStatus(enum.Enum):
    OK = None
    WARNING = None
    CRITICAL = None


class Interface(enum.Enum):
    enum.Enum.lower_case()

    OLD = None
    NEW = None


class EnvironmentFiles(enum.Enum):
    OWNERS_STORAGE = ".owners.json"


OAUTH_TOKEN_ENV_NAME = "OAUTH_TOKEN"  #: Name of environment variable with OAuth token
