""" Task-related constants. """

from __future__ import absolute_import, division, unicode_literals

import re
import functools
import collections

import six

from .. import enum
from .. import patterns
from .. import itertools as common_itertools


TASK_LOG_FORMAT = "%(asctime)s (%(delta)10.3fs) %(levelname)-6s (%(name)s) %(message)s"


class LogName(enum.Enum):
    COMMON = "common.log"
    DEBUG = "debug.log"


class SDKType(enum.Enum):
    enum.Enum.lower_case()

    SDK1 = None
    SDK2 = None
    BINARY = None
    TASKBOX = None


@functools.total_ordering
class Priority(object):
    """ Task priority. """

    class Class(enum.Enum):
        USER = 2
        SERVICE = 1
        BACKGROUND = 0

    class Subclass(enum.Enum):
        HIGH = 2
        NORMAL = 1
        LOW = 0

    def __init__(self, cls=None, scls=None):
        if cls is None:
            cls = self.Class.BACKGROUND
        if scls is None:
            scls = self.Subclass.LOW
        if cls not in self.Class or scls not in self.Subclass:
            raise ValueError("Incorrect priority values: class {}, subclass {}".format(cls, scls))
        self.cls = cls
        self.scls = scls

    def __int__(self):
        return 10 * self.cls + self.scls

    def __eq__(self, other):
        return int(self) == int(other)

    # Required because `total_ordering` in skynet's python is broken
    def __ne__(self, other):
        return int(self) != int(other)

    def __lt__(self, other):
        return int(self) < int(other)

    def __repr__(self):
        # noinspection PyUnresolvedReferences
        return "{cls}: {scls}".format(cls=self.Class.val2str(self.cls), scls=self.Subclass.val2str(self.scls))

    def __getstate__(self):
        # noinspection PyUnresolvedReferences
        return self.Class.val2str(self.cls), self.Subclass.val2str(self.scls)

    def __setstate__(self, state):
        if isinstance(state, int):
            self.cls, self.scls = divmod(state, 10)
            if self.cls not in self.Class or self.scls not in self.Subclass:
                raise ValueError("Invalid priority integer value: {}".format(state))
        elif (
            isinstance(state, (list, tuple)) and len(state) == 2 or
            isinstance(state, dict) and len(six.viewkeys(state) & {"class", "subclass"}) == 2
        ):
            cls, scls = (state["class"], state["subclass"]) if isinstance(state, dict) else state
            if cls:
                # noinspection PyUnresolvedReferences
                self.cls = cls if cls in self.Class else self.Class[cls]
            if scls:
                # noinspection PyUnresolvedReferences
                self.scls = scls if scls in self.Subclass else self.Subclass[scls]
        elif isinstance(state, Priority):
            self.cls = state.cls
            self.scls = state.scls
        else:
            raise ValueError(
                "Can't set state from {} value. Only support int or (Priority.Class, Priority.Subclass)".format(state)
            )
        return self

    def as_dict(self, legacy=True):
        fields = ("class", "subclass") if legacy else ("class_", "subclass")
        return dict(zip(fields, self.__getstate__()))

    @classmethod
    def make(cls, state):
        return cls().__setstate__(state)

    @property
    def next(self):
        """
        Get smallest priority bigger than given 'priority'

        :return: smallest priority, which bigger than given
        :rtype: Priority
        """
        result = None
        for p_cls in sorted(self.Class, reverse=True):
            for p_scls in sorted(self.Subclass, reverse=True):
                current = self.__class__(cls=p_cls, scls=p_scls)
                if current > self:
                    result = current
                if current == self:
                    return result or self

    @property
    def prev(self):
        """
        Get biggest priority lower than given 'priority'

        :return: biggest priority, which is smaller than given
        :rtype: Priority
        """
        result = None
        for p_cls in sorted(self.Class):
            for p_scls in sorted(self.Subclass):
                current = self.__class__(cls=p_cls, scls=p_scls)
                if current < self:
                    result = current
                if current == self:
                    return result or self


class RequestSource(enum.Enum):
    """ Source of request for changing of task history. """

    WEB = "web"
    API = "api"
    RPC = "rpc"
    SYS = "sys"
    TASK = "task"


# TODO: Remove after complete transition to new statuses [SANDBOX-2654]
class TaskStatus(enum.Enum):
    """ Task statuses. """

    NOT_READY = None
    ENQUEUED = None
    ENQUEUING = None
    EXECUTING = None
    FINISHED = None
    STOPPING = None
    STOPPED = None
    FAILURE = None
    WAIT_CHILD = None
    WAIT_DEPS = None
    UNKNOWN = None
    DELETED = None


class Status(enum.Enum):
    """ Task statuses. """

    class Group(enum.GroupEnum):
        """ Groups of task statuses """

        # Primary groups

        DRAFT = None
        QUEUE = None
        EXECUTE = None
        WAIT = None
        FINISH = None
        BREAK = None

        # Secondary groups

        # All - to query tasks in all statuses including DELETED
        ALL = None
        # non-wait-able statuses from all groups.
        NONWAITABLE = None
        # stop statuses
        STOP = None
        # suspend statuses
        SUSPEND = None
        # release statuses
        RELEASE = None
        # really executing statuses from all groups.
        REALEXECUTE = None
        # statuses for successfully executed tasks.
        SUCCEED = None
        # statuses that cause of scheduler to switch to status FAILURE.
        SCHEDULER_FAILURE = None
        # statuses that forced to switch to status FAILURE if fail_on_any_error == True
        FAIL_ON_ANY_ERROR = None
        # statuses that release semaphores by default
        SEMAPHORES_RELEASE = None
        # statuses that can be assigned to client for execution
        ASSIGNABLE = None
        # statuses that can be (re)started
        RESTARTABLE = None
        # statuses that can be used to suspend task from parameter suspend_on_status
        SUSPEND_ON_STATUS = None
        # statuses in which task states on host between terminal statuses
        TRANSIENT = None

    with Group.DRAFT:
        DRAFT = None

    with Group.QUEUE:
        ENQUEUING = None
        ENQUEUED = None

    with Group.EXECUTE:
        ASSIGNED = None
        PREPARING = None
        EXECUTING = None
        SUSPENDING = None
        SUSPENDED = None
        RESUMING = None
        TEMPORARY = None
        FINISHING = None
        STOPPING = None

    with Group.WAIT:
        WAIT_RES = None
        WAIT_TASK = None
        WAIT_TIME = None
        WAIT_OUT = None
        WAIT_MUTEX = None  # virtual status, in fact ENQUEUED, only for displaying in UI of tasks waiting semaphore(s)

    with Group.FINISH:
        SUCCESS = None
        RELEASING = None
        RELEASED = None
        NOT_RELEASED = None
        FAILURE = None
        DELETED = None

    with Group.BREAK:
        EXCEPTION = None
        NO_RES = None
        TIMEOUT = None
        STOPPED = None
        EXPIRED = None

    @patterns.classproperty
    def __transitions(self):
        transitions = {
            self.DRAFT: [
                self.ENQUEUING, self.ENQUEUED, self.EXCEPTION, self.NO_RES, self.DELETED
            ] + list(self.Group.WAIT),
            self.ENQUEUING: [
                self.ENQUEUED, self.STOPPING, self.STOPPED, self.FAILURE, self.EXCEPTION, self.NO_RES,
            ] + list(self.Group.WAIT),
            self.ENQUEUED: [self.STOPPING, self.STOPPED, self.EXCEPTION, self.DELETED, self.ASSIGNED],
            self.ASSIGNED: [self.PREPARING, self.DELETED, self.STOPPING, self.ENQUEUED, self.EXCEPTION, self.TEMPORARY],
            self.PREPARING: [
                self.EXECUTING, self.FINISHING, self.STOPPING, self.TEMPORARY, self.DELETED
            ],
            self.EXECUTING: [
                self.TEMPORARY, self.FINISHING, self.STOPPING, self.DELETED, self.SUSPENDING
            ],
            self.SUSPENDING: [self.SUSPENDED, self.EXCEPTION, self.TEMPORARY],
            self.SUSPENDED: [self.RESUMING, self.TEMPORARY, self.STOPPING, self.DELETED],
            self.RESUMING: [self.EXECUTING, self.EXCEPTION, self.TEMPORARY],
            self.TEMPORARY: [self.ENQUEUING, self.STOPPING, self.DELETED],
            self.FINISHING: [
                self.SUCCESS, self.FAILURE, self.TEMPORARY, self.EXCEPTION, self.TIMEOUT, self.DELETED
            ],
            self.WAIT_RES: [self.ENQUEUING, self.ENQUEUED, self.NO_RES, self.STOPPING, self.DELETED],
            self.WAIT_TASK: [self.ENQUEUING, self.ENQUEUED, self.STOPPING, self.DELETED],
            self.WAIT_TIME: [self.ENQUEUING, self.ENQUEUED, self.STOPPING, self.DELETED],
            self.WAIT_OUT: [self.ENQUEUING, self.ENQUEUED, self.STOPPING, self.DELETED],
            self.SUCCESS: [self.RELEASING, self.DELETED],
            self.RELEASING: [self.RELEASED, self.NOT_RELEASED, self.DELETED],
            self.RELEASED: [self.RELEASING, self.DELETED],
            self.FAILURE: [self.DELETED],
            self.DELETED: [],
            self.EXCEPTION: [self.ENQUEUING, self.DELETED],
            self.NO_RES: [self.ENQUEUING, self.DELETED],
            self.TIMEOUT: [self.ENQUEUING, self.DELETED],
            self.EXPIRED: [self.ENQUEUING, self.DELETED],
            self.STOPPING: [
                self.STOPPED, self.EXCEPTION, self.TIMEOUT, self.NO_RES, self.TEMPORARY
            ] + list(self.Group.WAIT),
            self.STOPPED: [self.ENQUEUING, self.DELETED],
            self.NOT_RELEASED: [self.RELEASING, self.DELETED]
        }

        self.__transitions = transitions
        return transitions

    # TODO: Remove after complete transition to new statuses [SANDBOX-2654]
    @patterns.classproperty
    def __old_statuses(self):
        old_statuses = {
            self.DRAFT: TaskStatus.NOT_READY,
            self.ENQUEUING: TaskStatus.ENQUEUING,
            self.ENQUEUED: TaskStatus.ENQUEUED,
            self.ASSIGNED: TaskStatus.ENQUEUED,
            self.PREPARING: TaskStatus.EXECUTING,
            self.EXECUTING: TaskStatus.EXECUTING,
            self.SUSPENDING: TaskStatus.EXECUTING,
            self.SUSPENDED: TaskStatus.EXECUTING,
            self.RESUMING: TaskStatus.EXECUTING,
            self.TEMPORARY: TaskStatus.EXECUTING,
            self.FINISHING: TaskStatus.EXECUTING,
            self.WAIT_RES: TaskStatus.WAIT_DEPS,
            self.WAIT_TASK: TaskStatus.WAIT_CHILD,
            self.WAIT_TIME: TaskStatus.WAIT_CHILD,
            self.SUCCESS: TaskStatus.FINISHED,
            self.RELEASING: TaskStatus.FINISHED,
            self.RELEASED: TaskStatus.FINISHED,
            self.NOT_RELEASED: TaskStatus.FINISHED,
            self.FAILURE: TaskStatus.FAILURE,
            self.DELETED: TaskStatus.DELETED,
            self.EXCEPTION: TaskStatus.UNKNOWN,
            self.NO_RES: TaskStatus.UNKNOWN,
            self.TIMEOUT: TaskStatus.UNKNOWN,
            self.EXPIRED: TaskStatus.UNKNOWN,
            self.STOPPING: TaskStatus.STOPPING,
            self.STOPPED: TaskStatus.STOPPED
        }
        self.__old_statuses = old_statuses
        return old_statuses

    # TODO: Remove after complete transition to new statuses [SANDBOX-2654]
    @patterns.classproperty
    def __new_statuses(self):
        new_statuses = {}
        for new, old in six.iteritems(self.__old_statuses):
            new_statuses.setdefault(old, []).append(new)
        self.__new_statuses = new_statuses
        return new_statuses

    @classmethod
    def can_switch(cls, from_status, to_status):
        """
        Check if one status can be switched to another

        :param from_status: status to switch from
        :param to_status: status to switch to
        :return: True if 'from_status' can be switched to 'to_status'
        """
        return to_status in cls.__transitions.get(from_status, [])

    # TODO: Remove after complete transition to new statuses [SANDBOX-2654]
    @classmethod
    def old_status(cls, status):
        return cls.__old_statuses.get(status, status)

    # TODO: Remove after complete transition to new statuses [SANDBOX-2654]
    @classmethod
    def new_status(cls, status):
        return {
            TaskStatus.NOT_READY: cls.DRAFT,
            TaskStatus.WAIT_DEPS: cls.WAIT_RES,
            TaskStatus.WAIT_CHILD: cls.WAIT_TASK,
            TaskStatus.FINISHED: cls.SUCCESS,
            TaskStatus.UNKNOWN: cls.EXCEPTION
        }.get(status, status)

    # TODO: Remove after complete transition to new statuses [SANDBOX-2654]
    @classmethod
    def new_statuses(cls, status):
        return cls.__new_statuses.get(status, [status])

    @patterns.classproperty
    def statuses_and_groups(self):
        return frozenset(six.moves.map(
            str,
            set(common_itertools.chain(six.moves.filter(lambda _: _.primary, self.Group), self))
        ))

Status.Group.ALL = tuple(Status)
Status.Group.NONWAITABLE = (
    Status.ENQUEUING, Status.ASSIGNED, Status.PREPARING, Status.EXECUTING, Status.STOPPING, Status.FINISHING,
    Status.RELEASING, Status.SUSPENDING, Status.RESUMING
)
Status.Group.REALEXECUTE = (
    Status.PREPARING, Status.EXECUTING, Status.SUSPENDING, Status.SUSPENDED, Status.RESUMING
)
Status.Group.STOP = (Status.STOPPING, Status.STOPPED)
Status.Group.SUSPEND = (Status.SUSPENDING, Status.SUSPENDED, Status.RESUMING)
Status.Group.RELEASE = (Status.RELEASING, Status.RELEASED, Status.NOT_RELEASED)
Status.Group.SUCCEED = (
    Status.SUCCESS, Status.RELEASING, Status.RELEASED, Status.NOT_RELEASED
)
Status.Group.SCHEDULER_FAILURE = (
    Status.FAILURE, Status.EXCEPTION, Status.NO_RES, Status.TIMEOUT, Status.EXPIRED,
)
Status.Group.FAIL_ON_ANY_ERROR = (
    Status.EXCEPTION, Status.NO_RES, Status.TIMEOUT, Status.EXPIRED, Status.TEMPORARY
)
Status.Group.SEMAPHORES_RELEASE = Status.Group.BREAK + Status.Group.FINISH + Status.Group.WAIT
Status.Group.ASSIGNABLE = (Status.ENQUEUED, Status.ASSIGNED)
Status.Group.RESTARTABLE = Status.Group.DRAFT + Status.Group.WAIT + Status.Group.BREAK + {Status.SUSPENDED}
Status.Group.SUSPEND_ON_STATUS = (
    Status.WAIT_RES, Status.WAIT_OUT, Status.WAIT_TASK, Status.WAIT_TIME, Status.SUCCESS,
    Status.EXCEPTION, Status.FAILURE
)
Status.Group.TRANSIENT = (
    Status.ASSIGNED, Status.PREPARING, Status.EXECUTING, Status.FINISHING, Status.STOPPING, Status.RELEASING
)

# Checking for intersection of names of statuses and groups
for group in iter(Status.Group):
    assert str(group) not in Status or list(group) == [str(group)], "group {!r} has wrong name".format(group)


class ReleaseStatus(enum.Enum):
    """ Release statuses. """
    enum.Enum.preserve_order()
    enum.Enum.lower_case()

    STABLE = None
    PRESTABLE = None
    TESTING = None
    UNSTABLE = None
    CANCELLED = None


class FieldValidationStatus(enum.Enum):
    """ Field validation status """

    SUCCESS = None
    WARNING = None
    ERROR = None


class ImageType(enum.Enum):
    """ Type of executor that performs the task """
    enum.Enum.lower_case()

    class Group(enum.GroupEnum):
        """ Groups of tasks images """
        #: Regular tasks image is prepared by Sandbox team and cannot fail on load, for example
        REGULAR = None
        #: Custom tasks image can be built by anybody so __any__ exception will treated as task error
        CUSTOM = None

        #: Secondary groups

        #: Extractable tasks bundle -- should be extracted before use
        EXTRACTABLE = None
        #: Image tasks bundle -- should be mounted before use
        IMAGE = None

    with Group.REGULAR:
        REGULAR_ARCHIVE = None
        REGULAR_IMAGE = None

    with Group.CUSTOM:
        CUSTOM_ARCHIVE = None
        CUSTOM_IMAGE = None
        BINARY = None
        INVALID = None


ImageType.Group.EXTRACTABLE = (ImageType.REGULAR_ARCHIVE, ImageType.CUSTOM_ARCHIVE)
ImageType.Group.IMAGE = (ImageType.REGULAR_IMAGE, ImageType.CUSTOM_IMAGE)


class SessionState(enum.Enum):
    """ Task session state """

    class Group(enum.GroupEnum):
        ACTIVE = None
        NON_ACTIVE = None

    with Group.ACTIVE:
        ACTIVE = None
        SUSPENDED = None
        UPDATED = None

    with Group.NON_ACTIVE:
        ABORTED = None
        EXPIRED = None


class IntervalType(enum.Enum):
    """ Type of task time interval """
    enum.Enum.preserve_order()
    enum.Enum.lower_case()

    EXECUTE = None
    QUEUE = None
    WAIT = None

    @classmethod
    def close_order(cls, interval_type):
        """ Returns order in which open intervals are closed. """
        types = list(cls)
        pos = types.index(interval_type) + 1
        if pos == len(types):
            return types
        else:
            return types[pos:] + types[:pos]


def relpath(task_id):
    """
    Determines task's relative directory path name for the common data directory by given task identifier.

    :param task_id: task id
    :return: Array of sub-directories names.
    """
    task_id = int(task_id)
    return [str(x) for x in (task_id % 10, task_id // 10 % 10, task_id)]


class Semaphores(collections.namedtuple("Semaphores", ("acquires", "release"))):
    class Acquire(collections.namedtuple("Acquire", ("name", "weight", "capacity", "public"))):
        def __new__(cls, name="", weight=1, capacity=0, public=False):
            if not isinstance(name, (six.text_type, six.binary_type)):
                raise ValueError("'name' must be string")
            if not name:
                raise ValueError("'name' cannot be empty")
            try:
                weight = int(weight)
                if weight <= 0:
                    raise ValueError("'weight' must be integer > 0")
            except (TypeError, ValueError):
                raise ValueError("'weight' must be integer")
            try:
                capacity = int(capacity)
                if capacity < 0:
                    raise ValueError("'capacity' must be integer >= 0")
            except (TypeError, ValueError):
                raise ValueError("'capacity' must be integer")
            return super(Semaphores.Acquire, cls).__new__(cls, name, weight, capacity, bool(public))

        def to_dict(self, with_public=True):
            ret = dict(self._asdict())
            if not with_public:
                ret.pop("public")
            return ret

    def __new__(cls, acquires=(), release=Status.Group.SEMAPHORES_RELEASE):
        acquires = tuple(
            (
                cls.Acquire(**item)
                if isinstance(item, dict) else
                cls.Acquire(*common_itertools.chain(item))
            )
            for item in common_itertools.chain(acquires)
        )
        if not acquires:
            raise ValueError("'acquires' cannot be empty")
        if not release:
            raise ValueError("'release' cannot be empty")
        release_ = frozenset(six.moves.map(str, Status.Group.collapse(release)))
        if release_ - Status.statuses_and_groups:
            raise ValueError("'release' must be a task status(es)/group(s) of statuses, {!r}".format(release))
        # noinspection PyMethodParameters
        return super(Semaphores, cls).__new__(cls, acquires, tuple(release_))

    # TODO: remove with_public after tasks binaries with age < 9 will be disabled
    def to_dict(self, with_public=True):
        return {
            k: [_.to_dict(with_public=with_public) for _ in v] if k == "acquires" else v
            for k, v in six.iteritems(self._asdict())
        }


class TaskTag(object):
    """ Common interface to test tag validity """

    REGEX = re.compile(r"\w[\w/.:-]{0,254}$")

    @classmethod
    def test(cls, tag):
        if not cls.REGEX.match(tag):
            raise ValueError("'{}' is not a valid task tag".format(tag))
        return True


class ParameterType(enum.Enum):
    """ Type of task parameter """
    enum.Enum.lower_case()

    BLOCK = None
    BOOLEAN = None
    CONTAINER = None
    FLOAT = None
    INTEGER = None
    MULTISELECT = None
    SELECT = None
    STRING = None
    RADIO = None
    RESOURCE = None
    TASK = None


class AuthMethod(enum.Enum):
    """ Authentication method """

    class Group(enum.GroupEnum):
        BLACKBOX = None
        TASK = None
        EXTERNAL_SESSION = None

    NONE = None

    with Group.BLACKBOX:
        OAUTH = None
        COOKIE = None
        TVM = None

    with Group.TASK:
        TASK = None

    with Group.EXTERNAL_SESSION:
        EXTERNAL_SESSION = None
