import os
import sys
import shutil
import logging
import datetime as dt
import subprocess as sp

from .. import data
from .. import config
from .. import patterns
from .. import threading as cth
from ..types import client as ctc

from .. import errors


class VCSCacheUnavailable(errors.TaskFailure):
    pass


class CacheRecord(patterns.Abstract):

    __slots__ = ("path", "task_id", "update_time", "is_permanent", "rep_type", "url")
    __defs__ = ("", None, None, False, None, None)

    def __eq__(self, other):
        for slot in self.__slots__:
            if getattr(self, slot) != getattr(other, slot):
                return False
        return True

    def dump(self):
        """
        Importing yaml takes a long time, so it's moved from module-level import to runtime
        """
        import yaml
        return yaml.dump(
            dict(self), default_flow_style=None, width=sys.maxsize
        )  # big number to guarantee storing one record per one line

    @classmethod
    def load(cls, s):
        try:
            return CacheRecord(**data.load_yaml(s))
        except Exception:
            # Do not init new fields using old metadata format
            return CacheRecord(*s.split("\t")[:3])


class CacheableVCS(object):
    METADATA_FILENAME = ".metadata"
    METADATA_MAX_RECORDS = 500
    base_cache_dir = None
    REP_TYPE = None
    DIRS_LIMIT = None
    NON_REMOVABLE_FILES = {METADATA_FILENAME, "arc_vcs"}

    @classmethod
    def raise_if_unavailable(cls):
        """ Check if cache usage is banned on a host """

        if ctc.Tag.MULTISLOT in config.Registry().client.tags:
            raise VCSCacheUnavailable(
                "VCS cache usage is banned on hosts where it may be altered by multiple tasks at once"
            )

    @patterns.classproperty
    def cache_metadata_path(cls):
        return os.path.join(cls.base_cache_dir, cls.METADATA_FILENAME)

    @patterns.classproperty
    def cache_metadata_lock(cls):
        return cth.FLock(cls.cache_metadata_path + ".lock")

    @classmethod
    def read_cache_records(cls):
        if not os.path.isfile(cls.cache_metadata_path):
            return []

        cache = {}
        records = []

        for line in open(cls.cache_metadata_path):
            try:
                rec = CacheRecord.load(line)
                exists = cache.get(rec.path, None)
                if exists is None:
                    exists = os.path.exists(rec.path)
                    cache[rec.path] = exists
                if exists:
                    records.append(rec)
            except (TypeError, ValueError):
                logging.exception("Failed to load cache value: %s", line)

        return records

    @classmethod
    def write_cache_records(cls, records):
        # don't use tempfile, because temporary file can be created on a different filesystem
        tmp = cls.cache_metadata_path + "~"
        with open(tmp, "w") as f:
            for record in records[-cls.METADATA_MAX_RECORDS:]:
                f.write(record.dump())
        shutil.move(src=tmp, dst=cls.cache_metadata_path)

    @classmethod
    def add_cache_metadata(cls, cache_dir, task_id, is_permanent=False, url=""):
        """ Save act of local VCS copy usage

        :param cache_dir: path to local vcs copy
        :param task_id: id of task that uses cache
        :param is_permanent: True if local vcs copy shouldn't be removed, usually it is trunk or master branch
        :param url: url to repository
        """
        with cls.cache_metadata_lock:
            records = cls.read_cache_records()
            records.append(CacheRecord(
                path=cache_dir,
                task_id=str(task_id),
                update_time=str(dt.datetime.now()),
                is_permanent=is_permanent,
                rep_type=cls.REP_TYPE,
                url=url,
            ))
            cls.write_cache_records(records)

    @classmethod
    def get_cache_dirs(cls):
        # metadata can contain a lot of duplicates
        return set(
            rec.path for rec in cls.read_cache_records()
            if rec.rep_type == cls.REP_TYPE
        )


class CacheableArcadia(CacheableVCS):
    REP_TYPE = "svn"
    DIRS_LIMIT = 5

    @patterns.singleton_classproperty
    def base_cache_dir(cls):
        return config.Registry().client.vcs.dirs.base_cache


class CacheableArcadiaTestData(CacheableArcadia):
    NON_REMOVABLE_FILES = {CacheableArcadia.METADATA_FILENAME}
    base_cache_dir = ""


class CacheableGit(CacheableVCS):
    REP_TYPE = "git"
    DIRS_LIMIT = 20

    @patterns.singleton_classproperty
    def base_cache_dir(cls):
        return config.Registry().client.vcs.dirs.base_cache


class CacheableHg(CacheableVCS):
    REP_TYPE = "hg"
    DIRS_LIMIT = 1
    RESOURCE_TYPE = "ARCADIA_HG_REPOSITORY"

    @patterns.singleton_classproperty
    def base_cache_dir(cls):
        return config.Registry().client.vcs.dirs.base_cache


class CacheableArcadiaApi(CacheableVCS):
    REP_TYPE = "aapi"
    DIRS_LIMIT = 1

    @patterns.singleton_classproperty
    def base_cache_dir(cls):
        return config.Registry().client.vcs.dirs.base_cache


class CacheableArcVcs(CacheableVCS):
    REP_TYPE = "arc_vcs"
    DIRS_LIMIT = 1
    RESOURCE_TYPE = "ARCADIA_ARC_REPOSITORY"

    @patterns.singleton_classproperty
    def base_cache_dir(cls):
        return config.Registry().client.vcs.dirs.base_cache

    @classmethod
    def raise_if_unavailable(cls):
        """ Check if cache usage is banned on a host """
        pass


class VCSCache(object):
    _LOG_NAME = "vcs"
    _cacheables = (
        CacheableArcadia, CacheableArcadiaTestData, CacheableGit,
        CacheableHg, CacheableArcadiaApi, CacheableArcVcs
    )

    def __init__(self, logger=None):
        self.logger = logger or logging.getLogger(self._LOG_NAME)

    def clean(self):
        for cacheable in self._cacheables:
            try:
                self._clean(cacheable)
            except Exception:
                self.logger.exception("Unable to clean %r in %s", cacheable.REP_TYPE, cacheable.base_cache_dir)

    def _rm(self, path):
        if not path:
            return
        try:
            with open(os.devnull, "wb") as devnull:
                sp.check_call(["/bin/rm", "-rf", path], stderr=devnull)
        except (sp.CalledProcessError, OSError) as ex:
            self.logger.error("Cannot remove %r: %s", path, ex)

    def cleanup_dir(self, dir_path, non_removable_files):
        for sub_path in os.listdir(dir_path):
            if sub_path not in non_removable_files:
                path = os.path.join(dir_path, sub_path)
                self.logger.info("Removing %r", path)
                self._rm(path)

    def _get_caches_to_delete(self, cacheable, all_records):
        known_types = [_.REP_TYPE for _ in self._cacheables]
        path_updates = {}
        paths_to_delete = []
        permanent = {}
        for rec in all_records:
            if rec.rep_type == cacheable.REP_TYPE:
                permanent[rec.path] = rec.is_permanent  # check only the last permanency flag
        for rec in all_records:
            if rec.rep_type not in known_types:
                paths_to_delete.append(rec.path)
            elif rec.rep_type == cacheable.REP_TYPE and not permanent[rec.path]:
                path_updates[rec.path] = rec.update_time or ""
        if len(path_updates) > cacheable.DIRS_LIMIT:
            caches = sorted(path_updates.keys(), key=lambda path: path_updates[path])
            paths_to_delete.extend(caches[:-cacheable.DIRS_LIMIT])
        return paths_to_delete

    def _clean(self, cacheable):  # type: (CacheableVCS) -> None
        cache_dir = cacheable.base_cache_dir
        if not os.path.exists(cache_dir):
            return

        with cacheable.cache_metadata_lock:
            records = cacheable.read_cache_records()
            non_removable_files = {os.path.relpath(r.path, cache_dir) for r in records} | cacheable.NON_REMOVABLE_FILES
            self.cleanup_dir(cache_dir, non_removable_files)
            caches_to_delete = self._get_caches_to_delete(cacheable, records)

            for cache_to_delete in set(caches_to_delete):
                cache_records = filter(lambda rec: rec.path == cache_to_delete, records)
                if cache_records:
                    self.logger.info("Deleting cache %r", cache_records[0])
                self._rm(cache_to_delete)

            cacheable.write_cache_records([rec for rec in records if rec.path not in caches_to_delete])
