from __future__ import absolute_import, unicode_literals

import os
import re
import six
import logging

from .. import patterns

from sandbox.common.windows import wsl

if six.PY2:
    import subprocess32 as sp
else:
    import subprocess as sp


class ImDisk(object):
    """
    Class for work with imdisk.exe - program and driver for images mounts in windows.
    """
    IMDISK = wsl.FS.platform_path(os.path.join(wsl.FS.SYSTEM32, "imdisk.exe"))
    IMDISK_SVC = wsl.FS.platform_path(os.path.join(wsl.FS.SYSTEM32, "imdsksvc.exe"))
    CMD = wsl.FS.platform_path(os.path.join(wsl.FS.SYSTEM32, "cmd.exe"))

    @classmethod
    def _get_output(cls, cmd, save_stderr=False):
        try:
            if save_stderr:
                return sp.check_output(cmd, stderr=sp.STDOUT)
            return sp.check_output(cmd)
        except sp.CalledProcessError as ex:
            return ex.output

    @classmethod
    def _get_exit_code(cls, cmd):
        try:
            return sp.check_call(cmd)
        except sp.CalledProcessError as ex:
            return ex.returncode

    @classmethod
    def _get_exit_code_output(cls, cmd):
        try:
            return (0, sp.check_output(cmd))
        except sp.CalledProcessError as ex:
            return (ex.returncode, ex.output)

    @patterns.classproperty
    def installed(cls):
        return os.path.isfile(cls.IMDISK) and os.path.isfile(cls.IMDISK_SVC)

    @classmethod
    def unmount(cls, block_unit):
        # Imdisk have 3 umount keys: -d (detach) -D (Force detach) -R (extreme remove)
        # NOTICE: block unit id can stay in imdisk -l -n for a 5-20 mins after umount.
        umount_actions = ["-d", "-D", "-R"]
        for action in umount_actions:
            cmd = [cls.IMDISK, action, "-u", str(block_unit)]
            exit_code = cls._get_exit_code(cmd)
            if exit_code <= 1:
                return True
        logging.warning("Can't unmount ramdrive unit %s", block_unit)
        return False

    @classmethod
    def unmount_all(cls):
        for unit in cls.mounted_block_units():
            cls.unmount(unit)

    @classmethod
    def mounted_block_units(cls):
        cmd = [cls.IMDISK, "-l", "-n"]
        return cls._get_output(cmd).splitlines()

    @classmethod
    def mount(cls, size, mountpoint, force=False):
        """
        Mount imdisk drive formatted to NTFS to random free disk letter and make symlink from mountpoint to that drive
        """
        win_mountpoint = wsl.FS.win_path(mountpoint)
        if not cls.installed:
            raise RuntimeError("Can't mount ramdisk, driver not installed")

        if os.path.exists(mountpoint):
            if force:
                sp.call([cls.CMD, "/C", "rmdir", "/S", "/Q", win_mountpoint])
            else:
                raise RuntimeError("Mountpoint {} ({}) should not exist".format(mountpoint, win_mountpoint))

        mount_cmd = [
            cls.IMDISK, "-a", "-t", "vm", "-s", str(size) + "M",
            "-m", "#:", "-p", "/fs:ntfs /q /y"
        ]

        mount_exit_code, mount_cmd_output = cls._get_exit_code_output(mount_cmd)
        drive_letter = re.findall(r"Formatting disk (\w):...", mount_cmd_output)[-1]
        symlink_cmd = [cls.CMD, "/C", "mklink", "/D", win_mountpoint, "{}:".format(drive_letter)]

        check_cmd = [cls.IMDISK, "-l", "-m", "{}:".format(drive_letter)]
        cls._get_exit_code(symlink_cmd)
        if cls._get_exit_code(check_cmd) > 0:
            cls.unmount_all()
            raise RuntimeError(
                "RamDrive Mount failed. All ramdrives unmounted. Mount command: {} ret code: {} output: {}".format(
                    mount_cmd, mount_exit_code, mount_cmd_output
                )
            )
