# coding: utf-8
from __future__ import absolute_import, unicode_literals

import os
import sys
import logging

import six
import psutil

if six.PY2:
    try:
        import subprocess32 as sp
    except:
        import subprocess as sp  # Fix for local Sandbox
else:
    import subprocess as sp

from .. import patterns


# TODO: change to constant CODEPAGE = "CP437"
def _get_codepage():
    chcp_path = "C:\\Windows\\System32\\chcp.com" if sys.platform == "win32" else "/mnt/c/Windows/System32/chcp.com"
    codepage_num = sp.check_output(chcp_path).split()[-1]
    return "CP" + codepage_num


class FS(object):
    """
    Helper class for working with Windows file system from WSL
    """
    DRIVE_C = "c:"
    WINDOWS_DRIVE_MOUNTPOINT = "/mnt/c/"
    SYSTEM32 = os.path.join(WINDOWS_DRIVE_MOUNTPOINT, "Windows/System32")
    POWERSHELL = os.path.join(WINDOWS_DRIVE_MOUNTPOINT, "Windows/System32/WindowsPowerShell/v1.0/powershell.exe")
    ICALCS = os.path.join(WINDOWS_DRIVE_MOUNTPOINT, "Windows/System32/icacls.exe")

    @patterns.singleton_classproperty
    def powershell(self):
        if sys.platform == "win32":
            return self.win_path(self.POWERSHELL)
        return self.POWERSHELL

    @patterns.singleton_classproperty
    def icacls(self):
        if sys.platform == "win32":
            return self.win_path(self.ICALCS)
        return self.ICALCS

    @patterns.singleton_classproperty
    def root(self):
        """ WSL rootfs in Windows """
        return sp.check_output([
            self.powershell,
            "-Command",
            "(Get-ChildItem HKCU:\\Software\\Microsoft\\Windows\\CurrentVersion\\Lxss | "
            "ForEach-Object {Get-ItemProperty $_.PSPath}) | "
            "select BasePath | "
            "Format-Table -HideTableHeaders"
        ]).strip() + "\\rootfs"

    @classmethod
    def win_path(cls, path, rootfs=None):
        """
        Transform path from WSL to Windows

        :param path: WSL path
        :param rootfs: WSL rootfs in Windows, by default value from property `rootfs`
        :return Windows path
        """
        path = six.ensure_str(path)
        if path.startswith(cls.WINDOWS_DRIVE_MOUNTPOINT):
            return "c:\\" + path[len(cls.WINDOWS_DRIVE_MOUNTPOINT):].replace("/", "\\")
        return (rootfs or cls.root) + path.replace("/", "\\")

    @classmethod
    def wsl_path(cls, path):
        """
        Transform path from Windows to WSL

        :param path: Windows path
        :return WSL path
        """
        if path.lower().startswith(cls.DRIVE_C):
            return cls.WINDOWS_DRIVE_MOUNTPOINT + path[len(cls.DRIVE_C):].replace("\\", "/").replace("//", "/")
        elif not os.path.isabs(path):
            return path.replace("\\", "/").replace("//", "/")
        return path.replace("//", "/")

    @classmethod
    def platform_path(cls, path):
        if sys.platform != "win32":
            return cls.wsl_path(path)
        return cls.win_path(path)

    @classmethod
    def acl_grant(cls, path, user, perms, rootfs=None, reset=False):
        """
        Grant NTFS permissions

        :param path: WSL path
        :param user: Windows user
        :param perms: permissions in format used by icacls.exe
        :param rootfs: WSL rootfs in Windows for rebase
        :param reset: if True, reset ACL to defaults before setting new ones
        """
        if sys.platform != "win32":
            path = cls.win_path(path, rootfs=rootfs)
        if reset:
            sp.check_call([cls.icacls, path, "/reset", "/Q"])
        try:
            sp.check_output(
                [cls.icacls, path, "/grant:r", "{}:{}".format(user.encode(_get_codepage()), perms)],
                stderr=sp.STDOUT
            )
        except sp.CalledProcessError as ex:
            logging.error(
                "Error changing permissions. Out: %s Ret.code: %s",
                ex.output.decode(_get_codepage()),
                ex.returncode
            )
            raise

    @classmethod
    def acl_get(cls, path, rootfs=None):
        """
        Return current path grants

        :param path: WSL path
        :param rootfs: WSL rootfs in Windows for rebase
        :return: list of sets [("username": "permissions"), ...]
        """
        if sys.platform != "win32":
            path = cls.win_path(path, rootfs=rootfs)
        try:
            out = sp.check_output([cls.icacls, path], stderr=sp.STDOUT).decode(_get_codepage()).splitlines()
            out[0] = out[0].split()[-1]
            out = out[:-2]
        except sp.CalledProcessError as e:
            logging.error(
                "icacls.exe failed with code %s and output %s",
                e.returncode,
                e.output.decode(_get_codepage())
            )
            raise

        return [(x.split(":")[0].split("\\")[-1].strip() if "\\" in x else x.split(":")[0], x.split(":")[-1].strip())
                for x in out]

    @classmethod
    def acl_get_by_user(cls, path, user, rootfs=None):
        """
        Return Users acl in path if user have minimum perm in path
        :param path: WSL path
        :param user: Windows user or group name
        :param rootfs: WSL rootfs in Windows for rebase
        :return: Str
        """
        user_acls = ""
        for u, a in cls.acl_get(path, rootfs=rootfs):
            if u == user:
                user_acls += " " + a
        return user_acls

    @classmethod
    def acl_deny(cls, path, user, perms, rootfs=None):
        """
        Deny NTFS permissions

        :param path: WSL path
        :param user: Windows user
        :param perms: permissions in format used by icacls.exe
        :param rootfs: WSL rootfs in Windows for rebase
        """
        if sys.platform != "win32":
            path = cls.win_path(path, rootfs=rootfs)
        sp.check_call([cls.icacls, path, "/deny", "{}:{}".format(user.encode(_get_codepage()), perms), "/Q"])

    @classmethod
    def acl_remove(cls, path, user, rootfs=None):
        """
        Remove all NTFS permissions for user

        :param path: WSL path
        :param user: Windows user
        :param rootfs: WSL rootfs in Windows for rebase
        """
        if sys.platform != "win32":
            path = cls.win_path(path, rootfs=rootfs)
        sp.check_call([cls.icacls, path, "/remove", "{}".format(user.encode(_get_codepage())), "/Q"])

    @classmethod
    def acl_reset(cls, path, rootfs=None):
        """
        Reset all NTFS permissions to defaults

        :param path: WSL path
        :param rootfs: WSL rootfs in Windows for rebase
        """
        if sys.platform != "win32":
            path = cls.win_path(path, rootfs=rootfs)
        sp.check_call([cls.icacls, path, "/reset", "/Q"])


class User(object):
    PSGETSID = os.path.join(FS.WINDOWS_DRIVE_MOUNTPOINT, "tools", "PsGetsid.exe")
    REG = os.path.join(FS.SYSTEM32, "reg.exe")
    NET = os.path.join(FS.SYSTEM32, "net.exe")
    TAKEOWN = os.path.join(FS.SYSTEM32, "takeown.exe")
    CMD = os.path.join(FS.SYSTEM32, "cmd.exe")
    QUERY = os.path.join(FS.SYSTEM32, "query.exe")
    LOGOFF = os.path.join(FS.SYSTEM32, "logoff.exe")

    @patterns.singleton_classproperty
    def psgetsid(self):
        if sys.platform == "win32":
            return FS.win_path(self.PSGETSID)
        return self.PSGETSID

    @patterns.singleton_classproperty
    def reg(self):
        if sys.platform == "win32":
            return FS.win_path(self.REG)
        return self.REG

    @patterns.singleton_classproperty
    def net(self):
        if sys.platform == "win32":
            return FS.win_path(self.NET)
        return self.NET

    @patterns.singleton_classproperty
    def takeown(self):
        if sys.platform == "win32":
            return FS.win_path(self.TAKEOWN)
        return self.TAKEOWN

    @patterns.singleton_classproperty
    def cmd(self):
        if sys.platform == "win32":
            return FS.win_path(self.CMD)
        return self.CMD

    @patterns.singleton_classproperty
    def query(self):
        if sys.platform == "win32":
            return FS.win_path(self.QUERY)
        return self.QUERY

    @patterns.singleton_classproperty
    def logoff(self):
        if sys.platform == "win32":
            return FS.win_path(self.LOGOFF)
        return self.LOGOFF

    @classmethod
    def win_user_sid(cls, username, timeout=60):
        """
        Return string with windows SID for username
        Require admin privileges

        :param username: String with windows username
        :param timeout: Timeout for subprocess
        """
        cmd = [cls.psgetsid, "/accepteula", "-nobanner", username]
        cmd_out = str()
        try:
            cmd_out = sp.check_output(cmd, stderr=sp.STDOUT, timeout=timeout)
        except(sp.CalledProcessError, sp.TimeoutExpired):
            logging.exception("Error while execute %s", " ".join(cmd))
            pass
        if "SID" in cmd_out.decode(_get_codepage()):
            sid = cmd_out.split()[-1]
            logging.debug("Found SID for user %s %s", username, sid)
            return sid
        return False

    @classmethod
    def win_user_exists(cls, username, timeout=60):
        """
        Check if windows user exists

        :param username: String with windows username
        :param timeout: Timeout for subprocess
        :return True if exists
        """
        cmd = [cls.net, "user", username]
        try:
            return not sp.call(cmd, timeout=timeout)
        except sp.TimeoutExpired:
            logging.error("Timeout expired while execute %s", " ".join(cmd))

    @classmethod
    def win_user_home_path(cls, username, timeout=60):
        """
        Return string with windows home_directory for username
        Require admin privileges

        :param username: String with windows username
        :param timeout: Timeout for subprocess
        """
        # TODO: Use _winreg for winpy2 winreg for winpy3 and something for wslpy2/3
        SID = cls.win_user_sid(username)
        cmd_out = list()
        if not SID:
            return False
        cmd = [
            cls.reg,
            "QUERY",
            "HKLM\\SOFTWARE\\Microsoft\\Windows NT\\CurrentVersion\\ProfileList\\{}".format(SID),
            "/v",
            "ProfileImagePath"
        ]
        try:
            cmd_out = sp.check_output(cmd, stderr=sp.STDOUT, timeout=timeout).split()
        except sp.TimeoutExpired:
            logging.debug("Timeout expired while execute %s", " ".join(cmd))
        except sp.CalledProcessError:
            logging.warning("Error while executing %s", " ".join(cmd))
            return False
        for line in cmd_out:
            if os.path.isdir(line) or os.path.isdir(FS.wsl_path(line)):
                return line
        return False

    @classmethod
    def win_remove_user(cls, username, timeout=60, logger=None):
        """
        Return bool status of removing windows local user operation
        Require admin privileges

        :param username: String with windows username
        :param timeout: Timeout for subprocess
        :param logger: Logger
        """

        if logger is None:
            logger = logging

        cmd = [cls.net, "user", "/DELETE", username]
        logger.debug("Trying to remove user %s \n via %s", username, " ".join(cmd))
        try:
            sp.check_call(cmd, timeout=timeout)
            logger.debug("User %s removed", username)
            return True
        except (sp.CalledProcessError, sp.TimeoutExpired) as e:
            logger.exception(
                "Can't remove user %s. cmd: %s\nReturned: %s",
                username, " ".join(cmd), str(e.output),
            )
            return False

    @classmethod
    def win_user_sessions(cls, username, timeout=60):
        """
        Return list of user sessions id's

        :param username: String with windows username
        :param timeout: Timeout for subprocess
        """
        cmd = [cls.query, "user"]
        user_sessions = []
        if not cls.win_user_exists(username):
            return user_sessions
        logging.debug("Trying to get all user session of %s \n via %s", username, " ".join(cmd))
        cmd_out = ""
        try:
            cmd_out = sp.check_output(cmd, stderr=sp.STDOUT, timeout=timeout).decode(_get_codepage()).splitlines()
        except sp.CalledProcessError as e:
            # Sometimes query.exe return non-zero exit code, but works
            cmd_out = e.output.decode(_get_codepage()).splitlines()
        except sp.TimeoutExpired:
            logging.warning("Timeout expired on %s", " ".join(cmd))
        for line in cmd_out:
            line = line.split()
            if line[0] == username or line[0] == ">" + username:
                user_sessions.append(line[2])
        return user_sessions

    @classmethod
    def win_logout_user(cls, username, timeout=60, logger=None):
        """
        Logout windows user from system and close sessoins
        Require admin privileges

        :param username: String with windows username
        :param timeout: Timeout for subprocess
        :param logger: Logger
        """

        if logger is None:
            logger = logging

        for session_id in cls.win_user_sessions(username=username):
            cmd = [cls.logoff, session_id]
            logger.debug("Logout user %s \n via %s", username, " ".join(cmd))
            try:
                sp.call(cmd, timeout=timeout)
            except sp.TimeoutExpired:
                logger.warning("Timeout expired on %s", " ".join(cmd))

    @classmethod
    def _try_rm_dir(cls, logger, dir_path, timeout):
        for cmd in (
            [cls.cmd, "/C", "del", "/F", "/S", "/Q", dir_path],
            [cls.cmd, "/C", "rmdir", "/S", "/Q", dir_path],
        ):
            logger.debug("Trying to run: %s", " ".join(cmd))
            try:
                sp.call(cmd, timeout=timeout)
            except sp.SubprocessError as exc:
                logger.error("Failed to run: %s", exc)
                return False
        return True

    @classmethod
    def _try_take_ownership(cls, logger, user_name, dir_path, timeout):
        for cmd in (
            [cls.takeown, "/R", "/A", "/F", dir_path, "/D", "Y"],
            [FS.icacls, dir_path, "/grant", "{}:F".format(user_name), "/T", "/C"],
        ):
            logger.debug("Trying to run: %s", " ".join(cmd))
            try:
                sp.call(cmd, timeout=timeout)
            except sp.SubprocessError as exc:
                logger.error("Failed to run: %s", exc)

    @classmethod
    def win_remove_home_dir(cls, username, system_user, win_home_dir, timeout=600, logger=None):
        """
        Remove windows user home directory.
        Require admin privileges

        :param username: String with windows username
        :param system_user: String with admin user account
        :param win_home_dir: Path to home directory to delete
        :param timeout: Timeout for subprocess
        :param logger: Logger
        """

        if logger is None:
            logger = logging
        if not win_home_dir:
            logger.debug("Home dir does not exist or not belongs to user, nothing to remove")
            return False
        wsl_home_dir = FS.wsl_path(win_home_dir)
        if os.path.exists(wsl_home_dir):
            if not cls._try_rm_dir(logger, wsl_home_dir, timeout):
                cls._try_take_ownership(logger, system_user, wsl_home_dir, timeout)
                cls._try_rm_dir(logger, wsl_home_dir, timeout)
        if os.path.exists(wsl_home_dir):
            logger.error("Failed to remove %s's home dir %s", username, win_home_dir)
            return False

    @classmethod
    def win_create_user(cls, username, timeout=60, logger=None):
        """
        Create user with username and password username
        Require admin privileges

        :param username: String with windows username
        :param timeout: Timeout for subprocess
        :param logger: Logger
        """

        if logger is None:
            logger = logging

        cmd = [cls.net, "user", "/ADD", username, username]
        logger.debug("Trying to create user %s \n via %s", username, " ".join(cmd))
        try:
            sp.check_call(cmd, timeout=timeout)
            logger.debug("User %s created", username)
        except (sp.CalledProcessError, sp.TimeoutExpired) as e:
            logger.error("Can't create user %s. cmd: %s\nReturned: %s", username, " ".join(cmd), str(e.output))
            return False


class Process(object):
    PSSUSPEND = os.path.join(FS.WINDOWS_DRIVE_MOUNTPOINT, "tools/pssuspend.exe")
    TASKLIST = os.path.join(FS.SYSTEM32, "tasklist.exe")
    TASKKILL = os.path.join(FS.SYSTEM32, "taskkill.exe")

    @patterns.singleton_classproperty
    def pssuspend(self):
        if sys.platform == "win32":
            return FS.win_path(self.PSSUSPEND)
        return self.PSSUSPEND

    @patterns.singleton_classproperty
    def tasklist(self):
        if sys.platform == "win32":
            return FS.win_path(self.TASKLIST)
        return self.TASKLIST

    @patterns.singleton_classproperty
    def taskkill(self):
        if sys.platform == "win32":
            return FS.win_path(self.TASKKILL)
        return self.TASKKILL

    @classmethod
    def win_user_processes(cls, username):
        """
        List all process id's of username

        :param username: String with windows username
        """
        process_list = []
        cmd = [cls.tasklist, "/FI", "USERNAME eq {}".format(username), "/FO", "LIST"]
        for line in sp.check_output(cmd, stderr=sp.STDOUT).decode(_get_codepage()).splitlines():
            if line.startswith("PID"):
                process_list.append(line.split()[-1])
        return process_list

    @classmethod
    def win_suspend_process(cls, ProcessId):
        """
        Suspend ProcessId
        You should be admin or owner of process

        :param ProcessId: string with id of windows process
        :param timeout: Timeout for subprocess
        """
        cmd = [cls.pssuspend, "/accepteula", ProcessId]
        sp.call(cmd)

    @classmethod
    def win_resume_process(cls, ProcessId):
        """
        Resume ProcessId
        You should be admin or owner of process

        :param ProcessId: string with id of windows process
        """
        cmd = [cls.pssuspend, "/accepteula", "-r", ProcessId]
        sp.call(cmd)

    @classmethod
    def win_kill_user_procs(cls, username, timeout=60, logger=None, exclude_self=False):
        """
        Kill all username process.
        You should be admin or owner of process.
        Windows os does not have SIGKILL.

        :param username: string with username to kill
        :param timeout: Timeout for subprocess
        :param logger: Logger
        :param exclude_self: if True, filter out own PID from taskkill
        """

        if logger is None:
            logger = logging
        cmd = [cls.taskkill, "/F", "/FI", "USERNAME eq {}".format(username)]
        if exclude_self:  # TODO: remove after SANDBOX-9583
            cmd.extend(["/FI", "PID ne {}".format(psutil.Process().pid)])
            cmd.extend(["/FI", "IMAGENAME ne conhost.exe"])
        logger.debug("Send term for all process owned by %s \n via %s", username, " ".join(cmd))
        try:
            sp.check_call(cmd, timeout=timeout)
        except sp.CalledProcessError:
            logger.error("Can't kill user processes with cmd: %s", " ".join(cmd))
        except sp.TimeoutExpired:
            logger.error("Timeout expired on %s", " ".join(cmd))

    @classmethod
    def win_kill_process(cls, pid, timeout=60, logger=None):
        if logger is None:
            logger = logging

        cmd = [cls.taskkill, "/PID", str(pid)]
        logger.debug('Going to kill process with cmd "%s"', " ".join(cmd))
        try:
            sp.check_call(cmd, timeout=timeout)
        except sp.CalledProcessError as ex:
            logger.error('Can''t kill process %s with cmd "%s": %s', pid, " ".join(cmd), ex)
        except sp.TimeoutExpired:
            logger.error("Timeout expired on %s", " ".join(cmd))
