from __future__ import unicode_literals

import re
import json
import six

from .. import crypto
from .. import encoding
from .. import patterns


class Secret(patterns.NamedTuple):

    SEC_RE = re.compile(r"^sec-[a-zA-Z0-9]+$")
    VER_SEP = "@"
    VER_RE = re.compile(r"^ver-[a-zA-Z0-9]+$")
    KEY_SEP = "#"

    __slots__ = ("secret_uuid", "version_uuid", "default_key")

    def __encode__(self):
        parts = [self.secret_uuid]
        if self.version_uuid is not None:
            parts.extend([self.VER_SEP, self.version_uuid])
        if self.default_key is not None:
            parts.extend([self.KEY_SEP, self.default_key])
        return "".join(parts)

    def __str__(self):
        return self.__encode__()

    def __nonzero__(self):
        return bool(self.secret_uuid)

    __bool__ = __nonzero__

    @classmethod
    def __decode__(cls, value):
        parts = value.split(cls.KEY_SEP, 1)
        default_key = parts[1] if len(parts) > 1 else None

        parts = parts[0].split(cls.VER_SEP, 1)
        secret_uuid = parts[0]

        version_uuid = parts[1] if len(parts) > 1 else None

        return cls(secret_uuid, version_uuid, default_key)

    @classmethod
    def create(cls, secret_uuid, version_uuid=None, default_key=None):
        secret_uuid = encoding.force_unicode(secret_uuid)
        if not cls.SEC_RE.match(secret_uuid):
            raise ValueError("invalid secret uuid: {}".format(secret_uuid))

        if version_uuid:
            version_uuid = encoding.force_unicode(version_uuid)
            if not cls.VER_RE.match(version_uuid):
                raise ValueError("invalid version uuid: {}".format(version_uuid))
        else:
            version_uuid = None

        if default_key is not None:
            default_key = encoding.force_unicode(default_key)

        return cls(secret_uuid, version_uuid, default_key)


class Data(patterns.NamedTuple):
    __slots__ = ("value", "version_uuid", "secret_values")

    class Error(Exception):
        pass

    @classmethod
    def from_dict(cls, secret_dict):  # type: (dict) -> Data
        value = {item["key"]: item["value"] for item in secret_dict["value"]}
        return Data(value, secret_dict["secret_version"], secret_dict["value"])

    def encrypt(self, key):
        """
        Encrypt secret value

        :param key: encryption key
        :type key: str
        :return: encrypted base64-encoded secret value
        :rtype str
        """

        try:
            data = json.dumps(self.value, ensure_ascii=False)
            data = six.ensure_binary(data)
        except (ValueError, TypeError):
            raise self.Error("secret to be encrypted is not JSON-serializable")

        data = crypto.AES(key).encrypt(data, use_base64=True, use_salt=True)

        return data

    @classmethod
    def decrypt(cls, key, data):
        """
        Decrypt secret value

        :param key: decryption key
        :type key: str
        :param data: base64-encoded encrypted data
        :type data: str
        :return: decrypted data
        """

        decrypted = crypto.AES(key).decrypt(data, use_base64=True)
        decrypted = six.ensure_text(decrypted)

        try:
            decrypted = json.loads(decrypted)
        except (ValueError, TypeError):
            raise cls.Error("decrypted secret is not JSON-serializable")

        return decrypted
