import os
import sys
import time
import uuid
import errno
import socket
import shutil
import httplib
import logging
import getpass
import urllib2
import tempfile
import contextlib
import subprocess
import collections

import py
import yaml
import pytest

SANDBOX_DIR = os.path.dirname(__file__)
sys.path.extend([os.path.dirname(SANDBOX_DIR), SANDBOX_DIR])

from sandbox import common
# this import is required if one wants to run tests via py.test
import sandbox.common.config  # noqa
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.user as ctu
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr

import sandbox.tests.common.db as tests_db
import sandbox.tests.common.utils as tests_utils
import sandbox.tests.common.config as tests_config
import sandbox.tests.common.network as tests_network

from sandbox.agentr import client as aclient
from sandbox.yasandbox.controller import dispatch


pytest_plugins = (
    "sandbox.tests.common",
    "sandbox.serviceq.tests.conftest",
)


collect_ignore = ["scripts/sandbox_monitoring_test.py"]

# Service user, group and token accumulator type
ServiceUGT_t = collections.namedtuple("ServiceUGs", ("user", "group", "token"))


def pytest_configure():
    import xdist.remote
    xdist.remote.__file__ = os.path.join(os.path.dirname(__file__), 'common', 'test', 'xdist_remote_patch.py')


def pytest_addoption(parser):
    parser.addoption(
        '--port',
        type='int',
        default=None,
        help='Sandbox server port.'
    )
    parser.addoption(
        '--mongo-uri',
        default=tests_db.DEFAULT_TEST_MONGO_URI,
        help='Mongo DB connection string.')
    parser.addoption(
        '--tasks',
        default=None,
        help='Path to Sandbox tasks source (projects folder).'
    )
    parser.addoption(
        '--runtime',
        default=None,
        help='Path to Sandbox runtime data.'
    )
    parser.addoption(
        '--local-arcadia-user',
        action='store_true',
        help='Use local user for arcadia.'
    )
    parser.addoption(
        '--hostname',
        # trim to 20 characters so that generated database name fits into 64 characters
        # https://docs.mongodb.com/manual/reference/limits/#Length-of-Database-Names
        default=tests_utils.hostname()[:20],
        help='Hostname alias.'
    )


@pytest.fixture()
def json_api_url(server, host, serviceapi_port):
    return "http://{}:{}/api/v1.0".format(host, serviceapi_port)


@pytest.fixture(scope="session")
def xmlrpc_url(host, serviceapi_port):
    return "http://{}:{}/sandbox/xmlrpc".format(host, serviceapi_port)


@pytest.fixture(scope="session")
def http_check_url(host, serviceapi_port):
    return "http://{}:{}/sandbox/http_check".format(host, serviceapi_port)


@pytest.fixture(scope="session")
def status_server_url(host, serviceapi_port):
    return "http://{}:{}/api/v1.0/service/status/server".format(host, serviceapi_port)


def __xmlrpc_server(xmlrpc_url, oauth_token=None):
    null_logger = logging.getLogger('null')
    null_logger.addHandler(logging.NullHandler())
    null_logger.propagate = False
    return common.proxy.ReliableServerProxy(
        xmlrpc_url, oauth_token=oauth_token, logger=null_logger, component=ctm.Component.TESTS
    )


def __api_session_ug(usr, grp, token, robot=False):
    from yasandbox import controller
    from yasandbox.database import mapping

    try:
        usr.save(force_insert=True)
    except mapping.NotUniqueError:
        pass
    controller.User.validated(usr.login, is_robot=robot)
    controller.OAuthCache.refresh(usr.login, token)
    try:
        grp.save(force_insert=True)
    except mapping.NotUniqueError:
        pass

    return ServiceUGT_t(usr, grp, token)


def __api_session(login, group, super_user=False, trusted=False, robot=False, token=None, roles=None):
    from yasandbox import controller

    # Create fake user
    usr = controller.User.Model(
        login=login,
        allowed_api_methods=['create_task'],
        super_user=super_user,
        robot=robot,
        roles=roles
    )
    grp = controller.Group.Model(name=group, users=[login])
    if trusted:
        grp.priority_limits = controller.Group.Model.PriorityLimits(
            api=ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.HIGH),
            ui=ctt.Priority(ctt.Priority.Class.USER, ctt.Priority.Subclass.HIGH),
        )
    # Generate fake OAuth token
    ugt = __api_session_ug(usr, grp, token or uuid.uuid4().hex, robot=robot)
    return ugt.token


def __xmlrpc_session(xmlrpc_url, login, group, super_user=False, trusted=False):
    return __xmlrpc_server(xmlrpc_url, __api_session(login, group, super_user, trusted))


def __gui_session(login, super_user=False):
    from sandbox.yasandbox import controller

    sid = uuid.uuid4().hex
    controller.User.Model(login=login, super_user=super_user).save()
    controller.User.set_session_id(login, sid)
    controller.User.validated(login)
    return {'Cookie': 'Session_id={}'.format(sid), 'Referer': 'localhost'}


def __rest_session(url, login, group, super_user=False, client_class=common.rest.Client, token=None, roles=None):
    return client_class(
        url, __api_session(login, group, super_user, token=token, roles=roles),
        component=ctm.Component.TESTS
    )


@pytest.yield_fixture()
def task_session(monkeypatch, client_node_id, api_session_login):
    import yasandbox.controller
    import yasandbox.database.mapping

    sessions = []

    def create_task_session(client, task_id, cid=client_node_id, login=api_session_login, vault_key=None):
        yasandbox.controller.User.validated(login)
        token = uuid.uuid4().hex
        oauth = yasandbox.database.mapping.OAuthCache(
            token=token,
            login=login,
            ttl=3600,
            source=":".join([ctu.TokenSource.CLIENT, cid]),
            app_id=str(task_id),
            task_id=task_id,
            state=str(ctt.SessionState.ACTIVE),
            vault=vault_key,
        )
        oauth.save()
        sessions.append(token)

        if isinstance(client, common.rest.Client):
            monkeypatch.setattr(client._Client__session, "auth", client.Auth(common.proxy.OAuth(oauth.token)))
        else:
            monkeypatch.setattr(client.transport, "auth", common.proxy.OAuth(oauth.token))

        return token

    yield create_task_session

    yasandbox.database.mapping.OAuthCache.objects(token__in=sessions).delete()


@pytest.fixture
def task_session_context(task_session):
    import yasandbox.database.mapping

    @contextlib.contextmanager
    def func(session, task_id):
        token = task_session(session, task_id)
        try:
            yield
        finally:
            yasandbox.database.mapping.OAuthCache.objects(token=token).delete()

    return func


@pytest.fixture()
def server(
    request, xmlrpc_url, status_server_url, api_session_login, api_session_group, initialize, clear_tests_dir,
    serviceapi
):
    # First of all, switch logging. To do this, clean up any loggers first.
    root = logging.getLogger()
    map(root.removeHandler, root.handlers[:])
    map(root.removeFilter, root.filters[:])

    settings = common.config.Registry()
    common.log.setup_log(
        os.path.join(settings.server.log.root, settings.server.log.name),
        settings.server.log.level
    )

    from sandbox.scripts import sandbox_ctl

    cfg, os.environ["SANDBOX_CONFIG"] = os.environ["SANDBOX_CONFIG"], os.environ["SANDBOX_WEB_CONFIG"]
    sandbox_ctl.rc_action("start", "server")

    def stop_server():
        sandbox_ctl.rc_action("stop", "server")
    request.addfinalizer(stop_server)

    os.environ["SANDBOX_CONFIG"] = cfg

    # wait while server finished projects initialization, etc
    for _ in xrange(100):
        try:
            urllib2.urlopen(status_server_url).read()
            break
        except (urllib2.URLError, httplib.BadStatusLine):
            time.sleep(.1)
    else:
        raise Exception("Failed to start web-server")

    return __xmlrpc_session(xmlrpc_url, api_session_login, api_session_group)


@pytest.fixture()
def new_layout_config(request, new_layout_config_path):
    import agentr.config

    config = agentr.config.Registry()
    config.reload()
    base = None
    for path in config._base + (py.path.local(new_layout_config_path), ):
        base = agentr.config.Registry._merge_recursive(base, yaml.load(path.open("rb")))

    def rollback_config():
        agentr.config.Registry().reload()

    request.addfinalizer(rollback_config)

    return config.query({}, base=base)


@pytest.fixture(scope="session")
def fileserver(fileserver_binary, host, fileserver_port, tests_dir, run_daemon):
    run_daemon([fileserver_binary])
    tests_network.wait_until_port_is_ready(host, fileserver_port, 10)


@pytest.fixture()
def agentr_daemon(
    request, initialize, clear_tests_dir, agentr_binary, run_daemon, rest_su_session_token, rest_su_session
):
    return _agentr_daemon(request, agentr_binary, run_daemon)


@pytest.fixture()
def new_layout_agentr_daemon(
    request, initialize, clear_tests_dir, new_layout_config, new_layout_config_path, agentr_binary, run_daemon,
    rest_su_session_token, rest_su_session
):
    return _agentr_daemon(request, agentr_binary, run_daemon, new_layout_config_path, new_layout_config)


def _agentr_daemon(request, agentr_binary, run_daemon, new_layout_config_path=None, new_layout_config=None):
    env = os.environ.copy()
    if new_layout_config_path:
        env["SANDBOX_CONFIG"] = new_layout_config_path
    env["AWS_ACCESS_KEY_ID"] = "1234567890"
    env["AWS_SECRET_ACCESS_KEY"] = "1234567890"

    run_daemon([agentr_binary], env)

    srv = _ensure_agentr_started(new_layout_config=new_layout_config)

    def terminate():
        try:
            srv.call("reset", hard=True).wait()
        except Exception:
            pass
        try:
            srv.call("shutdown").wait()
        except Exception:
            pass

    request.addfinalizer(terminate)
    return srv


@pytest.fixture
def agentr(agentr_daemon, s3_simulator):
    srv = _ensure_agentr_started()
    return srv


@pytest.fixture()
def new_layout_agentr(new_layout_agentr_daemon, new_layout_config):
    return _ensure_agentr_started(new_layout_config)


@pytest.fixture
def agentr_service(agentr):
    return aclient.Service(logging.getLogger(__name__))


@pytest.fixture
def agentr_session_maker(
    initialize, agentr, task_session, rest_session, rest_session_login, server, sdk2_dispatched_rest_session
):
    def make_session(task, iteration):
        token = task_session(rest_session, task.id, rest_session_login)
        return aclient.Session(token, iteration, 123456789, logging.getLogger(__name__))

    return make_session


def _ensure_agentr_started(new_layout_config=None):
    """Ensure AgentR has started. Returns RPCClient to connect to it."""
    import common.joint.client
    import agentr.config

    agentr.config.Registry().reload()
    config = (new_layout_config if new_layout_config else agentr.config.Registry()).agentr.daemon
    for _ in common.utils.progressive_yielder(.1, 1, 10):
        try:
            srv = common.joint.client.RPCClient(cfg=config.rpc, host=config.server.unix, port=None)
            srv.ping()
            srv.call("reset", hard=True).wait()
            return srv
        except socket.error:
            pass
    raise AssertionError("Unable to establish connection to AgentR daemon")


@pytest.fixture()
def api_session(initialize, xmlrpc_url, api_session_login, api_session_group):
    return __xmlrpc_session(xmlrpc_url, api_session_login, api_session_group)


@pytest.fixture()
def anonymous_session(initialize, server, xmlrpc_url):
    return __xmlrpc_server(xmlrpc_url)


@pytest.fixture()
def api_su_session(initialize, xmlrpc_url, api_su_session_login, api_su_session_group):
    return __xmlrpc_session(xmlrpc_url, api_su_session_login, api_su_session_group, super_user=True)


@pytest.fixture()
def api_su_session2(initialize, xmlrpc_url, api_su_session_login, api_su_session_group):
    return __xmlrpc_session(xmlrpc_url, api_su_session_login + "2", api_su_session_group, super_user=True)


@pytest.fixture()
def api_trusted_session(initialize, server, xmlrpc_url, api_trusted_session_login, api_trusted_session_group):
    return __xmlrpc_session(xmlrpc_url, api_trusted_session_login, api_trusted_session_group, trusted=True)


@pytest.fixture()
def gui_session(initialize, server, gui_session_login):
    return __gui_session(gui_session_login)


@pytest.fixture()
def gui_su_session(initialize, server, gui_su_session_login):
    return __gui_session(gui_su_session_login, True)


@pytest.fixture()
def service_user(initialize, service_login):
    return __api_session(service_login, common.config.Registry().common.service_group, super_user=True, robot=True)


@pytest.fixture()
def rest_session(initialize, server, rest_session_login, rest_session_group, rest_session_token, json_api_url):
    return __rest_session(json_api_url, rest_session_login, rest_session_group, token=rest_session_token)


@pytest.fixture()
def rest_session2(initialize, server, rest_session_login2, rest_session_group, json_api_url):
    return __rest_session(json_api_url, rest_session_login2, rest_session_group)


@pytest.fixture()
def rest_sudoer_session(
    initialize, server, sudoer_login, rest_session_group, rest_sudoer_token, json_api_url
):
    return __rest_session(
        json_api_url, sudoer_login, rest_session_group, token=rest_sudoer_token,
        roles=[ctu.Role.TASKS_SUDO],
    )


@pytest.fixture()
def rest_session_maker_session(
    initialize, server, session_maker_login, rest_session_group, rest_session_maker_token, json_api_url
):
    return __rest_session(
        json_api_url, session_maker_login, rest_session_group, token=rest_session_maker_token,
        roles=[ctu.Role.SESSION_MAKER]
    )


@pytest.fixture()
def rest_session_maker2_session(
    initialize, server, session_maker2_login, rest_session_group, rest_session_maker2_token, json_api_url
):
    return __rest_session(
        json_api_url, session_maker2_login, rest_session_group, token=rest_session_maker2_token,
        roles=[ctu.Role.SESSION_MAKER]
    )


@pytest.fixture()
def rest_noauth_session(initialize, server, json_api_url):
    return common.rest.Client(json_api_url, common.proxy.NoAuth(), component=ctm.Component.TESTS)


@pytest.fixture()
def rest_su_session(
    initialize, server, rest_su_session_login, rest_su_session_group, json_api_url, rest_su_session_token
):
    return __rest_session(json_api_url, rest_su_session_login, rest_su_session_group, True, token=rest_su_session_token)


@pytest.fixture(scope="session")
def settings(session_initialize):
    return common.config.Registry()


@pytest.fixture()
def client(
    initialize,
    server,
    agentr_daemon, json_api_url,
    rest_su_session_login, rest_su_session_group, rest_su_session_token, settings, client_node_id,
    monkeypatch, preexecutor_binary, s3_simulator
):
    import agentr.client
    import agentr.config
    agentr.config.Registry().reload()
    import yasandbox.manager

    agentr.client.Session(None, None, 0, logging.getLogger("")).reset()
    # Protect against multiple client initialization.
    skip_init = 'bin.client' in sys.modules

    import bin.client
    import sandbox.client

    if not skip_init:
        sandbox.client.system.PROCMAN_TAG += "_{}".format(os.getpid())
        sandbox.client.system.UNPRIVILEGED_USER = sandbox.client.system.SERVICE_USER = common.os.User(
            getpass.getuser(), os.path.expanduser("~" + getpass.getuser())
        )
        bin.client.launch.initialize()

    c = yasandbox.manager.client_manager.load(client_node_id, create=True)
    c["uuid"] = "c9c769c1c5314740ac350d5a022897c7"
    c.save()  # To avoid assigning "CLEANUP" task to the host
    data = common.utils.get_sysparams()
    data.update({
        "dc": "unk",
        "lxc": False,
        "root": False,
        "ram": data.pop("physmem"),
        "uuid": "c9c769c1c5314740ac350d5a022897c7",
        "age": sandbox.client.pinger.PingSandboxServerThread.CLIENT_AGE,
        "tags": map(str, tests_config.CLIENT_TAGS),
        "disk": {"free_space": 1 << 40, "total_space": 1 << 40, "status": "ok"},
    })
    rest = __rest_session(json_api_url, rest_su_session_login, rest_su_session_group, True, token=rest_su_session_token)
    rest.client[client_node_id](data)
    c = yasandbox.manager.client_manager.load(client_node_id)
    c.update_tags({ctc.Tag.NEW}, c.TagsOp.REMOVE)
    return c


@pytest.fixture()
def dummy_client(client_manager):
    c = client_manager.load("dummy_client", create=True)
    c.update_tags(tests_config.CLIENT_TAGS, c.TagsOp.SET)
    c.ram = sys.maxint
    c.ncpu = sys.maxint
    c.save()
    return c


@pytest.fixture()
def new_layout_config_path(request, initialize, config_path):
    nl_settings_file_path = os.path.join(os.path.dirname(config_path), "nl_settings.yaml")
    if os.path.exists(nl_settings_file_path):
        return nl_settings_file_path

    settings = yaml.load(open(config_path, "r"))

    new_layout = str(ctc.Tag.NEW_LAYOUT)
    if new_layout not in settings["client"]["tags"]:
        settings["client"]["tags"].append(new_layout)

    settings["agentr"]["log"]["name"] = "nl_agentr.log"
    settings["agentr"]["data"]["buckets"] = [
        "${{client.dirs.data}}/storage/{}".format(i) for i in xrange(4)
    ]
    settings["agentr"]["daemon"]["db"]["path"] = os.path.join(
        os.path.dirname(settings["agentr"]["daemon"]["db"]["path"]), "nl_agentr.db"
    )

    # will be cleaned up later in tests_dir finalizer
    agentr_socket = settings["agentr"]["daemon"]["server"]["unix"]
    settings["agentr"]["daemon"]["server"]["unix"] = agentr_socket.replace("agentr", "nl_agentr")

    yaml.dump(settings, open(nl_settings_file_path, "w"), default_flow_style=False)
    return nl_settings_file_path


def _create_tests_dir(ds, tests_path_getter):
    for d in ds:
        subdir = tests_path_getter(d)
        try:
            os.makedirs(subdir)
        except OSError as exc:
            if exc.errno == errno.EEXIST and os.path.isdir(subdir):
                pass
            else:
                raise


@pytest.fixture(scope="session")
def tests_dir(
    request, work_id, tests_path_getter, tests_common_path, config_path, web_config_path,
    agentr_socket, fileserver_socket, phazotron_socket,
):
    def clean_tmp_socks():
        for sock in (agentr_socket, fileserver_socket, phazotron_socket):
            common.fs.remove_path_safely(sock)

    request.addfinalizer(clean_tmp_socks)

    ds = ["etc", "srcdir", "testdata", "ccache", "logs", "logs/api", "mailout", "run", "tasks", "resources"]
    for i in xrange(5):
        ds.append("storage/" + str(i))
    _create_tests_dir(ds, tests_path_getter)

    os.environ["SANDBOX_CONFIG"] = config_path
    os.environ["SANDBOX_WEB_CONFIG"] = web_config_path
    return tests_path_getter()


@pytest.fixture(autouse=True)
def clear_tests_dir(tests_dir, tests_path_getter, config_path, web_config_path):
    ds = ["tasks", "resources"]
    for i in xrange(5):
        ds.append("storage/" + str(i))
    for d in ds:
        subdir = tests_path_getter(d)
        if os.path.exists(subdir):
            subprocess.Popen('chmod -R a+w {0}'.format(subdir), shell=True).wait()
            shutil.rmtree(subdir)

    _create_tests_dir(ds, tests_path_getter)

    os.environ["SANDBOX_CONFIG"] = config_path
    os.environ["SANDBOX_WEB_CONFIG"] = web_config_path
    common.config.Registry().reload()
    return tests_path_getter()


@pytest.fixture(scope="function")
def sandbox_copy_dir(tmpdir, sandbox_dir):
    dst = os.path.join(tmpdir, "sandbox")
    shutil.copytree(sandbox_dir, dst, symlinks=True)

    for root, _, files in os.walk(dst):
        for file in files:
            path = os.path.join(root, file)
            if os.path.islink(path):
                os.remove(path)
    return dst


@pytest.fixture()
def tasks_dir(clear_tests_dir):
    tasksdir = os.path.join(str(clear_tests_dir), 'tasks')
    subprocess.Popen('chmod -R a+w {0}'.format(tasksdir), shell=True).wait()
    shutil.rmtree(tasksdir)
    os.mkdir(tasksdir)

    yield str(tasksdir)

    for root, dirs, files in os.walk(tasksdir):
        for d in dirs + files:
            path = os.path.join(root, d)
            if os.path.islink(path):
                os.unlink(path)


@pytest.fixture()
def tmpdir(clear_tests_dir):
    """
    Redefines standart `tmpdir` fixture to create tmp dirs in test directory and NOT rely on `TEMPDIR` env variable
    """
    ret = tempfile.mkdtemp(prefix="tmp__", dir=clear_tests_dir)
    os.chmod(ret, 0o755)
    return ret


def reload_modules(module):
    from types import ModuleType
    for attr_name in dir(module):
        attr = getattr(module, attr_name, None)
        if type(attr) is ModuleType:
            reload(attr)
    reload(module)


def common_initialize(sandbox_tasks_dir, host, client_port, clear_mongo_uri, releaser):
    common.config.Registry().reload()
    from sandbox.yasandbox.database import mapping
    from sandbox.yasandbox import manager
    from sandbox.yasandbox import controller
    reload_modules(controller)
    from sandbox.common.projects_handler import load_project_types

    import sandbox.projects.resource_types as rt
    for rtype in rt.AbstractResource:
        lst = rtype.releasers or []
        rtype.releasers = list(lst) + [releaser]

    import sandbox.sandboxsdk.channel
    reload(sandbox.sandboxsdk.channel)
    mapping.ensure_connection(uri=clear_mongo_uri)
    manager.use_locally(detain_init=True)
    manager.initialize_locally()
    controller.initialize()
    from sandbox import sdk2
    sdk2.Task.__cache__.clear()
    sdk2.Resource.__cache__.clear()
    load_project_types(raise_exceptions=True)


@pytest.fixture(scope="session")
def session_initialize(mongo_uri, tests_dir, sandbox_tasks_dir, host, client_port, releaser):
    common_initialize(sandbox_tasks_dir, host, client_port, mongo_uri, releaser)


@pytest.fixture(autouse=True)
def initialize(clear_tests_dir, sandbox_tasks_dir, host, client_port, clear_mongo_uri, releaser):
    common_initialize(sandbox_tasks_dir, host, client_port, clear_mongo_uri, releaser)


@pytest.fixture()
def local_host_client(initialize, host, client_port, sandbox_tasks_dir):
    from sandbox.yasandbox import manager
    parameters = common.utils.get_sysparams()
    parameters['http_prefix'] = 'http://%s:%s' % (host, client_port)
    parameters['tasks_dir'] = sandbox_tasks_dir
    c = manager.client_manager.load(host).update({'system': parameters})
    c.update_tags(tests_config.CLIENT_TAGS, c.TagsOp.SET)


@pytest.fixture()
def signals_db(request, initialize):
    from sandbox.yasandbox.database import mapping
    db = mapping.ensure_connection().rw.connection[
        common.config.Registry().server.services.statistics_processor.database
    ]

    for collection in db.collection_names():
        if not collection.startswith("system"):
            db.drop_collection(collection)
    return db


@pytest.fixture()
def task_manager(initialize, serviceq):
    import yasandbox.manager
    return yasandbox.manager.task_manager


@pytest.fixture()
def client_manager(initialize):
    import yasandbox.manager
    return yasandbox.manager.client_manager


@pytest.fixture()
def resource_manager(initialize):
    import yasandbox.manager
    return yasandbox.manager.resource_manager


@pytest.fixture()
def release_manager(initialize, serviceq):
    import yasandbox.manager
    return yasandbox.manager.release_manager


@pytest.fixture()
def scheduler_controller(initialize, server, serviceq):
    import yasandbox.controller
    return yasandbox.controller.Scheduler()


@pytest.fixture()
def settings_controller(initialize):
    from sandbox.yasandbox import controller
    return controller.Settings()


@pytest.fixture()
def notification_controller(initialize):
    from sandbox.yasandbox import controller
    return controller.Notification


@pytest.fixture()
def notification_manager(initialize):
    import yasandbox.manager
    return yasandbox.manager.notification_manager


@pytest.fixture()
def oauth_controller(initialize, user_controller):
    import yasandbox.controller
    return yasandbox.controller.OAuthCache()


@pytest.fixture()
def user_controller(initialize):
    import yasandbox.controller
    return yasandbox.controller.User()


@pytest.fixture()
def group_controller(initialize):
    import yasandbox.controller
    return yasandbox.controller.Group()


@pytest.fixture()
def vault_controller(initialize):
    import yasandbox.controller
    return yasandbox.controller.Vault()


@pytest.fixture()
def semaphore_controller(initialize):
    import yasandbox.controller
    return yasandbox.controller.Semaphore


@pytest.fixture()
def notification_trigger_controller(initialize):
    import yasandbox.controller
    return yasandbox.controller.TaskStatusNotifierTrigger()


@pytest.fixture()
def statistics_controller(initialize):
    import yasandbox.controller
    return yasandbox.controller.Statistics()


@pytest.fixture()
def uinotification_controller(initialize):
    import yasandbox.controller
    return yasandbox.controller.UINotification()


@pytest.fixture()
def task_controller(initialize):
    import yasandbox.controller
    return yasandbox.controller.Task


@pytest.fixture()
def template_controller(initialize):
    from sandbox.yasandbox import controller
    return controller.Template


@pytest.fixture()
def clean_task_audit(initialize):
    pass


@pytest.fixture()
def state_controller(initialize):
    import sandbox.yasandbox.controller
    return sandbox.yasandbox.controller.State


@pytest.fixture()
def test_task(client):
    import projects.sandbox.test_task
    return projects.sandbox.test_task.TestTask


@pytest.fixture()
def test_task_2(initialize, client):
    from projects.sandbox.test_task_2 import TestTask2
    return TestTask2


@pytest.fixture()
def sdk2_dispatched_rest_session(request, rest_session):
    dc = common.rest.DispatchedClient.__enter__()
    dc(lambda *args, **kwargs: rest_session)
    request.addfinalizer(common.rest.DispatchedClient.__exit__)
    return rest_session


@pytest.fixture()
def sdk2_dispatched_rest_su_session(request, rest_su_session):
    dc = common.rest.DispatchedClient.__enter__()
    dc(lambda *args, **kwargs: rest_su_session)
    request.addfinalizer(common.rest.DispatchedClient.__exit__)
    return rest_su_session


@pytest.fixture()
def dispatched_rest_session(initialize, request, server, rest_session_login, rest_session_group, json_api_url):
    return __rest_session(
        json_api_url, rest_session_login, rest_session_group,
        client_class=dispatch.RestClient(None, rest_session_login, jailed=False)
    )


@pytest.fixture(scope="session")
def fake_agentr():
    import sandbox.agentr.client
    import common
    return lambda task: type(
        "FakeAgentR", (sandbox.agentr.client.Session,), {
            "meta": {
                "sdk_version": 1,
                "id": task.id,
                "type": task.type,
                "status": task.status,
                "author": task.author,
                "context": {}
            },
            "__init__": lambda _: None,
            "log_resource": common.rest.Client().resource.read(
                task_id=task.id, limit=1, type=ctr.TASK_LOG_RESOURCE_TYPE
            )["items"][0],
        }
    )()


@pytest.fixture()
def task_state_switcher(initialize, rest_su_session, server):
    from sandbox.services.modules import task_state_switcher as tts

    class TaskStateSwitcher(tts.TaskStateSwitcher):
        def __init__(self, *args, **kwargs):
            super(TaskStateSwitcher, self).__init__(*args, max_subworkers=1, **kwargs)
            self.__context = {}

        @property
        def _rest(self):
            # We do not enqueue multiple tasks in tests so its safe to use non-thread-safe object here
            return rest_su_session

        @property
        def context(self):
            return self.__context

        def enqueue(self, model, event, trigger=None):
            # Do real work
            self._enqueue(model, event, None)  # do not send signals during tests
            # Create empty future for return value
            import concurrent.futures
            future = concurrent.futures.Future()
            future.set_result(None)
            return future

        def tick(self):
            for target in self.targets:
                target.function()

    return TaskStateSwitcher()


@pytest.fixture
def task_status_notifier(initialize):
    from sandbox.services.modules import task_status_notifier as tsn
    svc = tsn.TaskStatusNotifier()
    svc.load_service_state()
    return svc


@pytest.fixture(autouse=True)
def print_pid():
    print("PID: {}".format(os.getpid()))
