from . import tvm
from . import base
from . import step
from . import proxy
from . import client
from . import server
from . import system
from . import layouts
from . import solomon
from . import services
from . import clickhouse
from . import unified_agent

from . import juggler

import hashlib
import logging
import itertools as it
import collections


SANDBOX_SECRETS_ALIAS = "sandbox_deploy_secrets"  # alias in yav for secrets scope


class HashableDict(dict):
    def __hash__(self):
        return id(self)


class Plugin(object):
    __name__ = "sandbox"

    Proxy = proxy.Proxy
    ServiceQ = server.ServiceQ
    ServerLauncher = server.ServerLauncher
    ServiceApi = server.ServiceApi
    Taskbox = server.Taskbox
    TVMTool = tvm.TVMTool
    RsyncLauncher = client.RsyncLauncher
    ClientLauncher = client.ClientLauncher
    AgentR = client.AgentR
    FileServer = client.FileServer
    SystemConfig = system.SystemConfig
    Atop = system.Atop
    Step = step.STEP
    RemToStep = step.RemToStep
    ClickHouse = clickhouse.ClickHouse
    Solomon = solomon.SolomonAgent
    UnifiedAgent = unified_agent.UnifiedAgent

    VIRTUAL_GROUPS = {
        "virtual_browser_experimental": {
            "browser-linux-sandbox-sas-31.haze.yandex.net",
            "browser-linux-sandbox-sas-32.haze.yandex.net",
            "browser-linux-sandbox-sas-33.haze.yandex.net",
            "browser-linux-sandbox-sas-34.haze.yandex.net",
            "browser-linux-sandbox-sas-35.haze.yandex.net",
            "browser-linux-sandbox-sas-36.haze.yandex.net",
            "browser-linux-sandbox-sas-37.haze.yandex.net",
            "browser-linux-sandbox-sas-38.haze.yandex.net",
            "browser-linux-sandbox-sas-39.haze.yandex.net",
        },
    }

    # A list of Conductor tags and groups, based on whose we choose layout for a given host.
    # Order matters: if the host has multiple tags and/or groups, a higher tag or group takes precedence.
    # Note: for production cluster, we choose layouts based on amount of hosts with the corresponding tag/group.
    PREPRODUCTION_SPECIAL_GROUPS = [
        "sandbox1_client_generic",
        "sandbox1_windows",
        "sandbox1_macos",
        "sandbox1_multislot",
        "sandbox1_multios",
        "yp_lite@sandbox3",
        "sandbox1_porto",
        "sandbox1_lxc",
        "sandbox1_server",
        "sandbox1_clickhouse",
        "sandbox1_stg",
        "sandbox1_client",
        "sandbox1"
    ]

    SERVER_GROUPS = {
        0: {HashableDict(name=base.ctag("sandbox_api"), solid=True)},
        1: {HashableDict(name=base.ctag("sandbox1_server"), solid=True)},
        2: set(),
        3: set(),
    }

    SERVICES_GROUPS = {
        0: {HashableDict(name=base.ctag("sandbox_api"), solid=True)},
        1: {HashableDict(name=base.ctag("sandbox1_server"), solid=True)},
        2: set(),
        3: set(),
    }

    PROXY_GROUPS = {
        0: {base.ctag("sandbox_proxy")},
        1: {base.ctag("sandbox1_proxy")},
        2: set(),
        3: set(),
    }

    CLIENT_GROUPS = {
        0: {
            base.ctag("sandbox_client"),
            base.ctag("sandbox_server"),
            base.ctag("sandbox_stg"),
        },
        1: {
            base.ctag("sandbox1_client"),
            base.ctag("sandbox1_server"),
            base.ctag("sandbox1_stg")
        },
        2: {base.ctag("yp_lite@sandbox2")},
        3: {base.ctag("yp_lite@sandbox3")},
    }

    STEP_GROUPS = {
        0: {HashableDict(name=base.ctag("sandbox_mongo"), solid=True)},
        1: {HashableDict(name=base.ctag("sandbox1_server"), solid=True)},
        2: set(),
        3: set(),
    }

    REM_TO_STEP_GROUPS = {
        0: {HashableDict(name=base.ctag("sandbox_zk"), solid=True)},
        1: {HashableDict(name=base.ctag("sandbox1_server"), solid=True)},
        2: set(),
        3: set(),
    }

    SOLOMON_GROUPS = {
        0: {HashableDict(name=base.ctag("sandbox_macos"), solid=True)},
        1: {HashableDict(name=base.ctag("sandbox1_macos"), solid=True)},
        2: set(),
        3: set(),
    }

    ATOP_GROUPS = {
        0: {
            base.ctag("sandbox_linux"),
        },
        1: {
            base.ctag("sandbox1_linux"),
        },
        2: {base.ctag("yp_lite@sandbox2")},
        3: {base.ctag("yp_lite@sandbox3")},
    }

    @property
    def server_groups(self):
        return self.SERVER_GROUPS[self.key]

    @property
    def services_groups(self):
        return self.SERVICES_GROUPS[self.key]

    @property
    def proxy_groups(self):
        return self.PROXY_GROUPS[self.key]

    @property
    def client_groups(self):
        return self.CLIENT_GROUPS[self.key]

    @property
    def step_groups(self):
        return self.STEP_GROUPS[self.key]

    @property
    def rem_to_step_groups(self):
        return self.REM_TO_STEP_GROUPS[self.key]

    @property
    def clickhouse_groups(self):
        return {clickhouse.ClickHouse.settings[self._is_test].ctag}

    @property
    def solomon_groups(self):
        return self.SOLOMON_GROUPS[self.key]

    @property
    def atop_groups(self):
        return self.ATOP_GROUPS[self.key]

    __deploy_config = None

    @staticmethod
    def __groups_without_prefix(groups):
        return {
            (
                group.replace(base.CTAG_PREFIX, "")
                if isinstance(group, basestring) else
                group["name"].replace(base.CTAG_PREFIX, "")
            )
            for group in groups
        }

    def get_services_servants(self):
        service_list = [
            services.CheckSemaphores,
            services.TaskQueueValidator,
            services.TasksEnqueuer,
            services.Cleaner,
            services.CleanResources,
            services.ClientAvailabilityChecker,
            services.ClientSlotsMonitor,
            services.GroupSynchronizer,
            services.MongoMonitor,
            services.MongoChecker,
            services.CpuBalancer,
            services.Scheduler,
            services.StatisticsProcessor,
            services.TasksStatistics,
            services.ResourcesStatistics,
            services.TaskStateSwitcher,
            services.ClientAvailabilityManager,
            services.NetworkMacrosResolver,
            services.ClientProcessor,
            services.HostsCacheUpdater,
            services.TaskStatusChecker,
            services.TaskStatusNotifier,
            services.AutoRestartTasks,
            services.Mailman,
            services.MessengerQBot,
            services.Juggler,
            services.UpdateRobotOwners,
            services.MDSCleaner,
            services.BackupResources,
            services.ReplicateResources,
            services.BackupResourcesToMds,
            services.TaskTagsChecker,
            # services.CopyResourcesToWarehouse,  # DC MAN disabled forever
            services.MdsLruCleaner,
            services.LogbrokerPublisher,
        ]
        if self.key == 0:
            service_list.append(services.TelegramBot)
            service_list.append(services.MetricsReporter)

        for service in service_list:
            yield service.__servant_name__

    @property
    def deploy_config(self):
        if self.__deploy_config is None:
            groups = self.client_groups | self.server_groups | self.proxy_groups
            hosts = list(it.chain.from_iterable(self.VIRTUAL_GROUPS.get(g, set()) for g in groups))
            groups = groups - set(self.VIRTUAL_GROUPS)
            self.__deploy_config = {
                "hosts": hosts,
                "groups": groups,
                "scenario": {
                    self.Proxy.__servant_name__: {
                        "operating_degrade_level": 0.5,
                        "stop_degrade_level": 0.5,
                        "dependences": [self.SystemConfig.__servant_name__],
                    },
                    self.ServiceQ.__servant_name__: {
                        "operating_degrade_level": 0.25,
                        "stop_degrade_level": 0.25,
                        "dependences": [self.SystemConfig.__servant_name__],
                    },
                    self.ServerLauncher.__servant_name__: {
                        "operating_degrade_level": 0.5,
                        "stop_degrade_level": 0.5,
                        "dependences": [self.SystemConfig.__servant_name__],
                    },
                    self.ServiceApi.__servant_name__: {
                        "operating_degrade_level": 0.5,
                        "stop_degrade_level": 0.5,
                        "dependences": [self.SystemConfig.__servant_name__],
                    },
                    self.TVMTool.__servant_name__: {
                        "operating_degrade_level": 0.5,
                        "stop_degrade_level": 0,
                        "dependences": [self.SystemConfig.__servant_name__],
                    },
                    self.Taskbox.__servant_name__: {
                        "operating_degrade_level": 0.5,
                        "stop_degrade_level": 0.5,
                        "dependences": [self.SystemConfig.__servant_name__],
                    },
                    self.ClientLauncher.__servant_name__: {
                        "operating_degrade_level": 0.65,
                        "stop_degrade_level": 0.5,
                        "dependences": [self.SystemConfig.__servant_name__],
                    },
                    self.AgentR.__servant_name__: {
                        "operating_degrade_level": 1,
                        "stop_degrade_level": 1,
                        "dependences": [self.SystemConfig.__servant_name__],
                    },
                    self.FileServer.__servant_name__: {
                        "operating_degrade_level": 0.5,
                        "stop_degrade_level": 0.5,
                        "dependences": [self.SystemConfig.__servant_name__],
                    },
                    self.RsyncLauncher.__servant_name__: {
                        "operating_degrade_level": 0.5,
                        "stop_degrade_level": 0.5,
                        "dependences": [self.SystemConfig.__servant_name__],
                    },
                    self.SystemConfig.__servant_name__: {
                        "operating_degrade_level": 1,
                        "stop_degrade_level": 0.9,
                        "dependences": []
                    },
                    self.Atop.__servant_name__: {
                        "operating_degrade_level": 1,
                        "stop_degrade_level": 0.9,
                        "dependences": [self.SystemConfig.__servant_name__]
                    },
                    self.Step.__servant_name__: {
                        "operating_degrade_level": 0.5,
                        "stop_degrade_level": 0.5,
                        "dependences": [self.SystemConfig.__servant_name__],
                    },
                    self.RemToStep.__servant_name__: {
                        "operating_degrade_level": 0.3,
                        "stop_degrade_level": 0.3,
                        "dependences": [self.SystemConfig.__servant_name__],
                    },
                    self.ClickHouse.__servant_name__: {
                        "operating_degrade_level": 1,
                        "stop_degrade_level": 1,
                        "dependences": [self.SystemConfig.__servant_name__],
                    },
                    self.Solomon.__servant_name__: {
                        "operating_degrade_level": 0.5,
                        "stop_degrade_level": 0.5,
                        "dependences": [self.SystemConfig.__servant_name__]
                    },
                    self.UnifiedAgent.__servant_name__: {
                        "operating_degrade_level": 0.65,
                        "stop_degrade_level": 0.5,
                        "dependences": [self.SystemConfig.__servant_name__],
                    },
                },
                "servants": {
                    self.Proxy.__servant_name__: {"groups": self.__groups_without_prefix(self.proxy_groups)},
                    self.ServiceQ.__servant_name__: {"groups": self.__groups_without_prefix(self.server_groups)},
                    self.ServerLauncher.__servant_name__: {"groups": self.__groups_without_prefix(self.server_groups)},
                    self.ServiceApi.__servant_name__: {"groups": self.__groups_without_prefix(self.server_groups)},
                    self.Taskbox.__servant_name__: {"groups": self.__groups_without_prefix(self.server_groups)},
                    self.Step.__servant_name__: {"groups": self.__groups_without_prefix(self.step_groups)},
                    self.RemToStep.__servant_name__: {"groups": self.__groups_without_prefix(self.rem_to_step_groups)},
                    self.ClickHouse.__servant_name__: {"groups": self.__groups_without_prefix(self.clickhouse_groups)},
                    self.Solomon.__servant_name__: {"groups": self.__groups_without_prefix(self.solomon_groups)},
                    self.Atop.__servant_name__: {"groups": self.__groups_without_prefix(self.atop_groups)},
                }
            }

            for servant in self.get_services_servants():
                self.__deploy_config["scenario"][servant] = {
                    "operating_degrade_level": 0.8,
                    "stop_degrade_level": 0.8,
                    "dependences": [self.SystemConfig.__servant_name__],
                }
                # Services are deployed alongside with servers
                self.__deploy_config["servants"][servant] = {
                    "groups": self.__groups_without_prefix(self.services_groups)
                }

        return self.__deploy_config

    def __init__(self, iface):
        self.iconf = iface.conf()
        self.key = iface.key()
        self._is_yp, self._is_test = divmod(self.key, 2)
        self.scenario = self.deploy_config["scenario"]
        self.layouts = {}

        from libs.sandbox import rest
        self.sandbox_api = rest.Client()

    def deploy_scenario(self):
        return self.scenario

    @classmethod
    def _get_host_groups(cls, iface):
        for host, info in iface.host_list():
            host_groups = set()
            for _ in ("conductorGroups", "conductorTags", "groups"):
                host_groups |= set(info.get(_, ()) or ())

            for virtual_group, group_hosts in cls.VIRTUAL_GROUPS.iteritems():
                if host in group_hosts:
                    host_groups.add(virtual_group)
            yield host, info, host_groups

    def _get_host_info_production(self, iface):
        host_groups = collections.defaultdict(set)
        group_counter = collections.Counter()

        for host, info, groups in self._get_host_groups(iface):
            host_groups[host] = groups
            for group in groups:
                group_counter[group] += 1

        preproduction_groups = set(self.PREPRODUCTION_SPECIAL_GROUPS)
        for host, info in iface.host_list():
            special_groups = host_groups[host]
            if set(special_groups) & preproduction_groups:
                continue

            count_groups = {}
            for group in special_groups:
                if group in self.layouts:
                    if group_counter[group] in count_groups:
                        raise AssertionError(
                            "There is a host '{}' with groups '{}' and "
                            "'{}' which assigned to equal number of hosts".format(
                                host, group, count_groups[group_counter[group]]
                            )
                        )
                    count_groups[group_counter[group]] = group

            matching_groups = [
                (group_counter[group], group)
                for group in it.chain((host,), special_groups) if group in self.layouts
            ]
            if not matching_groups:
                logging.error(
                    "Layout not found on host %s. Host tags and groups: %r. Layouts: %r.",
                    host, special_groups, self.layouts.keys()
                )
                continue
            layout = self.layouts.get(min(matching_groups)[1])

            yield host, info, layout, special_groups

    def _get_host_info_preproduction(self, iface):
        for host, info, host_groups in self._get_host_groups(iface):
            special_groups = filter(lambda _: _ in host_groups, self.PREPRODUCTION_SPECIAL_GROUPS)
            layout = next(it.ifilter(None, it.imap(self.layouts.get, it.chain((host,), special_groups))), None)
            yield host, info, layout, host_groups

    def _get_host_info(self, iface):
        if self._is_test:
            get_host_info = self._get_host_info_preproduction
        else:
            get_host_info = self._get_host_info_production
        for item in get_host_info(iface):
            yield item

    def gencluster(self, iface):
        self.layouts = layouts.build_layouts(iface.secrets[SANDBOX_SECRETS_ALIAS])
        juggler_token = (
            iface.secrets[SANDBOX_SECRETS_ALIAS]["juggler_token_preproduction"]
            if self._is_test else
            iface.secrets[SANDBOX_SECRETS_ALIAS]["juggler_token_production"]
        )
        juggler.upsert_checks(self.key, juggler_token)

        pkg_meta = self.sandbox_api.resource[str(iface.pkg_id()).replace("sbr:", "")].read()
        allowed_owners = ["SANDBOX"]
        release = pkg_meta["attributes"].get("released")
        if pkg_meta["owner"] not in allowed_owners:
            raise AssertionError("Only {!r} groups are allowed to deploy a package.".format(allowed_owners))
        if not self._is_test and release != "stable":
            raise AssertionError("Deployment of '{}' release to production is prohibited!".format(release))
        if self._is_test and release != "prestable":
            raise AssertionError("Deployment of '{}' release to pre-production is prohibited!".format(release))
        servants = self.deploy_config.get("servants")
        virtual_groups = set(self.VIRTUAL_GROUPS)

        tvm_secret = iface.secrets[SANDBOX_SECRETS_ALIAS][
            "preproduction_tvm_secret" if self._is_test else "production_tvm_secret"
        ]

        servant_objects = {}
        for host, info, layout, groups in self._get_host_info(iface):

            host_platform = info.get("platform")

            tvmtool = self.TVMTool(self.key, tvm_secret, host_platform)
            tvminfo = tvmtool.info
            yield host, tvmtool

            if host_platform == "linux":
                run_on_server = bool({"sandbox_server", "sandbox1_server"} & groups)
                run_on_proxy = bool({"sandbox_proxy", "sandbox1_proxy"} & groups)
                solomon_metrics = [
                    unified_agent.UnifiedAgent.LEGACY_WEBSERVER_METRIC,
                    unified_agent.UnifiedAgent.ZOOKEEPER_METRIC
                ] if run_on_server else []
                yield host, self.UnifiedAgent(self.key, tvm_secret, run_on_server, run_on_proxy, solomon_metrics)

            for servant in self.scenario:
                # YQL-7246
                if servant == self.Proxy.__servant_name__ and {clickhouse.ctag_for_key(self.key)} & groups:
                    continue

                if servant in (self.TVMTool.__servant_name__, self.UnifiedAgent.__servant_name__):
                    continue

                servant_config = servants.get(servant)
                if servant_config:
                    hosts = it.chain.from_iterable(
                        self.VIRTUAL_GROUPS.get(g, set()) for g in servant_config.get("groups")
                    )
                    servant_groups = servant_config.get("groups") - virtual_groups
                    if not (groups & servant_groups) and host not in hosts:
                        continue

                dc = info.get("location")
                if dc == "VIRTUAL":
                    dc = "UNK"
                object_hash = hashlib.md5()
                for item in (servant, host_platform, repr(sorted(groups)), dc, layout):
                    object_hash.update(str(item))
                object_hash = object_hash.digest()
                servant_object = servant_objects.get(object_hash)
                if servant_object is None:
                    servant_object = servant_objects[object_hash] = base.ServantMeta[servant](
                        self.key, host_platform, list(groups),
                        layout, dc, iface.push_solomon_descr, tvminfo
                    )
                yield host, servant_object

    def port_offset(self):
        return 30000

    def moderators(self):
        return ("staff:yandex_search_devtech_sandbox", "zomb-sandbox")

    @staticmethod
    def newsletter_address():
        return "sandbox-releases@yandex-team.ru"

    def conf(self):
        config = {
            "key": self.key,
            "root_partition": "/opt/sandbox",
            "garbage_limit": "1 gb",
            "environment_settings": {
                "coredumps_limit": "inf b",
            },
            "sb_resource_filter": {
                "attrs": {
                    "released": ("stable", "prestable")[self._is_test],
                },
            },
            "solomon": {
                "sensors_ttl_days": 30,
            },
            "append_samogon_admins_to_moderators": bool(self._is_test),
        }
        if not self._is_yp:
            config.update({
                "hosts": sorted(self.deploy_config.get("hosts", [])),
                "groups": sorted(self.deploy_config.get("groups", [])),
            })
        else:
            config.update({
                "yp_lite_bootstrap": {
                    "only_existing_pods": True,
                    "existing_pods_clusters": ["SAS", "VLA", "MYT"],
                    "labels_selector": {"is_alive": "true"}
                }
            })
            # Remove after SAMOGON-793
            patches = base.YP_LITE_PATCHES[self.key]
            yp_lite_allocation = {
                dc: base.Base.merge_dicts(
                    base.YP_LITE_ALLOCATION, patch
                )
                for dc, patch in patches.items()
            }
            config.update({
                "abc_service_id": 469,
                "yp_lite_replication_policy": {
                    "replication_method": "MOVE",
                },
                "yp_lite_allocation": yp_lite_allocation,
            })
        return config
