import os
import re
import sys
import abc
import pwd
import grp
import json
import copy
import uuid
import time
import shutil
import signal
import socket
import smtplib
import logging
import resource
import tempfile
import textwrap
import platform
import datetime as dt
import itertools as it
import subprocess as sp
import collections

import yaml

from six.moves import reduce

from . import unified_agent

CTAG_PREFIX = "CONDUCTORTAG@"
YP_LITE_TAGS = ["yp_lite@sandbox2", "yp_lite@sandbox3"]

VENV_PACKAGES = {
    # "BuildSandbox" task packs virtual environments in "venv_{platform_name}.tgz" archives,
    # and client then builds a list of environments from the inner dictionary's VALUES.
    # If you put already existing value for a different key (say, "20.04" -> "venv_linux_ubuntu_16.04_xenial.tgz"),
    # it will be lost
    "linux": {
        "12.04": "venv_linux_ubuntu_12.04_precise.tgz",
        "14.04": "venv_linux_ubuntu_14.04_trusty.tgz",
        "16.04": "venv_linux_ubuntu_16.04_xenial.tgz",
        "18.04": "venv_linux_ubuntu_18.04_bionic.tgz",
        "20.04": "venv_linux_ubuntu_20.04_focal.tgz",
        # "14.04_aarch64": "venv_linux_ubuntu_14.04_trusty_aarch64.tgz"
    },
    "darwin": {
        "17": "venv_osx_10.13_high_sierra.tgz",
        "18": "venv_osx_10.14_mojave.tgz",
        "19": "venv_osx_10.15_catalina.tgz",
        "20": "venv_osx_10.16_big_sur.tgz",
        "21": "venv_osx_12_monterey.tgz",
    },
    "cygwin_nt-6.3": {"6.3": "venv_cygwin_6.3.tgz"},
}

SANDBOX_RUNTIME_DIR = "/opt/sandbox/run"
SANDBOX_HOST = "sandbox.yandex-team.ru"

# Default write concern for mission-critical requests like new object creation:
# at least one secondary should confirm modification in maximum 2 seconds
WRITE_CONCERN = dict(
    w=2,            # TODO: SANDBOX-7540: Changing the plugin content to recreate configuration files
    j=False,
    wtimeout=15000  # FIXME: SANDBOX-5979: Temporary increased timeout
)


# Remove after SAMOGON-793
YP_LITE_BASE_VOLUMES = [
    {
        "mount_point": "/place",
        "disk_quota_megabytes": 1024,  # MiB
        "storage_class": "ssd",  # "hdd" or "ssd"
        "bandwidth_guarantee_megabytes_per_sec": 1024,  # MiB/S
        "bandwidth_limit_megabytes_per_sec": 1024  # MiB/S
    },
]


# Remove after SAMOGON-793
YP_LITE_ALLOCATION = {
    "cpu_guarantee": 1000,  # mcores
    "cpu_limit": 1000,  # mcores
    "memory_guarantee": 1024,  # MiB
    "memory_limit": 1028,  # MiB
    "replicas": 1,  # number of instances
    "rootfs": 1024,  # MiB
    "rootfs_storage_class": "ssd",
    "root_bandwidth_guarantee_megabytes_per_sec": 1024,  # MiB/S
    "root_bandwidth_limit_megabytes_per_sec": 1024,  # MiB/S
    "volumes": [],
    "pod_naming_mode": 1,
    "network_macro": "_CMSEARCHNETS_",
    "node_segment_id": "sandbox",
}


# Remove after SAMOGON-793
YP_LITE_PATCHES = {
    2: {
        "SAS": {
            "replicas": 1,
            "volumes": YP_LITE_BASE_VOLUMES,
        },
    },
    3: {
        "SAS": {
            "replicas": 1,
            "volumes": YP_LITE_BASE_VOLUMES,
        },
    },
}


def check_tags(tags, desired_tags=None, undesired_tags=None):
    tags = set(tags)
    return bool(
        (not desired_tags or tags & set(desired_tags)) and
        (not undesired_tags or not (set(undesired_tags) & tags))
    )


def ctag(name):
    return CTAG_PREFIX + name


def unfold_dict(d, base=None):
    return dict(
        it.chain.from_iterable(
            it.product(k, (v,)) if isinstance(k, tuple) else ((k, v),)
            for k, v in it.chain((base or {}).iteritems(), d.iteritems())
        )
    )


def unfold_list(l, base=None):
    result = copy.deepcopy(base or [])
    if base:
        l = unfold_list(l, None)
    for nd in l:
        insert_index = None
        for index, d in enumerate(result):
            for name in d.iterkeys():
                if name in nd:
                    insert_index = index
                    break
        if insert_index is None:
            result.append(unfold_dict(nd))
        else:
            result[insert_index] = unfold_dict(nd, base=result[insert_index])
    return result


def check_subprocess_status(p, logger=None, wrc=None):
    if logger is None:
        logger = logging
    stdout, stderr = map(str.strip, p.communicate())
    if p.returncode:
        msg = "Subprocess failed with code {}. {}".format(
            p.returncode, "Output follows:" if stdout or stderr else "No output taken."
        )
        if stderr:
            msg += "\n{hr}STDERR{hr}\n{out}".format(hr="-" * 40, out=stderr)
        if stdout:
            msg += "\n{hr}STDOUT{hr}\n{out}".format(hr="-" * 40, out=stdout)
        if wrc and p.returncode == wrc:
            logger.warning(msg)
            return True
        logger.error(msg)
    return False


class ServantMeta(abc.ABCMeta):
    # noinspection PyPep8Naming
    class __metaclass__(type):
        _servants = {}  # just for IDE

        def __getitem__(cls, servant_name):
            return cls._servants[servant_name]

    _servants = {}

    def __new__(mcs, name, bases, namespace):
        cls = super(mcs, mcs).__new__(mcs, name, bases, namespace)
        name = cls.__servant_name__
        if name:
            mcs._servants[name] = cls
        return cls

    @property
    def __servant_name__(cls):
        return cls.__dict__.get("__name__")


class CachingResolver(dict):
    def __init__(self, resolver):
        self._resolver = resolver
        super(CachingResolver, self).__init__()

    def __getitem__(self, item):
        if item in self:
            return super(CachingResolver, self).__getitem__(item)
        try:
            ret = self[item] = int(item)
        except ValueError:
            ret = self[item] = self._resolver(item)
        return ret


User = collections.namedtuple("User", "name uid group gid raise_uid_conflict raise_gid_conflict")
User.__new__.__defaults__ = (None,) * len(User._fields)


class Base(object):
    """Base servant"""
    __metaclass__ = ServantMeta

    __cfg_fmt__ = "yaml"

    _CONFIG = ""
    _CONFIG_PATCHES = {}
    _STOP_SIGNAL = signal.SIGINT

    _ROTATE_HOUR = 0
    _ROTATE_KEEP = 14
    _ROTATE_LOGS = []
    _ROTATE_SIGNAL = signal.SIGHUP
    _ROTATE_LOGS_SIZE = 3  # In Gb

    __MAIL_FROM = "sandbox-noreply@yandex-team.ru"
    __MAIL_TO = "sandbox-errors@yandex-team.ru"
    __HOSTNAME = socket.gethostname()

    __yaml_cache = {}
    _config = None
    CYSON_RESOURCE = [
        {
            "type": "naked",
            "source": "sbr:CYSON_FOR_SKYNET_PYTHON",
            "sb_resource_filter": {
                "attrs": {
                    "released": "stable"
                },
                "owner": "SANDBOX"
            },
            "alias": "_cyson_library",
            "unpack_it": False,
        }
    ]

    def __init__(self, key, host_platform, groups, layout, dc, solomon_descr, tvminfo=None):
        self._key = key
        self._host_platform = host_platform
        self._venv_packages = VENV_PACKAGES.get(host_platform, {})
        self._special_groups = groups
        self._layout = layout
        self._dc = dc
        self._solomon_descr = solomon_descr
        self._tvminfo = tvminfo
        self._clsname = self.__class__.__name__
        self._modname = self.__class__.__module__

    @staticmethod
    def _id_from_fqdn(fqdn):
        parts = fqdn.split(".")
        if len(parts) > 2 and parts[0].startswith("bootstrap-sandbox"):
            # for clients deployed to YP, example: bootstrap-sandbox2-1.sas.yp-c.yandex.net => bootstrap-sandbox2-sas-1
            id_parts = parts[0].split("-")
            id_parts.insert(-1, parts[1])
            return "-".join(id_parts)
        else:
            return parts[0]

    @property
    def __cfg__(self):
        if self._config is not None:
            return self._config
        host = socket.getfqdn()
        config = copy.deepcopy(self._load_yaml(self._CONFIG))
        if self._special_groups:
            for config_dict in self._CONFIG_PATCHES:
                for group, patch in config_dict.iteritems():
                    if group in self._special_groups:
                        config = self.merge_dicts(config, self._load_yaml(patch))
        config.setdefault("this", {})["fqdn"] = host
        if self._dc:
            config["this"].setdefault("dc", self._dc.lower())
        config["this"].setdefault("id", self._id_from_fqdn(host))
        self._config = config
        return config

    @__cfg__.setter
    def __cfg__(self, config):
        self._config = config

    def _write_to_config(self, config_section, values):
        section = reduce(
            lambda cfg, key: cfg.setdefault(key, {}),
            config_section.split("."),
            self.__cfg__
        )
        section.update(values)

    def _write_solomon_descr_to_config(self, config_section):
        solomon_conf = {
            "project": self._solomon_descr["project"],
            "cluster": self._solomon_descr["cluster_name"],
            "service": self._solomon_descr["service_name"],
        }
        self._write_to_config(config_section, solomon_conf)

    def _load_yaml(self, source):
        data = self.__yaml_cache.get(source)
        if data is None:
            data = self.__yaml_cache[source] = yaml.load(source) or {}
        return data

    @classmethod
    def merge_dicts(cls, d1, d2):
        for d in (d2, d1):
            if not isinstance(d, dict):
                return d
        result = {k: cls.merge_dicts(d1[k], v) if isinstance(d1.get(k), dict) else v for k, v in d2.iteritems()}
        for k, v in d1.iteritems():
            if k not in d2:
                result[k] = v
        return result

    @property
    def _service_user(self):
        return (
            User(name="zomb-sandbox", gid=1000)
            if sys.platform.startswith("darwin") else
            User(name="zomb-sandbox", group="sandbox", uid=-1, gid=3831)
        )

    @property
    def _sandbox_venv_path(self):
        version = None
        if self._host_platform == "linux":
            version = platform.linux_distribution()[1]
            processor = platform.processor()
            if processor and processor != "x86_64":
                version = "{}_{}".format(version, processor)
        elif self._host_platform in ("freebsd", "darwin"):
            version = platform.release().split(".")[0]
        elif self._host_platform.startswith("cygwin"):
            version = platform.system()[-3:]
        package = self._venv_packages.get(version)
        if not package:
            raise Exception("Cannot find venv package for version {}, host_platform={}, venv_packages={}".format(
                version,
                self._host_platform,
                self._venv_packages
            ))
        return self.get_pack(package)

    @staticmethod
    def _make_symlink(src, dst, force=False, ignore_errors=False):
        try:
            if force and os.path.islink(dst):
                dst_tmp = os.path.join(os.path.dirname(dst), uuid.uuid4().hex)
                logging.info("Recreate symlink %r -> %r", dst, src)
                os.symlink(src, dst_tmp)
                os.rename(dst_tmp, dst)
            else:
                logging.info("Create symlink %r -> %r", dst, src)
                os.symlink(src, dst)
        except OSError:
            logging.error("Cannot create symlink %r -> %r", dst, src)
            if not ignore_errors:
                raise

    @abc.abstractmethod
    def user(self):
        pass

    def packages(self):
        return self.CYSON_RESOURCE + sorted(set(self._venv_packages.values()))

    def stop(self):
        procs = {proc.stat()["uuid"]: proc for proc in self.procs()}
        for proc in procs.itervalues():
            proc.stopRetries()
            proc.signal(self._STOP_SIGNAL)
        while procs:
            time.sleep(1)
            for proc_uuid, proc in procs.items():
                if proc.status():
                    procs.pop(proc_uuid, None)

    def _write_config(self, fname, chmod, chown, template, *args, **kwargs):
        with open(fname, "w") as f:
            f.write(template.format(*args, **kwargs))

        if chmod or chown:
            self.change_permissions(fname, chmod=chmod, chown=chown)

    def config_as_file(self, *args, **kws):
        config_path = super(Base, self).config_as_file(*args, **kws)
        config_dir, config_name = os.path.split(config_path)
        config_base_name, config_ext = (config_name.rsplit(".", 1) + [None])[:2]
        config_alias = (
            ".".join(("current_config", config_ext))
            if config_ext is not None else
            config_base_name
        )
        alias_path = os.path.join(config_dir, config_alias)
        if os.path.exists(alias_path):
            os.remove(alias_path)
        os.symlink(config_name, alias_path)
        return config_path

    @staticmethod
    def _restart_service(name):
        logging.info("Restarting %s", name)
        p = sp.Popen(
            ["systemctl", "restart", name], stdout=sp.PIPE, stderr=sp.PIPE
        )
        if not check_subprocess_status(p):
            logging.info("Service %s successfully restarted", name)

    def _check_service(self, name, pid_fname):
        try:
            with open(pid_fname) as f:
                pids = map(int, f.read().strip().split())
        except Exception as ex:
            logging.error("Cannot read %s: %s", pid_fname, ex)
            self._restart_service(name)
            return False
        else:
            for pid in pids:
                try:
                    os.kill(pid, 0)
                except OSError:
                    logging.warning("Service %s is not executing", name)
                    self._restart_service(name)
                    return False
        return True

    def _rollover(self, logfile, now, yesterday, move, notify, new_date=False):
        dirname, basename = os.path.split(logfile)
        fnames = os.listdir(dirname)

        rotated = []
        by_dates = set()
        prefix = basename + "."
        plen = len(prefix)
        plen_date = plen + 10
        for f in fnames:
            if f[:plen] == prefix:
                rotated.append(os.path.join(dirname, f))
                by_dates.add(os.path.join(dirname, f[:plen_date]))
        rotated.sort()
        dates_sorted = sorted(by_dates)
        gzip_process = None
        to_unlink = set()
        for f in reversed(list(reversed(dates_sorted))[self._ROTATE_KEEP:]):
            to_unlink.add(f)

        full_len = plen_date + len(dirname) + 1
        for f in rotated:
            if f[:full_len] in to_unlink:
                logging.info("Dropping old log file %r", f)
                try:
                    os.unlink(f)
                except (OSError, IOError) as ex:
                    logging.warning("Unable to remove %r: %s", f, str(ex))

        suffix = yesterday.strftime("%Y-%m-%d") if new_date else now.strftime("%Y-%m-%d")
        new_logfile = ".".join([logfile, suffix])
        counts = 0
        for f in rotated:
            if f.startswith(new_logfile):
                counts += 1

        if counts:
            new_logfile += "_{}".format(counts)

        logging.info("Rotating %r log file.", logfile)
        if move:
            shutil.move(logfile, new_logfile)
        else:
            copied_logfile = new_logfile
            try:
                shutil.copy(logfile, copied_logfile)
                self.change_permissions(copied_logfile, chmod=0o664)
            except (OSError, IOError) as ex:
                logging.error("Unable to copy %s to %s: %s", logfile, copied_logfile, ex)
            open(logfile, "w").close()  # Truncate the original log file

        if rotated and not rotated[-1].endswith(".gz"):
            logging.info("Compressing %r log file.", rotated[-1])
            cmd = ["gzip", rotated[-1]]
            gzip_process = sp.Popen(cmd)
            gzip_process.cmd = " ".join(cmd)
        else:
            logging.info("No uncompressed rotated log files detected.")

        if notify:
            for proc in self.procs():
                pid = proc.stat().get("pid")
                if not pid:
                    continue
                pgid = os.getpgid(pid)
                logging.info("Senging signal HUP to the process group %s of the process %s", pgid, pid)
                os.killpg(pgid, self._ROTATE_SIGNAL)

        return gzip_process

    def _rotate_logs(self, move=False, notify=False):
        if self.persist is None:
            self.persist = {}
        self.persist.setdefault("log_rotated", dt.date.min)
        now = dt.datetime.now()
        today = now.date()
        yesterday = today - dt.timedelta(days=1)
        to_rotate = []
        new_date = False
        if (self._ROTATE_HOUR == now.hour and self.persist["log_rotated"] < today):
            to_rotate = self._ROTATE_LOGS
            new_date = True
        else:
            for logfile in self._ROTATE_LOGS:
                try:
                    if os.path.exists(logfile) and os.path.getsize(logfile) >> 30 >= self._ROTATE_LOGS_SIZE:
                        to_rotate.append(logfile)
                except OSError:
                    logging.exception("Skip log file %s", logfile)

        if to_rotate:
            gzip_processes = []
            for l in to_rotate:
                p = self._rollover(l, now, yesterday, move=move, notify=notify, new_date=new_date)
                if p is not None:
                    gzip_processes.append(p)
            for proc in gzip_processes:
                exit_code = proc.wait()
                if exit_code:
                    logging.error("Process '%s' failed with exit code %d.", proc.cmd, exit_code)
            if new_date:
                self.persist["log_rotated"] = today

    def _sendmail(self, subject, text, server_url="localhost", sender=__MAIL_FROM, to=__MAIL_TO):
        if self._key > 0:
            return  # prevent spam from test cluster
        logging.info("Sending email with subject %r", subject)
        headers = "From: {}\r\nTo: {}\r\nSubject: [{}] {}\r\n\r\n".format(sender, to, self.__HOSTNAME, subject)
        message = headers + text
        mail_server = smtplib.SMTP(server_url)
        mail_server.sendmail(sender, to, message)
        mail_server.quit()
        logging.info("Email has been sent to %r", to)

    def ulimit_resources(self):
        if sys.platform.startswith("darwin"):
            return {}
        else:
            return {resource.RLIMIT_NOFILE: (1000000, 1000000)}


class SandboxLauncher(Base):
    __cfg_fmt__ = "yaml"

    _SANDBOX_PKGS = []
    _CONFIG_COPY = None
    _CONFIG_LINK = None

    _CONFIG = textwrap.dedent("""
        common:
          network:
            fastbone: false
          zookeeper:
            enabled: true
            root: "/sandbox"
            hosts: "sandbox-server{{03,11,25}}.search.yandex.net:2181"

          installation: "PRODUCTION"

          statistics:
            enabled: true

          unified_agent:
            %(unified_agent)s

          abcd:
            d_tvm_service_id: "d-production"
            d_api_url: "https://d-api.yandex-team.ru/api/v1"

        server:
          api:
            quotas:
              enabled: true
              check: true
          daemonize: false
          log:
            level: "INFO"

          services:
            packages_updater:
              enabled: true
            clean_resources:
              enabled: true
            statistics_updater:
              enabled: true
            statistics_processor:
              enabled: true
            tasks_statistics:
              enabled: true
            juggler:
              enabled: true
            mailman:
              enabled: true
              whitelist: null
            check_semaphores:
              enabled: true
            cleaner:
              enabled: true
            client_availability_checker:
              enabled: true
            group_synchronizer:
              enabled: true
            tasks_hosts_updater:
              filter_enabled: true

          autoreload: false

          mongodb:
            default_read_preference: "readonly"
            connection_url: "file://${{common.dirs.service}}/.mongodb_rw"
            write_concern:
              %(write_concern)s

          auth:
            enabled: true
            use_blackbox: true

          web:
            rst2html: true
            use_web_helpers: true
            instances: 20
            threads_per_process: 5
            address:
              show_port: false
            static:
              root_path: "{int}/server_tgz/sandbox/web"

          upload:
            tmpdir: "/place/vartmp"

          autorestart:
            delay:
              - 1
              - 60
            timeout: 30

          profiler:
            performance:
              xmlrpc:
                dump_dir: "/var/tmp/sandbox/xmlrpc_prof/"
                threshold: 90000

        client:
          xmlrpc_url: "https://sandbox.yandex-team.ru/sandbox/xmlrpc"
          rest_url: "https://sandbox.yandex-team.ru/api/v1.0"
          daemonize: false
          sandbox_user: "sandbox"
          agentr:
            enabled: false
          fileserver:
            enabled: false
            proxy:
              host: "proxy.%(sandbox_host)s"
            shell:
              enabled: true
          auth:
            oauth_token: "file://~/oauth.token"
            docker_registry_token: "file://~/.docker_registry_token"
          sdk:
            svn:
              arcadia:
                key: "file://~/.arcadia_key"
                force_use_rw: false
            arc:
              token: "file://~/.common_arc_token"
          tasks:
            code_dir: "{srv}/packages/tasks"
          # TODO: remove after ensuring that common.unified_agent is used instead of client.unified_agent in binaries
          unified_agent:
            %(unified_agent)s

        agentr:
          daemon:
            server:
              unix: "${{common.dirs.runtime}}/agentr.sock"
    """ % dict(
        sandbox_host=SANDBOX_HOST,
        write_concern=json.dumps(WRITE_CONCERN),
        unified_agent=unified_agent.ua_sockets_config_for_client,
    ))

    _CONFIG_PATCHES = unfold_list([
        {("sandbox1", "yp_lite@sandbox3"): textwrap.dedent("""
            common:
              zookeeper:
                root: "/sandbox-pre-production"

              installation: "PRE_PRODUCTION"
              solomon:
                pull:
                  cluster: "sandbox_1"
                  service: "sandbox_1"
              mds:
                up:
                  url: "http://storage-int.mdst.yandex.net:1111"
                  tvm_service: "mds-testing"
                dl:
                  url: "https://storage-int.mdst.yandex.net"
                rb:
                  url: "http://rbtorrent.mdst.yandex.net"
                s3:
                  url: "http://s3.mdst.yandex.net"
                  idm:
                    url: "https://s3-idm.mdst.yandex.net"

              abcd:
                d_tvm_service_id: "d-testing"
                d_api_url: "https://d.test.yandex-team.ru/api/v1"

            server:
              log:
                level: "INFO"

              storage_hosts:
                - "sandbox-preprod08"

              services:
                packages_updater:
                  enabled: true
                  release_status: "prestable"
                mailman:
                  whitelist:
                    - "SANDBOX_ACCEPTANCE"  # Sandbox team tracks acceptance status
                    - "SANDBOX_CI_WEB4_PRIEMKA"  # Search Interfaces track acceptance task status
                    - "SANDBOX_LXC_IMAGE"  # Statinfra would like to know when their container is rebuilt
                cleaner:
                  enabled: true
                statistics_processor:
                  clickhouse:
                    database: "sandbox1"
                    cluster: "sandbox1"
                    connection_url: "https://clickhouse-sandbox1.n.yandex-team.ru"
                juggler:
                  enabled: true
                logbroker_publisher:
                  enabled: true
                  tasks_topic_name: "sandbox/preprod/entity_audit/topics/tasks"
                  rollout_percent: 100

              auth:
                oauth:
                  required_scope: ""

              web:
                address:
                  host: "www-sandbox1.n.yandex-team.ru"
                static:
                  root_path: "{int}/serviceapi_tgz/sandbox/web"

              mongodb:
                default_read_preference: "readonly"
                connection_url: "mongodb://localhost:22222/sandbox_restored"

            client:
              xmlrpc_url: "https://www-sandbox1.n.yandex-team.ru/sandbox/xmlrpc"
              rest_url: "https://www-sandbox1.n.yandex-team.ru/api/v1.0"
              fileserver:
                proxy:
                  host: "proxy-sandbox1.n.yandex-team.ru"
              agentr:
                enabled: false
        """)},

        {("sandbox_multios", "sandbox1_multios"): textwrap.dedent("""
            client:
              sandbox_home_cleanup_exclude:
                - "${{client.lxc.dirs.root}}/oauth.token"
              lxc:
                enabled: true
                venv:
            """ + "".join(("""
                  - arch: "{arch}"
                    path: "{{int}}/{dir_name}" """.format(
            arch=re.match(r"venv_(.+)\.tgz", pkg).group(1).replace(".", "_"), dir_name=pkg.replace(".", "_")
        ) for pkg in VENV_PACKAGES["linux"].values())))},

        {("sandbox_porto", "sandbox1_porto", "yp_lite@sandbox2", "yp_lite@sandbox3"): textwrap.dedent("""
            client:
              porto:
                network:
                  type: "VETH"
                enabled: true
                venv:
            """ + "".join(("""
                  - arch: "{arch}"
                    path: "{{int}}/{dir_name}" """.format(
            arch=re.match(r"venv_(.+)\.tgz", pkg).group(1).replace(".", "_"), dir_name=pkg.replace(".", "_")
        ) for pkg in VENV_PACKAGES["linux"].values())))},

        {tuple(YP_LITE_TAGS): textwrap.dedent("""
            client:
              porto:
                network:
                  veth_vlan: "veth"
        """)},

        {("sandbox_stg", "sandbox1_stg"): textwrap.dedent("""
            agentr:
              daemon:
                maintain:
                  limits:
                    extra_local: 50000
                    extra_actual_local: 75000
                    extra_remote: 5000
        """)},

        {("sandbox_proxy", "sandbox1_proxy"): textwrap.dedent("""
            common:
              unified_agent:
                {unified_agent}
        """.format(unified_agent=unified_agent.ua_sockets_config_for_proxy))},

        {"sandbox_server": textwrap.dedent("""
            client:
              idle_time: 10
              auto_cleanup:
                free_space_threshold: 122880  # reserved for MongoDB backups
            server:
              api:
                port: 8080
                workers: 200
        """)},

        {"sandbox1_server": textwrap.dedent("""
            server:
              api:
                port: 8080
                workers: 64
        """)},

        {("sandbox1_client", "sandbox_client"): textwrap.dedent("""
            client:
              auto_cleanup:
                free_space_threshold: 0.33
        """)},

        {("sandbox1_multislot", "yp_lite@sandbox3"): textwrap.dedent("""
            client:
              auto_cleanup:
                free_space_threshold: 0.33
        """)},

        {("sandbox_macos", "sandbox1_macos"): textwrap.dedent("""
            common:
              dirs:
                service: "/Users/zomb-sandbox"
            client:
              sandbox_user: "isandbox"
              sdk:
                svn:
                  confdir: "/Users/isandbox/.subversion"
        """)},

        {("sandbox_windows", "sandbox1_windows"): textwrap.dedent("""
            common:
              walle:
                token: ""
            client:
              dirs:
                data: "/mnt/c/place/sandbox-data"
              executor:
                log:
                  root: "/mnt/c/logs"
                dirs:
                  run: "/mnt/c/run"
            agentr:
              daemon:
                server:
                  host: ""
                  port: 13580
                  unix: null
        """)},

        {("sandbox_multislot", "yp_lite@sandbox2"): textwrap.dedent("""
            client:
              auto_cleanup:
                free_space_threshold: 0.25
        """)},

        {"sandbox_rem": textwrap.dedent("""
            client:
              max_job_slots: 64
        """)},

        {"mac_browser_sandbox": textwrap.dedent("""
            client:
              auto_cleanup:
                free_space_threshold: 0.1
        """)},

        {"sandbox_browser_twoslots": textwrap.dedent("""
            client:
              max_job_slots: 2
        """)},

        {"sandbox_browser_linux_testing": textwrap.dedent("""
            common:
              zookeeper:
                enabled: false
            client:
              max_job_slots: 16
        """)},

        {"sandbox_browser_linux": textwrap.dedent("""
            common:
              zookeeper:
                enabled: false
            client:
              sandbox_user: "teamcity"
        """)},

        {"virtual_browser_experimental": textwrap.dedent("""
            client:
              sandbox_user: "sandbox"
        """)},

        {"search_instrum-sandbox_antispam_multislot": textwrap.dedent("""
            client:
              max_job_slots: 5
        """)},
    ])

    _SYSDEPS = {
        "deb": [
            "yandex-coroner",
            "yandex-search-common",
            "yandex-search-common-apt",
            "yandex-search-common-sysctl",  # SANDBOX-3405
            "yandex-search-common-rsyslog",  # SANDBOX-4503
            "config-juggler-search",  # juggler agent deployment moved from cfengine to samogon
            "juggler-client",  # JUGGLERSUPPORT-234: Force juggler-client update to latest stable version
            "yandex-environment-intranet",
            "yandex-internal-root-ca",
            "wall-e-checks-bundle",  # SANDBOX-6964
            "yandex-wall-e-agent",
            "yandex-hw-watcher",
            "nvme-cli=1.4-1",  # required by yandex-hw-watcher but apt is dumb to install required version by dependency
            "yandex-cauth",  # SANDBOX-4558
            "yandex-search-hw-watcher-walle-config",  # Wall-E integration with hw-watcher
            "yandex-hbf-agent-static",  # SANDBOX-5247
            "yandex-hbf-agent-init",  # SANDBOX-5247
            "atop",
            "lsscsi",
            "m4",
            "rsync",  # SANDBOX-8718
            "nano",
            "ntpdate",
            "tshark",
            "wget",
            "curl",
            "tzdata",
            "openssl",
            "vmtouch",
            "patch",
            "ncdu",  # Service tool
            "dupload",
            "fuse",  # SANDBOX-5262
        ]
    }

    _SYSDEPS_PATCHES = unfold_dict({
        "sandbox_server": {
            "deb": [
                "config-caching-dns",
                "yandex-environment-production",
                "yandex-yasmagent",
                "statbox-push-client-daemon",
                "pixz",  # SANDBOX-4732
            ]
        },
        ("sandbox1_server", "sandbox_proxy", "sandbox1_proxy"): {
            "deb": [
                "config-caching-dns"
            ]
        },
        "sandbox_client": {
            "deb": [
                "yandex-environment-production",
                "yandex-netconfig",
                "yandex-push-client=6.73.1",
            ]
        },
        "sandbox1": {
            "deb": [
                "yandex-environment-prestable",
                "yandex-netconfig",
                "yandex-push-client=6.73.1",
            ]
        },
        ("sandbox_client_generic", "sandbox1_client"): {
            "deb": [
                "realpath",
                "yandex-yasmagent",         # GOLOVANSRE-245 need version >=1.291-1
            ]
        },
        ("sandbox_multios", "sandbox1_multios"): {
            "deb": [
                "lxc",
            ]
        },
        "sandbox_stg": {
            "deb": [
                "yandex-netconfig",
                "pixz",
            ]
        },
        "sandbox_zk": {  # SANDBOX-4539 - Deploy zookeeper package and configuration files
            "deb": [
                "zookeeper=3.4.8-1",
            ]
        },
        "sandbox_focal": {
            "deb": [
                "bc",
                "binutils",
                "bsd-mailx",
                "curl",
                "daemon",
                "dbus",
                "debsums",
                "dmidecode",
                "dnsutils",
                "dstat",
                "edac-utils",
                "ethtool",
                "file",
                "git",
                "git-core",
                "hdparm",
                "htop",
                "hwloc-nox",
                "iotop",
                "iperf",
                "ipmitool",
                "ipset",
                "iptables",
                "iptraf",
                "language-pack-en",
                "less",
                "lldpd",
                "lm-sensors",
                "logrotate",
                "lsof",
                "lsscsi",
                "ltrace",
                "mc",
                "mcelog",
                "mdadm",
                "ndisc6",
                "netcat-openbsd",
                "net-tools",
                "ngrep",
                "ntp",
                "openipmi",
                "openssh-server",
                "pciutils",
                "postfix",
                "psmisc",
                "pv",
                "python3",
                "python2.7",
                "rsync",
                "screen",
                "smartmontools",
                "strace",
                "subversion",
                "sudo",
                "sysstat",
                "systemtap",
                "tcpdump",
                "tcsh",
                "telnet",
                "time",
                "tmux",
                "traceroute",
                "tree",
                "vim",
                "vlan",
                "xfsprogs",
                "yandex-cauth",
                "yandex-config-hostsaccess",
                "yandex-coroner",
                "yandex-search-common-apt",
                "yandex-search-common-rsyslog",
                "yandex-search-common-settings",
                "yandex-search-user-monitor",
                "yandex-search-user-root",
                "yandex-search-user-skynet",
                "yandex-search-user-tcpdump",
                "yandex-ubuntu-archive-apt",
                "zsh",
            ]
        },
    })

    # HOSTMAN-215 - Deny any packages installation on RTC hosts
    _SYSDEPS_EXCLUDE_ALL = ["sandbox_porto", "sandbox1_porto"]
    # SANDBOX-7624 - Do not install any packages on WSL hosts
    _SYSDEPS_EXCLUDE_ALL += ["sandbox_windows", "sandbox1_windows"]
    # do not install any packages on YP hosts
    _SYSDEPS_EXCLUDE_ALL += YP_LITE_TAGS
    _SYSDEPS_EXCLUDE = unfold_dict({
        "sandbox_focal": {
            "deb": [
                "realpath",
                "yandex-search-common",
                "nvme-cli=1.4-1",
            ]
        },
    })

    def update_sandbox_data(self, config):
        config_tags = config.get("client", {}).get("tags", [])
        if {"OSX_BIG_SUR", "OSX_MONTEREY"} & set(config_tags):
            config["client"].setdefault("dirs", {})["data"] = "/usr/local/place/sandbox-data"

    @staticmethod
    def extradirs():
        return ["{srv}/packages", "{srv}/downloads"]

    @staticmethod
    def _filter_packages(pkgs):
        """returns a list of packages with the latest version mentioned"""
        package_to_ver = dict()
        for pkg in pkgs:
            package, version = (pkg.split("=", 1) + [None])[:2]
            package_to_ver[package] = version

        versioned_packages = []
        for pkg in pkgs:
            package, version = (pkg.split("=", 1) + [None])[:2]
            if package in package_to_ver and package_to_ver[package] == version:
                versioned_packages.append(pkg)
                del package_to_ver[package]
        return versioned_packages

    @property
    def _sandbox_user(self):
        user = self.__cfg__["client"]["sandbox_user"]
        if sys.platform.startswith("darwin"):
            # On OS X sandbox user should be created upon host deploy (SANDBOX-5230)
            return User(name=user, uid=-1, gid=1000)
        return User(name=user, uid=3831, group="sandbox", gid=3831)

    def additional_users(self):
        return {
            self._service_user.name: dict(
                filter(lambda _: _[0] != "name", self._service_user._asdict().iteritems()),
                create_home=True
            ),
            self._sandbox_user.name: dict(
                filter(lambda _: _[0] != "name", self._sandbox_user._asdict().iteritems()),
                create_home=True
            )
        }

    def sysdeps(self):
        deps = copy.deepcopy(self._SYSDEPS)
        if self._special_groups:
            for group in self._special_groups:
                if group in self._SYSDEPS_EXCLUDE_ALL:
                    return {"deb": []}
                # Add some group-specific packages
                patch = self._SYSDEPS_PATCHES.get(group)
                if patch:
                    for pkg_type, pkgs in patch.iteritems():
                        deps.setdefault(pkg_type, []).extend(pkgs)
            # Exclude some group-specific packages
            for group in self._special_groups:
                patch = self._SYSDEPS_EXCLUDE.get(group)
                if patch:
                    for pkg_type, pkgs in patch.iteritems():
                        for pkg in pkgs:
                            deps.setdefault(pkg_type, []).remove(pkg)
        for pkg_type, pkg in deps.iteritems():
            deps[pkg_type] = self._filter_packages(pkg)
        return deps

    def postinstall(self):
        usr = self._sandbox_user
        if sys.platform.startswith("darwin"):
            patch = "#includedir /etc/sudoers.d"
            patched = False
            output = []
            sudoers = "/etc/sudoers"
            with open(sudoers) as f:
                for line in f:
                    if line.strip() == patch:
                        patched = True
                    output.append(line)
            if not patched:
                output.append("\n{}\n".format(patch))
                fd, sudoers_tmp = tempfile.mkstemp(prefix=sudoers)
                os.close(fd)
                with open(sudoers_tmp, "w") as f:
                    f.write("".join(output))
                self.change_permissions(sudoers_tmp, 0o440)
                os.rename(sudoers_tmp, sudoers)
        elif (
            sys.platform.startswith("linux") and
            "Microsoft" not in platform.platform() and
            check_tags(self._special_groups, undesired_tags=YP_LITE_TAGS)
        ):
            with open("/proc/sys/kernel/shmmax", "w") as fh:
                fh.write("68719476736")
            with open("/proc/sys/kernel/shmall", "w") as fh:
                fh.write("16777216")

        uid = pwd.getpwnam(usr.name).pw_uid
        gid = grp.getgrnam(usr.group).gr_gid if usr.group else usr.gid
        for _ in self.extradirs():
            _ = self.substval(_)
            os.chown(_, uid, gid)
            os.chmod(_, 0o775)

    @abc.abstractmethod
    def start(self):
        pass
