import os
import re
import errno
import socket
import logging
import textwrap
import datetime as dt
import subprocess as sp
import collections as cs

import requests

from . import base
from . import utils
from .utils import clickhouse


ZOOKEEPER_CTAG = "sandbox_zk"
ZOOKEEPER_HOSTS_EXPR = base.ctag(ZOOKEEPER_CTAG)

# To figure what every parameter means, head over to https://clickhouse.yandex/docs/en/operations/server_settings/
# YQL-7246: backends and balancer should listen on the same port for "YQL over ClickHouse" tools to work
CONFIG_TEMPLATE = textwrap.dedent(
    """
    <yandex>
        <listen_host>::</listen_host>
        <http_port>{http_port}</http_port>
        <path>{db_path}</path>
        <tmp_path>{tmp_data_path}</tmp_path>
        <default_database>{default_database}</default_database>
        <system_profile>{su_profile}</system_profile>

        <logger>
            <level>debug</level>
        </logger>

        <merge_tree>
            <!-- see CLICKHOUSE-4024 and https://clickhouse.yandex/docs/en/operations/server_settings/settings/ -->
            <use_minimalistic_part_header_in_zookeeper>1</use_minimalistic_part_header_in_zookeeper>
        </merge_tree>

        <remote_servers>
            <{cluster_name}>
                {shards_block}
            </{cluster_name}>
        </remote_servers>

        <zookeeper>
            {zk_blocks}
        </zookeeper>

        <macros>
            <shard>{shard_name}</shard>
            <replica>{replica_name}</replica>
        </macros>
    </yandex>
    """
).strip()

USERS_TEMPLATE = textwrap.dedent(
    """
    <yandex>
        <profiles>
            <common>
                <max_memory_usage>20000000000</max_memory_usage>
                <max_query_size>1048576</max_query_size>
                <insert_distributed_sync>1</insert_distributed_sync>
                <add_http_cors_header>1</add_http_cors_header>
                <log_queries>1</log_queries>
                <max_ast_elements>100000</max_ast_elements>
            </common>
            <{default_profile}>
                <profile>common</profile>
                <readonly>2</readonly>
            </{default_profile}>
            <{su_profile}>
                <profile>common</profile>
                <readonly>0</readonly>
            </{su_profile}>
        </profiles>
        <users>
            {users_block}
        </users>
    </yandex>
    """
).strip()

CLIENT_CONFIG_TEMPLATE = textwrap.dedent(
    """
    <config>
        <user>{su_login}</user>
        <password>{su_password}</password>
    </config>
    """
).strip()

# Listening on ports lower than 1024 requires either of root privilegies and special capability
CAPABILITIES = [
    "CAP_NET_ADMIN",
    "CAP_NET_BIND_SERVICE",
    "CAP_SYS_NICE",
]
SYSTEMD_EXTENSION_TEMPLATE = textwrap.dedent(
    """
    [Service]
    CapabilityBoundingSet={}
    """
).strip().format(" ".join(CAPABILITIES))

ClusterSettings = cs.namedtuple("ClusterSettings", ("ctag", "hosts_expr", "default_db", "cluster"))
UserSettings = cs.namedtuple("UserSettings", ("user", "profile", "quota", "password", "access_management"))


def ctag_for_key(key):
    return "{}_clickhouse".format(utils.prefix_for_key(key))


class ClickHouse(base.Base):
    __name__ = "clickhouse"
    __cfg_fmt__ = "raw"  # gonna be converted into XML later on

    _LOGFILES = [
        "/".join(("/var/log/clickhouse-server", logname))
        for logname in ["clickhouse-server.log", "clickhouse-server.err.log", "stdout", "stderr"]
    ]
    _ROTATE_LOGS = _LOGFILES[0:2]

    INTERSERVER_PORT = 9000
    HTTP_PORT = 80
    # Time to wait for reply from clickhouse-server's HTTP interface, seconds
    __PING_TIMEOUT = 5

    # Time to ping clickhouse-server for before declaring it dead, seconds
    # This value should be relatively high, as ClickHouse doesn't respond via HTTP until replication is done.
    __RESTART_TIMEOUT = 300

    __CLICKHOUSE_SHARD_TAG_RE = re.compile(r"sandbox\d?_clickhouse_s(?P<shard_no>\d)")

    __CONFIG_FILE_NAME = "sandbox.xml"
    __CONFIG_DIRECTORY = "/etc/clickhouse-server/conf.d"

    __CLIENT_CONFIG_FILE_NAME = "config.xml"
    __CLIENT_CONFIG_DIRECTORY = "/home/zomb-sandbox/.clickhouse-client"

    __USERS_FILE_NAME = "users.xml"
    __USERS_DIRECTORY = "/etc/clickhouse-server/users.d"

    DB_PATH = "/place/clickhouse/data/"
    TMP_DATA_PATH = "/place/clickhouse/tmp/"

    SU_PROFILE = "su"
    DEFAULT_PROFILE = "default"

    DEFAULT_QUOTA = "default"

    SU_LOGIN = "sandbox"
    SU_PASSWORD_PATH = "/home/zomb-sandbox/.clickhouse_password"

    YQL_LOGIN = "yql"

    DATALENS_LOGIN = "datalens"
    DATALENS_PASSWORD_PATH = "/home/zomb-sandbox/.clickhouse_datalens_password"
    SYSTEMD_SERVICE_EXTENSION_FILE = "/etc/systemd/system/clickhouse-server.service.d/network-cap-override.conf"

    # For settings block below, 0 means production (Samogon cluster key), 1 means preproduction

    # WARNING: you're gonna need to reshard the data on the whole cluster if you change shard count.
    # Details: https://clickhouse.yandex/docs/ru/single/#table-engines-resharding

    # You also need to create all the necessary tables on the new replicas if you change replica count,
    # or, alternatively, run a distributed query.
    # Details: https://clickhouse.yandex/docs/ru/single/index.html#ddl-on-cluster

    PACKAGE_VERSION = "20.9.3.45"

    settings = {
        0: ClusterSettings(
            ctag=ctag_for_key(0),
            hosts_expr=base.ctag(ctag_for_key(0)),
            default_db=utils.prefix_for_key(0),
            cluster=utils.prefix_for_key(0),
        ),
        1: ClusterSettings(
            ctag=ctag_for_key(1),
            hosts_expr=base.ctag(ctag_for_key(1)),
            default_db=utils.prefix_for_key(1),
            cluster=utils.prefix_for_key(1),
        ),
    }

    @property
    def _su_password(self):
        return open(self.SU_PASSWORD_PATH).read().strip()

    @property
    def _datalens_password(self):
        if not os.path.exists(self.DATALENS_PASSWORD_PATH):
            return ""
        return open(self.DATALENS_PASSWORD_PATH).read().strip()

    def make_config(self):
        """
        As config requires password file's existence, please do not call this function before unpacking the layout.
        """

        name2shard, shard2name = {}, {}
        clickhouse_fqdns = utils.sort_hosts(utils.resolve_ctag(self.settings[self._key % 2].ctag))

        for fqdn in clickhouse_fqdns:
            matches = filter(
                None,
                map(
                    lambda tag: re.match(self.__CLICKHOUSE_SHARD_TAG_RE, tag),
                    utils.host_tags(fqdn)
                )
            )
            if len(matches) > 1:
                raise RuntimeError("Host {} has multiple shard-defining tags, which is unacceptable".format(fqdn))

            match = next(iter(matches), None)
            if match is None:
                raise RuntimeError(
                    "Incorrect cluster layout -- shard-defining tag is missing on {}".format(fqdn)
                )

            shard_no = int(match.group("shard_no"))
            name2shard[fqdn] = shard_no
            shard2name.setdefault(shard_no, []).append(fqdn)

        zk_hosts = utils.sort_hosts(utils.resolve_ctag(ZOOKEEPER_CTAG))
        zk_blocks = "\n".join((
            clickhouse.zk_block(index, fqdn)
            for index, fqdn in enumerate(zk_hosts, start=1)
        ))

        shards_block = "\n".join((
            clickhouse.shard_block(fqdns, self.INTERSERVER_PORT, self.SU_LOGIN, self._su_password)
            for _, fqdns in sorted(shard2name.items())
        ))

        host = socket.getfqdn()
        replaced_config = CONFIG_TEMPLATE.format(
            http_port=self.HTTP_PORT,
            db_path=self.DB_PATH,
            tmp_data_path=self.TMP_DATA_PATH,
            default_database=self.settings[self._key].default_db,
            shards_block=shards_block,
            zk_blocks=zk_blocks,
            cluster_name=self.settings[self._key].cluster,
            shard_name="shard{}".format(name2shard[host]),
            replica_name=host,
            su_profile=self.SU_PROFILE,
        )
        return utils.prettify_xml(replaced_config)

    def config_as_file(self):
        self.__cfg__ = self.make_config()
        return super(ClickHouse, self).config_as_file()

    def sysdeps(self):
        return {
            "repos": [
                "deb http://repo.yandex.ru/clickhouse/deb/stable/ main/"
            ],
            "deb": [
                "clickhouse-server={}".format(self.PACKAGE_VERSION),
                "clickhouse-common-static={}".format(self.PACKAGE_VERSION),
                "clickhouse-client={}".format(self.PACKAGE_VERSION),
            ]
        }

    def extradirs(self):
        return [
            {
                "path": path,
                "chmod": 0755,
                "chown": self.user(),
                "recursive": True,
                "do_not_remove": True,
            }
            for path in [self.DB_PATH, self.TMP_DATA_PATH, self.__CONFIG_DIRECTORY, self.__USERS_DIRECTORY]
        ]

    def user(self):
        return "clickhouse"

    def balancer(self):
        return {
            "domain": "clickhouse",
            "location": "/",
            "port": self.HTTP_PORT,
            "balancing_options": {
                "backend_timeout": "60s",
                "balancer_retries_timeout": "120s",
            }
        }

    @staticmethod
    def logfiles():
        return [
            {
                "path": logpath,
                "external": True
            }
            for logpath in ClickHouse._LOGFILES
        ]

    def ensure_config(self):
        config_path = self.config_as_file()
        self.change_permissions(config_path, chmod=0400, chown=self.user())
        destination = os.path.join(self.__CONFIG_DIRECTORY, self.__CONFIG_FILE_NAME)
        self._make_symlink(config_path, destination, force=True, ignore_errors=True)

    def ensure_client_config(self):
        if not os.path.exists(self.__CLIENT_CONFIG_DIRECTORY):
            os.mkdir(self.__CLIENT_CONFIG_DIRECTORY)

        with open(os.path.join(self.__CLIENT_CONFIG_DIRECTORY, self.__CLIENT_CONFIG_FILE_NAME), "w") as client_config:
            client_config.write(self.make_client_config())

    def make_client_config(self):
        formatted_client_config = CLIENT_CONFIG_TEMPLATE.format(
            su_login=self.SU_LOGIN,
            su_password=self._su_password,
        )
        return utils.prettify_xml(formatted_client_config)

    def ping(self):
        def is_http_interface_available():
            try:
                response = requests.get("http://localhost:{}".format(self.HTTP_PORT), timeout=self.__PING_TIMEOUT)
                response.raise_for_status()
                return True
            except (requests.ConnectionError, requests.HTTPError) as exc:
                logging.error("HTTP interface is down: %s", exc)
                return False

        self._rotate_logs(move=False, notify=False)
        if self.persist is None:
            self.persist = {}

        utcnow = dt.datetime.utcnow()
        successful_ping = self.persist.get("successful_ping", utcnow)

        delta = (utcnow - successful_ping).total_seconds()
        alive = is_http_interface_available()
        if alive:
            self.persist["successful_ping"] = utcnow

        logging.info(
            "Alive: %s / Last successful ping: %s UTC",
            alive, self.persist.get("successful_ping", "never")
        )
        return alive or delta <= self.__RESTART_TIMEOUT

    def __extend_capabilities(self):
        try:
            os.makedirs(os.path.dirname(self.SYSTEMD_SERVICE_EXTENSION_FILE))
        except OSError as exc:
            if exc.errno == errno.EEXIST:
                pass

        with open(self.SYSTEMD_SERVICE_EXTENSION_FILE, "w") as fileobj:
            fileobj.write(SYSTEMD_EXTENSION_TEMPLATE)

        # setcap only recognizes capabilities in lowercase. =+ doesn't reset already existing capabilities
        capabilities = ",".join(_.lower() for _ in CAPABILITIES)
        sp.check_call(["setcap", "{}=+ep".format(capabilities), "/usr/bin/clickhouse"])
        sp.check_call(["systemctl", "daemon-reload"])

    def start(self):
        self.ensure_config()
        self.__extend_capabilities()
        sp.check_call(["service", "clickhouse-server", "restart"])
        self.create(["/bin/sleep", "infinity"])

    @property
    def clickhouse_users(self):
        return (
            UserSettings(
                user=self.SU_LOGIN, profile=self.SU_PROFILE, quota=self.DEFAULT_QUOTA, password=self._su_password,
                access_management=1
            ),
            UserSettings(
                user=self.YQL_LOGIN, profile=self.DEFAULT_PROFILE, quota=self.DEFAULT_QUOTA, password=None,
                access_management=0
            ),
            UserSettings(
                user=self.DATALENS_LOGIN, profile=self.DEFAULT_PROFILE, quota=self.DEFAULT_QUOTA,
                password=self._datalens_password, access_management=0
            ),
        )

    def make_users(self):
        users_block = "\n".join((
            clickhouse.user_block(user, profile, quota, password, access_management)
            for user, profile, quota, password, access_management in self.clickhouse_users
        ))

        formatted_users = USERS_TEMPLATE.format(
            users_block=users_block,
            default_profile=self.DEFAULT_PROFILE,
            su_profile=self.SU_PROFILE,
        )
        return utils.prettify_xml(formatted_users)

    def setup_users(self):
        """
        Setup users, their profiles and quotas.
        All users, except `sandbox`, are only allowed to read from the database
        """

        if not os.path.exists(self.__USERS_DIRECTORY):
            os.mkdir(self.__USERS_DIRECTORY)

        with open(os.path.join(self.__USERS_DIRECTORY, self.__USERS_FILE_NAME), "w") as users_settings:
            users_settings.write(self.make_users())

    def postinstall(self):
        for log_dict in self.logfiles():
            path = log_dict["path"]
            if not os.path.exists(path):
                open(path, "a").close()
            self.change_permissions(path, chmod=0774, chown=self.user())
        self.ensure_config()
        self.setup_users()
        self.ensure_client_config()

    def stop(self):
        sp.check_call(["service", "clickhouse-server", "stop"])
