import os
import re
import sys
import pwd
import glob
import time
import yaml
import shutil
import signal
import base64
import socket
import random
import tarfile
import logging
import datetime as dt
import textwrap
import tempfile
import platform
import itertools as it
import subprocess as sp
import multiprocessing as mp

from . import base
from . import utils
from . import disk_devices

# Total RAM on host, in MiB
TOTAL_RAM = (os.sysconf("SC_PAGE_SIZE") * os.sysconf("SC_PHYS_PAGES")) >> 20
# Total CPU cores on host
TOTAL_CORES = mp.cpu_count()
# RAM shared by dynamic slots, 1GiB reserved for sandbox processes, in MiB
SHARED_RAM_LXC = TOTAL_RAM - 1024
# same as for SHARED_RAM_LXC, minus 2GiB margin, see https://rtc.yandex-team.ru/docs/containers/porto-examples#memory
SHARED_RAM_PORTOD = SHARED_RAM_LXC - 2048
# CPU cores shared by dynamic slots, 1 core reserved
SHARED_CORES = TOTAL_CORES - 1
# Nanny infra tax in cores from total machine cores
YP_INFRA_TAX = 3


class RsyncLauncher(object):
    __metaclass__ = base.ServantMeta
    __name__ = "rsync_launcher"

    __MEMORY_LIMIT = "8G"  # Memory limit for servant's cgroup

    __CONFIGS = {
        "linux": textwrap.dedent("""
            uid                     = nobody
            gid                     = nogroup

            use chroot              = yes
            strict modes            = yes
            incoming chmod          = Dg+rwx,Fg+rw

            log file                = {srv}/log/rsyncd.log
            transfer logging        = yes

            exclude                 = lost+found/
            ignore nonreadable      = yes
            dont compress           = *.gz *.tgz *.zip *.z *.Z *.xz *.bz2

            log format = %h %o %f %l %b

            [sandbox-tasks]
                    uid = root
                    gid = root
                    use chroot = no
                    max connections = 100
                    path = /place/sandbox-data/tasks
                    read only = true

            [coredumps]
                    uid = root
                    gid = root
                    use chroot = no
                    max connections = 100
                    path = /coredumps
                    read only = true
        """),
        "windows": textwrap.dedent("""
            uid                     = nobody
            gid                     = nogroup

            use chroot              = yes
            strict modes            = yes
            incoming chmod          = Dg+rwx,Fg+rw

            log file                = {srv}/log/rsyncd.log
            transfer logging        = yes

            exclude                 = lost+found/
            ignore nonreadable      = yes
            dont compress           = *.gz *.tgz *.zip *.z *.Z *.xz *.bz2

            log format = %h %o %f %l %b

            [sandbox-tasks]
                    uid = root
                    gid = root
                    use chroot = no
                    max connections = 100
                    path = /mnt/c/place/sandbox-data/tasks
                    read only = true

            [coredumps]
                    uid = root
                    gid = root
                    use chroot = no
                    max connections = 100
                    path = /coredumps
                    read only = true
            """),
        "linux_new_layout": textwrap.dedent("""
            uid                     = nobody
            gid                     = nogroup

            use chroot              = yes
            strict modes            = yes
            incoming chmod          = Dg+rwx,Fg+rw

            log file                = {srv}/log/rsyncd.log
            transfer logging        = yes

            exclude                 = lost+found/
            ignore nonreadable      = yes
            dont compress           = *.gz *.tgz *.zip *.z *.Z *.xz *.bz2

            log format = %h %o %f %l %b

            [sandbox-resources]
                    uid = root
                    gid = root
                    use chroot = no
                    max connections = 1000
                    path = /place/sandbox-data/resources
                    read only = true

            [coredumps]
                    uid = root
                    gid = root
                    use chroot = no
                    max connections = 100
                    path = /coredumps
                    read only = true
        """),
        "storage": textwrap.dedent("""
            uid                     = nobody
            gid                     = nogroup

            use chroot              = yes
            strict modes            = yes
            incoming chmod          = Dg+rwx,Fg+rw

            log file                = {srv}/log/rsyncd.log
            transfer logging        = yes

            exclude                 = lost+found/
            ignore nonreadable      = yes
            dont compress           = *.gz *.tgz *.zip *.z *.Z *.xz *.bz2

            log format = %h %o %f %l %b

            [sandbox-tasks]
                    uid = root
                    gid = root
                    use chroot = no
                    max connections = 1000
                    path = /storage/sandbox-tasks
                    read only = true

            [coredumps]
                    uid = root
                    gid = root
                    use chroot = no
                    max connections = 100
                    path = /coredumps
                    read only = true
        """),
        "freebsd": textwrap.dedent("""
            uid = root
            gid = wheel
            transfer logging = true
            log format = %h %o %f %l %b
            log file = {srv}/log/rsyncd.log
            [sandbox-tasks]
                    path = /place/sandbox-data/tasks
                    read only = true
            [coredumps]
                    path = /coredumps
                    read only = true
        """),
        "darwin": textwrap.dedent("""
            uid = root
            gid = wheel
            transfer logging = true
            log format = %h %o %f %l %b
            log file = {srv}/log/rsyncd.log
            [sandbox-tasks]
                    path = /place/sandbox-data/tasks
                    read only = true
            [coredumps]
                    path = /coredumps
                    read only = true
        """),
        "cygwin": textwrap.dedent("""
            uid                     = nobody
            gid                     = None

            use chroot              = yes
            strict modes            = yes
            incoming chmod          = Dg+rwx,Fg+rw

            log file                = {srv}/log/rsyncd.log
            transfer logging        = yes

            exclude                 = lost+found/
            ignore nonreadable      = yes
            dont compress           = *.gz *.tgz *.zip *.z *.Z *.xz *.bz2

            log format = %h %o %f %l %b

            [sandbox-tasks]
                    uid = root
                    gid = root
                    use chroot = no
                    max connections = 100
                    path = /place/sandbox-data/tasks
                    read only = true

            [coredumps]
                    uid = root
                    gid = root
                    use chroot = no
                    max connections = 100
                    path = /coredumps
                    read only = true
        """),
    }

    def __init__(self, key, _, groups, *__):
        self._key = key
        self.__is_semistorage = "sandbox_stg_semi" in groups
        self.__is_new_layout = "sandbox_stg" in groups or "sandbox1_stg" in groups

    @staticmethod
    def user():
        return "root"

    @staticmethod
    def extradirs():
        return ["{srv}/log"]

    @staticmethod
    def logfiles():
        return ["{srv}/log/rsyncd.log"]

    @classmethod
    def cgroup_settings(cls):
        return {"memory": {"memory.limit_in_bytes": cls.__MEMORY_LIMIT}}

    @property
    def __cfg__(self):
        key = "unknown"
        if sys.platform.startswith("linux"):
            key = "linux"
            if "Microsoft" in platform.platform():
                key = "windows"
            elif self.__is_semistorage:
                key = "storage"
            elif self.__is_new_layout:
                key = "linux_new_layout"
        elif sys.platform.startswith("freebsd"):
            key = "freebsd"
        elif sys.platform.startswith("darwin"):
            key = "darwin"
        elif sys.platform.startswith("cygwin"):
            key = "cygwin"
        return self.__CONFIGS.get(key, "")

    def start(self):
        rsync_path = None
        if sys.platform.startswith("linux") or sys.platform.startswith("darwin"):
            rsync_path = "/usr/bin/rsync"
        elif sys.platform.startswith("freebsd"):
            rsync_path = "/usr/local/bin/rsync"
        if rsync_path:
            self.create([rsync_path, "--daemon", "--no-detach", "--config", self.config_as_file()])

    @staticmethod
    def postinstall():
        if sys.platform.startswith("linux"):
            os.system("/usr/sbin/update-rc.d -f rsync remove && /etc/init.d/rsync stop")
        elif sys.platform.startswith("freebsd"):
            changed = False
            output = []
            inetd_conf = "/etc/inetd.conf"
            with open(inetd_conf) as f:
                for line in f:
                    if line.startswith("rsync"):
                        changed = True
                        line = "#{}".format(line)
                    output.append(line)
            if changed:
                inetd_conf_tmp = inetd_conf + "~"
                with open(inetd_conf_tmp, "w") as f:
                    f.write("".join(output))
                os.rename(inetd_conf_tmp, inetd_conf)
                os.system("/etc/rc.d/inetd reload")


class ConfigTags(object):
    _PURPOSE_TAGS = base.unfold_dict({
        (
            "sandbox_client_generic", "sandbox_macos_generic",
            "sandbox1_client_generic", "sandbox1_macos_generic",
            "yp_lite@sandbox2", "yp_lite@sandbox3",
        ): ["GENERIC"],
        ("sandbox_api", "sandbox1_server"): ["SERVER"],
        ("sandbox_stg_semi", "sandbox_stg", "sandbox1_stg"): ["STORAGE"],
        (
            "search_instrum-sandbox_lxc_browser",
            "sandbox_browser_linux_experimental",
            "sandbox_browser_linux_testing",
            "mac_browserbuild_only", "mac_browser_sandbox",
            "search_instrum-sandbox_windows_browser",
        ): ["BROWSER"],
        "sandbox_client_ott": ["OTT"],
        "sandbox_client_yabs": ["YABS"],
        "search_instrum-sandbox_ukrop": ["UKROP"],
        "search_instrum-sandbox_antispam_multislot": ["ANTISPAM"],
        "search_instrum-sandbox_oxygen": ["OXYGEN"],
        "sandbox_sdc": ["SDC"],
        "search_instrum-sandbox_verticals": ["VERTICALS"],
        "ape-infra-build": ["COCAINE"],
        "sandbox_market": ["MARKET"],
        ("sandbox1_client", "yp_lite@sandbox3"): [
            "ACCEPTANCE", "BROWSER", "YABS"
        ],
        "mac_mobile_sandbox": ["MOBILE_MONOREPO"],
    })

    _DENSITY_TAGS = base.unfold_dict({
        ("sandbox_multislot", "sandbox1_multislot", "yp_lite@sandbox2", "yp_lite@sandbox3"): ["MULTISLOT"],
    })

    _VIRTUAL_TAGS = base.unfold_dict({
        ("sandbox_multios", "sandbox1_multios"): "LXC",
        ("sandbox_porto", "sandbox1_porto", "yp_lite@sandbox2", "yp_lite@sandbox3"): "PORTOD",
    })

    _OS_TAGS = {
        "LINUX": {
            "10.04": "LINUX_LUCID",
            "12.04": "LINUX_PRECISE",
            "14.04": "LINUX_TRUSTY",
            "16.04": "LINUX_XENIAL",
            "18.04": "LINUX_BIONIC",
            "20.04": "LINUX_FOCAL",
        },
        "OSX": {
            "13": "OSX_MAVERICKS",
            "14": "OSX_YOSEMITE",
            "15": "OSX_EL_CAPITAN",
            "16": "OSX_SIERRA",
            "17": "OSX_HIGH_SIERRA",
            "18": "OSX_MOJAVE",
            "19": "OSX_CATALINA",
            "20": "OSX_BIG_SUR",
            "21": "OSX_MONTEREY",
        }
    }

    _CPU_TAGS = {
        "Intel": {
            "E5-2683": "INTEL_E5_2683",
            "E5-2650": "INTEL_E5_2650",
            "E5-2660": "INTEL_E5_2660",
            "E5-2667": "INTEL_E5_2667",
            "E5645": "INTEL_E5645",
            "E312xx": "INTEL_E312XX",
            "X5675": "INTEL_X5675",
            "Gold 6230": "INTEL_GOLD_6230",
            "Gold 6230R": "INTEL_GOLD_6230R",
            "Gold 6338": "INTEL_GOLD_6338",
            "i5-4278U": "INTEL_4278U",
            "i7-3720QM": "INTEL_3720QM",
            "i7-4578U": "INTEL_4578U",
            "i7-8700B": "INTEL_8700B",
        },
        "AMD": {
            "6176": "AMD6176",
        }
    }

    _CPU_NARROW_TAGS = {
        "Intel": {
            "E5-2660": {
                "0": "V1",
                "v4": "V4",
            },
            "E5-2650": {
                "v2": "V2",
            },
            "E5-2683": {
                "v4": "V4",
            },
            "E5-2667": {
                "v2": "V2",
                "v4": "V4",
            },
        }
    }

    _DC_TAGS = {
        "iva": "IVA",
        "sas": "SAS",
        "myt": "MYT",
        "vla": "VLA",
        "man": "MAN",
        "unk": "UNK"
    }

    _FEATURES_TAGS = base.unfold_dict({
        ("sandbox_stg", "sandbox1_stg"): ["NEW_LAYOUT"],
    })

    __CONFIG_PH_RE = re.compile(r"\$\{+([a-z0-9_.-]+)}+", re.IGNORECASE)

    _CONFIG_PATCHES_BY_TAGS = {}

    __client_tags = None

    __INTEL_NOISY_WORDS = ["Intel", "Xeon", "CPU", "Core", "(R)", "(TM)"]

    @property
    def __density_tags(self):
        tags = set(it.chain.from_iterable(filter(None, it.imap(
            lambda _: self._DENSITY_TAGS.get(_), self._special_groups
        ))))
        for tag in tags:
            yield tag
        if not tags:
            cores = TOTAL_CORES
            for fixed_cores in (16, 24, 32, 56, 64):
                if cores >= fixed_cores:
                    yield "CORES{}".format(fixed_cores)

    @property
    def __purpose_tags(self):
        return set(it.chain.from_iterable(filter(None, it.imap(
            lambda _: self._PURPOSE_TAGS.get(_), self._special_groups
        ))))

    @property
    def __virtual_tags(self):
        if self._host_platform == "linux":
            for tag in it.ifilter(None, it.imap(lambda _: self._VIRTUAL_TAGS.get(_), self._special_groups)):
                yield tag
            system = next(utils.check_output(["/usr/sbin/dmidecode", "-s", "system-product-name"]), "").lower()
            if "openstack" in system:
                yield "OPENSTACK"

    @property
    def __os_tags(self):
        if self._host_platform == "linux":
            if "Microsoft" in platform.platform():
                yield "WINDOWS"
            else:
                yield self._OS_TAGS["LINUX"].get(platform.linux_distribution()[1], "LINUX")
        elif self._host_platform.startswith("darwin"):
            yield self._OS_TAGS["OSX"].get(platform.release().split(".")[0], "OSX")
        elif self._host_platform.startswith("freebsd"):
            yield "FREEBSD"
        elif self._host_platform.startswith("cygwin"):
            yield "CYGWIN"

    @property
    def __cpu_tags(self):
        cpu_model = None
        if self._host_platform.startswith("freebsd"):
            cpu_model = next(utils.check_output(["/sbin/sysctl", "hw.model"]), "")
            cpu_model = cpu_model.replace("hw.model:", "").strip()
        elif self._host_platform == "linux" or self._host_platform.startswith("cygwin"):
            with open("/proc/cpuinfo") as f:
                for line in (_ for _ in f.readlines() if _.startswith("model name")):
                    cpu_model = line.partition(":")[2].strip()
                    break

        elif self._host_platform.startswith("darwin"):
            cpu_model = next(utils.check_output(["/usr/sbin/sysctl", "-n", "machdep.cpu.brand_string"]), "")

        if cpu_model == "Common KVM processor":
            yield "KVM"
        elif "AMD" in cpu_model:
            g = re.search("Processor *([^ ]*)", cpu_model)
            if g:
                cpu_model = self._CPU_TAGS["AMD"].get(g.group(1))
                if cpu_model:
                    yield cpu_model
        elif "Intel" in cpu_model:
            for word in self.__INTEL_NOISY_WORDS:
                cpu_model = cpu_model.replace(word, "")
            cpu_model = cpu_model.split("@")[0].strip()

            # Intel processors are usually named as follows: Intel E5-2660 v4. In client tags,
            # we use processor's model and version as separate tags, hence the example translates to
            # INTEL_E5_2660 and INTEL_E5_2660V4
            if cpu_model:
                # cover both presence and absence of version (like "E-2650 v2" vs "E5645")
                model, _, version = cpu_model.partition(" ")
                possible_tags = map(self._CPU_TAGS["Intel"].get, (model, cpu_model))
                cpu_tag = next(iter(filter(None, possible_tags)), None)
                version_suffix = self._CPU_NARROW_TAGS["Intel"].get(model, {}).get(version)
                if cpu_tag:
                    yield cpu_tag
                if version_suffix:
                    yield cpu_tag + version_suffix
        elif "M1" in cpu_model:
            yield "M1"

    @property
    def __dc_tags(self):
        yield self._DC_TAGS.get(self.__cfg__["this"].get("dc"), "UNK")

    @property
    def __features_tags(self):
        return it.chain.from_iterable(filter(None, it.imap(
            lambda _: self._FEATURES_TAGS.get(_), self._special_groups
        )))

    def __get_config_value(self, config, key):
        if not key:
            return
        value = config
        for k in key.split("."):
            value = value.get(k, {})
        if value == {}:
            return
        while True:
            match = self.__CONFIG_PH_RE.search(value)
            if not match:
                break
            value = "".join((
                value[:match.start()],
                str(self.__get_config_value(config, match.group(1))),
                value[match.end():]
            ))
        return value

    @property
    def __disk_tags(self):
        settings_path = os.path.join(self.actual_pack(), "sandbox", "etc", ".settings.yaml")
        with open(settings_path) as f:
            config = yaml.load(f)
        config = self.merge_dicts(config, self.__cfg__)
        tasks_data_dir = self.__get_config_value(config, "client.tasks.data_dir")

        # Running in porto container
        if os.path.exists(utils.ISS_SOCKET):
            cyson_lib_path = self.get_package("_cyson_library")
            for tag in disk_devices.Parser.parse_portod(tasks_data_dir, cyson_lib_path):
                yield tag
            return

        for tag in disk_devices.Parser.parse_linux(tasks_data_dir):
            yield tag

    @property
    def __net_tags(self):
        myname = self.__cfg__["this"]["fqdn"]
        meta = filter(lambda x: x[0] == socket.AF_INET, socket.getaddrinfo(myname, None))
        for family, socktype, proto, canonname, sockaddr in meta:
            try:
                logging.debug("Checking IPv4 at %r is working..", sockaddr[0])
                s = socket.socket(family, socktype, proto)
                s.bind(sockaddr)
                yield "IPV4"
            except socket.error:
                continue
        yield "IPV6"

    @property
    def client_tags(self):
        if self.__client_tags is None:
            self.__client_tags = set(it.chain(
                self.__density_tags,
                self.__purpose_tags,
                self.__virtual_tags,
                self.__os_tags,
                self.__cpu_tags,
                self.__features_tags,
                self.__disk_tags,
                self.__net_tags,
                self.__dc_tags
            ))
        return self.__client_tags

    def _patch_config(self):
        client_tags = self.client_tags

        def match(tag):
            if tag in self._OS_TAGS:
                # Expand OS group and check for presence of any tag from this group
                os_tags = set(self._OS_TAGS[tag].itervalues())
                return bool(os_tags & client_tags)
            return tag in client_tags

        for tags, patch in self._CONFIG_PATCHES_BY_TAGS.iteritems():
            tags = tags.split()
            if tags and all(map(match, tags)):
                self.__cfg__ = self.merge_dicts(self.__cfg__, self._load_yaml(patch))


class ClientLauncher(base.SandboxLauncher, ConfigTags):
    __name__ = "client_launcher"
    # Dummy version field. If changed only code - inc this.
    V = 25

    __STATBOX_PUSH_CLIENT_PID_FILE = "/var/run/statbox/push-client.pid"

    _CONFIG_PATCHES_BY_TAGS = base.unfold_dict({
        ("YABS SSD", "OSX"): textwrap.dedent("""
            client:
              auto_cleanup:
                free_space_threshold: 0.5
        """),
        ("VERTICALS",): textwrap.dedent("""
            client:
              auto_cleanup:
                free_space_threshold: 0.3
        """),
        ("MULTISLOT LXC",): textwrap.dedent("""
            client:
              max_job_slots: {cores}
              dynamic_slots:
                shared_ram: {ram}
                shared_cores: {cores}
        """).format(ram=SHARED_RAM_LXC, cores=SHARED_CORES),
        ("MULTISLOT PORTOD",): textwrap.dedent("""
            client:
              max_job_slots: {cores}
              dynamic_slots:
                shared_ram: {ram}
                shared_cores: {cores}
        """).format(ram=SHARED_RAM_PORTOD, cores=SHARED_CORES),
    })

    @property
    def __place_capacity(self):
        try:
            result = os.statvfs("/place")
            return result.f_blocks * result.f_frsize
        except OSError:
            return 0

    def user(self):
        return "root"

    def cgroup_available(self):
        return os.uname()[0] == "Linux" and base.check_tags(self._special_groups, undesired_tags=base.YP_LITE_TAGS)

    def client_pack(self):
        pl = platform.platform()
        if pl.startswith("Darwin") and "arm64-arm" in pl:
            return self.get_pack("client-default-darwin-arm64.tgz")
        return self.get_pack("client.tgz")

    def actual_pack(self):
        return self.client_pack()

    def preexecutor_pack(self):
        pl = platform.platform()
        if pl.startswith("Darwin") and "arm64-arm" in pl:
            return self.get_pack("preexecutor-default-darwin-arm64.tgz")
        if sys.platform.startswith("linux") and "Microsoft" in pl:
            return self.get_pack("preexecutor-windows.tgz")
        return self.get_pack("preexecutor.tgz")

    def packages(self):
        packages = super(ClientLauncher, self).packages() + [
            "preexecutor.tgz", "preexecutor-windows.tgz", "client.tgz"
        ]
        if self._host_platform.startswith("darwin"):
            packages += ["preexecutor-default-darwin-arm64.tgz", "client-default-darwin-arm64.tgz"]
        return packages

    def postinstall(self):
        super(ClientLauncher, self).postinstall()

        if self._layout:
            layout_path = self.raw_data_as_file(base64.b64decode(self._layout), "tar.bz2")
            layout_root = "/"
            layout_tar = tarfile.open(layout_path)

            # Save sandbox-user home dir
            sandbox_user_home_path = os.path.relpath(
                os.path.expanduser("~{}".format(self._sandbox_user.name)),
                layout_root
            )

            home_backup_path = os.path.expanduser("~{}/sandbox_home.tar".format(self._service_user.name))
            fd, home_backup_path_tmp = tempfile.mkstemp(prefix=home_backup_path)
            os.close(fd)
            with tarfile.open(home_backup_path_tmp, mode="w") as home_tar:
                for tarinfo in layout_tar:
                    if tarinfo.name.startswith(sandbox_user_home_path):
                        f = layout_tar.extractfile(tarinfo)
                        tarinfo.name = os.path.relpath(tarinfo.name, sandbox_user_home_path)
                        home_tar.addfile(tarinfo, f)
            self.change_permissions(home_backup_path_tmp, chmod=0o444, chown=self._sandbox_user.name)
            os.rename(home_backup_path_tmp, home_backup_path)

        if sys.platform.startswith("darwin"):
            sp.call(["/bin/launchctl", "load", "-w", "com.zerowidth.launched.rm_check_stop_on_boot.plist"])

        if (
            (sys.platform.startswith("darwin") or sys.platform.startswith("linux")) and
            "Microsoft" not in platform.platform()
        ):
            preexecutor_binary = os.path.join(self.preexecutor_pack(), "preexecutor")
            preexecutor_binary_current = os.path.expanduser("~{}/preexecutor".format(self._service_user.name))
            preexecutor_binary_new = os.path.expanduser("~{}/.preexecutor_new".format(self._service_user.name))
            if os.path.exists(preexecutor_binary_new):
                os.remove(preexecutor_binary_new)
            shutil.copy(preexecutor_binary, preexecutor_binary_new)
            shutil.move(preexecutor_binary_new, preexecutor_binary_current)

        if sys.platform.startswith("linux"):
            if "Microsoft" in platform.platform():
                preexecutor_binary = os.path.join(self.preexecutor_pack(), "preexecutor.exe")
                preexecutor_binary_copy = os.path.join("/mnt/c/Users/zomb-sandbox", "preexecutor.exe")
                if os.path.exists(preexecutor_binary_copy):
                    os.remove(preexecutor_binary_copy)
                shutil.copy(preexecutor_binary, preexecutor_binary_copy)
            else:
                if base.check_tags(self._special_groups, {"sandbox_client", "sandbox1"}):
                    self._restart_service("statbox-push-client")
                # Ensure 512 loop devices are available
                for i in range(512):
                    if not os.path.exists("/dev/loop" + str(i)):
                        sp.call(["/bin/mknod", "-m660", "/dev/loop" + str(i), "b", "7", str(i)])

    def start(self):
        if sys.platform.startswith("linux"):
            scaling_available_governors = "/sys/devices/system/cpu/cpu0/cpufreq/scaling_available_governors"
            cpu_value = "performance"
            update_scaling_governors = False
            if os.path.isfile(scaling_available_governors):
                with open(scaling_available_governors) as f:
                    update_scaling_governors = cpu_value in f.read().strip().split()

            if update_scaling_governors:
                for cpu in glob.glob("/sys/devices/system/cpu/cpu*/cpufreq/scaling_governor"):
                    with open(cpu) as f:
                        current_value = f.read().strip()

                    if current_value != cpu_value:
                        with open(cpu, "wb") as f:
                            f.write(cpu_value + "\n")

        tags = self.client_tags
        postexecute_tags = {"GENERIC", "WINDOWS"} | set(self._OS_TAGS["OSX"].values())
        if (postexecute_tags & tags) and not ({"MULTISLOT", "PORTOD"} & tags):
            tags.add("POSTEXECUTE")

        if "LXC" in tags:
            tags.discard("IPV4")  # Remove IPV4 tag since tasks will be executed in IPv6-only LXC
        self.__cfg__.setdefault("client", {})["tags"] = list(tags)
        self.__cfg__["client"]["cgroup_available"] = self.cgroup_available()
        self._patch_config()
        if base.check_tags(self._special_groups, base.YP_LITE_TAGS):
            self.__cfg__["client"]["max_job_slots"] = SHARED_CORES - YP_INFRA_TAX
            self.__cfg__["client"]["dynamic_slots"]["shared_cores"] = SHARED_CORES - YP_INFRA_TAX
        self.update_sandbox_data(self.__cfg__)
        config_path = self.config_as_file()
        settings_path = os.path.join(self.client_pack(), "sandbox", "etc", "settings.yaml")
        if os.path.exists(settings_path):
            os.unlink(settings_path)
        shutil.copyfile(config_path, settings_path)
        self.change_permissions(settings_path, chmod=0o644, chown=self._service_user.name)

        gosky_path = "/var/run/gosky.nocron"
        try:
            open(gosky_path, "a").close()
            os.chmod(gosky_path, 0o664)
        except OSError as os_ex:
            logging.warning("%s file wasn't touched. Error message: %s", gosky_path, os_ex)

        venv_bindir = os.path.join(self._sandbox_venv_path, "bin")
        env = {
            "SANDBOX_DIR": os.path.join(self.client_pack(), "sandbox"),
            "EXECUTABLE": os.path.join(venv_bindir, "python"),
            "LANG": "en_US.UTF-8",
            "SANDBOX_CONFIG": settings_path
        }

        self.create(
            [
                os.path.join(self.client_pack(), "client")
            ],
            env=env,
            resources=self.ulimit_resources()
        )

    def check_cgroup(self):
        if not self.cgroup_available():
            return True
        if os.path.exists("/sys/fs/cgroup/memory"):
            logging.debug("Check cgroup: Cgroups mounted")
        else:
            logging.debug("Check cgroup: Cgroups not mounted, but available")
            return False
        return True

    @staticmethod
    def mount_cgroup():
        logging.debug("Trying to mount cgroups")
        if os.path.exists("/etc/init/cgconfig.conf"):
            # Mount cgmanager
            try:
                sp.check_output(["/usr/sbin/service", "cgconfig", "start"], stderr=sp.STDOUT)
                logging.debug("Trying to mount cgroups: sucess")
                return True
            except sp.CalledProcessError as ex:
                logging.error("Error while trying to mount cgmanager: %s", ex.output)
        elif os.path.exists("/lib/systemd/system/cgroup-lite.service"):
            # Mount cgroup-lite on xenial with systemd
            try:
                sp.check_output(["/usr/sbin/service", "cgroup-lite.service", "start"], stderr=sp.STDOUT)
                logging.debug("Trying to mount cgroups: sucess")
                return True
            except sp.CalledProcessError as ex:
                logging.error("Error while trying to mount cgroup-lite.service: %s", ex.output)
        elif os.path.exists("/etc/init/cgroup-lite.conf"):
            # Mount cgroup-lite for trusty
            try:
                sp.check_output(["/usr/sbin/service", "cgroup-lite", "start"], stderr=sp.STDOUT)
                logging.debug("Trying to mount cgroups: sucess")
                return True
            except sp.CalledProcessError as ex:
                logging.error("Error while trying to mount cgroup-lite: %s", ex.output)
        return False

    def check_mount_cgroup(self):
        if "Microsoft" in platform.platform() or self.check_cgroup():
            return True
        if not self.mount_cgroup():
            return False
        return self.check_cgroup()

    def ping(self):
        if (
            base.check_tags(self._special_groups, {"sandbox_client", "sandbox1"}) and
            sys.platform.startswith("linux") and
            "Microsoft" not in platform.platform()
        ):
            self._check_service("statbox-push-client", self.__STATBOX_PUSH_CLIENT_PID_FILE)
        if os.path.exists(os.path.join(base.SANDBOX_RUNTIME_DIR, "client_check_stop")) or not self.check_mount_cgroup():
            self.disable()
            return False
        else:
            self.enable()
            return True


class Escapee(base.SandboxLauncher, utils.EscapeeMixin):
    __name__ = None

    def stop(self):
        for proc in self.procs():
            logging.info("Resetting tags and sending SIGINT to process %r", proc)
            proc.deleteTags(proc.stat()["tags"])
            proc.stopRetries()
            proc.signal(signal.SIGINT)
            self.move_from_cgroups()

    def get_childs(self):
        return []

    def hard_stop(self):
        pass

    def postinstall(self):
        self.prepare(True)

    def actual_pack(self):
        return None

    def prepare(self, drop_prev=False):
        home = os.path.expanduser("~zomb-sandbox")
        link = os.path.join(home, self.__name__)
        persistent = ".".join((link, self.uniq_tag()))
        src = self.actual_pack()
        if os.path.exists(persistent):
            if drop_prev:
                logging.info("Dropping previous persistent copy at %r", persistent)
                shutil.rmtree(persistent)
            else:
                return persistent

        prev = os.readlink(link) if os.path.lexists(link) else ""
        if prev != persistent:
            self._remove_stale_copies(home, link)

        logging.debug("Copying servant files %r to persistence location %r", src, persistent)
        shutil.copytree(src, persistent)

        logging.debug("Making link %r to %r", link, persistent)
        if os.path.lexists(link):
            os.unlink(link)
        os.symlink(persistent, link)
        return persistent

    def _remove_stale_copies(self, home, prev):
        if not os.path.exists(home):
            return

        lsof = "/usr/bin/lsof"
        lsof = lsof if os.path.exists(lsof) else "/usr/sbin/lsof"

        logging.debug("Checking for stale persistent copies.")
        for f in os.listdir(home):
            path = os.path.join(home, f)
            if not os.path.isdir(path) or os.path.islink(path) or not f.startswith(self.__name__) or prev.endswith(f):
                continue
            cmd = [lsof, "+D", path]
            try:
                p = sp.Popen(cmd, stdout=sp.PIPE, stderr=sp.PIPE)
                lsof_out, lsof_err = p.communicate()
                logging.debug("Command %r stdout:\n%s\nstderr:\n%s", cmd, lsof_out, lsof_err)
                if lsof_out.strip():
                    logging.debug("Location %r is still used", path)
                    continue
            except (OSError, sp.CalledProcessError) as ex:
                logging.debug("Command %r error: %s", cmd, ex)
            logging.info("Considering location %r as unused - remove it.", path)
            shutil.rmtree(path)


class FileServer(Escapee, ConfigTags):
    __name__ = "fileserver"

    def user(self):
        return "root"

    def packages(self):
        packages = self.CYSON_RESOURCE + ["fileserver.tgz"]
        if self._host_platform.startswith("darwin"):
            packages += ["fileserver-default-darwin-arm64.tgz"]
        return packages

    def fileserver_pack(self):
        pl = platform.platform()
        if pl.startswith("Darwin") and "arm64-arm" in pl:
            return self.get_pack("fileserver-default-darwin-arm64.tgz")
        return self.get_pack("fileserver.tgz")

    def actual_pack(self):
        return self.fileserver_pack()

    def start(self):
        persistent = self.prepare()

        self.__cfg__.setdefault("client", {})["tags"] = list(self.client_tags)
        self.update_sandbox_data(self.__cfg__)
        config_path = self.config_as_file()
        settings_path = os.path.join(self.fileserver_pack(), "sandbox", "etc", "settings.yaml")
        if os.path.exists(settings_path):
            os.unlink(settings_path)
        shutil.copyfile(config_path, settings_path)
        self.change_permissions(settings_path, chmod=0o644, chown=self._service_user.name)

        self.create(
            [os.path.join(persistent, "sandbox", "fileserver", "fileserver")],
            env={"SANDBOX_CONFIG": settings_path},
            cgroup="",
            resources=self.ulimit_resources()
        )


class AgentR(Escapee, ConfigTags):
    __name__ = "agentr"

    _LOGFILE = "/var/log/sandbox/agentr.log"
    _ROTATE_LOGS = [_LOGFILE]
    _MAX_FAIL_INTERVAL = 60 * 10  # in seconds
    _MAX_FAILS = 5
    _ESCAPE_TIMEOUT = 15  # in seconds

    def user(self):
        return "root"

    def agentr_pack(self):
        # TODO: SANDBOX-9024 uncomment in py3 agentr
        # pl = platform.platform()
        # if pl.startswith("Darwin") and "arm64-arm" in pl:
        #     return self.get_pack("agentr-default-darwin-arm64.tgz")
        return self.get_pack("agentr.tgz")

    def actual_pack(self):
        return self.agentr_pack()

    def packages(self):
        packages = super(AgentR, self).packages() + ["agentr.tgz"]

        if self._host_platform.startswith("darwin"):
            packages += ["agentr-default-darwin-arm64.tgz"]

        return packages

    def set_coredumper(self):
        if (
            sys.platform.startswith("linux") and
            "Microsoft" not in platform.platform() and
            base.check_tags(self._special_groups, undesired_tags=base.YP_LITE_TAGS)
        ):
            with open("/proc/sys/kernel/core_pattern", "w") as fh:
                # core_pattern limited by 127 symbols
                fh.write("|{} {} %e %p %g %u %s %P %c".format(
                    os.path.expanduser("~{}/venv/bin/python".format(self._service_user.name)),
                    os.path.expanduser("~{}/client/sandbox/bin/coredumper.py".format(self._service_user.name))
                ))

    def postinstall(self):
        self.set_coredumper()
        super(AgentR, self).postinstall()

    def ping(self):
        self._rotate_logs(move=True, notify=False)
        self.set_coredumper()
        if self.persist is None:
            self.persist = {}

        self.persist.setdefault("last_not_fail", dt.datetime.now())
        self.persist.setdefault("fail_count", 0)

        ping_value = str(random.randint(1, 1000000))
        alive = True
        try:
            srv_path = self.agentr_pack()
            sp.check_output([
                os.path.join(srv_path, "bctl"), "ping", "--ping-value", ping_value],
                stderr=sp.STDOUT
            )
        except sp.CalledProcessError as ex:
            logging.error("Error while pinging server: %s", ex.output)
            alive = False

        now = dt.datetime.now()
        if alive:
            self.persist["last_not_fail"] = now
            self.persist["fail_count"] = 0
        else:
            fail_interval = (now - self.persist["last_not_fail"]).total_seconds()
            self.persist["fail_count"] += 1
            logging.warning(
                "AgentR ping failed - fail count: %s, fail interval: %ss",
                self.persist["fail_count"], self.persist["last_not_fail"]
            )
            if fail_interval >= self._MAX_FAIL_INTERVAL and self.persist["fail_count"] >= self._MAX_FAILS:
                self.persist["last_not_fail"] = now
                self.persist["fail_count"] = 0
                logging.error("Stopping hardly: AgentR hanged up")
                super(Escapee, self).hard_stop()
                self.start()

        return alive

    def _socket_pid(self):
        pid = None
        try:
            srv_path = self.agentr_pack()
            pid = int(sp.check_output([
                os.path.join(srv_path, "bctl"), "socket_pid"],
                stderr=sp.STDOUT
            ).strip() or 0)
        except sp.CalledProcessError as ex:
            logging.error("Error while getting socket pid: %s", ex.output)
        return pid

    def stop(self):
        pgids = set()
        for proc in self.procs():
            pid = proc.stat().get("pid")
            if not pid:
                continue
            pgids.add(os.getpgid(pid))
        old_socket_pid = self._socket_pid()
        super(AgentR, self).stop()
        deadline = time.time() + self._ESCAPE_TIMEOUT
        while time.time() < deadline:
            socket_pid = self._socket_pid()
            if not socket_pid or socket_pid != old_socket_pid:
                break
            time.sleep(1)
        else:
            if old_socket_pid < 0:  # AgentR deadlocked
                for pgid in pgids:
                    try:
                        os.killpg(pgid, signal.SIGKILL)
                    except OSError:
                        pass
            elif old_socket_pid:
                try:
                    os.killpg(os.getpgid(old_socket_pid), signal.SIGKILL)
                except OSError:
                    pass

    def start(self):
        persistent = self.prepare()
        os_family = os.uname()[0]
        self.__cfg__["client"]["cgroup_available"] = os_family == "Linux"
        config_path = self.config_as_file()
        settings_path = os.path.join(self.agentr_pack(), "sandbox", "etc", "settings.yaml")
        if os.path.exists(settings_path):
            os.unlink(settings_path)
        shutil.copyfile(config_path, settings_path)
        self.change_permissions(config_path, chmod=0o644, chown=self._service_user.name)

        env = {
            "LANG": "en_US.UTF-8",
            "SANDBOX_CONFIG": config_path,
        }
        if os_family == "Darwin":
            # workaround for https://bugs.python.org/issue30385 after forking process
            env["NO_PROXY"] = "*"
            # workaround for https://bugs.python.org/issue33725 after forking process
            env["OBJC_DISABLE_INITIALIZE_FORK_SAFETY"] = "YES"

        self.create(
            [os.path.join(persistent, "agentr")],
            env=env,
            cgroup="",
            resources=self.ulimit_resources()
        )

    def prepare(self, drop_prev=False):
        uid = pwd.getpwnam(self._service_user.name).pw_uid
        if self._special_groups and {"sandbox_stg", "sandbox1_stg"} & set(self._special_groups):
            buckets = []
            with open("/etc/fstab", "r") as fh:
                prev = ""
                for l in fh:
                    l = l.strip()
                    if l.startswith("#"):
                        prev = l
                        continue
                    l = l.split()
                    if len(l) < 5:
                        continue
                    if l[1].startswith("/storage/") and "bucket" in prev:
                        buckets.append(l[1])
                        try:
                            os.chown(l[1], uid, self._service_user.gid)
                        except OSError as ex:
                            logging.warning("Error changing ownership on bucket #%r: %s", l[1], ex)
            logging.info("Detected buckets: %r", buckets)
            self.__cfg__.setdefault("agentr", {}).setdefault("data", {})["buckets"] = buckets

        self.__cfg__.setdefault("client", {})["tags"] = list(self.client_tags)
        self.update_sandbox_data(self.__cfg__)
        self.config_as_file()  # Fix configuration file modifications
        return super(AgentR, self).prepare(drop_prev)
