import base64
import json
import logging
import os
import re
import socket
import sys
from . import utils


class Parser(object):

    __DISK_DEV_NVME_RE = re.compile(r"^/dev/(nvme\w+)p\d+$")
    __DISK_DEV_RE = re.compile(r"^/dev/([a-z]+)\d*$")
    __UNABLE_DETERMINE_SSD = "Unable to determine whether SSD is on board"

    @staticmethod
    def parse_linux(tasks_data_dir):
        def _rotational(d):
            with open("/sys/block/{}/queue/rotational".format(d)) as _:
                return _.read().strip()

        def _block_name_by_partition(partition):
            if partition.startswith("sd"):
                return partition[:len("sda")]
            elif partition.startswith("nvme"):
                return partition[:len("nvme1n1")]
            else:
                logging.error("{}: cannot parse partition prefix {}", Parser.__UNABLE_DETERMINE_SSD, partition)
                return None

        df_output = list(utils.check_output(["df", tasks_data_dir]))
        if len(df_output) < 2:
            return
        dev = df_output[1].split(" ", 1)[0]
        if dev.startswith("/dev/md"):
            try:
                with open("/proc/mdstat") as f:
                    mdstat = {_[0].strip(): _[2].strip() for _ in map(lambda _: _.partition(":"), f.readlines())}

                _, _, dev = dev.partition("/dev/")
                raid_config = mdstat.get(dev)
                if raid_config:
                    rotational = None
                    for part in {_ for _ in raid_config.split() if _.startswith("sd") or _.startswith("nvme")}:
                        block_name = _block_name_by_partition(part)
                        if not block_name:
                            yield "HDD"

                        _ = _rotational(dev)
                        if rotational is None:
                            rotational = _
                        elif rotational != _:
                            logging.error(Parser.__UNABLE_DETERMINE_SSD)
                            return
                    if rotational is None:
                        logging.warning(Parser.__UNABLE_DETERMINE_SSD)
                    yield "SSD" if rotational == "0" else "HDD"
            except IOError:
                logging.exception(Parser.__UNABLE_DETERMINE_SSD)
        else:
            try:
                nvme_match = Parser.__DISK_DEV_NVME_RE.match(dev)
                dev_match = Parser.__DISK_DEV_RE.match(dev)

                filesystem_short = None
                if nvme_match:
                    filesystem_short = nvme_match.group(1)
                if dev_match:
                    filesystem_short = dev_match.group(1)

                if filesystem_short:
                    yield "SSD" if _rotational(filesystem_short) == "0" else "HDD"
                else:
                    logging.warning(Parser.__UNABLE_DETERMINE_SSD)
            except IOError:
                logging.exception(Parser.__UNABLE_DETERMINE_SSD)

    @staticmethod
    def parse_portod(tasks_data_dir, cyson_lib_path):
        def _get_mp(p):
            p = os.path.abspath(p)
            while not os.path.ismount(p):
                p = os.path.dirname(p)
            return p

        if cyson_lib_path not in sys.path:
            sys.path.insert(0, cyson_lib_path)
        import _cyson

        data_dir_mp = _get_mp(tasks_data_dir)
        pod_spec = Parser.__get_pod_spec()
        if not pod_spec:
            return

        disk_volume_requests = pod_spec.get("diskVolumeRequests", [])
        for v_r in disk_volume_requests:
            storage_class = v_r.get("storageClass", "hdd")
            if storage_class != "ssd":
                storage_class = "hdd"
            volume_type = ""
            mount_point = ""
            for attribute in v_r.get("labels", {}).get("attributes", []):
                if attribute.get("key") == "mount_path":
                    mount_point = _cyson.loads(base64.b64decode(attribute.get("value")))
                if attribute.get("key") == "volume_type":
                    volume_type = _cyson.loads(base64.b64decode(attribute.get("value")))
                if volume_type == "persistent" and mount_point == data_dir_mp:
                    yield storage_class.upper()

    @staticmethod
    def __get_pod_spec():
        sock = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
        sock.connect(utils.ISS_SOCKET)
        sock.send(utils.ISS_POD_SPEC_REQ)
        pod_spec = []
        while True:
            buff = sock.recv(16)
            pod_spec.append(buff)
            if len(buff) < 16:
                break
        pod_spec = b"".join(pod_spec).decode()
        sock.close()

        headers = []
        for line in pod_spec.splitlines():
            if not line:
                break
            headers.append(line)

        for header in headers:
            if "Content-Length" in header:
                c_l = int(header.split()[-1])
                pod_spec = pod_spec[-c_l:]
                break
        else:
            return {}

        try:
            return json.loads(pod_spec)
        except ValueError:
            return {}
