#!/skynet/python/bin/python

from __future__ import print_function

import os
import sys
import time
import json
import errno
import signal
import syslog
import socket
import httplib
import threading as th

import psutil
import requests

from kernel.util import console
from kernel.util.sys import user


RUNAS = "nobody"
NAME = os.path.basename(__file__).split(".")[0]
PIDFILE = os.path.join("/opt/sandbox/run/", NAME + ".pid")
# Its important to keep state file on journal-enabled partition (rootfs typically)
STATEFILE = os.path.join("/opt/sandbox/run/", NAME + ".state")
WALLE_TOKEN = os.path.join("/home/zomb-sandbox/.walle.token")
WALLE_API_URL = "https://api.wall-e.yandex-team.ru/v1"


def _readfile(fname):
    if not os.path.exists(fname):
        return
    with open(fname, "r") as fh:
        return fh.read().strip()


def _readintfile(fname):
    try:
        return int(_readfile(fname))
    except (TypeError, ValueError):
        return


def _chkpid(pid):
    try:
        os.kill(pid, 0)
        return True
    except OSError as ex:
        return ex.errno != errno.ESRCH


def _log(lvl_or_msg, msg=None):
    if msg is None:
        lvl, msg = syslog.LOG_INFO, lvl_or_msg
    else:
        lvl = lvl_or_msg
    print(msg, file=sys.stderr)
    syslog.syslog(lvl, msg)


def redeploy_self_or_halt():
    _log(syslog.LOG_EMERG, "Unclear shutdown event detected!")
    _log(syslog.LOG_EMERG, "Asking Wall-E to redeploy self.")
    r = None
    token = _readfile(WALLE_TOKEN) or "UNKNOWN"
    for i in xrange(10):
        r = requests.post(
            "/".join([WALLE_API_URL, "hosts", socket.gethostname(), "redeploy"]),
            headers={"Authorization": "OAuth " + token, "Content-Type": "application/json"},
            data=json.dumps({"reason": "Unclear shutdown event detected"}),
        )
        if r.status_code == httplib.OK:
            break
        _log("Wall-E reported HTTP code {}: {}. Retry in 30 seconds.".format(r.status_code, r.text.strip()))
        time.sleep(30)

    if r.status_code == httplib.OK:
        _log("Wall-E reported OK. Waiting 10 minutes.")
        time.sleep(600)

    _log(
        syslog.LOG_EMERG,
        "Wall-E reported HTTP code {}: {}. Halting the system.".format(r.status_code, r.text.strip())
    )
    while True:
        with open("/proc/sysrq-trigger", "w") as fh:
            fh.write("b")
        time.sleep(1)


def daemonize():
    # decouple from parent environment
    os.chdir("/")
    os.setsid()
    os.umask(0)

    # redirect standard file descriptors
    sys.stdout.flush()
    sys.stderr.flush()
    si = open(os.devnull, "r")
    so = open(os.devnull, "w")
    se = open(os.devnull, "w")
    os.dup2(si.fileno(), sys.stdin.fileno())
    os.dup2(so.fileno(), sys.stdout.fileno())
    os.dup2(se.fileno(), sys.stderr.fileno())


def daemon_loop():
    _log("Daemon started normally.")

    console.setProcTitle(NAME)

    stopev = th.Event()
    signalhndlr = lambda *_: stopev.set()
    signal.signal(signal.SIGINT, signalhndlr)
    signal.signal(signal.SIGTERM, signalhndlr)

    while not stopev.is_set():
        stopev.wait(86400)

    pid = _readintfile(PIDFILE)
    if pid and (pid == os.getpid() or not _chkpid(pid)):
        with user.UserPrivileges():
            os.unlink(PIDFILE)
            os.unlink(STATEFILE)

    _log("Daemon stopped normally.")


def main():
    syslog.openlog(NAME)

    pid = _readintfile(PIDFILE)
    if pid:
        if _chkpid(pid):
            _log("Another instance of the script with PID {} already running.".format(pid))
            return 1
        _log("Stale pidfile with PID {} detected. Continue.".format(pid))

    bttime = _readintfile(STATEFILE)
    if bttime is not None and psutil.get_boot_time() - 1 > bttime:
        redeploy_self_or_halt()

    pid = os.fork()
    if not pid:
        daemonize()
        with user.UserPrivileges(RUNAS):
            daemon_loop()
    else:
        with open(PIDFILE, "w") as pidfh, open(STATEFILE, "w") as statefh:
            pidfh.write(str(pid))
            statefh.write(str(int(psutil.get_boot_time())))

    return 0


if __name__ == "__main__":
    if os.getuid():
        print("This script should run under root", file=sys.stderr)
        sys.exit(-1)
    sys.exit(main())
