import os
import io
import yaml
import copy
import base64
import tarfile
import contextlib
import collections


LAYOUTS_FOLDER = "layout"
FULL_LAYOUTS_FOLDER = os.path.join(os.path.dirname(os.path.abspath(__file__)), LAYOUTS_FOLDER)
DEFAUL_FILE_PERMISSION = 0o664
DEFAULT_FOLDER_PERMISSION = 0o755
FULL_PERMISSIONS = 0o777
LAYOUT_METAFILE_NAME = "metainfo.yaml"
EXCLUDED_FILES = {"ya.make", "a.yaml"}
INHERITANCE_KEYS = ("uids", "gids", "owners")


class Layout(object):
    def __init__(self, group, name, permissions, linkname="", value=None, binary=None):
        self.group = group
        self.name = name
        self.permissions = permissions
        self.linkname = linkname
        self.value = value
        self.binary = binary


def ro_property(meth):
    return property(meth).setter(lambda *_: None)


class TarInfoBase(tarfile.TarInfo):
    __layout_metainfo__ = None

    @property
    def __uids(self):
        return self.__layout_metainfo__["uids"]

    @property
    def __gids(self):
        return self.__layout_metainfo__["gids"]

    @property
    def __owners(self):
        return self.__layout_metainfo__["owners"]

    @property
    def __owner(self):
        return self.__owners[sorted(
            path
            for path in (
                os.path.commonprefix((os.path.sep + self.name, path))
                for path in self.__owners
            )
            if path in self.__owners
        )[-1]].split(":")

    @ro_property
    def uid(self):
        try:
            uid = int(self.uname)
        except ValueError:
            uid = self.uname
        return self.__uids.get(self.uname, uid)

    @ro_property
    def uname(self):
        return self.__owner[0]

    @ro_property
    def gid(self):
        try:
            gid = int(self.gname)
        except ValueError:
            gid = self.gname
        return self.__gids.get(self.gname, gid)

    @ro_property
    def gname(self):
        return self.__owner[1]


def read_layouts(layouts_tar):
    tarinfo = collections.defaultdict(list)
    metainfo = {}
    for member in layouts_tar:
        name, group = (list(os.path.split("".join(reversed(member.name)))) + [""])[:2]
        group = "".join(reversed(group))
        name = "".join(reversed(name))
        if not name:
            continue
        member.name = name
        if name == LAYOUT_METAFILE_NAME:
            metainfo[group] = member
        else:
            tarinfo[group].append(member)
    return tarinfo, metainfo


def read_layouts_folder():
    layouts = collections.defaultdict(list)
    metainfo = {}
    root_index = len(FULL_LAYOUTS_FOLDER.split(os.sep))

    for root, internal_folders, files in os.walk(FULL_LAYOUTS_FOLDER):
        path = root.split(os.sep)
        if root_index == len(path):
            continue
        group, member = path[root_index], os.path.join(*path[root_index + 1:]) if len(path) > root_index + 1 else ""
        layouts[group].append(member)

        for filename in files:
            if filename in EXCLUDED_FILES:
                continue
            if filename == LAYOUT_METAFILE_NAME:
                with open(os.path.join(root, filename)) as f:
                    metainfo[group] = yaml.safe_load(f)
            else:
                layouts[group].append(os.path.join(member, filename))
    return layouts, metainfo


def read_security_files(layouts_tar):
    tar_info = read_layouts(layouts_tar)[0]
    layouts = collections.defaultdict(list)
    layouts_map = collections.defaultdict(dict)

    for group, members in tar_info.iteritems():
        for member in members:
            layouts[group].append(member.name)
            layouts_map[group][member.name] = member

    return layouts, layouts_map


def _expand_layouts(group, layout_names, metainfo, layouts):
    if group in layouts:
        return
    parent = metainfo[group].get("parent")
    layouts_dict = {}
    secrets = set(metainfo[group].get("secured", {}).keys())
    group_names = metainfo[group].get("name", {})
    inheritance_keys_dict = {key: metainfo[group].get(key, {}) for key in INHERITANCE_KEYS}

    if parent:
        if parent not in layouts:
            _expand_layouts(parent, layout_names, metainfo, layouts)
        for member in layouts[parent]:
            layouts_dict[member.name] = copy.deepcopy(member)
        for name, real_name in metainfo[parent].get("name", {}).iteritems():
            if name not in group_names:
                group_names[name] = real_name

        for key in INHERITANCE_KEYS:
            for name, value in metainfo[parent].get(key, {}).iteritems():
                if name not in inheritance_keys_dict[key]:
                    inheritance_keys_dict[key][name] = value

    metainfo[group]["name"] = group_names

    for production_type in ("production", "preproduction"):
        for member_name in metainfo[group].get(production_type, []):
            layouts_dict[member_name] = copy.deepcopy(layouts[production_type][member_name])

    for member_name in layout_names[group]:
        if member_name in layouts_dict:
            layouts_dict[member_name].group = group
        else:
            is_folder = (
                member_name not in secrets and os.path.isdir(os.path.join(FULL_LAYOUTS_FOLDER, group, member_name))
            )

            layout = Layout(group, member_name, DEFAUL_FILE_PERMISSION)
            if is_folder:
                layout.permissions = DEFAULT_FOLDER_PERMISSION
            if member_name in metainfo[group].get("symlinks", {}):
                layout.linkname = metainfo[group]["symlinks"][member_name]
            layouts_dict[member_name] = layout

    layouts[group] = layouts_dict.values()
    for member in layouts[group]:
        if member.name in metainfo[group].get("access", {}):
            member.permissions = metainfo[group]["access"][member.name]
        if member.name in secrets:
            member.permissions = metainfo[group]["secured"][member.name]["access"]
            member.value = metainfo[group]["secured"][member.name]["value"]
            member.binary = metainfo[group]["secured"][member.name].get("binary")
            secrets.remove(member.name)
        if member.linkname:
            member.permissions = FULL_PERMISSIONS
    for key in INHERITANCE_KEYS:
        metainfo[group][key] = inheritance_keys_dict[key]

    if secrets:
        raise IOError(
            "Secured file {} declared in secured section for group {} but not found.".format(
                list(secrets)[0],
                group
            )
        )
    layouts[group].sort(key=lambda x: x.name)


def _build_layouts(members, metainfo):
    class TarInfo(TarInfoBase):
        __layout_metainfo__ = metainfo

    buf = io.BytesIO()
    with contextlib.closing(tarfile.open(mode="w:bz2", tarinfo=TarInfo, fileobj=buf)) as tar_file:
        for member in members:
            name = os.path.join(member.group, member.name)
            path = os.path.join(FULL_LAYOUTS_FOLDER, name)
            fileobj = None
            if member.value is not None:
                member_info = tarfile.TarInfo()
                if member.binary:
                    member.value = base64.b64decode(member.value)
                    fileobj = io.BytesIO(member.value)
                    member_info.size = len(member.value)
                else:
                    data = member.value.encode("utf8") if isinstance(member.value, unicode) else member.value
                    fileobj = io.BytesIO(data)
                    member_info.size = len(data)
            else:
                member_info = tar_file.gettarinfo(os.path.join(FULL_LAYOUTS_FOLDER, name))

            member_info.name = metainfo["name"][member.name] if member.name in metainfo.get("name", {}) else member.name
            member_info.mode = member.permissions
            member_info.__class__ = TarInfo
            member_info.linkname = member.linkname
            member_info.mtime = 0

            if member_info.linkname:
                member_info.type = tarfile.SYMTYPE

            if member.value is not None or not os.path.isfile(path) or member_info.linkname:
                tar_file.addfile(member_info, fileobj)
            else:
                with open(path) as fileobj:
                    tar_file.addfile(member_info, fileobj)
    return base64.b64encode(buf.getvalue())


def build_layouts(secrets):
    layout_names, metainfo = read_layouts_folder()
    security_files = {}

    for group, info in metainfo.iteritems():
        security_files[group] = list(info.get("secured", {}).keys())
        for name, description in info.get("secured", {}).iteritems():
            key = description.get("key")
            if key is None:
                raise KeyError("There is no yav key in secret section. Group {}, name {}.".format(group, name))
            if key not in secrets:
                raise KeyError("There is no yav secret with key {}. Group {}, name {}.".format(key, group, name))
            description["value"] = secrets[key]
        layout_names[group].extend(security_files[group])

    for group, filenames in security_files.iteritems():
        for filename in filenames:
            if group in metainfo and filename in metainfo[group].get("secured", {}):
                layout_names[group].append(filename)

    layouts = collections.defaultdict(list)
    for production_type in ("production", "preproduction"):
        _expand_layouts(production_type, layout_names, metainfo, layouts)
        layouts[production_type] = {member.name: member for member in layouts[production_type]}

    for group in layout_names:
        _expand_layouts(group, layout_names, metainfo, layouts)

    result = {}
    for group, members in layouts.iteritems():
        if metainfo[group]["real"]:
            result[group] = _build_layouts(members, metainfo[group])
    return result
